(****************************************************************************)
(*                                                                          *)
(* Copyright 1997-2000 University of Cambridge and University of Edinburgh  *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : conditionals.sml                                         *)
(* DESCRIPTION   : Examples for HOL/Clam interface that involve implication.*)
(*                 (Based on examples in the Clam 2.7.0 library.)           *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 21st November 1997                                       *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 17th July 2000                                           *)
(****************************************************************************)

start_clam {clam = "in_emacs clam-server"};
(* or 
   start_clam {clam = "xterm -e clam-server"};
*)

(*---------------------------------------------------------------------------
    A lot of the examples below are trivial if Clam has access 
    to HOL's decision procedure. So turn iteration off, by default.
 ---------------------------------------------------------------------------*)
iteration false;
Definitions.auto_send := false;
Schemes.auto_send := false;

val _ = new_theory "conditionals" handle _ => ();

send_scheme("num_INDUCTION",theorem "num" "INDUCTION");

val NOT_SUC = theorem "num" "NOT_SUC";
val NOT_0_SUC = (GEN_ALL o EQF_INTRO o SPEC_ALL o GSYM) NOT_SUC
and NOT_SUC_0 = (GEN_ALL o EQF_INTRO o SPEC_ALL) NOT_SUC;
send_rule ("NOT_0_SUC",NOT_0_SUC);
send_rule ("NOT_SUC_0",NOT_SUC_0);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: cnc_s                                              *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>x=y in pnat=>s(x)=s(y)in pnat                    *)
(*--------------------------------------------------------------------------*)

val CANCEL_SUC =
   prove
      (--`!x y. (x = y) ==> (SUC x = SUC y)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: cnc_pred                                           *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>x=y in pnat=>pred(x)=pred(y)in pnat              *)
(*--------------------------------------------------------------------------*)

val CANCEL_PRE =
   prove
      (--`!x y. (x = y) ==> (PRE x = PRE y)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: cnc_pl2                                            *)
(* Dependencies beyond needs.pl: def(plus)                                  *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>u1:pnat=>u2:pnat=>v:pnat=>                                       *)
(*         u1=u2 in pnat=>plus(u1,v)=plus(u2,v)in pnat                      *)
(*--------------------------------------------------------------------------*)

val CANCEL_PLUS_LEFT =
   prove
      (--`!u1 u2 v. (u1 = u2) ==> (u1 + v = u2 + v)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: cnc_pr                                             *)
(* Dependencies beyond needs.pl: def(plus)                                  *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>u1:pnat=>u2:pnat=>v:pnat=>                                       *)
(*         u1=u2 in pnat=>plus(v,u1)=plus(v,u2)in pnat                      *)
(*--------------------------------------------------------------------------*)

val CANCEL_PLUS_RIGHT =
   prove
      (--`!u1 u2 v. (u1 = u2) ==> (v + u1 = v + u2)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: cnc_cml                                            *)
(* Dependencies beyond needs.pl: def(times)                                 *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>u1:pnat=>v:pnat=>w:pnat=>                                        *)
(*         v=w in pnat=>times(v,u1)=times(w,u1)in pnat                      *)
(*--------------------------------------------------------------------------*)

send_definition("ADD", definition "arithmetic" "ADD");
send_definition("MULT",definition "arithmetic" "MULT");

val CANCEL_TIMES_LEFT =
   prove
      (--`!u v w. (v = w) ==> (v * u = w * u)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: cnc_cmr                                            *)
(* Dependencies beyond needs.pl: def(times)                                 *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>u1:pnat=>v:pnat=>w:pnat=>                                        *)
(*         v=w in pnat=>times(u1,v)=times(u1,w)in pnat                      *)
(*--------------------------------------------------------------------------*)

val CANCEL_TIMES_RIGHT =
   prove
      (--`!u v w. (v = w) ==> (u * v = u * w)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: zeroplus                                           *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>plus(x,y)=0 in pnat=>x=0 in pnat#y=0 in pnat     *)
(*--------------------------------------------------------------------------*)

iteration true;

val ZERO_PLUS =
   prove
      (--`!x y. (x + y = 0) ==> (x = 0) /\ (y = 0)`--,
       CLAM_TAC);

iteration false;

(*--------------------------------------------------------------------------*)
(* Name in Clam library: zerotimes                                          *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>times(x,y)=0 in pnat=>x=0 in pnat\y=0 in pnat    *)
(*--------------------------------------------------------------------------*)

send_rule("ZERO_PLUS",ZERO_PLUS);

(* Gets stuck in a loop if extended interaction is used! *)
val ZERO_TIMES =
   prove
      (--`!x y. (x * y = 0) ==> (x = 0) \/ (y = 0)`--,
       CLAM_TAC);

(* More challenges! *)
(*
   set_goal([],--`!x y z. (x + y = x + z) ==> (y=z)`--);
   set_goal([],--`!x y z. (y + x = z + x) ==> (y=z)`--);
   set_goal([],--`!x y z. (x + y = x + z) = (y=z)`--);
   set_goal([],--`!x y z. (y + x = z + x) = (y=z)`--);
*)

(*--------------------------------------------------------------------------*)
(* Name in Clam library: zerotimes1                                         *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>x=0 in pnat=>times(x,y)=0 in pnat                *)
(*--------------------------------------------------------------------------*)

val ZERO_TIMES1 =
   prove
      (--`!x y. (x = 0) ==> (x * y = 0)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: zerotimes2                                         *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>y=0 in pnat=>times(x,y)=0 in pnat                *)
(*--------------------------------------------------------------------------*)

val ZERO_TIMES2 =
   prove
      (--`!x y. (y = 0) ==> (x * y = 0)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: leqtrans                                           *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>(leq(x,y)#leq(y,z))=>leq(x,z)            *)
(*--------------------------------------------------------------------------*)

val LEQ1 = decisionLib.DECIDE (--`!y. (0 <= y) = T`--);
val LEQ2 = decisionLib.DECIDE (--`!x. (SUC x <= 0) = F`--);
val LEQ3 = decisionLib.DECIDE (--`!x y. (SUC x <= SUC y) = (x <= y)`--);
send_rule("LEQ1",LEQ1);
send_rule("LEQ2",LEQ2);
send_rule("LEQ3",LEQ3);

val LEQ_ZERO = decisionLib.DECIDE (--`!y. (y <= 0) = (y = 0)`--);
send_rule("LEQ_ZERO",LEQ_ZERO);

val NUM_NUM =
   prove
      (--`!P. (!y. P 0 y) ==> (!x. P x 0) ==>
              (!x y. P x y ==> P (SUC x) (SUC y)) ==> !x y. P x y`--,
       GEN_TAC THEN REPEAT DISCH_TAC THEN INDUCT_TAC THENL
       [ASM_REWRITE_TAC [],
        INDUCT_TAC THENL
        [ASM_REWRITE_TAC [],
         FIRST_ASSUM MATCH_MP_TAC THEN ASM_REWRITE_TAC []]]);
send_scheme ("NUM_NUM",NUM_NUM);

val LEQ_TRANS =
   prove
      (--`!x y z. x <= y /\ y <= z ==> x <= z`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: lesseq                                             *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>((x=y in pnat=>void)#leq(x,y))=>less(x,y)        *)
(*--------------------------------------------------------------------------*)

val LESS1 = decisionLib.DECIDE (--`!x. (x < 0) = F`--);
val LESS2 = decisionLib.DECIDE (--`!y. (0 < SUC y) = T`--);
val LESS3 = decisionLib.DECIDE (--`!x y. (SUC x < SUC y) = (x < y)`--);
send_rule("LESS1",LESS1);
send_rule("LESS2",LESS2);
send_rule("LESS3",LESS3);

val LESS_ZERO = decisionLib.DECIDE (--`!x. (0 < x) = ~(x = 0)`--);
send_rule("LESS_ZERO",LESS_ZERO);

send_rule("CANCEL_SUC",CANCEL_SUC);

val LESS_EQ =
   prove
      (--`!x y. (~(x = y) /\ (x <= y)) ==> (x < y)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: lessleq1                                           *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>(leq(x,y)=>void)=>less(y,x)                      *)
(*--------------------------------------------------------------------------*)

val LESS_LEQ1 =
   prove
      (--`!x y. ~(x <= y) ==> (y < x)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: lesstrans1                                         *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>                                         *)
(*         ((less(z,x)=>void)#less(x,y)=>void)=>less(z,y)=>void             *)
(*--------------------------------------------------------------------------*)

val LESS_TRANS1 =
   prove
      (--`!x y z. (~(z < x) /\ ~(x < y)) ==> ~(z < y)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: lesstrans2                                         *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>(leq(z,x)#less(x,y))=>leq(z,y)           *)
(*--------------------------------------------------------------------------*)

val LESS_TRANS2 =
   prove
      (--`!x y z. ((z <= x) /\ (x < y)) ==> (z <= y)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: lesstrans3                                         *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>(less(y,z)#less(x,y))=>less(x,z)         *)
(*--------------------------------------------------------------------------*)

val SUC_LEMMA =
   prove
      (--`!y. ~(y = 0) ==> ?x. y = SUC x`--,
       INDUCT_TAC THENL
       [REWRITE_TAC [],
        REWRITE_TAC [NOT_SUC_0] THEN EXISTS_TAC (--`y:num`--) THEN REFL_TAC]);
send_rule("SUC_LEMMA",SUC_LEMMA);

val LESS_TRANS3 =
   prove
      (--`!x y z. y < z /\ x < y ==> x < z`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: notlesstrans                                       *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>leq(z,y)=>less(x,z)=>less(x,y)           *)
(*--------------------------------------------------------------------------*)

delete_rule "SUC_LEMMA";

val NOT_LESS_TRANS =
   prove
      (--`!x y z. z <= y ==> x < z ==> x < y`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: notlesstrans2                                      *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>(less(x,z)#leq(y,x))=>less(y,z)          *)
(*--------------------------------------------------------------------------*)

val NOT_LESS_TRANS2 =
   prove
      (--`!x y z. x < z /\ y <= x ==> y < z`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: notlesstrans3                                      *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>(leq(z,y)#leq(y,x))=>leq(z,x)            *)
(*--------------------------------------------------------------------------*)

(* Clam times out
val NOT_LESS_TRANS3 =
   prove
      (--`!x y z. z <= y /\ y <= x ==> z <= x`--,
       CLAM_TAC);
*)

(*--------------------------------------------------------------------------*)
(* Name in Clam library: geqtrans                                           *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>(geq(x,y)#geq(y,z))=>geq(x,z)            *)
(*--------------------------------------------------------------------------*)

val GEQ1 = decisionLib.DECIDE (--`!x. (x >= 0) = T`--);
val GEQ2 = decisionLib.DECIDE (--`!x. (0 >= SUC x) = F`--);
val GEQ3 = decisionLib.DECIDE (--`!x y. (SUC x >= SUC y) = (x >= y)`--);
send_rule("GEQ1",GEQ1);
send_rule("GEQ2",GEQ2);
send_rule("GEQ3",GEQ3);

val GEQ_ZERO = decisionLib.DECIDE (--`!x. (0 >= x) = (x = 0)`--);
send_rule("GEQ_ZERO",GEQ_ZERO);

(* Clam times out
val GEQ_TRANS =
   prove
      (--`!x y z. x >= y /\ y >= z ==> x >= z`--,
       CLAM_TAC);
*)

(*--------------------------------------------------------------------------*)
(* Name in Clam library: greaterplus                                        *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>greater(x,y)=>greater(plus(z,x),y)       *)
(*--------------------------------------------------------------------------*)

val GREATER1 = decisionLib.DECIDE (--`!x. (0 > x) = F`--);
val GREATER2 = decisionLib.DECIDE (--`!x. (SUC x > 0) = T`--);
val GREATER3 = decisionLib.DECIDE (--`!x y. (SUC x > SUC y) = (x > y)`--);
send_rule("GREATER1",GREATER1);
send_rule("GREATER2",GREATER2);
send_rule("GREATER3",GREATER3);

val GREATER_PLUS =
   prove
      (--`!x y z. x > y ==> z + x > y`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: greaterplus2                                       *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>greater(x,y)=>greater(plus(x,z),y)       *)
(*--------------------------------------------------------------------------*)

val GREATER_PLUS2 =
   prove
      (--`!x y z. x > y ==> x + z > y`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: greaters2                                          *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>greater(x,y)=>greater(s(x),y)                    *)
(*--------------------------------------------------------------------------*)

val GREATERS2 =
   prove
      (--`!x y. x > y ==> SUC x > y`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: greatertrans                                       *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>                                         *)
(*         (greater(x,y)#greater(y,z))=>greater(x,z)                        *)
(*--------------------------------------------------------------------------*)

val GREATER_TRANS =
   prove
      (--`!x y z. x > y /\ y > z ==> x > z`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: geqnotlessp                                        *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>geq(x,y)=>less(x,y)=>void                        *)
(*--------------------------------------------------------------------------*)

val GEQ_NOT_LESSP =
   prove
      (--`!x y. (x >= y) ==> ~(x < y)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: plusless                                           *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>z:pnat=>less(x,y)=>less(x,plus(y,z))             *)
(*--------------------------------------------------------------------------*)

val PLUS_LESS =
   prove
      (--`!x y z. (x < y) ==> (x < y + z)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: minmaxgeq                                          *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>a:pnat=>b:pnat=>c:pnat=>d:pnat=>                                 *)
(*         (geq(a,c)#geq(b,d))=>geq(max(a,b),min(c,d))                      *)
(*--------------------------------------------------------------------------*)

val MIN = new_definition ("MIN",--`!x y. MIN x y = ((x < y) => x | y)`--);
val MIN1 = prove (--`!y. MIN 0 y = 0`--,
                  REWRITE_TAC [MIN] THEN decisionLib.DECIDE_TAC);
val MIN2 = prove (--`!x. MIN x 0 = 0`--,
                  REWRITE_TAC [MIN] THEN decisionLib.DECIDE_TAC);
val MIN3 = prove (--`!x y. MIN (SUC x) (SUC y) = SUC (MIN x y)`--,
                  REWRITE_TAC [MIN] THEN decisionLib.DECIDE_TAC);
send_rule("MIN1",MIN1);
send_rule("MIN2",MIN2);
send_rule("MIN3",MIN3);

val MAX = new_definition ("MAX",--`!x y. MAX x y = ((x > y) => x | y)`--);
val MAX1 = prove (--`!y. MAX 0 y = y`--,
                  REWRITE_TAC [MAX] THEN decisionLib.DECIDE_TAC);
val MAX2 = prove (--`!x. MAX x 0 = x`--,
                  REWRITE_TAC [MAX] THEN decisionLib.DECIDE_TAC);
val MAX3 = prove (--`!x y. MAX (SUC x) (SUC y) = SUC (MAX x y)`--,
                  REWRITE_TAC [MAX] THEN decisionLib.DECIDE_TAC);
send_rule("MAX1",MAX1);
send_rule("MAX2",MAX2);
send_rule("MAX3",MAX3);

(* Clam times out
val MIN_MAX_GEQ =
   prove
      (--`!a b c d. ((a >= c) /\ (b >= d)) ==> (MAX a b >= MIN c d)`--,
       CLAM_TAC);
*)

(*--------------------------------------------------------------------------*)
(* Name in Clam library: evenp                                              *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>(even(x)#even(y))=>even(plus(x,y))               *)
(*--------------------------------------------------------------------------*)

val EVEN_BASE :: EVEN_STEP :: _ = CONJUNCTS (definition "arithmetic" "EVEN");
val EVEN1 = EVEN_BASE
and EVEN2 = EQF_INTRO (REWRITE_RULE [EVEN_BASE] (SPEC (--`0`--) EVEN_STEP))
and EVEN3 = GEN_ALL (CONV_RULE (RAND_CONV (REWRITE_CONV [EVEN_STEP]))
                               (SPEC (--`SUC x`--) EVEN_STEP));
send_rule("EVEN1",EVEN1);
send_rule("EVEN2",EVEN2);
send_rule("EVEN3",EVEN3);

val TWOS =
   prove
      (--`!P. P 0 ==> P (SUC 0) ==> (!x. P x ==> P (SUC (SUC x))) ==>
              !z. P z`--,
       REPEAT STRIP_TAC THEN
       DISJ_CASES_TAC
          (SPEC (--`z:num`--) (theorem "arithmetic" "EVEN_OR_ODD")) THENL
       [POP_ASSUM (MP_TAC o REWRITE_RULE [theorem "arithmetic" "EVEN_EXISTS"]),
        POP_ASSUM (MP_TAC o REWRITE_RULE [theorem "arithmetic" "ODD_EXISTS"])]
       THEN
       CONV_TAC LEFT_IMP_EXISTS_CONV THEN
       GEN_TAC THEN DISCH_THEN (fn th => REWRITE_TAC [th]) THEN
       REWRITE_TAC [theorem "arithmetic" "TIMES2"] THEN
       SPEC_TAC (--`m:num`--,--`m:num`--) THEN INDUCT_TAC THEN
       REWRITE_TAC [theorem "arithmetic" "ADD_CLAUSES"] THENL
       [ASM_REWRITE_TAC [],RES_TAC,ASM_REWRITE_TAC [],RES_TAC]);
send_scheme("TWOS",TWOS);

val EVEN_P =
   prove
      (--`!x y. EVEN x /\ EVEN y ==> EVEN (x + y)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: evenm                                              *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>n:pnat=>m:pnat=>even(n)=>even(times(n,m))                        *)
(*--------------------------------------------------------------------------*)

val TIMES_2_RIGHT = prove (--`!x y. x * SUC y = x + (x * y)`--,CLAM_TAC);
send_rule("TIMES_2_RIGHT",TIMES_2_RIGHT);
send_rule("EVEN_P",EVEN_P);

val EVEN_M =
   prove
      (--`!n m. EVEN n ==> EVEN (n * m)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: evenodd1                                           *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>even(x)=>odd(x)=>void                                    *)
(*--------------------------------------------------------------------------*)

val ODD_BASE :: ODD_STEP :: _ = CONJUNCTS (definition "arithmetic" "ODD");
val ODD1 = ODD_BASE
and ODD2 = EQT_INTRO (REWRITE_RULE [ODD_BASE] (SPEC (--`0`--) ODD_STEP))
and ODD3 = GEN_ALL (CONV_RULE (RAND_CONV (REWRITE_CONV [ODD_STEP]))
                               (SPEC (--`SUC x`--) ODD_STEP));
send_rule("ODD1",ODD1);
send_rule("ODD2",ODD2);
send_rule("ODD3",ODD3);

val EVEN_ODD1 =
   prove
      (--`!x. EVEN x ==> ~(ODD x)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: evenodd2                                           *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>odd(x)=>even(x)=>void                                    *)
(*--------------------------------------------------------------------------*)

val EVEN_ODD2 =
   prove
      (--`!x. ODD x ==> ~(EVEN x)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: cnc_half                                           *)
(* Dependencies beyond needs.pl: def(half)                                  *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:pnat=>y:pnat=>x=y in pnat=>half(x)=half(y)in pnat              *)
(*--------------------------------------------------------------------------*)

val HALF = new_definition ("HALF",--`!n. HALF n = n DIV 2`--);
val HALF1 =
   prove
      (--`HALF 0 = 0`--,
       REWRITE_TAC [HALF] THEN
       MATCH_MP_TAC (theorem "arithmetic" "ZERO_DIV") THEN
       decisionLib.DECIDE_TAC);
val HALF2 =
   prove
      (--`HALF (SUC 0) = 0`--,
       REWRITE_TAC [HALF] THEN
       MATCH_MP_TAC (theorem "arithmetic" "DIV_UNIQUE") THEN
       EXISTS_TAC (--`1`--) THEN decisionLib.DECIDE_TAC);
val HALF3 =
   prove
      (--`!n. HALF (SUC (SUC n)) = SUC (HALF n)`--,
       REWRITE_TAC [HALF] THEN GEN_TAC THEN
       MATCH_MP_TAC (theorem "arithmetic" "DIV_UNIQUE") THEN
       EXISTS_TAC (--`n MOD 2`--) THEN
       REWRITE_TAC [decisionLib.DECIDE
                       (--`(SUC (SUC x) = SUC y * 2 + z) = (x = y * 2 + z)`--)]
       THEN
       MATCH_MP_TAC (definition "arithmetic" "DIVISION") THEN
       decisionLib.DECIDE_TAC);
send_rule("HALF1",HALF1);
send_rule("HALF2",HALF2);
send_rule("HALF3",HALF3);

val CANCEL_HALF =
   prove
      (--`!x y. (x = y) ==> (HALF x = HALF y)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: doublehalf                                         *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>n:pnat=>even(n)=>double(half(n))=n in pnat                       *)
(*--------------------------------------------------------------------------*)

val DOUBLE =
   new_recursive_definition
      {name = "DOUBLE",fixity = Prefix,
       rec_axiom = theorem "prim_rec" "num_Axiom",
       def = --`(DOUBLE 0 = 0) /\
                (!n. DOUBLE (SUC n) = SUC (SUC (DOUBLE n)))`--};
send_definition("DOUBLE",DOUBLE);

val DOUBLE_HALF =
   prove
      (--`!n. EVEN n ==> (DOUBLE (HALF n) = n)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: prod_times                                         *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>u:pnat=>v:pnat list=>                                            *)
(*         w:pnat=>prod(v)=w in pnat=>prod(u::v)=times(u,w)in pnat          *)
(*--------------------------------------------------------------------------*)

val PROD =
   new_recursive_definition
      {name = "PROD",fixity = Prefix,
       rec_axiom = theorem "list" "list_Axiom",
       def = --`(PROD [] = SUC 0) /\
                (!h t. PROD (CONS h t) = h * (PROD t))`--};
send_definition("PROD",PROD);

val PROD_TIMES =
   prove
      (--`!u v w. (PROD v = w) ==> (PROD (CONS u v) = u * w)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: cnc_cons1                                          *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>h:int=>t1:int list=>t2:int list=>                                *)
(*         t1=t2 in int list=>h::t1=h::t2 in int list                       *)
(*--------------------------------------------------------------------------*)

val CANCEL_CONS1 =
   prove
      (--`!(h:'a) t1 t2. (t1 = t2) ==> (CONS h t1 = CONS h t2)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: cnc_cons2                                          *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>h1:int=>h2:int=>t:int list=>                                     *)
(*         h1=h2 in int list=>h1::t=h2::t in int list                       *)
(*--------------------------------------------------------------------------*)

val CANCEL_CONS2 =
   prove
      (--`!(h1:'a) h2 t. (h1 = h2) ==> (CONS h1 t = CONS h2 t)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: cnc_app                                            *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:int list=>y:int list=>z:int list=>                             *)
(*         y=z in int list=>app(x,y)=app(x,z)in int list                    *)
(*--------------------------------------------------------------------------*)

val CANCEL_APPEND =
   prove
      (--`!(x:'a list) y z. (y = z) ==> (APPEND x y = APPEND x z)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: memapp1                                            *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:int=>y:int list=>z:int list=>member(x,y)=>member(x,app(y,z))   *)
(*--------------------------------------------------------------------------*)

val MEMBER =
   new_recursive_definition
      {name = "MEMBER",fixity = Prefix,
       rec_axiom = theorem "list" "list_Axiom",
       def = --`(!x. MEMBER x [] = F) /\
                (!x h t. MEMBER x (CONS h t) = ((x = h) \/ MEMBER x t))`--};

val MEMBER1 = prove (--`!x. MEMBER x [] = F`--,REWRITE_TAC [MEMBER]);
val MEMBER2 =
   prove (--`!x h l. (x = h) ==> (MEMBER x (CONS h l) = T)`--,
          REPEAT STRIP_TAC THEN ASM_REWRITE_TAC [MEMBER]);
val MEMBER3 =
   prove (--`!x h l. ~(x = h) ==> (MEMBER x (CONS h l) = MEMBER x l)`--,
          REPEAT STRIP_TAC THEN ASM_REWRITE_TAC [MEMBER]);
send_definition("MEMBER",CONJ MEMBER1 (CONJ MEMBER2 MEMBER3));

send_definition("APPEND",definition "list" "APPEND");

send_scheme("list_INDUCTION",theorem "list" "list_INDUCT");

val MEMBER_APPEND1 =
   prove
      (--`!x y z. MEMBER x y ==> MEMBER x (APPEND y z)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: memapp2                                            *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:int=>y:int list=>z:int list=>member(x,z)=>member(x,app(y,z))   *)
(*--------------------------------------------------------------------------*)

val MEMBER_APPEND2 =
   prove
      (--`!x y z. MEMBER x z ==> MEMBER x (APPEND y z)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: memapp3                                            *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>e:int=>l1:int list=>l2:int list=>                                *)
(*         (member(e,l1)\member(e,l2))=>member(e,app(l1,l2))                *)
(*--------------------------------------------------------------------------*)

val MEMBER_APPEND3 =
   prove
      (--`!e l1 l2.
           (MEMBER e l1 \/ MEMBER e l2) ==> MEMBER e (APPEND l1 l2)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: memrev                                             *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:int=>y:int list=>member(x,y)=>member(x,rev(y))                 *)
(*--------------------------------------------------------------------------*)

val REV =
   new_recursive_definition
      {name = "REV",fixity = Prefix,
       rec_axiom = theorem "list" "list_Axiom",
       def = --`(REV [] = []) /\
                (!h l. REV (CONS h l) = APPEND (REV l) [h])`--};
send_definition("REV",REV);

val MEMBER_REV =
   prove
      (--`!x y. MEMBER x y ==> MEMBER x (REV y)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: meminsert1                                         *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>a:int=>b:int=>l:int list=>                                       *)
(*         a=b in int=>member(a,insert(b,l))={true}in u(1)                  *)
(*--------------------------------------------------------------------------*)

val INSERT =
   new_recursive_definition
      {name = "INSERT",fixity = Prefix,
       rec_axiom = theorem "list" "list_Axiom",
       def = --`(!n. INSERT n [] = [n]) /\
                (!n h t.
                   INSERT n (CONS h t) =
                   ((n < h) => CONS n (CONS h t) | CONS h (INSERT n t)))`--};

val INSERT1 = prove (--`!n. INSERT n [] = [n]`--,REWRITE_TAC [INSERT]);
val INSERT2 =
   prove (--`!n h t. (n < h) ==> (INSERT n (CONS h t) = CONS n (CONS h t))`--,
          REPEAT STRIP_TAC THEN ASM_REWRITE_TAC [INSERT]);
val INSERT3 =
   prove
      (--`!n h t. ~(n < h) ==> (INSERT n (CONS h t) = CONS h (INSERT n t))`--,
       REPEAT STRIP_TAC THEN ASM_REWRITE_TAC [INSERT]);
send_definition("INSERT",CONJ INSERT1 (CONJ INSERT2 INSERT3));

val MEMBER_INSERT1 =
   prove
      (--`!a b l. (a = b) ==> (MEMBER a (INSERT b l) = T)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: meminsert2                                         *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>a:int=>b:int=>l:int list=>                                       *)
(*         (a=b in int=>void)=>member(a,insert(b,l))=member(a,l)in u(1)     *)
(*--------------------------------------------------------------------------*)

val MEMBER_INSERT2 =
   prove
      (--`!a b l. ~(a = b) ==> (MEMBER a (INSERT b l) = MEMBER a l)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: memintersect                                       *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:int=>a:int list=>b:int list=>                                  *)
(*         (member(x,a)#member(x,b))=>member(x,intersect(a,b))              *)
(*--------------------------------------------------------------------------*)

val INTERSECT =
   new_recursive_definition
      {name = "INTERSECT",fixity = Prefix,
       rec_axiom = theorem "list" "list_Axiom",
       def = --`(!l2. INTERSECT [] l2 = []) /\
                (!h t l2. INTERSECT (CONS h t) l2 =
                          ((MEMBER h l2) => CONS h (INTERSECT t l2)
                                          | INTERSECT t l2))`--};
val INTERSECT1 = el 1 (CONJUNCTS INTERSECT);
val INTERSECT2 =
   prove
      (--`!l1. INTERSECT l1 [] = []`--,
       INDUCT_THEN (theorem "list" "list_INDUCT") ASSUME_TAC THEN
       REWRITE_TAC [INTERSECT] THEN ASM_REWRITE_TAC [MEMBER]);
val INTERSECT3 =
   prove
      (--`!h t l2. MEMBER h l2 ==>
                   (INTERSECT (CONS h t) l2 = CONS h (INTERSECT t l2))`--,
       REWRITE_TAC [INTERSECT] THEN REPEAT STRIP_TAC THEN
       ASM_REWRITE_TAC []);
val INTERSECT4 =
   prove
      (--`!h t l2. ~(MEMBER h l2) ==>
                   (INTERSECT (CONS h t) l2 = INTERSECT t l2)`--,
       REWRITE_TAC [INTERSECT] THEN REPEAT STRIP_TAC THEN
       ASM_REWRITE_TAC []);
send_definition
   ("INTERSECT",
    CONJ INTERSECT1 (CONJ INTERSECT2 (CONJ INTERSECT3 INTERSECT4)));

val MEMBER_INTERSECT =
   prove
      (--`!x a b. (MEMBER x a /\ MEMBER x b) ==> MEMBER x (INTERSECT a b)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: memunion1                                          *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:int=>a:int list=>b:int list=>member(x,a)=>member(x,union(a,b)) *)
(*--------------------------------------------------------------------------*)

val UNION =
   new_recursive_definition
      {name = "UNION",fixity = Prefix,
       rec_axiom = theorem "list" "list_Axiom",
       def = --`(!l2. UNION [] l2 = l2) /\
                (!h t l2. UNION (CONS h t) l2 =
                          ((MEMBER h l2) => UNION t l2
                                          | CONS h (UNION t l2)))`--};
val UNION1 = el 1 (CONJUNCTS UNION);
val UNION2 =
   prove
      (--`!l1. UNION l1 [] = l1`--,
       INDUCT_THEN (theorem "list" "list_INDUCT") ASSUME_TAC THEN
       REWRITE_TAC [UNION] THEN ASM_REWRITE_TAC [MEMBER]);
val UNION3 =
   prove
      (--`!h l1 l2. MEMBER h l2 ==>
                    (UNION (CONS h l1) l2 = UNION l1 l2)`--,
       REWRITE_TAC [UNION] THEN REPEAT STRIP_TAC THEN
       ASM_REWRITE_TAC []);
val UNION4 =
   prove
      (--`!h t l2. ~(MEMBER h l2) ==>
                   (UNION (CONS h t) l2 = CONS h (UNION t l2))`--,
       REWRITE_TAC [UNION] THEN REPEAT STRIP_TAC THEN
       ASM_REWRITE_TAC []);
send_definition("UNION",CONJ UNION1 (CONJ UNION2 (CONJ UNION3 UNION4)));

val MEMBER_UNION1 =
   prove
      (--`!x a b. MEMBER x a ==> MEMBER x (UNION a b)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: memunion2                                          *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:int=>a:int list=>b:int list=>member(x,b)=>member(x,union(a,b)) *)
(*--------------------------------------------------------------------------*)

val MEMBER_UNION2 =
   prove
      (--`!x a b. MEMBER x b ==> MEMBER x (UNION a b)`--,
       CLAM_TAC);

val MEMBER_UNION3 =
   prove
      (--`!x a b. MEMBER x a \/ MEMBER x b ==> MEMBER x (UNION a b)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: memsort1                                           *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:int=>l:int list=>member(x,sort(l))=>member(x,l)                *)
(*--------------------------------------------------------------------------*)

val SORT =
   new_recursive_definition
      {name = "SORT",fixity = Prefix,
       rec_axiom = theorem "list" "list_Axiom",
       def = --`(SORT [] = []) /\
                (!h t. SORT (CONS h t) = INSERT h (SORT t))`--};
send_definition("SORT",SORT);

val MEMBER_SORT1 =
   prove
      (--`!x l. MEMBER x (SORT l) ==> MEMBER x l`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: memsort2                                           *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:int=>l:int list=>member(x,l)=>member(x,sort(l))                *)
(*--------------------------------------------------------------------------*)

val MEMBER_SORT2 =
   prove
      (--`!x l. MEMBER x l ==> MEMBER x (SORT l)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: nthmem                                             *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:int=>n:pnat=>l:int list=>member(x,nth(n,l))=>member(x,l)       *)
(*--------------------------------------------------------------------------*)

val NTH =
   new_recursive_definition
      {name = "NTH",fixity = Prefix,
       rec_axiom = theorem "list" "list_Axiom",
       def = --`(!n. NTH n [] = []) /\
                (!n (h:'a) t. NTH n (CONS h t) =
                              ((n = 0) => CONS h t | NTH (PRE n) t))`--};
val NTH1 =
   prove
      (--`!(l:'a list). NTH 0 l = l`--,
       INDUCT_THEN (theorem "list" "list_INDUCT") ASSUME_TAC THEN
       REWRITE_TAC [NTH]);
val NTH2 = el 1 (CONJUNCTS NTH);
val NTH3 =
   prove
      (--`!n (h:'a) t. NTH (SUC n) (CONS h t) = NTH n t`--,
       REWRITE_TAC [NTH,theorem "prim_rec" "PRE",theorem "num" "NOT_SUC"]);
send_rule("NTH1",NTH1);
send_rule("NTH2",NTH2);
send_rule("NTH3",NTH3);

val NUM_LIST_PAIR =
   prove (--`!P. (!x. P x []) ==> 
                 (!y. P 0 y)  ==>
                 (!a (h:'a) t. P a t ==> P (SUC a) (CONS h t)) ==>
                 !p l. P p l`--,
          GEN_TAC THEN REPEAT DISCH_TAC THEN INDUCT_TAC THENL
          [ASM_REWRITE_TAC [],
           INDUCT_THEN (theorem "list" "list_INDUCT") ASSUME_TAC THENL
           [ASM_REWRITE_TAC [],
            FIRST_ASSUM MATCH_MP_TAC THEN ASM_REWRITE_TAC []]]);
send_scheme ("NUM_LIST_PAIR",NUM_LIST_PAIR);

val NTH_MEMBER =
   prove
      (--`!x n l. MEMBER x (NTH n l) ==> MEMBER x l`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: subsetintersect                                    *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>a:int list=>b:int list=>                                         *)
(*         subset(a,b)=>intersect(a,b)=a in int list                        *)
(*--------------------------------------------------------------------------*)

val SUBSET =
   new_recursive_definition
      {name = "SUBSET",fixity = Prefix,
       rec_axiom = theorem "list" "list_Axiom",
       def = --`(!l. SUBSET [] l = T) /\
                (!h t l. SUBSET (CONS h t) l = (MEMBER h l /\ SUBSET t l))`--};

val SUBSET1 = prove (--`!l. SUBSET [] l = T`--,REWRITE_TAC [SUBSET]);
val SUBSET2 =
   prove (--`!h t l. ~(MEMBER h l) ==> (SUBSET (CONS h t) l = F)`--,
          REPEAT STRIP_TAC THEN ASM_REWRITE_TAC [SUBSET]);
val SUBSET3 =
   prove (--`!h t l. (MEMBER h l) ==> (SUBSET (CONS h t) l = SUBSET t l)`--,
          REPEAT STRIP_TAC THEN ASM_REWRITE_TAC [SUBSET]);
send_definition("SUBSET",CONJ SUBSET1 (CONJ SUBSET2 SUBSET3));

send_rule("CANCEL_CONS1",CANCEL_CONS1);

val SUBSET_INTERSECT =
   prove
      (--`!a b. SUBSET a b ==> (INTERSECT a b = a)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: subsetunion                                        *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>a:int list=>b:int list=>subset(a,b)=>union(a,b)=b in int list    *)
(*--------------------------------------------------------------------------*)

val SUBSET_UNION =
   prove
      (--`!a b. SUBSET a b ==> (UNION a b = b)`--,
       CLAM_TAC);

(*--------------------------------------------------------------------------*)
(* Name in Clam library: ordered_cons                                       *)
(* Original Clam example:                                                   *)
(*                                                                          *)
(*    []==>x:int=>y:int list=>ordered(x::y)=>ordered(y)                     *)
(*--------------------------------------------------------------------------*)

val ORDERED =
   new_recursive_definition
      {name = "ORDERED",fixity = Prefix,
       rec_axiom = theorem "list" "list_Axiom",
       def = --`(ORDERED [] = T) /\
                (!x l. ORDERED (CONS x l) =
                       ((NULL l) => T | (HD l < x) => F | ORDERED l))`--};
val ORDERED1 = el 1 (CONJUNCTS ORDERED);
val ORDERED2 =
   prove
      (--`!x. ORDERED [x] = T`--,
       REWRITE_TAC [ORDERED,definition "list" "NULL_DEF"]);
val ORDERED3 =
   prove
      (--`!x y t. (y < x) ==> (ORDERED (CONS x (CONS y t)) = F)`--,
       REPEAT STRIP_TAC THEN REWRITE_TAC [ORDERED] THEN
       ASM_REWRITE_TAC [definition "list" "NULL_DEF",definition "list" "HD"]);
val ORDERED4 =
   prove
      (--`!x y t. ~(y < x) ==>
                  (ORDERED (CONS x (CONS y t)) = ORDERED (CONS y t))`--,
       REPEAT STRIP_TAC THEN REWRITE_TAC [ORDERED] THEN
       ASM_REWRITE_TAC [definition "list" "NULL_DEF",definition "list" "HD"]);
send_definition
   ("ORDERED",CONJ ORDERED1 (CONJ ORDERED2 (CONJ ORDERED3 ORDERED4)));

val TWO_CONS =
   prove
      (--`!P. P [] ==>
              (!h. P [h]) ==>
              (!(h1:'a) h2 t. P (CONS h2 t) ==> P (CONS h1 (CONS h2 t)))
              ==>
              !l. P l`--,
       GEN_TAC THEN REPEAT DISCH_TAC THEN
       INDUCT_THEN (theorem "list" "list_INDUCT") ASSUME_TAC THENL
       [ASM_REWRITE_TAC [],
        SPEC_TAC (--`l:'a list`--,--`l:'a list`--) THEN
        INDUCT_THEN (theorem "list" "list_INDUCT")
           (fn th => REPEAT GEN_TAC THEN ASSUME_TAC (SPEC_ALL th)) THENL
        [ASM_REWRITE_TAC [],RES_TAC THEN ASM_REWRITE_TAC []]]);
send_scheme("TWO_CONS",TWO_CONS);

val ORDERED_CONS =
   prove
      (--`!x y. ORDERED (CONS x y) ==> ORDERED y`--,
       CLAM_TAC);

(*
end_clam ();
*)
