(****************************************************************************)
(*                                                                          *)
(* Copyright 1998, 1999 University of Cambridge and University of Edinburgh *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : definitions.sml                                          *)
(* DESCRIPTION   : Finding and sending definitions.                         *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 31st July 1998                                           *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 13th October 1999                                        *)
(****************************************************************************)

structure Definitions =
struct

local

open CLaReTPortable;

in

fun theory_of_const name = #theory (Theory.const_decl name);

(* Relies on format used to encode datatype constructors *)
fun is_constructor_def tm =
   let fun result_type ty =
          let val {Args = [_,rty],Tyop = "fun"} = dest_type ty
          in  result_type rty
          end
          handle HOL_ERR _ => ty | Pattern => ty
       val {lhs,rhs} = (dest_eq o snd o strip_forall) tm
       val ty = (#Ty o dest_const o fst o strip_comb) lhs
       val result_ty = result_type ty
       val ty_name = #Tyop (dest_type result_ty)
       val name = (#Name o dest_const o fst o strip_comb) rhs
   in  name = "ABS_" ^ ty_name
   end
   handle HOL_ERR _ => false;

datatype theory_kind = Theory of string | BasicLogic | NoTheory;

fun subjects_of_definition th =
   CallClam.subjects_of_clauses ((strip_conj o snd o strip_forall o concl) th);

fun subjects_and_definitions names theory =
   let fun sub [] = []
         | sub ((name,th)::defs) =
          let fun filter [] = sub defs
                | filter (s::ss) =
                 if (member s names)
                 then (s,(name,Theory theory,th)) :: filter ss
                 else filter ss
          in  filter (subjects_of_definition th)
          end
   in  sub (definitions theory)
   end;

fun definitions_for_constants names =
   let val num_ty = mk_type {Args = [],Tyop = "num"}
       fun mk_num_const Name = mk_const {Name = Name,Ty = num_ty}
       fun is_positive NONE = false | is_positive (SOME n) = (n > 0)
       val (numeric_names,names') =
          Lib.partition (is_positive o Int.fromString) names
       val basic_data =
          mapfilter (fn n => (n,HOLClamBasicLogic.data_for_constant n)) names'
       val basic_defs =
          mapfilter
             (fn (n,ds) =>
                 let val (c,th) = hd (HOLClamBasicLogic.definitions ds)
                 in  (n,(c,BasicLogic,th))
                 end)
             basic_data
       val names'' = subtract names' (map fst basic_defs)
       val names_and_theories = map (fn n => (n,theory_of_const n)) names''
       val theories = setify (map snd names_and_theories)
       val names_and_defs =
          map (fn n => (n,("NUM" ^ n,NoTheory,num_CONV (mk_num_const n))))
             numeric_names
          @
          basic_defs
          @
          flat (map (fn t => subjects_and_definitions
                                (map fst (filter (fn (n,t') => t' = t)
                                             names_and_theories)) t)
                   theories)
   in  map (fn n => (n,SOME (assoc n names_and_defs) handle NOT_FOUND => NONE))
          names
   end;

val built_in_constants =
   ["~","/\\","\\/","==>","=","!","?","?!","@","T","F"];

fun constants_in_term consts tm =
   (case (dest_term tm)
    of CONST {Name,Ty} =>
          if (member Name consts) orelse (member Name built_in_constants)
          then consts
          else Name :: consts
     | VAR _ => consts
     | COMB {Rator,Rand} =>
          constants_in_term (constants_in_term consts Rator) Rand
     | LAMB {Bvar,Body} => constants_in_term consts Body);

fun constants_in_terms consts [] = consts
  | constants_in_terms consts (tm::tms) =
   constants_in_terms (constants_in_term consts tm) tms;

fun send_definitions_in_terms ignore tms =
   let fun select_def (_,NONE) = true
         | select_def (_,SOME (_,_,th)) = not (is_constructor_def (concl th))
       fun process_def (c,NONE) =
          Lib.say ("No definition found for constant `" ^ c ^ "'\n")
         | process_def (c,SOME (name,thy,th)) =
          let val thystr = case thy of Theory s => "theory `" ^ s ^ "'"
                                     | BasicLogic => "basic logic"
                                     | NoTheory => "no theory"
          in  (Lib.say ("Sending definition `" ^ name ^ "' (" ^ thystr ^
                        ") for constant `" ^ c ^ "'\n");
               CallClam.send_definition (name,th))
          end
       fun terms_of_defs [] = []
         | terms_of_defs ((_,NONE)::defs) = terms_of_defs defs
         | terms_of_defs ((_,SOME (_,_,th))::defs) =
          concl th :: terms_of_defs defs
       fun send _ _ [] = ()
         | send known processed tms =
          let val consts = constants_in_terms [] tms
              val unprocessed_consts = subtract consts processed
              val defs = definitions_for_constants unprocessed_consts
              val selected_defs = filter select_def defs
              val selected_consts = map fst selected_defs
              val unknown_consts = subtract selected_consts known
              val unknown_defs =
                 filter (fn (c,_) => member c unknown_consts) selected_defs
          in  (* Look for other constants in ALL the definitions ... *)
              send (unknown_consts @ known) (unprocessed_consts @ processed)
                 (terms_of_defs selected_defs);
              (* ... but only process the definitions for unknown constants. *)
              map process_def unknown_defs;
              ()
          end
   in  send (ClamDatabase.known_hol_constants ()) ignore tms
   end;

val auto_send = ref true;

local

val disabled = ref ([]:string list);

in

fun disable c = (disabled := c :: !disabled);

fun enable c =
   (disabled := CLaReTPortable.filter (fn s => not (s = c)) (!disabled));

fun send_definitions gl =
   if !auto_send
   then let val (tms,tm) = gl
        in  send_definitions_in_terms (!disabled) (tm :: tms)
        end
   else ();

val disabled = fn () => !disabled;

end;

end;

end; (* Definitions *)
