(*--------------------------------------------------------------------------*)
(* Load the LR parser and build a structure for use with the lexer          *)
(*--------------------------------------------------------------------------*)

use "clam.grm.sig";
use "clam.grm.sml";
structure ClamTablesTokens = 
   ClamLrValsFun (structure Token = LrParser.Token);

(*--------------------------------------------------------------------------*)
(* Load the lexer                                                           *)
(*--------------------------------------------------------------------------*)

use "clam.lex.sml";

(*--------------------------------------------------------------------------*)
(* Main structure for Clam                                                  *)
(*--------------------------------------------------------------------------*)

structure ClamParse =
struct

(*--------------------------------------------------------------------------*)
(* Instantiate the lexer                                                    *)
(*--------------------------------------------------------------------------*)

structure L = ClamLexFun (structure Tokens = ClamTablesTokens.Tokens);

(*--------------------------------------------------------------------------*)
(* Build structure for parser                                               *)
(*--------------------------------------------------------------------------*)

structure P =
   JoinWithArg
      (structure ParserData = ClamTablesTokens.ParserData
       structure Lex = L
       structure LrParser = LrParser);

(*--------------------------------------------------------------------------*)
(* Parsing functions for Clam                                               *)
(*--------------------------------------------------------------------------*)

val error = L.UserDeclarations.error;

(*--------------------------------------------------------------------------*)
(* General parser                                                           *)
(*--------------------------------------------------------------------------*)

fun parse {filename} {line_number} read_function =
   let val _ = (L.UserDeclarations.linenum := line_number)
       val lexer = P.makeLexer read_function filename
       fun perror (text,first,last) = error (text,(filename,(first,last)));
       val (res,_) = P.parse (0,lexer,perror,filename)
   in  res
   end;

(*--------------------------------------------------------------------------*)
(* Parsing of strings                                                       *)
(*--------------------------------------------------------------------------*)

fun parse_strings line_number ss =
   let val ss_ref = ref ss
       fun get_chars _ =
          case !ss_ref
          of [] => ""
           | s::ss => (ss_ref := ss; s)
   in  parse {filename = ""} line_number get_chars
   end;

(*--------------------------------------------------------------------------*)
(* Parsing a stream                                                         *)
(*--------------------------------------------------------------------------*)

fun parse_stream stream =
   parse {filename = ""} {line_number = 1}
      (fn n => TextIO.inputN (stream,n));

(*--------------------------------------------------------------------------*)
(* Parsing a file                                                           *)
(*--------------------------------------------------------------------------*)

fun parse_file filename =
   let val instr = TextIO.openIn filename
       val res = parse {filename = filename} {line_number = 1}
                    (fn n => TextIO.inputN (instr,n))
                 handle e => (TextIO.closeIn instr; raise e)
   in  TextIO.closeIn instr; res
   end;

end; (* ClamParse *)
