(*---------------------------------------------------------------------------*
 * FILE: vistactic.sml                                                       *
 *                                                                           *
 * This structure provides implementations of "visible" HOL tactics - ones   *
 * that come with a prettyprinter attached.                                  *
 *---------------------------------------------------------------------------*)

functor VisibleTactic (val clam_theorem : string -> thm
                       val scheme : string -> thm) : VisibleTactic_sig =

struct

type term    = Term.term
type vtactic = (PrettyPrint.ppstream -> unit) * tactic

open PrettyPrint;
open Tactic;
open Rsyntax;

val scheme       = scheme
val clam_theorem = clam_theorem ;

fun tac_err(func,mesg) = HOL_ERR{origin_structure="PlantoTactic",
                                 origin_function = func,
                                 message = mesg};

datatype direction = LEFT | RIGHT;


(*---------------------------------------------------------------------------*
 * Miscellaneous.                                                            *
 *---------------------------------------------------------------------------*)
fun W f x = f x x;

fun paren s = String.concat["(",s,")"];

local fun remove1 P =
        let fun del [] = []
              | del (h::t) = if P h then t else h::del t
        in del end
in
fun WEAKEN_TAC tm :tactic = fn (asl,w) =>
 ([(remove1 (aconv tm) asl,w)], 
  fn [th] => ADD_ASSUM tm th)
end;

(*---------------------------------------------------------------------------*
 * Projection functions for vtactics.                                        *
 *---------------------------------------------------------------------------*)
fun pp_of (pp,t) = pp
fun tactic_of (pp,t) = t;

(*---------------------------------------------------------------------------
 * Prettyprinting support.
 *---------------------------------------------------------------------------*)

fun pr_list pfun dfun bfun =
  let fun pr [] = ()
        | pr [i] = pfun i
        | pr (i::rst) = ( pfun i ; dfun() ; bfun() ; pr rst )
  in  pr   end;

fun pp_list pp_item ppstrm L =
  ( begin_block ppstrm CONSISTENT 0;
    add_string ppstrm "["; 
      begin_block ppstrm INCONSISTENT 0;
        pr_list (pp_item ppstrm)
                (fn () => add_string ppstrm ",") 
                (fn () => add_break ppstrm (1,0)) L;
      end_block ppstrm;
    add_string ppstrm "]";
    end_block ppstrm);

fun quote_pp strm s =
   (add_string strm "`"; add_string strm s; add_string strm "`");

fun string_literal_pp strm s = add_string strm (Lib.quote s);

(*---------------------------------------------------------------------------
 * Prettyprint a term so that it is reparsable. Uses SML/NJ quotations.
 *---------------------------------------------------------------------------*)
fun parsable_pp ppstrm tm = 
  let val stypes = !Globals.show_types
      val _ = Globals.show_types := true
  in 
    begin_block ppstrm INCONSISTENT 2;
    add_string ppstrm "`";
    pp_term ppstrm tm;
    add_string ppstrm ("`");
    end_block ppstrm;
    Globals.show_types := stypes
   end;


(*---------------------------------------------------------------------------*
 *                                                                           *
 *                        INDUCTION SCHEMES                                  *
 *                                                                           *
 *---------------------------------------------------------------------------*)

fun CONSEQ_CONV conv tm =
   if (is_imp tm) andalso not (is_neg tm)
   then RAND_CONV conv tm
   else NO_CONV tm;

(* ``c ==> !x1 ... xn y1 ... ym. b`` --> ``!x1 ... xn. c ==> !y1 ... ym. b`` *)
fun ONCE_MOVE_FORALL_OUT_CONV n tm =
   if (n < 1)
   then ALL_CONV tm
   else (RIGHT_IMP_FORALL_CONV THENC
         DecisionSupport.BINDER_CONV (ONCE_MOVE_FORALL_OUT_CONV (n - 1))) tm;

(* ``c1 ==> ... ==> ck ==> !x1 ... xn y1 ... ym. b`` -->
   ``!x1 ... xn. c1 ==> ... ==> ck ==> !y1 ... ym. b``   *)
fun MOVE_FORALL_OUT_CONV n tm =
   TRY_CONV (CONSEQ_CONV (MOVE_FORALL_OUT_CONV n) THENC
             ONCE_MOVE_FORALL_OUT_CONV n) tm;

(*---------------------------------------------------------------------------*
 * Apply an induction theorem to a goal.                                     *
 *---------------------------------------------------------------------------*)
fun RINDUCT_TAC thm =
  let val {Bvar,Body} = dest_forall (concl thm)
      val conc = (#2 o strip_forall o #2 o strip_imp) Body
      val (ants,pconc) = ((I ## (#2 o strip_forall)) o strip_imp) conc
      val args = #2 (strip_comb pconc)
      fun ndest_forall n trm =
          let fun dest (0,tm,V) = (rev V,tm)
                | dest (n,tm,V) = 
                   let val {Bvar,Body} = dest_forall tm
                   in dest(n-1,Body, Bvar::V)
                   end
          in dest(n,trm,[])
          end
      fun ndest_imp n trm =
          let fun dest (0,tm,A) = (rev A,tm)
                | dest (n,tm,A) = 
                   let val {ant,conseq} = dest_imp tm
                   in dest(n-1,conseq, ant::A)
                   end
          in dest(n,trm,[])
          end
      fun tac (asl,w) =
       let val (V1,body1) = ndest_forall (length args) w
           val (V2,body2) = strip_forall body1
           val (_,body3) = ndest_imp (length ants) body2
           val P = list_mk_abs(V1,list_mk_forall(V2,body3))
           val thm' =
              CONV_RULE
                 (DEPTH_CONV GEN_BETA_CONV THENC
                  RAND_CONV (DecisionSupport.DEPTH_FORALL_CONV
                                (MOVE_FORALL_OUT_CONV (length V2))))
                 (ISPEC P thm)
       in MATCH_MP_TAC thm' (asl,w)
       end handle _ => raise tac_err("RINDUCT_TAC","")
  in tac
  end;

(*---------------------------------------------------------------------------
 * Generate concrete syntax for obtaining the induction scheme.
 *---------------------------------------------------------------------------*)
fun pp_scheme name ppstrm 
    = (begin_block ppstrm INCONSISTENT 0;
        add_string ppstrm "scheme";
        add_break ppstrm (1,0);
        add_string ppstrm (Lib.quote name);
       end_block ppstrm)

(*---------------------------------------------------------------------------
 * Generate concrete syntax for induction tactic applications.
 *---------------------------------------------------------------------------*)
fun pp_induct conjnum name terms vars ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        case conjnum
          of NONE => add_string ppstrm "QIND_TAC"
           | SOME n => (add_string ppstrm "QMULTI_PRED_IND_TAC";
                        add_break ppstrm (1,0);
                        add_string ppstrm (Lib.int_to_string n));
        add_break ppstrm (1,0);
        add_string ppstrm "(";
        begin_block ppstrm CONSISTENT 2;
          add_string ppstrm "QRENAME";
          add_break ppstrm (1,0);
          add_string ppstrm "("; pp_scheme name ppstrm; add_string ppstrm ")";
          add_break ppstrm (1,0);
          pp_list (pp_list parsable_pp) ppstrm terms;
        end_block ppstrm;
        add_string ppstrm ")";
        add_break ppstrm (1,0);
        pp_list parsable_pp ppstrm vars;
       end_block ppstrm);

(*---------------------------------------------------------------------------
 * IND_TAC (RENAME (scheme <name>) <terms>) <variables> 
 *---------------------------------------------------------------------------*)

(* Identify the clauses that have inductive hypotheses *)
fun step_cases thm = 
  let val Body = snd (strip_forall (concl thm))
      val (ants,_) = strip_imp Body
  in
      map (is_imp o snd o strip_forall) ants
  end;


fun GEN_IND_TAC scheme thm V (asl,w) =
   let val (prefix, body) = strip_forall w
       val unused = subtract prefix V
       fun GEN_NEW_TAC (g as (_,w)) =
          let val {Bvar,Body} = dest_forall w
          in  if (Lib.mem Bvar unused) then NO_TAC g else GEN_TAC g
          end
   in (REPEAT GEN_TAC 
        THEN MAP_EVERY Tactic.SPEC_TAC (map (fn x => (x,x)) (rev unused))
        THEN MAP_EVERY Tactic.SPEC_TAC (map (fn x => (x,x)) (rev V))
        THEN RINDUCT_TAC thm
        THEN REPEAT Tactic.CONJ_TAC
        THENL (map (fn true  => REPEAT GEN_TAC THEN
                                DISCH_THEN (EVERY o map ASSUME_TAC o CONJUNCTS)
                     | false => REPEAT GEN_NEW_TAC) (step_cases scheme)))
      (asl,w)
   end;

fun IND_TAC scheme =
   GEN_IND_TAC scheme (Rewrite.PURE_REWRITE_RULE [AND_IMP_INTRO] scheme);

fun QIND_TAC scheme V = IND_TAC scheme (map Parse.term_parser V);

(*---------------------------------------------------------------------------
 * MULTI_PRED_IND_TAC (RENAME (scheme <name>) <terms>) <lambdas> 
 *---------------------------------------------------------------------------*)

(* Drop all unwanted conjuncts in the conclusion. *)
fun WEAKEN_CONCL n thm =
   let val (Ps,Body) = strip_forall (concl thm)
       val ants = fst (strip_imp Body)
   in  (GENL Ps o C (foldr (uncurry DISCH)) ants o Lib.el n o CONJUNCTS o
        UNDISCH_ALL o SPEC_ALL) thm
   end;

fun MULTI_PRED_IND_TAC n scheme lambdas =
   let val thm1 = WEAKEN_CONCL n scheme
       val thm2 = Rewrite.PURE_REWRITE_RULE [AND_IMP_INTRO] thm1
       val Ps = fst (strip_forall (concl thm2))
       val P = Lib.el n Ps
       val V = fst (strip_abs (Lib.el n lambdas))
       fun insts i [] = []
         | insts i (l::ls) = (if (i = n) then P else l) :: insts (i + 1) ls
       val thm3 = GEN P (SPECL (insts 1 lambdas) thm2)
       val thm4 = CONV_RULE (DEPTH_CONV GEN_BETA_CONV) thm3
   in  GEN_IND_TAC scheme thm4 V
   end;

fun QMULTI_PRED_IND_TAC n scheme lambdas =
   MULTI_PRED_IND_TAC n scheme (map Parse.term_parser lambdas);

(*---------------------------------------------------------------------------
 * Renaming of variables in an induction scheme.
 *---------------------------------------------------------------------------*)

fun mk_name_map {redex,residue} =
   (#Name (dest_var redex),#Name (dest_var residue));

fun rename_var name_map v =
   let val {Name,Ty} = dest_var v
   in  mk_var {Name = assoc Name name_map handle NOT_FOUND => Name,Ty = Ty}
   end;

fun rename S tm = 
  let val (V,body) = strip_forall tm
      val conseq = snd (strip_imp body)
      val args = snd (strip_comb conseq)
      val bindings = map2 match_term args S
      val name_map = map mk_name_map (Lib.flatten (map fst bindings))
      val theta = map (fn v => {redex = v,residue = rename_var name_map v}) V
      val V' = map (subst theta) V
  in list_mk_forall(V',subst theta body) 
  end;

fun RENAME thm termsl =
  let val (Ps,Body) = strip_forall (concl thm)
      val (ants,c) = strip_imp Body
      val ants' = map2 rename termsl ants
      val tm = list_mk_forall (Ps,list_mk_imp(ants',c))
  in EQ_MP (ALPHA (concl thm) tm) thm
  end;

fun QRENAME thm fragsl = RENAME thm (map (map Parse.term_parser) fragsl);

(*---------------------------------------------------------------------------*
 * Apply an induction scheme. There are a few steps in this.                
 *
 *    1. Use the first projections to permute the universal prefix 
 *       of the goal so that it matches the binding structure of the 
 *       scheme. We assume that the order the induction variables come in 
 *       is the same as the order they should be in the new goal.
 *
 *    2. Extract the induction scheme from the database. 
 *
 *    3. Rename variables in the scheme into those that Clam used in 
 *       planning. This is necessary, because explicit terms are given when
 *       generalizing goals, and the goals that Clam and HOL are working on
 *       must agree on their constants and free variables.
 * 
 *    4. To enable fertilization to succeed as often as possible, especially
 *       in nested inductions, it may be necessary to maintain track of 
 *       inductive hypotheses so that there is no (or at least less) confusion 
 *       about which i.h. is to be used. (This has not been implemented yet.)
 *---------------------------------------------------------------------------*)

fun IND s V iterms =
  let val scheme' = RENAME (scheme s) iterms
  in 
     (pp_induct NONE s iterms V, IND_TAC scheme' V)
  end;

fun MULTI_PRED_IND n s lambdas iterms =
  let val scheme' = RENAME (scheme s) iterms
  in 
     (pp_induct (SOME n) s iterms lambdas,
      MULTI_PRED_IND_TAC n scheme' lambdas)
  end;

(*---------------------------------------------------------------------------
 * Dealing with universally quantified variables.
 *---------------------------------------------------------------------------*)
fun GENL_TAC L = 
   let fun pp ppstrm 
    = (begin_block ppstrm INCONSISTENT 2;
        add_string ppstrm "MAP_EVERY";
        add_break ppstrm (1,0);
        add_string ppstrm "Q.X_GEN_TAC";
        add_break ppstrm (1,0);
        add_string ppstrm "[";
          begin_block ppstrm CONSISTENT 2;
           pr_list (quote_pp ppstrm)
                   (fn () => add_string ppstrm ",") 
                   (fn () => add_break ppstrm (0,0)) L;
          end_block ppstrm;
        add_string ppstrm "]"; 
       end_block ppstrm)
   in
    (pp, MAP_EVERY Q.X_GEN_TAC (map (fn s => `s`) L))
   end;



(*---------------------------------------------------------------------------
 * Blow away trivial subgoals with ELEM_TAC. Finishing tactics are invoked 
 * in order of (roughly) increasing cost.
 *---------------------------------------------------------------------------*)
val CONS_NOT_CYCLIC1 =
   prove
      (--`!h t. (CONS h t = t) = F`--,
       GEN_TAC THEN
       INDUCT_THEN (theorem "list" "list_INDUCT") ASSUME_TAC THEN
       decisionLib.DECIDE_TAC);

val CONS_NOT_CYCLIC2 =
   prove
      (--`!h t. (CONS h t = t) = F`--,
       REWRITE_TAC [CONS_NOT_CYCLIC1]);

val rws = [theorem"num" "NOT_SUC", GSYM (theorem"num" "NOT_SUC"),
           theorem"prim_rec" "SUC_ID", GSYM(theorem"prim_rec" "SUC_ID"),
           theorem"prim_rec" "INV_SUC_EQ",
           theorem"list" "CONS_11",
           theorem"list""NOT_CONS_NIL", theorem"list" "NOT_NIL_CONS",
           CONS_NOT_CYCLIC1,CONS_NOT_CYCLIC2];

fun BRING (asl,w) = 
  let val (_,body) = strip_forall w
      fun occurs M tm = (free_in tm M orelse 
                         (free_in (dest_neg tm) M handle _ => false))
      fun disch [] = ALL_TAC
        | disch [x] = UNDISCH_TAC x
        | disch (x::rst) = UNDISCH_TAC x THEN disch rst
  in 
    disch (filter (occurs body) asl)
  end;

(*---------------------------------------------------------------------------
 * Each tactic in the ORELSE chain must either finish off the proof, or fail.
 *---------------------------------------------------------------------------*)
fun ELEM_TAC g = 
  (REPEAT GEN_TAC 
   THEN (REFL_TAC
        ORELSE (BRING g
                THEN ((Rewrite.REWRITE_TAC rws THEN NO_TAC)
                      ORELSE
                      tautLib.TAUT_TAC 
                      ORELSE 
                      decisionLib.DECIDE_TAC)))) g;


val ELEMENTARY = (C add_string "ELEM_TAC", ELEM_TAC);


(*---------------------------------------------------------------------------
 * Term rewriting with explicit occurrences.
 *
 * The code assumes that the term is in some sort of standard form, i.e., no 
 * beta-redices, binders get applied to lambdas, etc. Paired bindings 
 * currently are not dealt with properly. Quantifiers magically disappear.
 * Equality has a special extra argument.
 *---------------------------------------------------------------------------*)

val LIST_MK_COMB = rev_itlist (Lib.C AP_THM);
val LIST_MK_ABS = itlist ABS;

fun remake_comb (h::t, th, R) = LIST_MK_COMB R (AP_TERM (list_mk_comb(h,t)) th)
  | remake_comb ([], _, _) = raise tac_err("remake_comb","");

fun breakup i L =
  let fun break 0 (h::t) L = (rev L, h, t)
        | break n (h::t) L = break(n-1) t (h::L)
        | break _ [] _ = raise tac_err("break", "");
  in break i L []
  end;

(*---------------------------------------------------------------------------
 * Occurrence lists have an extra entry for equalities, so we skip past the
 * entry, unless the equality was translated to a bi-implication (iff).
 *---------------------------------------------------------------------------*)
fun skip_eq f p L = 
   if (let val {Name,Ty} = dest_const f
       in  (Name = "=") andalso not (hd (#Args (dest_type Ty)) = bool)
       end
       handle _ => false)
   then case L 
        of (_::h1::t) => (h1,t) 
         | others     => raise tac_err("skip_eq", "")
   else p;

(*---------------------------------------------------------------------------
 * This may well be wrong for descending into lambdas. I don't know what
 * Clam does for that.
 *---------------------------------------------------------------------------*)
fun OCC_CONV conv = 
 let fun trav [] tm = conv tm
       | trav (path as (occ::rst)) tm =
          case (dest_term tm)
           of LAMB{Bvar,Body} (* there's only one way to go into a lambda *)
              => ABS Bvar (trav path Body) 
            | COMB{Rator,Rand}
              => if (is_binder (#Name(dest_const(Rator))) handle _ => false)
                 then remake_comb([Rator], trav path Rand, []) 
                 else let val (f,args) = strip_comb tm
                          val (occ', rst') = skip_eq f (occ,rst) path
                          val (L,a,R) = breakup occ' (f::args)
                      in remake_comb(L, trav rst' a, R)    end
            | CONST _ => trav rst tm (* Could make more checks *)
            | VAR   _ => raise tac_err("OCC_CONV", "VAR")
 in trav o rev
 end;

fun exp_at tm occlist = 
 let fun trav [] tm = tm
       | trav (path as (occ::rst)) tm =
          case (dest_term tm)
           of LAMB{Bvar,Body} (* there's only one way to go into a lambda *)
              => trav path Body
            | COMB{Rator,Rand}
              => if (is_binder (#Name(dest_const(Rator))) handle _ => false)
                 then trav path Rand
                 else let val (f,args) = strip_comb tm
                          val (occ', rst') = skip_eq f (occ,rst) path
                          val (L,a,R) = breakup occ' (f::args)
                      in trav rst' a end
            | CONST _ => trav rst tm (* Could make more checks *)
            | VAR   _ => raise tac_err("exp_at", "VAR")
 in trav (rev occlist) tm
 end;

fun goal_at occlist = exp_at (snd(top_goal())) occlist;

(*---------------------------------------------------------------------------
 * tests for occurrence finding.

     set_goal(dest_thm (theorem "arithmetic" "ADD_ASSOC"));
     e INDUCT_TAC;
     val [plus1,plus2] = CONJUNCTS (definition"arithmetic" "ADD");
     val tm = snd(top_goal());

     (OCC_CONV (REWR_CONV plus1) [1,1] 
     THENC 
     OCC_CONV (REWR_CONV plus1) [1,2,1]) tm;

     Prolog.
     exp_at(s(plus(m,plus(n,p))) = s(plus(plus(m,n),p)) [1,1], X).

 *---------------------------------------------------------------------------*)


(*---------------------------------------------------------------------------
 * Given a term ``(C1 /\ ... /\ Cn) ==> E`` where ``C1 /\ ... /\ Cn`` is a
 * conjunction tree (i.e. it does not need to be linear), the following
 * theorem is returned:
 * 
 *    |- ((C1 /\ ... /\ Cn) ==> E) = (C1 ==> ... ==> Cn ==> E)
 *---------------------------------------------------------------------------*)
fun NORMALIZE_ONE_IMP_CONV tm =
   if (is_imp tm) andalso (is_conj (rand (rator tm)))
   then (REWR_CONV (GSYM AND_IMP_INTRO) THENC
         RAND_CONV NORMALIZE_ONE_IMP_CONV THENC
         NORMALIZE_ONE_IMP_CONV) tm
   else ALL_CONV tm;

(*---------------------------------------------------------------------------
 * For a term ``Cs1 ==> ... ==> Csn ==> E`` where Cs1, ..., Csn are
 * conjunction trees, this conversion normalizes all the conjunction trees
 * as above.
 *---------------------------------------------------------------------------*)
fun NORMALIZE_IMP_CONV tm =
   if (is_imp tm)
   then (RAND_CONV NORMALIZE_IMP_CONV THENC NORMALIZE_ONE_IMP_CONV) tm
   else ALL_CONV tm;


(*---------------------------------------------------------------------------
 * Given a rule P ==> ...==> Z ==> (Q = R) and a term theta(Q), we return
 * 
 *   theta(P), ..., theta(Z) |- theta(Q) = theta(R)
 *
 * It is up to the caller to eliminate the assumptions.
 *---------------------------------------------------------------------------*)
fun RUNDISCH_ALL th =
   if is_imp (concl th)
   then RUNDISCH_ALL 
         (let val {ant,conseq} = dest_imp (concl th)
          in if is_eq ant
             then let val {lhs,rhs} = dest_eq ant
                  in if (lhs = rhs)
                     then MP th (REFL lhs)
                     else UNDISCH th
                  end
             else UNDISCH th
          end)
   else th;

fun CREWR_CONV thl th tm =
  let val pth = CONV_RULE NORMALIZE_IMP_CONV (GSPEC th)
      val (_,eq) = strip_imp (concl pth)
      val ith = RUNDISCH_ALL(INST_TY_TERM (Match.match_term (lhs eq) tm) pth)
      val ith' = itlist PROVE_HYP (Lib.flatten (map CONJUNCTS thl)) ith
      val thl2 = mapfilter decisionLib.DECIDE (hyp ith')
      val ith'' = itlist PROVE_HYP thl2 ith'
      val l = lhs(concl ith'')
  in 
    if (l = tm) then ith'' else TRANS (ALPHA tm l) ith''
  end 
  handle _ => raise tac_err ("CREWR_CONV", "");


fun ORW_TAC thm path lr (g as (asl,_)) = 
 let val DEPTH_FORALL_CONV = DecisionSupport.DEPTH_FORALL_CONV
     fun DEPTH_CONSEQUENT_CONV conv tm =
        if (is_imp tm)
        then RAND_CONV (DEPTH_CONSEQUENT_CONV conv) tm
        else conv tm
     fun SYM_CONSEQUENT th =
        CONV_RULE (DEPTH_FORALL_CONV (DEPTH_CONSEQUENT_CONV SYM_CONV)) th
     val thl = map ASSUME asl
     val thm' = case lr of LEFT => thm | RIGHT => SYM_CONSEQUENT thm
 in CONV_TAC (DEPTH_FORALL_CONV (OCC_CONV (CREWR_CONV thl thm') path)) g
 end;

fun ORW (name, thm, path, lr) =
   let fun pp ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "ORW_TAC";
        add_break ppstrm (1,0);
        add_string ppstrm name;
        add_break ppstrm (1,0);
        add_string ppstrm "[";
          begin_block ppstrm CONSISTENT 2;
           pr_list (add_string ppstrm o Lib.int_to_string)
                   (fn () => add_string ppstrm ",") 
                   (fn () => add_break ppstrm (0,0)) path;
          end_block ppstrm;
        add_string ppstrm "]"; 
        add_break ppstrm (1,0);
        add_string ppstrm (case lr of LEFT => "LEFT" | RIGHT => "RIGHT");
       end_block ppstrm)
   in
   (pp, ORW_TAC thm path lr)
   end;

fun OCC_RW_TAC s = ORW_TAC (clam_theorem s);

fun OCC_RW (name, path, lr) = 
   let fun pp ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "OCC_RW_TAC";
        add_break ppstrm (1,0);
        add_string ppstrm (Lib.quote name);
        add_break ppstrm (1,0);
        add_string ppstrm "[";
          begin_block ppstrm CONSISTENT 2;
           pr_list (add_string ppstrm o Lib.int_to_string)
                   (fn () => add_string ppstrm ",") 
                   (fn () => add_break ppstrm (0,0)) path;
          end_block ppstrm;
        add_string ppstrm "]"; 
        add_break ppstrm (1,0);
        add_string ppstrm (case lr of LEFT => "LEFT" | RIGHT => "RIGHT");
       end_block ppstrm)
   in
   (pp, OCC_RW_TAC name path lr)
   end;

(*---------------------------------------------------------------------------
 * Weak fertilization. The goal is an equality. This "FIRST_ASSUM" business 
 * will eventually fail when more inductive hypotheses (or ordinary 
 * assumptions) are present. What is needed is to have some aspects of Clam's
 * indexing of inductive hypotheses communicated to HOL.
 *---------------------------------------------------------------------------*)

local 
   fun GSPEC th =
    let val w = concl th 
    in if (is_forall w)
       then GSPEC (SPEC(genvar(type_of(#Bvar(dest_forall w)))) th)
       else th
    end

   fun trim th = 
    let val (asl,c) = dest_thm th
        val (V,body) = strip_forall c
        val {lhs,rhs} = dest_eq body
        val lhsvars = free_vars lhs
    in 
      case Lib.set_diff (free_vars rhs) lhsvars
       of [] => th   (* std. rewrite rule *)
        | _  => GENL (Lib.set_diff lhsvars (free_varsl asl)) (SPEC_ALL th)
    end

   fun rewrCONV th tm =
    let val pth = GSPEC (trim th)
        val eqn = INST_TY_TERM (Match.match_term (lhs(concl pth)) tm) pth
        val l = lhs(concl eqn)
    in 
      if (l = tm) then eqn else TRANS (ALPHA tm l) eqn
    end 
    handle _ => raise tac_err("rewrCONV","")

in

fun RW_TAC thm path lr = 
   let val thm' = case lr of LEFT => thm | RIGHT => GSYM thm
   in  CONV_TAC
          (DecisionSupport.DEPTH_FORALL_CONV (OCC_CONV (rewrCONV thm') path))
   end

fun WFERT_EQ_TAC (path,dir) = 
   FIRST_ASSUM (fn th => RW_TAC th path dir THEN WEAKEN_TAC (concl th))

end;

fun WFERT_EQ (path,dir) = 
   let fun pp ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "WFERT_EQ_TAC";
        add_break ppstrm (0,0);
        add_string ppstrm "(";
        begin_block ppstrm INCONSISTENT 2;
          pp_list (C(C add_string o Lib.int_to_string)) ppstrm path;
          add_string ppstrm ","; 
          add_break ppstrm (0,0);
          add_string ppstrm (case dir of LEFT => "LEFT" | RIGHT => "RIGHT"); 
        end_block ppstrm;
        add_string ppstrm ")"; 
       end_block ppstrm)
   in
   (pp, WFERT_EQ_TAC(path,dir))
   end;


(*---------------------------------------------------------------------------
 * Strong fertilization.
 *---------------------------------------------------------------------------*)
fun FINISHED_TAC (g as (_,w)) =
   let val (_,body) = strip_forall w
   in  if DecisionSupport.is_T body
       then (REPEAT GEN_TAC THEN CONV_TAC ALL_CONV) g
       else NO_TAC g
   end;

fun SFERT_TAC path =
   FIRST_ASSUM (fn th => RW_TAC (EQT_INTRO (SPEC_ALL th)) path LEFT THEN
                         TRY FINISHED_TAC THEN WEAKEN_TAC (concl th));

fun SFERT path =
   let fun pp ppstrm =
      (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "SFERT_TAC";
        add_break ppstrm (1,0);
        pp_list (C(C add_string o Lib.int_to_string)) ppstrm path;
       end_block ppstrm)
   in
   (pp, SFERT_TAC path)
   end;

(*---------------------------------------------------------------------------
 * Kind of strong fertilization used by multi-predicate induction.
 *---------------------------------------------------------------------------*)

fun USE_HYPS_TAC tms =
   let val select_assums =
          map EQT_INTRO o Lib.filter (C Lib.mem tms o concl) o map SPEC_ALL
   in  ASSUM_LIST (REWRITE_TAC o select_assums)
   end;

fun QUSE_HYPS_TAC qtms = USE_HYPS_TAC (map Parse.term_parser qtms);

fun USE_HYPS tms =
   let fun pp ppstrm = 
      (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "QUSE_HYPS_TAC";
        add_break ppstrm (1,0);
        pp_list parsable_pp ppstrm tms;
       end_block ppstrm)
   in  (pp, USE_HYPS_TAC tms)
   end;

(*---------------------------------------------------------------------------
 * Generalization. There is some hokey-pokey so that universal quantifiers 
 * binding the term to be generalized are removed.
 *---------------------------------------------------------------------------*)
fun QGENERALISE_TAC (tm,x) (asl,w) =
  let val tm' = Parse.term_parser tm
      val x' = Parse.term_parser x
  in
  ((if (free_in tm' w) then ALL_TAC 
   else let val (V,body) = strip_forall w
           val W = intersect (free_vars tm') V
           val Y = set_diff V W
        in REPEAT GEN_TAC THEN
           MAP_EVERY Tactic.SPEC_TAC (map (fn x => (x,x)) (rev Y))
        end)
   THEN Tactic.SPEC_TAC (tm',x')) (asl,w)
   end;


fun GENERALISE (tm,x) = 
  let fun pp ppstrm = 
          ( begin_block ppstrm CONSISTENT 2;
            add_string ppstrm "QGENERALISE_TAC";
            add_break ppstrm (1,0);
            add_string ppstrm "(";
              begin_block ppstrm CONSISTENT 1;
                parsable_pp ppstrm tm;
                add_string ppstrm ",";
                add_break ppstrm (0,0);
                parsable_pp ppstrm x;
              end_block ppstrm;
              add_string ppstrm ")"; 
            end_block ppstrm)
  in
     (pp, QGENERALISE_TAC (`^tm`,`^x`)) 
  end;


(*---------------------------------------------------------------------------
 * Case splitting.
 *---------------------------------------------------------------------------*)
val CASES_THM = 
 ONCE_REWRITE_RULE 
   [decisionLib.DECIDE(Parse.term_parser`x \/ y = y \/ x`)]
   (REWRITE_RULE [] BOOL_CASES_AX);

(*---------------------------------------------------------------------------
   Sometimes qtm is a negation and Clam expects the ensuing double 
   negation to be normalized. I assume that this can only happen in
   the left disjunct.
 ---------------------------------------------------------------------------*)
fun QCASE_TAC qtm (g as (_,w)) =
  let val tm' = Parse.term_parser qtm
      val th = ISPEC tm' CASES_THM
      val th1 = GEN_REWRITE_RULE (RATOR_CONV o RAND_CONV)
                                 empty_rewrites [CONJUNCT1 NOT_CLAUSES] th
                handle _ => th
      val non_frees = subtract (fst (strip_forall w)) (free_vars tm')
  in
  (REPEAT GEN_TAC THEN
   MAP_EVERY Tactic.SPEC_TAC (map (fn x => (x,x)) (rev non_frees)) THEN
   DISJ_CASES_THEN
      (fn th => ASSUME_TAC th THEN
                (* Remove duplicate assumptions *)
                UNDISCH_TAC (concl th) THEN DISCH_TAC) th1) g
  end;


fun CASE tm = 
  let fun pp ppstrm = 
          ( begin_block ppstrm CONSISTENT 2;
            add_string ppstrm "QCASE_TAC";
            add_break ppstrm (1,0);
            parsable_pp ppstrm tm;
            end_block ppstrm)
  in
     (pp, QCASE_TAC `^tm`) 
  end;


(*---------------------------------------------------------------------------
 * Boolean case analysis.
 *---------------------------------------------------------------------------*)
(* Assumes the universal quantifier, if present, is outermost. *)
fun QBOOL_CASES_TAC qtm (asl,w) =
  let val v = Parse.term_parser qtm
      val (vars,_) = strip_forall w
  in  ((if (Lib.mem v vars) then X_GEN_TAC v else ALL_TAC) THEN
       DISJ_CASES_THEN
          (fn th => PURE_REWRITE_TAC [th] THEN ASSUME_TAC th THEN
                    (* Remove duplicate assumptions *)
                    UNDISCH_TAC (concl th) THEN DISCH_TAC)
          (SPEC v EXCLUDED_MIDDLE)) (asl,w)
  end;

fun BOOL_CASES tm =
  let fun pp ppstrm = 
          ( begin_block ppstrm CONSISTENT 2;
            add_string ppstrm "QBOOL_CASES_TAC";
            add_break ppstrm (1,0);
            parsable_pp ppstrm tm;
            end_block ppstrm)
  in
     (pp, QBOOL_CASES_TAC `^tm`) 
  end;


(*---------------------------------------------------------------------------
 * Term cancellation.
 *---------------------------------------------------------------------------*)

local

val num_ty = NumArith.ArithCons.NumberHOLType.num_ty;

fun summands tm =
   let val (tm1,tm2) = NumArith.ArithCons.dest_plus tm
   in  union (summands tm1) (summands tm2)
   end
   handle HOL_ERR _ => [tm];

fun summands_of_eq tm =
   let val {lhs,rhs} = dest_eq tm
   in  union (summands lhs) (summands rhs)
   end;

(* Assumes all summands in tm2 also occur in tm1. If not, the resulting *)
(* theorem will contain genvars and the test at the end will fail.      *)
fun EQUAL_SUMS (tm1,tm2) =
   let val tms = filter (not o NumArith.ArithCons.is_num_const)
                    (union (summands_of_eq tm1) (summands_of_eq tm2))
       val vars = map (fn _ => genvar num_ty) tms
       val sub = map2 (fn tm => fn v => {redex = tm,residue = v}) tms vars
       val rew = decisionLib.DECIDE (subst sub (mk_eq {lhs = tm1,rhs = tm2}))
       val th = REWR_CONV rew tm1
   in  if (rhs (concl th) = tm2) then th else raise tac_err ("EQUAL_SUMS","")
   end;

in

fun QTERM_CANCEL_TAC qtm (asl,w) =
  let val tm' = Parse.term_parser qtm
      val (_,tm1) = strip_forall w
      and (_,tm2) = strip_forall tm'
      val th = EQUAL_SUMS (tm1,tm2)
  in  PURE_ONCE_REWRITE_TAC [th] (asl,w)
  end;

end;

fun TERM_CANCEL tm =
  let fun pp ppstrm = 
          ( begin_block ppstrm CONSISTENT 2;
            add_string ppstrm "QTERM_CANCEL_TAC";
            add_break ppstrm (1,0);
            parsable_pp ppstrm tm;
            end_block ppstrm)
  in
     (pp, QTERM_CANCEL_TAC `^tm`) 
  end;


(*---------------------------------------------------------------------------
 * Shows when unknown methods are encountered in the plan.
 *---------------------------------------------------------------------------*)
fun HUH_TAC s = ALL_TAC;

fun HUH s = 
  let fun pp ppstrm = 
          ( begin_block ppstrm CONSISTENT 0;
              add_string ppstrm "HUH_TAC";
              add_break ppstrm (1,0);
              add_string ppstrm (Lib.quote s); end_block ppstrm)
  in
     (pp, HUH_TAC s) 
  end;


(*---------------------------------------------------------------------------
 * Trivial methods.
 *---------------------------------------------------------------------------*)
val MK_GEN_TAC   = 
  let fun pp ppstrm = 
          ( begin_block ppstrm CONSISTENT 0;
              add_string ppstrm "TRY"; add_break ppstrm (1,0);
              add_string ppstrm "GEN_TAC"; end_block ppstrm)
  in 
     (pp, Tactical.TRY Tactic.GEN_TAC)
  end;

val MK_DISCH_TAC =  (C add_string "DISCH_TAC", Tactic.DISCH_TAC) ;
val MK_ALL_TAC   =  (C add_string "ALL_TAC",   Tactical.ALL_TAC);


(*---------------------------------------------------------------------------
 * Need more info for this to be solid, i.e., which assumption is getting
 * broken apart. As it is, I just do all of them. The info is sort of there
 * in the plan, because names are given for this tactic. However, plans in
 * general are insufficiently well-annotated to determine the names 
 * attached to assumptions.
 *---------------------------------------------------------------------------*)
val ASM_CONJ_TAC = ASSUM_LIST (fn athl => 
 MAP_EVERY (fn tm => UNDISCH_TAC tm THEN STRIP_TAC)
           (map concl (filter (can dest_conj o concl) athl)));

val ASM_CONJ =  (C add_string "ASM_CONJ_TAC", ASM_CONJ_TAC);


(*---------------------------------------------------------------------------
 * Using hypotheses.
 *---------------------------------------------------------------------------*)
fun var_eq tm = 
   let val {lhs,rhs} = dest_eq tm
   in 
     (is_var lhs andalso not (Lib.mem lhs (free_vars rhs)))
     orelse 
     (is_var rhs andalso not (Lib.mem rhs (free_vars lhs)))
   end
   handle _ => false;

(*---------------------------------------------------------------------------
 * When invoked, we know that th is an equality, at least one side of which
 * is a var.
 *---------------------------------------------------------------------------*)
fun orient th = 
  let val c = concl th
      val {lhs,rhs} = dest_eq c
  in if (is_var lhs)
     then if (is_var rhs) 
          then if term_lt lhs rhs (* both vars, rewrite to textually smaller *)
               then SYM th 
               else th
          else th
     else SYM th
  end;

fun VSUBST_TAC tm = 
   UNDISCH_TAC tm THEN 
   DISCH_THEN (SUBST_ALL_TAC o orient);

val ASM_EQ_TAC = 
  W (fn (asl,_) => MAP_EVERY VSUBST_TAC (filter var_eq asl));

val ASM_EQ =  (C add_string ("ASM_EQ_TAC"), ASM_EQ_TAC) ;

(*---------------------------------------------------------------------------
        The following chunks of code do conditional replacement 
        in the goal, at depth. If we are working in a "negative" 
        context, like on the left-hand side of the top implication,
        then we do forward inference. In positive contexts, we are 
        doing backwards inference, i.e., applying tactics.
 ---------------------------------------------------------------------------*)
 
(*---------------------------------------------------------------------------
   Make a forward inference at depth inside a theorem. Once at the
   occurrence, perform the operation and sew everything back up again 
   on the way out. Needs to be extended to disjunctions.
 ---------------------------------------------------------------------------*)

fun on_thm impth [] th = MATCH_MP impth th
  | on_thm impth (oc::rst) th = 
      let val tm = concl th
      in if (is_conj tm)
         then let val (th1,th2) = (CONJUNCT1 th, CONJUNCT2 th)
              in case oc of 
                  1 => CONJ (on_thm impth rst th1) th2
                | 2 => CONJ th1 (on_thm impth rst th2)
                | _ => raise tac_err ("on_thm:is_conj","bad path")
              end 
         else if (is_neg tm)
              then on_thm (CONV_RULE 
                     (ONCE_DEPTH_CONV CONTRAPOS_CONV) impth) rst th
         else if (is_imp tm)
              then raise tac_err ("on_thm:is_imp","not top level")
         else raise tac_err ("on_thm", "fall through")
      end;


fun AT_CONJ1 tac :tactic = fn (asl,w) =>
  let val {conj1,conj2} = dest_conj w
  in 
   case tac (asl,conj1)
    of ([(_,conj1')], just) =>
       ([(asl, mk_conj{conj1=conj1', conj2=conj2})],
        fn [th] => CONJ (just [CONJUNCT1 th]) (CONJUNCT2 th))
     | _ =>  raise tac_err ("AT_CONJ1", "expected a single (internal) subgoal")
  end;

fun AT_CONJ2 tac :tactic = fn (asl,w) =>
  let val {conj1,conj2} = dest_conj w
  in 
   case tac (asl,conj2)
    of ([(_,conj2')], just) =>
       ([(asl, mk_conj{conj1=conj1, conj2=conj2'})],
        fn [th] => CONJ (CONJUNCT1 th) (just [CONJUNCT2 th]))
     | _ => raise tac_err ("AT_CONJ2", "expected a single (internal) subgoal")
  end;



(*---------------------------------------------------------------------------
         Backchaining at depth in a goal. 
 ---------------------------------------------------------------------------*)

fun on_goal th occs = 
  let fun go [] g = MATCH_MP_TAC th g
        | go (oc::rst) (g as (asl,w)) =
           if is_conj w
           then case oc of
                   1 => AT_CONJ1 (go rst) g
                 | 2 => AT_CONJ2 (go rst) g
                 | _ => raise tac_err ("on_goal:is_conj","bad path")
           (* else if is_disj tm ... *)
           else if is_neg w
                then case oc 
                      of 1 => (DISCH_THEN (fn thm =>
                                 MP_TAC (on_thm th rst thm)
                                 THEN MATCH_MP_TAC F_IMP)) g
                       | _ => raise tac_err ("on_goal:is_neg",
                                       "path should have 1 in this case")
           else if is_imp w
                then raise tac_err ("on_goal:is_imp","not top level")
           else raise tac_err ("on_goal", "fall through")
  in 
    go occs
  end

fun chase th0 (1::occ) = DISCH_THEN (MP_TAC o on_thm th0 occ)
  | chase th0 (2::occ) = DISCH_THEN (fn th => on_goal th0 occ THEN MP_TAC th)
  | chase th0 (_::occ) = raise tac_err("chase", "occ. not in {1,2}")
  | chase th0 []       = MATCH_MP_TAC th0;

fun in_forall tac (g as (_,c)) =
   let val (vars,body) = strip_forall c
   in  (REPEAT GEN_TAC THEN
        itlist (fn v => fn t => t THEN SPEC_TAC (v,v)) vars tac) g
   end;

fun OCC_ANT th0 occs g = in_forall (chase th0 (rev occs)) g;

fun ARW_TAC th occ LEFT  g = OCC_ANT th occ g
  | ARW_TAC th occ RIGHT g = OCC_ANT th occ g

fun ANT_RW_TAC s occ lr g = ARW_TAC (clam_theorem s) occ lr g;

fun WFERT_IMP_TAC (path,dir) = 
   FIRST_ASSUM (fn th => ARW_TAC th path dir THEN WEAKEN_TAC (concl th));

fun WFERT_IMP (path,dir) = 
   let fun pp ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "WFERT_IMP_TAC";
        add_break ppstrm (0,0);
        add_string ppstrm "(";
        begin_block ppstrm INCONSISTENT 2;
          pp_list (C(C add_string o Lib.int_to_string)) ppstrm path;
          add_string ppstrm ","; 
          add_break ppstrm (0,0);
          add_string ppstrm (case dir of LEFT => "LEFT" | RIGHT => "RIGHT"); 
        end_block ppstrm;
        add_string ppstrm ")"; 
       end_block ppstrm)
   in
   (pp, WFERT_IMP_TAC(path,dir))
 end;


fun ARW (name, thm, path, lr) =
   let fun pp ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "ARW_TAC";
        add_break ppstrm (1,0);
        add_string ppstrm name;
        add_break ppstrm (1,0);
        add_string ppstrm "[";
          begin_block ppstrm CONSISTENT 2;
           pr_list (add_string ppstrm o Lib.int_to_string)
                   (fn () => add_string ppstrm ",") 
                   (fn () => add_break ppstrm (0,0)) path;
          end_block ppstrm;
        add_string ppstrm "]"; 
        add_break ppstrm (1,0);
        add_string ppstrm (case lr of LEFT => "LEFT" | RIGHT => "RIGHT");
       end_block ppstrm)
   in
   (pp, ARW_TAC thm path lr)
   end;

fun ANT_RW (thname, path, lr) =
   let fun pp ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "ANT_RW_TAC";
        add_break ppstrm (1,0);
        add_string ppstrm (Lib.quote thname);
        add_break ppstrm (1,0);
        add_string ppstrm "[";
          begin_block ppstrm CONSISTENT 2;
           pr_list (add_string ppstrm o Lib.int_to_string)
                   (fn () => add_string ppstrm ",") 
                   (fn () => add_break ppstrm (0,0)) path;
          end_block ppstrm;
        add_string ppstrm "]"; 
        add_break ppstrm (1,0);
        add_string ppstrm (case lr of LEFT => "LEFT" | RIGHT => "RIGHT");
       end_block ppstrm)
   in
   (pp, ANT_RW_TAC thname path lr)
   end;

(*---------------------------------------------------------------------------*
 *  Support for extended interaction                                         *
 *---------------------------------------------------------------------------*)

(* This code is conservative about which hypotheses to consider. *)
fun DECIDE_GOAL gl =
   TAC_PROOF (gl,decisionLib.DECISION_TAC decisionLib.DECIDE_CONV
                    (can (#Discriminator DecideNum.num_proc)));

fun QDECIDE_GOAL (qs,q) =
   DECIDE_GOAL (map Parse.term_parser qs,Parse.term_parser q);

type vdecider = (ppstream -> unit) * (term list * term -> thm)

fun pp_quoted_term ppstrm tm =
   (add_string ppstrm "`";
    Lib.with_flag Globals.show_types true (pp_term ppstrm) tm;
    add_string ppstrm "`");

fun pp_quoted_goal ppstrm (tms,tm) =
   (begin_block ppstrm CONSISTENT 1;
    add_string ppstrm "([";
    pr_list (pp_quoted_term ppstrm) (fn () => add_string ppstrm ",")
       (fn () => add_break ppstrm (0,1)) tms;
    add_string ppstrm "],";
    add_break ppstrm (0,0);
    pp_quoted_term ppstrm tm;
    add_string ppstrm ")";
    end_block ppstrm);

fun ppPROVED s ppstrm = 
     ( begin_block ppstrm CONSISTENT 0;
       add_string ppstrm "PROVED_TAC"; add_break ppstrm (1,0);
       add_string ppstrm s; end_block ppstrm);

fun ppCONJECTURE gl ppstrm = 
   (begin_block ppstrm CONSISTENT 0;
    add_string ppstrm "CONJECTURE_TAC";
    add_break ppstrm (1,0);
    pp_quoted_goal ppstrm gl;
    end_block ppstrm);

fun ppREFUTED ppstrm = 
    ( begin_block ppstrm CONSISTENT 0;
      add_string ppstrm "FAIL_TAC \"REFUTED!!\""; end_block ppstrm);

fun PROVED_TAC thm = MATCH_ACCEPT_TAC thm;
fun CONJECTURE_TAC M = ALL_TAC;

fun ppFAIL s ppstrm =
    ( begin_block ppstrm CONSISTENT 0;
      add_string ppstrm ("FAIL_TAC \"" ^ s ^ "\""); end_block ppstrm);

fun FAIL s = (ppFAIL s,FAIL_TAC s);

(*---------------------------------------------------------------------------*
 * When at a theorem node in the plan, look up the theorem in the Fact base, *
 * and add it to the reference cell.                                         *
 *---------------------------------------------------------------------------*)
local open Fact
in
fun proved r s = 
  case (get s)
  of SOME(f as (_,Fact.Proved (thm, _))) => (r := (OLD f :: !r); thm)
   | SOME _ => raise tac_err ("proved", "unproved")
   | NONE   => raise tac_err ("proved", "no such fact")
end;

fun PROVED r s = (ppPROVED s, PROVED_TAC (proved r s));

(*---------------------------------------------------------------------------*
 * When at a conjecture, classify it. If it's proved then generate a name    *
 * for it, and attach that with it in the fact dbase. The system will find   * 
 * it on the next go-round.                                                  *
 *---------------------------------------------------------------------------*)

type classifier = term list * term -> string * Fact.fact;

local
   open Fact
in

fun conjecture vtacf classify r M = 
  let val (factoid as (s,fact)) = classify M
  in 
     r := (NEW factoid :: !r);
     vtacf factoid
  end;

fun CONJECTURE classify r M =
   conjecture
      (fn (s,Proved (thm,_)) => (ppPROVED s,     PROVED_TAC thm)
        | (_,CantProve M)    => (ppCONJECTURE M, CONJECTURE_TAC M)
        | (_,Refuted M)      => (ppREFUTED,      FAIL_TAC"REFUTED!!"))
      classify r M;

end;

end; (* functor VisibleTactic *)
