/****************************************************************************/
/*                                                                          */
/* Copyright 1997-1999 University of Cambridge and University of Edinburgh  */
/*                                                                          */
/*                           All rights reserved.                           */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/* FILE          : hol-clam.pl                                              */
/* DESCRIPTION   : File for creating a Clam executable for use with HOL.    */
/*                 (Load into a Clam executable.)                           */
/*                                                                          */
/* WRITES FILES  : hol-clam                                                 */
/*                                                                          */
/* AUTHOR        : R.J.Boulton                                              */
/* DATE          : 4th March 1997                                           */
/*                                                                          */
/* LAST MODIFIED : R.J.Boulton                                              */
/* DATE          : 14th December 1999                                       */
/****************************************************************************/

/* Prolog should not interactively query when predicates are redefined */
:- prolog_flag(redefine_warnings,_,off).

/* Set up HOL types */
:- retractall(boolean_type(_,_,_)),
   asserta(boolean_type(holbool,{true},void)).
:- retractall(natural_type(_,_,_)),
   asserta(natural_type(holnum,hol0,holSUC)).
:- retractall(natural_plus(_)),
   asserta(natural_plus(holPLUS)).
:- retractall(list_type(_,_,_)),
   asserta(list_type(hollist,holNIL,holCONS)).

/* Redefine guess_type/3 to work with functions that haven't been defined in
 * terms of p_ind, etc.
 */
guess_type(True,_,Bool):-
    boolean_type(Bool,True,_), !.
guess_type(False,_,Bool):-
    boolean_type(Bool,_,False), !.
guess_type(Zero,_,Nat):-
    natural_type(Nat,Zero,_), !.
guess_type(App,_,Nat):-
    functor(App,Suc,1),
    natural_type(Nat,_,Suc), !.
guess_type(Nil,_,ListType):-
    list_type(List,Nil,_),
    ListType =.. [List,_], !.
guess_type(App,H,ListType):-
    App =.. [Cons,Hd,Tl],
    list_type(List,_,Cons),
    ((guess_type(Hd,H,ArgType),ListType =.. [List,ArgType]);
     guess_type(Tl,H,ListType)), !.
guess_type(App,_,Type) :-
    functor(App,F,N),
    functor(MetaApp,F,N),
    recorded(theorem,theorem(_,eqn,Eqn,_),_),
    matrix(_,MetaApp=_ in Type,Eqn).
guess_type(X, H,XType) :-
    memberchk(X:XType,H).
guess_type( lambda(V,Term),H, Vt => TermType) :-
    guess_type(Term, [V:Vt|H],TermType).
guess_type( F of Arg,H,CoDomain) :-
    % Account for dependant and independant types
    guess_type(F,  H,FType),
    once((FType = (_:Domain=>CoDomain);
          FType =   (Domain=>CoDomain))),!,
    meta_try guess_type(Arg, H, Domain),
    meta_try guess_type(Arg, H, CoDomain).
guess_type(F of Arg,H,XType) :-
    \+ noprove =: _,
    eval(F of Arg,Y), (F of Arg) \= Y,
    guess_type(Y,  H,XType).

/* Turn warnings back on */
:- prolog_flag(redefine_warnings,_,on).


/* Set up pretty-printing for HOL notation */
:- once(retract(portray(lambda(_,_),_))),
   asserta(portray(lambda(V,T),[tex:[99-['(\lambda ', V, '.\ ', T, ')']],
                                _:[_-['(\', V, '. ', T, ')']]])).
:- once(retract(portray(_#_,_))),
   asserta(portray(T1#T2,[tex:[99-['(', T1, ' \wedge ', T2, ')']],
                          _:[_-['(', T1, ' /\ ', T2, ')']]])).
:- asserta(portray(T1\T2,[tex:[99-['(', T1, ' \vee ', T2, ')']],
                          _:[_-['(', T1, ' \/ ', T2, ')']]])).
:- once(retract(portray(_=>void,_))),
   once(retract(portray(_:_=>_,_))),
   once(retract(portray(_=>_,_))),
   asserta(portray(T1=>T2,[tex:[99-['(', T1, ' \Rightarrow ', T2, ')']],
                           _:[_-['(', T1, ' => ', T2, ')']]])),
   asserta(portray(T=>void,[tex:[99-['\neg ', T]],
                            _:[_-['~', T]]])),
   asserta(portray(V:T1=>T2,[tex:[99-[V, ':', T1, ' \Rightarrow ', T2]],
                             _:[_-[V, ':', T1, ' => ', T2]]])).
:- asserta(portray(T1<=>T2,[tex:[99-['(', T1, '\Leftrightarrow ', T2, ')']],
                            _:[_-['(', T1, ' <=> ', T2, ')']]])).
:- once(retract(portray(T1 = T2 in _Type,L1))),
   L2 = [_:[_-['(', T1, ' = ', T2, ')']]], append(L1,L2,L3),
   asserta(portray(T1 = T2 in Type, L3)).


/* Predicates for `user-defined' pretty-printing in HOL notation */

hol_commas([],[]).
hol_commas([H|T],[',',H|L]):-
    hol_commas(T,L).

hol_format(Name,[T1,T2],infix,['(', T1, ' ', Name, ' ', T2, ')']):- !.
hol_format(Name,[],_,[Name]).
hol_format(Name,[Arg|Args],_,[Name, '(', Arg | Rest]):-
    hol_commas(Args,L),
    append(L,[')'],Rest).

/*
 * Fixity should be either 'prefix' or 'infix'. Otherwise 'prefix' is assumed.
 */
new_syntax([]).
new_syntax([(HOLName-ClamName-Arity-Fixity)|Syntax]):-
    length(Args,Arity),
    ClamForm =.. [ClamName|Args],
    hol_format(HOLName,Args,Fixity,HOLForm),
    assertz(portray(ClamForm,[_:[_-HOLForm]])),
    !, new_syntax(Syntax).

delete_syntax([]).
delete_syntax([(_HOLName-ClamName-Arity-_Fixity)|Syntax]):-
    length(Args,Arity),
    ClamForm =.. [ClamName|Args],
    once(retract(portray(ClamForm,_))),
    !, delete_syntax(Syntax).


/* Introduce new logical objects */

new_scheme(Name,G):-
    ctheorem(Name):=problem([]==>G,_,_,_),cpos(Name):=[],
    record_thm(Name,scheme),
    add_induction_scheme(Name).

new_eqn(Name,G):-
    ctheorem(Name):=problem([]==>G,_,_,_),cpos(Name):=[],
    record_thm(Name,eqn).

new_eqns(Name,G,Names):-
    new_eqns(Name,G,1,Names,_).
new_eqns(Name,G1#G2,M,Names,N):-
    new_eqns(Name,G1,M,Names1,P),
    new_eqns(Name,G2,P,Names2,N),
    append(Names1,Names2,Names).
new_eqns(Name,G,M,[NameM],N):-
    concat_atom([Name,M],NameM),
    new_eqn(NameM,G),
    N is M + 1.

new_def(Name,G):-
    new_def(Name,[],G).
new_def(Name,Syntax,G):-
    new_eqns(Name,G,Names),
    add_rewrite_rules_list(Names),
    add_complementary_sets(Names),
    add_reduction_rules(Names),
    !, new_syntax(Syntax).

new_rule(Name,G):-
    ctheorem(Name):=problem([]==>G,_,_,_),cpos(Name):=[],
    record_thm(Name,wave),
    add_rewrite_rules_list([Name]),
    add_reduction_rules([Name]),
    add_cancel_rule(Name).

new_transitivity_rule(Name,G):-
    ctheorem(Name):=problem([]==>G,_,_,_),cpos(Name):=[],
    record_thm(Name,thm),
    add_transitivity_rule(Name).

new_goal(Name,H==>G):-
    ctheorem(Name):=problem(H==>G,_,_,_),cpos(Name):=[],
    record_thm(Name,thm).


/* Delete logical objects */

delete_scheme(Name):-
    if(lib_present(scheme(Name)),lib_delete(scheme(Name))).

delete_def(Name):-
    lib_delete_aux(Name,1).

delete_rule(Name):-
    if(lib_present(thm(Name)),lib_delete(thm(Name))),
    if(lib_present(wave(Name)),lib_delete(wave(Name))),
    if(lib_present(red(Name)),lib_delete(red(Name))).

delete_transitivity_rule(Name):-
    if(lib_present(trans(Name)),lib_delete(trans(Name))).

delete_goal(Name):-
    if(lib_present(thm(Name)),lib_delete(thm(Name))).

quiet_delete_method(M/A):-
    lib_present(mthd(M/A)),
    delete_method(M/A),
    writef('Deleted method %t\n',[M/A]).
quiet_delete_method(_).

quiet_delete_submethod(M/A):-
    lib_present(smthd(M/A)),
    delete_submethod(M/A),
    writef('Deleted submethod %t\n',[M/A]).
quiet_delete_submethod(_).


/* Calling the planner */

:- dynamic planner_time_limit/1.
planner_time_limit(60).

prove_goal(Name,proof_plan(H==>G,Name,TimeTaken,Plan,Planner)):-
    select(Name),
    planner_time_limit(TimeLimit),
    dplan(TimeLimit,Plan),
    print_plan(Plan),
    dplan:=Plan,
    recorded(proof_plan,proof_plan(H==>G,Name,TimeTaken,Plan,Planner),_).

prove_and_save(Name,File,Plan):-
    prove_goal(Name,Plan),
    tell(File),
    writeq(Plan),write('.'),nl,
    told.

apply_methods(Name,Methods,Choice,
              proof_plan(H==>G,Name,TimeTaken,Plan,Planner)):-
    select(Name),
    planner_time_limit(TimeLimit),
    hyp_list(H), goal(G),
    external_apply_method_choice(TimeLimit,Methods,Plan-TimeTaken,Choice),
    Planner = none,
    print_plan(Plan).

apply_methods_and_save(Name,Methods,Choice,File,Plan):-
    apply_methods(Name,Methods,Choice,Plan),
    tell(File),
    writeq(Plan),write('.'),nl,
    told.

apply_any_method(Name,Choice,Plan):-
    list_methods(Methods),
    apply_methods(Name,Methods,Choice,Plan).

apply_any_method_and_save(Name,Choice,File,Plan):-
    apply_any_method(Name,Choice,Plan),
    tell(File),
    writeq(Plan),write('.'),nl,
    told.

apply_any_submethod(Name,Choice,Plan):-
    list_submethods(Methods),
    apply_methods(Name,Methods,Choice,Plan).

apply_any_submethod_and_save(Name,Choice,File,Plan):-
    apply_any_submethod(Name,Choice,Plan),
    tell(File),
    writeq(Plan),write('.'),nl,
    told.


/* Process a term that represents some operation */
process_term(new_scheme(Name,G),[]):-
    new_scheme(Name,G).
process_term(new_def(Name,Syntax,G),[]):-
    new_def(Name,Syntax,G).
process_term(new_rule(Name,G),[]):-
    new_rule(Name,G).
process_term(new_transitivity_rule(Name,G),[]):-
    new_transitivity_rule(Name,G).
process_term(new_goal(Name,H==>G),[]):-
    new_goal(Name,H==>G).
process_term(new_fact(Name,Type,HG),[]):-
    new_external_fact(Name,Type,HG).
process_term(new_syntax(Syntax),[]):-
    new_syntax([Syntax]).
process_term(delete_scheme(Name),[]):-
    delete_scheme(Name).
process_term(delete_def(Name),[]):-
    delete_def(Name).
process_term(delete_rule(Name),[]):-
    delete_rule(Name).
process_term(delete_transitivity_rule(Name),[]):-
    delete_transitivity_rule(Name).
process_term(delete_goal(Name),[]):-
    delete_goal(Name).
process_term(delete_fact(Name),[]):-
    delete_external_fact(Name).
process_term(delete_all_facts,[]):-
    delete_all_external_facts.
process_term(delete_syntax(Syntax),[]):-
    delete_syntax([Syntax]).
process_term(prove_goal(Name),Plan):-
    prove_goal(Name,Plan).
process_term(prove_and_save(Name,File),Plan):-
    prove_and_save(Name,File,Plan).
process_term(apply_methods(Name,Methods,Choice),Plan):-
    apply_methods(Name,Methods,Choice,Plan).
process_term(apply_methods_and_save(Name,Methods,Choice,File),Plan):-
    apply_methods_and_save(Name,Methods,Choice,File,Plan).
process_term(apply_any_method(Name,Choice),Plan):-
    apply_any_method(Name,Choice,Plan).
process_term(apply_any_method_and_save(Name,Choice,File),Plan):-
    apply_any_method_and_save(Name,Choice,File,Plan).
process_term(apply_any_submethod(Name,Choice),Plan):-
    apply_any_submethod(Name,Choice,Plan).
process_term(apply_any_submethod_and_save(Name,Choice,File),Plan):-
    apply_any_submethod_and_save(Name,Choice,File,Plan).
process_term(method_configuration(Name),[]):-
    load_ind_plan(Name).
process_term(trace_plan(New),Current):-
    trace_plan(Current,New).
process_term(time_limit(New),Current):-
    planner_time_limit(Current),
    retract(planner_time_limit(_)),
    asserta(planner_time_limit(New)).
process_term(iteration(on),[]):-
    load_method(external_decision/1,first),
    load_method(external_lemma/1,first),
    load_submethod(external_wave/3,last).
process_term(iteration(off),[]):-
    quiet_delete_method(external_decision/1),
    quiet_delete_method(external_lemma/1),
    quiet_delete_submethod(external_wave/3).


/* Load new methods */
/*
:- load_method(classic_elem/1,first).
:- load_method(classic_simp/1,last).
*/


/* chop_list_at(+X,+L1,?L2)
 *
 * List L2 is the suffix of L1 from the first occurrence of X.
 * L2 is empty if X does not appear in L1.
 */
chop_list_at(_,[],[]).
chop_list_at(X,[X|T],T):- !.
chop_list_at(X,[_|T],NewT):-
    chop_list_at(X,T,NewT).

/* dirname(+Filename,?Dirname)
 *
 * Like the Unix dirname command.
 * Dirname is the directory component of Filename or '.' if Filename has
 * no directory component. Examples:
 *
 *    dirname('/usr/lib/foo','/usr/lib').
 *    dirname('foo','.').
 */
dirname(Filename,Dirname):-
    atom_chars('/',[Slash]),
    atom_chars(Filename,Chars1),
    reverse(Chars1,Chars2),
    chop_list_at(Slash,Chars2,Chars3),
    reverse(Chars3,Chars4),
    atom_chars(Dirname,Chars4),
    \+ Dirname = '', !.
dirname(_,'.').

/* Get arguments when a normal SICStus process */
server_arguments(Type,Name,TimeOut,Debug,InEmacs):-
    prolog_flag(argv,[Type,Name,TimeOut,Debug,InEmacs]).
/* Get arguments when a saved runtime */
server_arguments(Type,Name,TimeOut,Debug,InEmacs):-
    prolog_flag(argv,[ProcessName,Type,Name,TimeOut,Debug,InEmacs]),
    dirname(ProcessName,Dirname),
    concat(Dirname,'/clam-lib',Library),
    lib_set(dir([Library])).

/* Predicate to execute on start-up */
start:-
    ((server_arguments(Type,Name,TimeOut,Debug,InEmacs),
      atom_chars(TimeOut,Chars), number_chars(Secs,Chars),
/*
      use_module(library(sockets)),
*/
      ((InEmacs = in_emacs) -> push_portray_type(emacs); true),
      server(Type,Name,Secs:0,Debug));
     (write(server_failed),nl)).

/* Save a new Clam executable */
:- save_program('hol-clam',start).
