(****************************************************************************)
(*                                                                          *)
(*    Copyright 1999 University of Cambridge and University of Edinburgh    *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : basic_logic.sml                                          *)
(* DESCRIPTION   : Clam-friendly definitions, etc., for basic logic of HOL. *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 29th January 1999                                        *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 13th October 1999                                        *)
(****************************************************************************)

structure HOLClamBasicLogic =
struct

structure Thms =
struct

val COND_DEF' =
   prove (--`(!b (x:'a) y. b ==> (COND b x y = x)) /\
             (!b (x:'a) y. ~b ==> (COND b x y = y))`--,
          REPEAT STRIP_TAC THEN ASM_REWRITE_TAC []);

val LESS_DEF' =
   prove (--`(!x. (x < 0) = F) /\
             (!y. (0 < SUC y) = T) /\
             (!x y. (SUC x < SUC y) = (x < y))`--,
          REPEAT CONJ_TAC THENL
          [REWRITE_TAC [theorem "prim_rec" "NOT_LESS_0"],
           REWRITE_TAC [theorem "prim_rec" "LESS_0"],
           REWRITE_TAC [theorem "arithmetic" "LESS_MONO_EQ"]]);

val LESS_ZERO = decisionLib.DECIDE (--`!y. (0 < y) = ~(y = 0)`--);

val LESS_EQ_DEF' =
   prove (--`(!y. (0 <= y) = T) /\
             (!x. (SUC x <= 0) = F) /\
             (!x y. (SUC x <= SUC y) = (x <= y))`--,
          REPEAT CONJ_TAC THENL
          [REWRITE_TAC [theorem "arithmetic" "ZERO_LESS_EQ"],
           REWRITE_TAC [theorem "arithmetic" "NOT_SUC_LESS_EQ_0"],
           REWRITE_TAC [theorem "arithmetic" "LESS_EQ_MONO"]]);

val LESS_EQ_ZERO = decisionLib.DECIDE (--`!x. (x <= 0) = (x = 0)`--);

val GREATER_DEF' =
   prove (--`(!x. (0 > x) = F) /\
             (!x. (SUC x > 0) = T) /\
             (!x y. (SUC x > SUC y) = (x > y))`--,
          REWRITE_TAC [definition "arithmetic" "GREATER_DEF",LESS_DEF']);

val GREATER_ZERO = decisionLib.DECIDE (--`!x. (x > 0) = ~(x = 0)`--);

val GREATER_EQ_DEF' =
   prove (--`(!x. (x >= 0) = T) /\
             (!x. (0 >= SUC x) = F) /\
             (!x y. (SUC x >= SUC y) = (x >= y))`--,
          REWRITE_TAC [theorem "arithmetic" "GREATER_EQ",LESS_EQ_DEF']);

val GREATER_EQ_ZERO = decisionLib.DECIDE (--`!y. (0 >= y) = (y = 0)`--);

val num_INDUCTION = theorem "num" "INDUCTION";

val num_num_INDUCT =
   prove (--`!P. (!y. P 0 y) ==> (!x. P x 0) ==>
                 (!x y. P x y ==> P (SUC x) (SUC y)) ==> !x y. P x y`--,
          GEN_TAC THEN REPEAT DISCH_TAC THEN INDUCT_TAC THENL
          [ASM_REWRITE_TAC [],
           INDUCT_TAC THENL
           [ASM_REWRITE_TAC [],
            FIRST_ASSUM MATCH_MP_TAC THEN ASM_REWRITE_TAC []]]);

val NOT_0_SUC =
   (GEN_ALL o EQF_INTRO o SPEC_ALL o GSYM) (theorem "num" "NOT_SUC")
and NOT_SUC_0 =
   (GEN_ALL o EQF_INTRO o SPEC_ALL) (theorem "num" "NOT_SUC");

val SUC_11 = theorem "prim_rec" "INV_SUC_EQ";

val (SUC_ONE_ONE,SUC_FUN) =
   ((GEN_ALL ## GEN_ALL) o EQ_IMP_RULE o SPEC_ALL)
      (theorem "prim_rec" "INV_SUC_EQ");

val list_INDUCT = theorem "list" "list_INDUCT";

val NOT_NIL_CONS =
   (GEN_ALL o EQF_INTRO o SPEC_ALL) (theorem "list" "NOT_NIL_CONS")
and NOT_CONS_NIL =
   (GEN_ALL o EQF_INTRO o SPEC_ALL) (theorem "list" "NOT_CONS_NIL");

val CONS_11 = theorem "list" "CONS_11";

val (CONS_ONE_ONE,CONS_FUN) =
   ((GEN_ALL ## GEN_ALL) o EQ_IMP_RULE o SPEC_ALL) (theorem "list" "CONS_11");

end; (* Thms *)

datatype data = Scheme of string * thm
              | Definition of string * thm
              | Rule of string * thm
              | TransitivityRule of string * thm;

fun send_data (Scheme st) = CallClam.send_scheme st
  | send_data (Definition st) = CallClam.send_definition st
  | send_data (Rule st) = CallClam.send_rule st
  | send_data (TransitivityRule st) = CallClam.send_transitivity_rule st;

fun delete_data (Scheme (s,_)) = CallClam.delete_scheme s
  | delete_data (Definition (s,_)) = CallClam.delete_definition s
  | delete_data (Rule (s,_)) = CallClam.delete_rule s
  | delete_data (TransitivityRule (s,_)) = CallClam.delete_transitivity_rule s;

fun schemes [] = []
  | schemes (Scheme st :: ds) = st :: schemes ds
  | schemes (_ :: ds) = schemes ds;

fun definitions [] = []
  | definitions (Definition st :: ds) = st :: definitions ds
  | definitions (_ :: ds) = definitions ds;

fun rules [] = []
  | rules (Rule st :: ds) = st :: rules ds
  | rules (_ :: ds) = rules ds;

fun transitivity_rules [] = []
  | transitivity_rules (TransitivityRule st :: ds) =
       st :: transitivity_rules ds
  | transitivity_rules (_ :: ds) = transitivity_rules ds;

exception NoData of string;

local

val type_data =
   ref [("num",[Scheme ("num_INDUCTION",Thms.num_INDUCTION),
                Scheme ("num_num_INDUCT",Thms.num_num_INDUCT),
                Rule ("NOT_0_SUC",Thms.NOT_0_SUC),
                Rule ("NOT_SUC_0",Thms.NOT_SUC_0),
                Rule ("SUC_11",Thms.SUC_11),
                Rule ("SUC_FUN",Thms.SUC_FUN),
                Rule ("SUC_ONE_ONE",Thms.SUC_ONE_ONE)]),
        ("list",[Scheme ("list_INDUCT",Thms.list_INDUCT),
                 Rule ("NOT_NIL_CONS",Thms.NOT_NIL_CONS),
                 Rule ("NOT_CONS_NIL",Thms.NOT_CONS_NIL),
                 Rule ("CONS_11",Thms.CONS_11),
                 Rule ("CONS_FUN",Thms.CONS_FUN),
                 Rule ("CONS_ONE_ONE",Thms.CONS_ONE_ONE)])];

val constant_data =
   ref [("COND",[Definition ("COND_DEF'",Thms.COND_DEF')]),
        ("<",[Definition ("LESS_DEF'",Thms.LESS_DEF')]),
        ("<=",[Definition ("LESS_EQ_DEF'",Thms.LESS_EQ_DEF')]),
        (">",[Definition ("GREATER_DEF'",Thms.GREATER_DEF')]),
        (">=",[Definition ("GREATER_EQ_DEF'",Thms.GREATER_EQ_DEF')])];

fun add_data name data_list [] = [(name,data_list)]
  | add_data name data_list ((n,ds)::nds) =
   if (n = name)
   then (n,ds @ data_list) :: nds
   else (n,ds) :: add_data name data_list nds;

fun delete_data name = filter (fn (n,_) => not (n = name));

in

fun all_type_data () = !type_data;

fun all_constant_data () = !constant_data;

fun data_for_type name =
   assoc name (!type_data) handle NOT_FOUND => raise NoData name;

fun data_for_constant name =
   assoc name (!constant_data) handle NOT_FOUND => raise NoData name;

fun add_type_data name data_list =
   type_data := add_data name data_list (!type_data);

fun add_constant_data name data_list =
   constant_data := add_data name data_list (!constant_data);

fun delete_type_data name =
   type_data := delete_data name (!type_data);

fun delete_constant_data name =
   constant_data := delete_data name (!constant_data);

fun send_all () =
   (map (map send_data o snd) (!type_data);
    map (map send_data o snd) (!constant_data);
    ());

end;

end; (* HOLClamBasicLogic *)
