/*
 * @(#)$Id: external.pl,v 1.1.2.8 1999/12/15 11:58:48 rjb Exp $
 *
 * $Log: external.pl,v $
 * Revision 1.1.2.8  1999/12/15 11:58:48  rjb
 * Fixed bug in single-method code and added a time-limit parameter.
 * Plan returned by external_apply predicates now includes the time taken.
 *
 * Revision 1.1.2.7  1999/10/18 16:19:39  rjb
 * Using hypotheses in extended interaction.
 * Weakening restrictions on what hypotheses are sent to an external system.
 *
 * Revision 1.1.2.6  1999/04/27 14:06:57  rjb
 * Support for applying one Clam method at a time.
 *
 * Revision 1.1.2.5  1999/02/11 10:23:55  rjb
 * Bugfix: Wrong part of induction hypotheses being extracted by external_hyps/2.
 *
 * Revision 1.1.2.4  1999/02/05 15:46:40  rjb
 * external_decision now recognizes conditional expressions.
 * Hypotheses now passed to external prover.
 *
 * Revision 1.1.2.3  1998/08/05 11:18:11  rjb
 * Added operation for deleting all external facts.
 *
 * Revision 1.1.2.2  1998/05/22 13:08:16  rjb
 * New method for external lemmas.
 * Corrected matching.
 * Added operation for deleting facts.
 *
 * Revision 1.1.2.1  1998/05/12 16:08:24  rjb
 * New methods for co-operating with external systems.
 *
 */

/* external_fact(?Name,?Type,?Hyps==>?Formula,?Vars,?Matrix)
 */
:- dynamic external_fact/5.

/* new_external_fact/3
 *
 * new_external_fact(+Name,+Type,+Hyps==>+Formula)
 */
new_external_fact(Name,Type,Hyps==>Formula):-
    \+ external_fact(Name,_,_,_,_),
    matrix(Vars,Matrix,Formula),
    asserta(external_fact(Name,Type,Hyps==>Formula,Vars,Matrix)).

/* delete_external_fact/1
 */
delete_external_fact(Name):-
    retract(external_fact(Name,_,_,_,_)).

/* delete_all_external_facts/0
 */
delete_all_external_facts:-
    retractall(external_fact(_,_,_,_,_)).

/* find_external_fact(+Instance,?Name,?Type,?Formula,?Hyps).
 */
find_external_fact(Instance,Name,Type,Formula,Hyps):-
    matrix(_,InstanceMatrix,Instance),
    external_fact(Name,Type,Hyps1==>Formula,Vars,Matrix),
    match_consequent(Matrix,InstanceMatrix,Binding,Hyps2),
    check_binding(Vars,Binding),
    map_list(Hyps1,H:=>IH,instantiated(H,Binding,IH),IHyps1),
    union(IHyps1,Hyps2,Hyps).

/* match_consequent(+Pattern,+Term,?Binding,?Hyps).
 */
match_consequent(Patt,Term,Binding,[]):-
    match_term(Patt,Term,Binding).
match_consequent(H=>Patt,Term,Binding,[Hyp|Hyps]):-
    match_consequent(Patt,Term,Binding,Hyps),
    instantiated(H,Binding,Hyp).

/* check_binding/2.
 *
 * Non-universally quantified variables in the pattern must match to an
 * identical object in the target.
 */
check_binding(_,[]).
check_binding(Vars,[X-X|Binding]):-
    !, check_binding(Vars,Binding).
check_binding(Vars,[X-_|Binding]):-
    member(X:_,Vars),
    !, check_binding(Vars,Binding).

/* external/3.
 */
external(H,G,thm(Name)):-
    find_external_fact(G,Name,proved,_,Hyps),
    external_hyps(H,EH),
    subset(Hyps,EH).
/* Could be relaxed to allow conjectures with the same conclusion as a
   previous one, but different hypotheses. */
external(H,G,conjecture(EH,G)):-
    \+ find_external_fact(G,_,_,_,_),
    external_hyps(H,EH).

/* Tests whether a term is a genvar as generated by genvar/1 in oyster.pl.
 */
is_genvar(X):-
    atom(X),
    atom_chars(X,[118|Z]),
    number_chars(_,Z).

/* external_hyps/2.
 * 
 * Removes annotations from hypotheses and discards hypotheses that can be
 * seen to simply assert the type of a variable. For genvars it is not
 * possible to tell whether the genvar is a hypothesis label or a term
 * variable.
 */
external_hyps([],[]):- !.
external_hyps([ih:[_,(_:H)]|Hs],[H|EHs]):-
    external_hyps(Hs,EHs),
    !.
external_hyps([ch:[(_:H)]|Hs],[H|EHs]):-
    external_hyps(Hs,EHs),
    !.
external_hyps([(V:H)|Hs],[H|EHs]):-
    is_genvar(V),
    external_hyps(Hs,EHs),
    !.
external_hyps([(_:_)|Hs],EHs):-
    external_hyps(Hs,EHs),
    !.
external_hyps([H|Hs],[H|EHs]):-
    external_hyps(Hs,EHs).

/* external_goal/2.
 *
 * Removes all annotations from a goal and discards hypotheses
 * that can be seen to simply assert the type of a variable.
 */
external_goal(H==>G,HH==>GG):-
    external_hyps(H,HH),
    unannotated(G,GG).

/* external_goals/2.
 *
 * Removes all annotations from a list of goals and discards hypotheses
 * that can be seen to simply assert the type of a variable.
 */
external_goals([],[]).
external_goals([AnnHG|AnnR],[HG|R]):-
    external_goal(AnnHG,HG),
    external_goals(AnnR,R).

/* external_condition/3.
 */
external_condition(decidable(hol),_H,G):-
    hol_presburger_formula(G).
external_condition(fail,_,_):-   % Failure made explicit for clarity.
    !, fail.


hol_presburger_formula(_:_=>F):-
    hol_presburger_formula(F).
hol_presburger_formula(F1#F2):-
    hol_presburger_formula(F1),
    hol_presburger_formula(F2).
hol_presburger_formula(F1\F2):-
    hol_presburger_formula(F1),
    hol_presburger_formula(F2).
hol_presburger_formula(F=>void):-
    hol_presburger_formula(F).
hol_presburger_formula(F1=>F2):-
    hol_presburger_formula(F1),
    hol_presburger_formula(F2).
hol_presburger_formula(F1<=>F2):-
    hol_presburger_formula(F1),
    hol_presburger_formula(F2).
hol_presburger_formula(T1=T2 in holnum):-
    hol_presburger_term(T1),
    hol_presburger_term(T2).
hol_presburger_formula(holCOND(F1,F2,F3)):-
    hol_presburger_formula(F1),
    hol_presburger_formula(F2),
    hol_presburger_formula(F3).
hol_presburger_formula(holCOND(F,T1,T2)):-
    hol_presburger_formula(F),
    hol_presburger_term(T1),
    hol_presburger_term(T2).
hol_presburger_formula(holLESS(T1,T2)):-
    hol_presburger_term(T1),
    hol_presburger_term(T2).
hol_presburger_formula(holLESSEQ(T1,T2)):-
    hol_presburger_term(T1),
    hol_presburger_term(T2).
hol_presburger_formula(holGREAT(T1,T2)):-
    hol_presburger_term(T1),
    hol_presburger_term(T2).
hol_presburger_formula(holGREATEQ(T1,T2)):-
    hol_presburger_term(T1),
    hol_presburger_term(T2).
hol_presburger_formula({true}).
hol_presburger_formula(void).

hol_presburger_term(holSUC(T)):-
    hol_presburger_term(T).
hol_presburger_term(holPRE(T)):-
    hol_presburger_term(T).
hol_presburger_term(holPLUS(T1,T2)):-
    hol_presburger_term(T1),
    hol_presburger_term(T2).
hol_presburger_term(holMINUS(T1,T2)):-
    hol_presburger_term(T1),
    hol_presburger_term(T2).
hol_presburger_term(holTIMES(T1,T2)):-
    hol_presburger_const_struct(T1),
    hol_presburger_term(T2).
hol_presburger_term(holTIMES(T1,T2)):-
    hol_presburger_term(T1),
    hol_presburger_const_struct(T2).
hol_presburger_term(holCOND(F,T1,T2)):-
    hol_presburger_formula(F),
    hol_presburger_term(T1),
    hol_presburger_term(T2).
hol_presburger_term(T):-
    atom(T).

hol_presburger_const_struct(T):-
    hol_num_const(T).
hol_presburger_const_struct(holSUC(T)):-
    hol_presburger_const_struct(T).
hol_presburger_const_struct(holPRE(T)):-
    hol_presburger_const_struct(T).
hol_presburger_const_struct(holPLUS(T1,T2)):-
    hol_presburger_const_struct(T1),
    hol_presburger_const_struct(T2).
hol_presburger_const_struct(holMINUS(T1,T2)):-
    hol_presburger_const_struct(T1),
    hol_presburger_const_struct(T2).
hol_presburger_const_struct(holTIMES(T1,T2)):-
    hol_presburger_const_struct(T1),
    hol_presburger_const_struct(T2).
hol_presburger_formula(holCOND(F,T1,T2)):-
    hol_presburger_formula(F),
    hol_presburger_const_struct(T1),
    hol_presburger_const_struct(T2).

hol_num_const(T):-
    atom(T),
    atom_chars(T,[104,111,108|Cs]),
    number_chars(N,Cs),
    number(N).


/* external_apply_any_method(?TimeLimit,?Method,?Plan).
 */
external_apply_any_method(TL,Method,Plan):-
    hyp_list(H), goal(G),
    external_apply_any_method(TL,H==>G,Method,Plan).
/*
 * external_apply_any_method(?TimeLimit,+Goal,?Method,?Plan) looks for any
 * method or submethod that is applicable to the current goal. ?Method may be
 * instantiated to restrict the choices. The resulting Plan treats unsolved
 * subgoals as applications of the external method (conjectures).
 */
external_apply_any_method(TL,Goal,Method,Plan-TimeTaken):-
    statistics(runtime,[MilliSecs|_]),
    if(\+ var(TL), EndTime is TL * 1000 + MilliSecs),
    retractall(end_time(_)),
    assert(end_time(EndTime)),
    statistics(runtime,[StartMS|_]),
    on_exception(time_limit_exceeded,
                 applicable_anymethod(Goal,Method,_,AnnSubGoals),
                 (clam_warning('Time limit of %ts exceeded.',[TL]),fail)),
    statistics(runtime,[EndMS|_]),
    TimeTaken is EndMS - StartMS,
    clam_info('Time taken for method is %t milliseconds.\n',[TimeTaken]),
    external_goals(AnnSubGoals,SubGoals),
    map_list(SubGoals,(H==>G):=>external([],[],conjecture(H,G)),Conjectures),
    ((Conjectures = []) -> Plan = Method; Plan = (Method then Conjectures)).

/* external_apply_any_method_choice(?TimeLimit,?Method,?Plan,+N).
 */
external_apply_any_method_choice(TL,Method,Plan,N):-
    hyp_list(H), goal(G),
    external_apply_any_method_choice(TL,H==>G,Method,Plan,N).
/*
 * external_apply_any_method_choice(?TimeLimit,+Goal,?Method,?Plan,+N) behaves
 * like external_apply_any_method/4 except that the Nth (counting from 0)
 * choice under backtracking is selected.
 */
external_apply_any_method_choice(TL,Goal,Method,Plan,N):-
    number(N),
    nth_choice(external_apply_any_method(TL,Goal,Method,Plan),N).

/* external_apply_one_method(?TimeLimit,+Goal,+MethodName,?MethodArity,?Plan)
 * behaves like external_apply_any_method/4 with the method argument
 * instantiated to MethodName(...). +MethodName should be an atom and
 * leaving ?MethodArity uninstantiated is not recommended but might be
 * useful in some circumstances.
 */
external_apply_one_method(TL,Goal,MethodName,MethodArity,Plan):-
    atom(MethodName),
    length(Ms,MethodArity),
    Method =.. [MethodName|Ms],
    external_apply_any_method(TL,Goal,Method,Plan).

/* external_apply_method(?TimeLimit,+MethodNamesAndArities,?Plan).
 */
external_apply_method(TL,MethodNamesAndArities,Plan):-
    hyp_list(H), goal(G),
    external_apply_method(TL,H==>G,MethodNamesAndArities,Plan).
/*
 * external_apply_method(+TimeLimit,+Goal,+MethodNamesAndArities,?Plan) behaves
 * like external_apply_any_method/4 with the method argument instantiated
 * according to one of the elements of the +MethodNamesAndArities list.
 * An example of the list is [ind_strat/1,generalise/2]. The methods are
 * tried in the order of the list.
 */
external_apply_method(_TL,_Goal,MethodNamesAndArities,_Plan):-
    var(MethodNamesAndArities), !, fail.
external_apply_method(TL,Goal,[(Name/Arity)|_Others],Plan):-
    external_apply_one_method(TL,Goal,Name,Arity,Plan).
external_apply_method(TL,Goal,[_|Others],Plan):-
    external_apply_method(TL,Goal,Others,Plan).

/* external_apply_method_choice(?TimeLimit,+MethodNamesAndArities,?Plan,+N).
 */
external_apply_method_choice(TL,MethodNamesAndArities,Plan,N):-
    hyp_list(H), goal(G),
    external_apply_method_choice(TL,H==>G,MethodNamesAndArities,Plan,N).
/*
 * external_apply_method_choice(?TimeLimit,+Goal,
 *                              +MethodNamesAndArities,?Plan,+N)
 * behaves like external_apply_method/4 except that the Nth (counting from 0)
 * choice under backtracking is selected.
 */
external_apply_method_choice(TL,Goal,MethodNamesAndArities,Plan,N):-
    number(N),
    nth_choice(external_apply_method(TL,Goal,MethodNamesAndArities,Plan),N).


/* nth_choice/2.
 *
 * nth_choice(+G,+N) finds the Nth successful call of the goal G under
 * backtracking (counting from 0). So, nth_choice(G,0) is equivalent to
 * call(G) except that nth_choice won't backtrack. All the calls upto the
 * Nth are executed but later calls are not. This is useful if there are
 * an infinite (or large) number of choices. Without the final cut
 * duplicate solutions are obtained on backtracking.
 */
:- dynamic(nth_choice_counter/1).
nth_choice(G,N):-
    retractall(nth_choice_counter(_)),
    asserta(nth_choice_counter(0)),
    if(call(G),
       (nth_choice_counter(I),
        J is I + 1,
        asserta(nth_choice_counter(J))),
       fail),
    J > N, !.
