/****************************************************************************/
/*                                                                          */
/* Copyright 1997, 1998 University of Cambridge and University of Edinburgh */
/*                                                                          */
/*                           All rights reserved.                           */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/* FILE          : socket_server.pl                                         */
/* DESCRIPTION   : A server for calling Prolog goals through sockets.       */
/*                                                                          */
/* AUTHOR        : R.J.Boulton                                              */
/* DATE          : 21st March 1997                                          */
/*                                                                          */
/* LAST MODIFIED : R.J.Boulton                                              */
/* DATE          : 31st July 1998                                           */
/****************************************************************************/

/* Load this file using consult/1 */
/* Assumes the predicate process_term/1 has been defined to call terms. */

/* Required by SICStus 3
:- ensure_loaded(library(sockets)).
*/

/* Dynamic predicates for information required to restart the server. */
:- dynamic server_debug/1.
:- dynamic server_stream/1.

process_data(Data,Result,_):-
    process_term(Data,Result), !.
process_data(interrupt,_,_):-
    abort.
process_data(quit,_,_):-
    halt.
process_data(Term,succeeded,debug):-
    on_exception(_,call(Term),fail), !.
process_data(_,failed,_):-
    write('!!! Command failed.'), nl.

/*
create_socket(unix,File,Socket):-
    socket('AF_UNIX',Socket),
    socket_bind(Socket,'AF_UNIX'(File)).
create_socket(inet,File,Socket):-
    socket('AF_INET',Socket),
    socket_bind(Socket,'AF_INET'(Host,Port)),
    tell(File), write(Host), nl, write(Port), nl, flush_output, told.
create_socket(_,_,_):-
    write('invalid socket type; use "unix" or "inet".'), nl,
    halt.

wait_for_connection(Socket,Stream,TimeOut):-
    socket_select(Socket,Stream,TimeOut,[],_),
    % Check that Stream is instantiated, i.e. that there was no time-out.
    nonvar(Stream), !.
wait_for_connection(_,_,_):-
    write('timed out.'), nl,
    halt.
*/

/* Instantiate TimeOut to `off' for no time-out. */
/* Instantiate Debug to `debug' to allow arbitrary predicates to be called. */
server(Type,Name,TimeOut,Debug):-
/*
    create_socket(Type,Name,Socket),
    socket_listen(Socket,1),
    nl, write('Server started. Waiting for connection ... '), flush_output,
    wait_for_connection(Socket,Stream,TimeOut),
    write('connection established.'), nl,
*/
    retractall(server_debug(_)),
    asserta(server_debug(Debug)),
    start_server(Type,Name,TimeOut,Stream),
    retractall(server_stream(_)),
    asserta(server_stream(Stream)),
    server_loop(Stream,Debug).

server_loop(Stream,Debug):-
    write('Waiting for commands ...'), nl, nl, flush_output,
    repeat,
        on_exception(_,read(Stream,Term),Term=syntax_error),
        write('| ?- '), writeq(Term), write('.'), nl,
        process_data(Term,Result,Debug),
        nl,
        handshake(Stream,Result),
    fail.

handshake(Stream,Result):-
    write(Stream,Result), write(Stream,'.'), flush_output(Stream).
handshake(Result):-
    server_stream(Stream),
    handshake(Stream,Result).
handshake:-
    handshake(failed).

restart_server:-
    server_debug(Debug),
    server_stream(Stream),
    server_loop(Stream,Debug).

reset_server:-
    handshake,
    restart_server.

/* Wait for 5 minutes for a connection. */
server(Type,Name,Debug):-
    server(Type,Name,300:0,Debug).

/* The default is to use local Unix sockets, no debugging. */
server(Name):-
    server(unix,Name,no_debug).
