(****************************************************************************)
(*                                                                          *)
(* Copyright 1997-1999 University of Cambridge and University of Edinburgh  *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : load.sml                                                 *)
(* DESCRIPTION   : Main file for HOL/Clam system.                           *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 23rd January 1997                                        *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 27th April 1999                                          *)
(****************************************************************************)

load_library_in_place decision_lib;
load_library_in_place taut_lib;

use "Q.sig"; use "Q.sml";

val list_induct = prove(Parse.term_parser
`!P:'a list -> bool. P[] /\ (!h t. P t ==> P (CONS h t)) ==> !l. P l`,
GEN_TAC THEN STRIP_TAC
   THEN MATCH_MP_TAC (theorem "list" "list_INDUCT")
   THEN ASM_REWRITE_TAC[]
   THEN REPEAT STRIP_TAC THEN FIRST_ASSUM MATCH_MP_TAC
   THEN ASM_REWRITE_TAC[]);

(* Bring in CLaReT-generated stuff. *)
structure HOL_Portable = Portable;
val concat = String.concat;
val mlsyn_dir = "claret";
use (mlsyn_dir ^ "/runtime/mlsyn_runtime.sml");
use "clam.ast.sml";
nonfix THEN;
use "clam.parse.sml";
infix THEN;
use "clam.print.sml";
val concat = Lib.concat;
structure CLaReTPortable = Portable;
structure Portable = HOL_Portable;

use "socket_server.sml";
use "hol_to_clam.sml";
use"Btree.sig"; use "Btree.sml";
use"Fact.sig";  use"Fact.sml";
use"vistactic.sig";   use"vistactic.sml";
use"clam_to_hol.sig"; use"clam_to_hol.sml";
use"clam2client.sig"; 
use"plan2tactic.sig"; use"plan2tactic.sml"; 
use "database.sml";
use"call_clam.sml";
use"ei.sig"; use "ei.sml";
use "basic_logic.sml";
use "definitions.sml";
use "schemes.sml";

(*---------------------------------------------------------------------------*
 * Generate a tactic from a tacticAST                                        *
 *---------------------------------------------------------------------------*)

local open CallClam.Plan2Tactic
     nonfix THEN THENL
    fun AST2Tactic NO_TAC         = Tactical.NO_TAC
      | AST2Tactic ALL_TAC        = Tactical.ALL_TAC
      | AST2Tactic (PRIM p)       = VisibleTactic.tactic_of p
      | AST2Tactic (THEN (t1,t2)) = Tactical.THEN(AST2Tactic t1,AST2Tactic t2)
      | AST2Tactic (THENL(t,L))   = Tactical.THENL(AST2Tactic t, 
                                                   map AST2Tactic L)
in
  val CLAM_TAC :tactic = fn g => 
        (Definitions.send_definitions g;
         Schemes.send_schemes g;
         CallClam.CLAM_TAC0 CallClam.prove_goal
            (EI.std_classifier()) AST2Tactic g);

  fun ALT_CLAM_TAC choice :tactic = fn g =>
        (Definitions.send_definitions g;
         Schemes.send_schemes g;
         CallClam.CLAM_TAC0 (fn s => CallClam.apply_any_method s choice)
            (EI.std_classifier()) AST2Tactic g);

  val ONCE_CLAM_TAC = ALT_CLAM_TAC 0;

  fun ALT_CLAM_INDUCT_TAC choice :tactic = fn g =>
        (Definitions.send_definitions g;
         Schemes.send_schemes g;
         CallClam.CLAM_TAC0
            (fn s => CallClam.apply_methods s ["induction/1"] choice)
            (EI.std_classifier()) AST2Tactic g);

  val CLAM_INDUCT_TAC = ALT_CLAM_INDUCT_TAC 0;

  val EI_TAC = EI.EI (AST2Tactic o fst)

  (*-------------------------------------------------------------------------*
   * The `EI.alwaysFail' classifier always says "CantProve"; it should       *
   * thus force Clam to act the old way.                                     *
   *-------------------------------------------------------------------------*)
  val OLD_CLAM_TAC :tactic = EI.EI0 EI.alwaysFail (AST2Tactic o fst)
end;

(*---------------------------------------------------------------------------*
 * Generate concrete syntax for tactics                                      *
 *---------------------------------------------------------------------------*)
fun with_ppstream ppstrm = 
  let open PrettyPrint
  in
          {add_break=add_break ppstrm, 
         add_string=add_string ppstrm, 
        begin_block=begin_block ppstrm, 
          end_block=fn () => end_block ppstrm, 
        add_newline=fn () => add_newline ppstrm, 
     clear_ppstream=fn () => clear_ppstream ppstrm, 
     flush_ppstream=fn () => flush_ppstream ppstrm}
  end;


fun pr_list pfun dfun bfun =
  let fun pr [] = ()
        | pr [i] = pfun i
        | pr (i::rst) = ( pfun i ; dfun() ; bfun() ; pr rst )
  in  pr   end;


fun pp_tacticAST ppstrm = 
  let nonfix THEN THENL
      open CallClam.Plan2Tactic
      val {add_break, add_newline, add_string,
           begin_block, end_block, ...} = with_ppstream ppstrm
      val CB = PrettyPrint.CONSISTENT
      val IB = PrettyPrint.INCONSISTENT
      fun pp NO_TAC        = add_string"NO_TAC"
        | pp ALL_TAC       = add_string"ALL_TAC"
        | pp (PRIM p)      = VisibleTactic.pp_of p ppstrm
        | pp (THEN(t1,t2)) = (begin_block IB 0;
                              pp t1; add_break (1,0);
                              add_string "THEN"; add_break(1,0);
                              pp t2;
                              end_block())
        | pp (THENL(t,L)) = 
            (begin_block IB 0;
             pp t; add_break (1,0);
             add_string "THENL"; add_break(1,0);
             add_string"[";
               begin_block CB 0;
                    pr_list pp (fn () => add_string",") 
                               (fn () => add_break(0,0)) L;
               end_block();
             add_string"]";
             end_block())
  in fn x => (begin_block CB 0; pp x; end_block())
  end;

(*---------------------------------------------------------------------------*
 * Shows the tactic arising from the plan.                                   *
 *---------------------------------------------------------------------------*)
fun tac_of0 f () = 
  let val (tacAST,facts) = f () (top_goal())
      open PrettyPrint
      val ppstrm = mk_ppstream
             {consumer = fn s => Portable.output (Portable.std_out,s),
              flush = fn () => Portable.flush_out Portable.std_out,
              linewidth = 79}
  in
   (case facts
     of [] => pp_tacticAST ppstrm tacAST
      |  _ => let val {add_break, add_newline, add_string,
             begin_block, end_block,...} = with_ppstream ppstrm
           in
              begin_block CONSISTENT 0;
              add_string "let"; add_break (1,0);
              begin_block CONSISTENT 0;
              pr_list 
                (fn (s,Fact.Proved(_,f)) =>
                       ( begin_block INCONSISTENT 2;
                         add_string "val"; add_break (1,0);
                         add_string  s;    add_break (1,0);
                         add_string "=";   add_break (1,0);
                         f ppstrm; end_block())
                    | _ => ())
                (fn () => ())
                (fn () => add_break(0,0)) (rev facts);
              end_block(); 
              add_break (1,0);
              add_string "in"; add_break(1,2); pp_tacticAST ppstrm tacAST;
              add_break(1,0); add_string "end"; 
              end_block()
            end);
      flush_ppstream ppstrm;
      Portable.output(Portable.std_out,"\n")
     end;

(* Analogue of CLAM_TAC -- prints out tactic in ML *)
val tac_of = 
  let val g = CallClam.CLAM_TAC_AST0 CallClam.prove_goal o EI.std_classifier
      fun h x = (I ## map Fact.dest_age) o g x
  in tac_of0 h
  end;

(* Analogue of OLD_CLAM_TAC -- prints out tactic in ML *)
val old_tac_of = 
  let fun g0 x y = EI.EI0 EI.alwaysFail x y
      fun h () = g0 K
  in tac_of0 h
  end;

(* Analogue of EI_TAC -- prints out tactic in ML *)
val ei_tac_of = 
  let fun g0 x y = EI.EI0 EI.std_classifier x y
      fun h () = g0 K
  in tac_of0 h
  end;

fun plan_of () = CallClam.CLAM_PLAN CallClam.prove_goal (top_goal());

open CallClam;
open VisibleTactic;

fun SPEC_ALL_THEN tac (h,g) =
   foldr (fn (v,tac) => Tactic.SPEC_TAC (v,v) THEN tac)
      tac (free_vars g) (h,g);

val _ = Compiler.PPTable.install_pp 
          ["CallClam","Plan2Tactic","tacticAST"]
           pp_tacticAST;

val _ = Compiler.PPTable.install_pp  
          ["ClamAST","plan"]
           (Lib.C (ClamPP.PP.PPBoxes.write_ppbox 
                   o ClamPP.print_plan DefaultStringTable.string_table));

(* 
val _ = Compiler.PPTable.install_pp 
          ["ClamAST","method"]
           (Lib.C (ClamPP.PP.PPBoxes.write_ppbox 
                   o ClamPP.print_method DefaultStringTable.string_table));

val _ = Compiler.PPTable.install_pp  
          ["ClamAST","argument"]
           (Lib.C (ClamPP.PP.PPBoxes.write_ppbox 
                   o ClamPP.print_argument DefaultStringTable.string_table));
*)
