(****************************************************************************)
(*                                                                          *)
(*               Copyright 1994, 1995 University of Cambridge               *)
(*               Copyright 1997, 1998 University of Edinburgh               *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : lex_support.sml                                          *)
(* DESCRIPTION   : Support code for lexical analysis.                       *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 24th May 1994                                            *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 17th February 1998                                       *)
(****************************************************************************)

structure LexSupport =
struct

local

fun val_of_digits [] = raise Fail "val_of_digits"
  | val_of_digits cs =
   let fun val_of_digit c =
          if (#"0" <= c andalso c <= #"9")
          then ord c - ord #"0"
          else raise Fail "val_of_digits"
   in  foldl (fn (c,n) => val_of_digit c + 10 * n) 0 cs
   end;

fun digits_of_val 0 = ""
  | digits_of_val i =
   digits_of_val (i div 10) ^ String.str (chr (ord(#"0") + (i mod 10)));

in

fun string_of_string (s:string) = s;

fun character_of_string s =
   case (explode s)
   of [#"'",c,#"'"] => c
    | _ => raise Fail "character_of_string";

fun boolean_of_string "true" = true
  | boolean_of_string "false" = false
  | boolean_of_string _ = raise Fail "boolean_of_string";

fun natural_of_string s =
   Natural.nat_of_int (val_of_digits (explode s))
   handle _ => raise Fail "natural_of_string";

fun integer_of_string s =
   (case explode s
    of [] => raise Fail ""
     | (#"-"::cs) => ~(val_of_digits cs)
     | (#"~"::cs) => ~(val_of_digits cs)
     | cs => val_of_digits cs)
   handle _ => raise Fail "integer_of_string";

fun rational_of_string s = raise Fail "rational_of_string";

fun real_of_string s = raise Fail "real_of_string";

fun string_of_character c = "'" ^ String.str c ^ "'";

fun string_of_boolean true = "true"
  | string_of_boolean false = "false";

fun string_of_natural n =
   let val i = Natural.int_of_nat n
   in  if (i = 0)
       then "0"
       else digits_of_val i
   end;

fun string_of_integer i =
   if (i = 0)
   then "0"
   else if (i < 0)
        then "~" ^ digits_of_val (~i)
        else digits_of_val i;

fun string_of_rational r = raise Fail "string_of_rational";

fun string_of_real r = raise Fail "string_of_real";

fun double (right,s) =
   let val rsize = size right
       fun double_right s =
          let val ssize = size s
          in  if (ssize < rsize) orelse (ssize = 0)
              then s
              else if (substring (s,0,rsize) = right)
                   then right ^ right ^
                        (double_right (substring (s,rsize,ssize - rsize)))
                   else substring (s,0,1) ^
                        (double_right (substring (s,1,ssize - 1)))
          end
   in  double_right s
   end;

fun escape (esc,right,s) =
   let val esize = size esc
       and rsize = size right
       fun escape_esc_right s =
          let val ssize = size s
          in  if (ssize = 0) then s
              else if (ssize >= esize) andalso (substring (s,0,esize) = esc)
                   then esc ^ esc ^
                        (escape_esc_right (substring (s,esize,ssize - esize)))
              else if (ssize >= rsize) andalso (substring (s,0,rsize) = right)
                   then esc ^ right ^
                        (escape_esc_right (substring (s,rsize,ssize - rsize)))
              else substring (s,0,1) ^
                   (escape_esc_right (substring (s,1,ssize - 1)))
          end
   in  escape_esc_right s
   end;

end;

end; (* LexSupport *)
