(****************************************************************************)
(*                                                                          *)
(* Copyright 1997-1999 University of Cambridge and University of Edinburgh  *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : call_clam.sml                                            *)
(* DESCRIPTION   : Calling Clam to prove a theorem.                         *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 27th January 1997                                        *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 4th May 1999                                             *)
(****************************************************************************)

structure CallClam =
struct

exception CallClam of string;

local

open ClamDatabase;

fun error s = (Lib.say (s ^ "\n"); raise CallClam s);

structure PPStream = ClamPP.PP.PPBoxes.PPStream;

datatype connection = FilePrefix of string
                    | Socket of SocketServer.socket * PrettyPrint.ppstream;

val clam_process = ref (NONE:(string * connection) option);

fun clam_connection () =
   (case (!clam_process)
    of NONE => error "No Clam process"
     | SOME c => c)

fun check_connected () = (clam_connection (); ());

fun clam_socket () = case (clam_connection ()) of (_,Socket x) => x;

val plan_file = ref (NONE:string option);

fun output (os,s) = TextIO.outputSubstr (os,Substring.all s);

val system = Portable.system;

fun mk_ppstream_from_socket socket =
   PPStream.mk_ppstream
      {consumer = fn s => (SocketServer.send_string (socket,s); ()),
       flush = (fn () => ()),
       linewidth = 79};

fun write_string ppstream s =
   let fun trim s = let val c =
                           substring (s,size s - 1,1) handle Subscript => ""
                    in  if (c = ".") orelse (c = "\n")
                        then trim (substring (s,0,size s - 1))
                        else s
                    end
       val consumer = #consumer (PPStream.dest_ppstream ppstream)
   in  consumer (trim s ^ ".\n\n")
   end;

fun comma [] = ""
  | comma (s::ss) = s ^ String.concat (map (fn s => "," ^ s) ss);

fun write_simple_prolog ppstream (predicate,strings) =
   let val args = if (null strings) then "" else "(" ^ comma strings ^ ")"
   in  write_string ppstream (predicate ^ args ^ "\n")
   end;

fun write_prolog ppstream (predicate,strings,dataf) =
   let val consumer = #consumer (PPStream.dest_ppstream ppstream)
   in  consumer (predicate ^ "(" ^
                 String.concat (map (fn s => s ^ ",") strings) ^ "\n");
       dataf ppstream;
       PPStream.flush_ppstream ppstream;
       consumer ").\n\n"
   end;

fun write_data ppstream (Scheme (_,{clam_name,term})) =
   write_prolog ppstream
      ("new_scheme",[clam_name],
       ClamPP.print_typ DefaultStringTable.string_table term)
  | write_data ppstream (Definition (_,{clam_name,term})) =
   write_prolog ppstream
      ("new_def",[clam_name],
       ClamPP.print_typ DefaultStringTable.string_table term)
  | write_data ppstream (Rule (_,{clam_name,term})) =
   write_prolog ppstream
      ("new_rule",[clam_name],
       ClamPP.print_typ DefaultStringTable.string_table term)
  | write_data ppstream (Trans (_,{clam_name,term})) =
   write_prolog ppstream
      ("new_transitivity_rule",[clam_name],
       ClamPP.print_typ DefaultStringTable.string_table term)
  | write_data ppstream (Goal (_,{clam_name,goal})) =
   write_prolog ppstream
      ("new_goal",[clam_name],
       ClamPP.print_goal DefaultStringTable.string_table goal)
  | write_data ppstream (Fact ({hol_name,fact},{clam_name,term})) =
   write_prolog ppstream
      ("new_fact",[clam_name,
                   case fact of Fact.Proved _    => "proved"
                              | Fact.CantProve _ => "cant_prove"
                              | Fact.Refuted _   => "refuted"],
       ClamPP.print_typ DefaultStringTable.string_table term);

fun wait_on_socket socket =
   let val s = SocketServer.receive_string socket
       val ss = if ((substring (s,size s - 1,1) = ".") handle _ => false)
                then [s]
                else s :: wait_on_socket socket
   in  if (ss = ["failed."])
       then raise CallClam "Remote failure"
       else ss
   end;

fun write_clam_socket data =
   (case (!clam_process)
    of SOME (_,Socket (socket,pps)) =>
          (write_data pps data; wait_on_socket socket; ())
     | _ => ());

fun write_clam_input (source,target) (operation,args) =
   let val os = TextIO.openOut source
       val pps =
          PPStream.mk_ppstream
             {consumer = (fn s => output (os,s)),
              flush = (fn () => ()),
              linewidth = 79}
   in  map (write_data pps) (clam_data ());
       write_simple_prolog pps (operation,args @ ["'" ^ target ^ "'"]);
       TextIO.closeOut os
   end;

fun read_clam_output filename =
   ClamParse.parse_file filename handle _ => error "No plan found";

fun parse_plan ss =
   ClamParse.parse_strings {line_number = 0} ss
   handle _ => error "Badly formed plan";

val clam_name = HOLtoClam.translate_metalanguage_name;

in

fun call_clam (operation,op_with_save) name args =
   let val (clam,connection) = clam_connection ()
       val (gl,translations) = find_goal name
   in  case connection
       of FilePrefix file_prefix =>
             let val source = file_prefix ^ ".source"
                 and target = file_prefix ^ ".target"
             in  (write_clam_input (source,target)
                     (op_with_save,clam_name name :: args);
                  system ("rm " ^ target ^ " 2>/dev/null");
                  system (clam ^ " <" ^ source);
                  let val plan = read_clam_output target
                  in  system ("rm " ^ source ^ " " ^ target);
                      (plan, (translations:HOLtoClam.translations))
                  end)
                 handle e =>
                 (system ("rm -f " ^ source ^ " " ^ target); Raise e)
             end
        | Socket (socket,pps) =>
             (case (!plan_file)
              of NONE => ()
               | SOME target => (system ("rm " ^ target ^ " 2>/dev/null"); ());
              if (clam = "") (* Clam process is not local *)
              then (write_simple_prolog pps (operation,clam_name name :: args);
                    let val plan = parse_plan (wait_on_socket socket)
                    in  case (!plan_file)
                        of NONE => ()
                         | SOME target =>
                              ClamPrint.print_to_file target 79 plan;
                        (plan,(translations:HOLtoClam.translations))
                    end)
              else (case (!plan_file)
                    of NONE => write_simple_prolog pps
                                  (operation,clam_name name :: args)
                     | SOME target =>
                          write_simple_prolog pps
                             (op_with_save,
                              clam_name name :: args @ ["'" ^ target ^ "'"]);
                    let val plan = parse_plan (wait_on_socket socket)
                    in  (plan,(translations:HOLtoClam.translations))
                    end))
   end;

fun start_clam {clam} =
   (case (!clam_process)
    of SOME _ => error "Clam already started"
     | NONE =>
       let val socket = SocketServer.start_server clam
           val pps = mk_ppstream_from_socket socket
       in  reset_data (); clam_process := SOME (clam,Socket (socket,pps))
       end
       handle _ =>
       let val pid = Word32.toString
                        (Posix.Process.pidToWord (Posix.ProcEnv.getpid ()))
           val file_prefix = "/tmp/clam." ^ pid
       in  reset_data ();
           clam_process := SOME (clam,FilePrefix file_prefix)
       end);

fun end_clam () =
   (case (clam_connection ())
    of (_,FilePrefix _) => clam_process := NONE
     | (_,Socket (socket,pps)) =>
          (clam_process := NONE;
           PPStream.flush_ppstream pps;
           PPStream.clear_ppstream pps;
           SocketServer.send_string (socket,"quit.");
           SocketServer.close socket));

fun interrupt_clam () =
   (case (clam_connection ())
    of (_,FilePrefix _) => error "Can't interrupt Clam process"
     | (_,Socket (socket,pps)) =>
          (PPStream.flush_ppstream pps;
           PPStream.clear_ppstream pps;
           write_simple_prolog pps ("interrupt",[])));

fun connect_to_local_clam {socket_file} =
   (case (!clam_process)
    of SOME _ => error "Clam already connected"
     | NONE =>
       let val socket = SocketServer.connect_to_local_server socket_file
           val pps = mk_ppstream_from_socket socket
       in  reset_data (); clam_process := SOME ("",Socket (socket,pps))
       end);

fun connect_to_clam {machine_name,port} =
   (case (!clam_process)
    of SOME _ => error "Clam already connected"
     | NONE =>
       let val socket =
              SocketServer.connect_to_server {host = machine_name,port = port}
           val pps = mk_ppstream_from_socket socket
       in  reset_data (); clam_process := SOME ("",Socket (socket,pps))
       end);

fun subjects_of_scheme tm =
   let fun components tm =
          let val {fst,snd} = dest_pair tm
          in  components fst @ components snd
          end
          handle HOL_ERR _ => [tm]
       val consequent = (snd o strip_imp o snd o strip_forall) tm
       val patterns = map (snd o strip_forall) (strip_conj consequent)
       val arguments = CLaReTPortable.flat (map (snd o strip_comb) patterns)
       val variables = CLaReTPortable.flat (map components arguments)
   in  CLaReTPortable.setify (map type_of variables)
   end;

local

val DEPTH_FORALL_CONV = DecisionSupport.DEPTH_FORALL_CONV;

fun ANTE_CONV conv tm =
   if (is_imp tm) andalso not (is_neg tm)
   then RATOR_CONV (RAND_CONV conv) tm
   else NO_CONV tm;

fun CONSEQ_CONV conv tm =
   if (is_imp tm) andalso not (is_neg tm)
   then RAND_CONV conv tm
   else NO_CONV tm;

fun MOVE_FORALL_OUT_CONV tm =
   TRY_CONV (DEPTH_FORALL_CONV (CONSEQ_CONV MOVE_FORALL_OUT_CONV) THENC
             REPEATC (DEPTH_FORALL_CONV RIGHT_IMP_FORALL_CONV)) tm;

fun NORMALIZE_SCHEME_CONV tm =
   TRY_CONV (DEPTH_FORALL_CONV (CONSEQ_CONV NORMALIZE_SCHEME_CONV THENC
                                ANTE_CONV MOVE_FORALL_OUT_CONV)) tm;

in

fun send_scheme (name,th) =
   let val th' = CONV_RULE NORMALIZE_SCHEME_CONV
                    (GEN_ALL (REWRITE_RULE [GSYM AND_IMP_INTRO] th))
       val (term,translations) =
          HOLtoClam.translate_top_level_formula (concl th')
       val data = Scheme ({hol_name = name,thm = th',
                           hol_types = subjects_of_scheme (concl th')},
                          {clam_name = clam_name name,term = term})
   in  add_data check_connected data;
       add_translations translations;
       write_clam_socket data
   end;

end;

fun subjects_of_clauses clauses =
   let fun subjects subs [] = rev subs
         | subjects subs (c::cs) =
          let val name = (#Name o dest_const o fst o strip_comb o
                          lhs o snd o strip_imp o snd o strip_forall) c
          in  if (CLaReTPortable.member name subs)
              then subjects subs cs
              else subjects (name :: subs) cs
          end
          handle HOL_ERR _ => subjects subs cs
   in  subjects [] clauses
   end;

local

val COND_CONJ_IMP =
   prove (--`!b x y z. (z = COND b x y) =
                       ((b ==> (z = x)) /\ (~b ==> (z = y)))`--,
          REPEAT GEN_TAC THEN BOOL_CASES_TAC (--`b:bool`--) THEN
          REWRITE_TAC []);

val IMP_NORM =
   prove (--`!b1 b2 b3 x y. (b1 ==> ((b2 ==> x) /\ (b3 ==> y))) =
                            ((b1 /\ b2 ==> x) /\ (b1 /\ b3 ==> y))`--,
          REPEAT GEN_TAC THEN EQ_TAC THEN REPEAT STRIP_TAC THEN RES_TAC);

fun COND_NORM th =
   map GEN_ALL
      (CONJUNCTS (PURE_REWRITE_RULE [COND_CONJ_IMP,IMP_NORM] (SPEC_ALL th)));

in

fun send_definition (name,th) =
   let val ths = Lib.flatten
                    (map (COND_NORM o GEN_ALL) (CONJUNCTS (SPEC_ALL th)))
       val th' = end_itlist CONJ ths
       val tms = map concl ths
       val (terms,translations) =
          HOLtoClam.bindN_map
             (HOLtoClam.translate_top_level_formula,tms,HOLtoClam.unitN)
       val term = end_itlist (fn x => fn y => ClamAST.Prop (x,"#",y)) terms
       val cname = clam_name name
       val cnames = map (fn n => cname ^ Lib.int_to_string n)
                        (CLaReTPortable.upto 1 (length terms))
       val data = Definition
                   ({hol_name = name,thm = th',defs = Lib.zip cnames ths,
                     hol_constant_names = subjects_of_clauses tms},
                    {clam_name = cname,term = term})
   in  add_data check_connected data;
       add_translations translations;
       write_clam_socket data
   end;

end;

fun send_rule (name,th) =
   let val th' = GEN_ALL th
       val (term,translations) =
          HOLtoClam.translate_top_level_formula (concl th')
       val cname = clam_name name
       val data = Rule ({hol_name = name, thm = th', rule = (cname,th')},
                        {clam_name = cname, term = term})
   in  add_data check_connected data;
       add_translations translations;
       write_clam_socket data
   end;

fun send_transitivity_rule (name,th) =
   let val th' = GEN_ALL th
       val (term,translations) =
          HOLtoClam.translate_top_level_formula (concl th')
       val cname = clam_name name
       val data = Trans ({hol_name = name,thm = th',rule = (cname,th')},
                         {clam_name = cname,term = term})
   in  add_data check_connected data;
       add_translations translations;
       write_clam_socket data
   end;

fun send_goal (name,gl) =
   let val (goal,translations) = HOLtoClam.translate_goal gl
       val data = Goal ({hol_name = name,goal = gl},
                        {clam_name = clam_name name,goal = goal})
   in  add_data check_connected data;
       add_translations translations;
       write_clam_socket data
   end;

fun send_fact (name,fact) =
   let val (term,translations) =
          HOLtoClam.translate_top_level_formula
             (case fact
              of Fact.Proved (th,pp) => concl th
               | Fact.CantProve tm => tm
               | Fact.Refuted tm => tm)
       val cname = clam_name name
       val data = Fact ({hol_name = name, fact = fact},
                        {clam_name = cname, term = term})
   in  add_data check_connected data;
       add_translations translations;
       write_clam_socket data
   end;

fun delete_scheme name =
   let val (socket,pps) = clam_socket ()
   in  delete_scheme_data check_connected name;
       write_simple_prolog pps ("delete_scheme",[clam_name name]);
       wait_on_socket socket; ()
   end;

fun delete_definition name =
   let val (socket,pps) = clam_socket ()
   in  delete_definition_data check_connected name;
       write_simple_prolog pps ("delete_def",[clam_name name]);
       wait_on_socket socket; ()
   end;

fun delete_rule name =
   let val (socket,pps) = clam_socket ()
   in  delete_rule_data check_connected name;
       write_simple_prolog pps ("delete_rule",[clam_name name]);
       wait_on_socket socket; ()
   end;

fun delete_transitivity_rule name =
   let val (socket,pps) = clam_socket ()
   in  delete_transitivity_data check_connected name;
       write_simple_prolog pps ("delete_transitivity_rule",[clam_name name]);
       wait_on_socket socket; ()
   end;

fun delete_goal name =
   let val (socket,pps) = clam_socket ()
   in  delete_goal_data check_connected name;
       write_simple_prolog pps ("delete_goal",[clam_name name]);
       wait_on_socket socket; ()
   end;

fun delete_fact name =
   let val (socket,pps) = clam_socket ()
   in  delete_fact_data check_connected name;
       write_simple_prolog pps ("delete_fact",[clam_name name]);
       wait_on_socket socket; ()
   end;

fun delete_all_facts () =
   let fun select (Fact _) = true | select _ = false
       val (socket,pps) = clam_socket ()
   in  delete_data check_connected select;
       write_simple_prolog pps ("delete_all_facts",[]);
       wait_on_socket socket; ()
   end;

fun trace_plan n =
   let val new = Lib.int_to_string n
       val (socket,pps) = clam_socket ()
       val current =
          (write_simple_prolog pps ("trace_plan",[new]);
           CLaReTPortable.implode (wait_on_socket socket))
   in  Lib.string_to_int (substring (current,0,size current - 1))
   end;

fun time_limit n =
   let val new = Lib.int_to_string n
       val (socket,pps) = clam_socket ()
       val current =
          (write_simple_prolog pps ("time_limit",[new]);
           CLaReTPortable.implode (wait_on_socket socket))
   in  Lib.string_to_int (substring (current,0,size current - 1))
   end;

fun iteration b =
   let val (socket,pps) = clam_socket ()
   in  write_simple_prolog pps ("iteration",[if b then "on" else "off"]);
       wait_on_socket socket; ()
   end;

fun call s =
   let val (socket,pps) = clam_socket ()
       val result = (write_string pps s;
                     CLaReTPortable.implode (wait_on_socket socket))
   in  if (result = "succeeded.") then () else error "Call to Clam failed"
   end;

fun prove_goal name = call_clam ("prove_goal","prove_and_save") name [];

fun apply_methods name methods choice =
   call_clam ("apply_methods","apply_methods_and_save")
      name ["[" ^ comma methods ^ "]",Lib.int_to_string choice];

fun apply_any_method name choice =
   call_clam ("apply_any_method","apply_any_method_and_save")
      name [Lib.int_to_string choice];

fun apply_any_submethod name choice =
   call_clam ("apply_any_submethod","apply_any_submethod_and_save")
      name [Lib.int_to_string choice];

fun set_plan_file sopt =
   let val old = !plan_file
   in  plan_file := sopt; old
   end;

fun get_plan_file () = !plan_file;

(*---------------------------------------------------------------------------*
 * This alias is used by the tactics generated from tacticAST because it's   *
 * shorter than "find_scheme". This helps the generated tactics look nicer   *
 * when prettyprinted. Some reason, huh? Anyway, if you want to change this  *
 * name, then you'd better change the corresponding syntax in vistactic.sml  *
 *---------------------------------------------------------------------------*)
val scheme = find_scheme;

structure VisibleTactic = 
                 VisibleTactic(val clam_theorem = find_theorem
                               val scheme = find_scheme);
structure Plan2Tactic = 
  PLAN2TAC(structure VisibleTactic = VisibleTactic
           structure Clam2Client = 
            struct 
                  type client_type = Type.hol_type
                  type client_term = Term.term

                  val client_bool     = Parse.type_parser`:bool`
                  val client_var_name = Clam2HOL.clam_to_hol_var
                  val mk_client_var   = Term.mk_var
                  val mk_client_type  = Clam2HOL.mk_hol_type
                  val mk_client_term  = Clam2HOL.mk_hol_term
            end);

local

val count = ref 0;

in

fun CLAM_PLAN f = fn gl =>
   let val name = (count := !count + 1; int_to_string (!count))
       val _ = send_goal (name,gl)
   in  f name
   end;

fun CLAM_TAC_AST0 f classifier g =
   let val (p,ts) = CLAM_PLAN f g
   in 
      Plan2Tactic.tactic_of classifier (p,ts)
   end;

fun CLAM_TAC0 f classifier ast2tac :tactic = fn g =>
   let val (tacAST,facts) = CLAM_TAC_AST0 f classifier g
   in 
     ast2tac tacAST g
   end;

end;

end;

end; (* structure CallClam *)
