(****************************************************************************)
(*                                                                          *)
(*    Copyright 1999 University of Cambridge and University of Edinburgh    *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : database.sml                                             *)
(* DESCRIPTION   : Database of information sent to Clam.                    *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 21st April 1999                                          *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 21st April 1999                                          *)
(****************************************************************************)

structure ClamDatabase =
struct

exception ClamDatabase of string;

local

fun error s = (Lib.say (s ^ "\n"); raise ClamDatabase s);

in

datatype clam_data = Scheme of {hol_name : string,thm : thm,
                                hol_types : hol_type list} *
                               {clam_name : string,term : ClamAST.typ}
                   | Definition of {hol_name : string, thm:thm,
                                    defs : (string * thm) list,
                                    hol_constant_names : string list} *
                                   {clam_name : string,term : ClamAST.typ}
                   | Rule of {hol_name : string,thm:thm, rule:string*thm} *
                             {clam_name : string,term : ClamAST.typ}
                   | Trans of {hol_name : string,thm:thm, rule:string*thm} *
                              {clam_name : string,term : ClamAST.typ}
                   | Goal of {hol_name : string,goal : goal} *
                             {clam_name : string,goal : ClamAST.goal}
                   | Fact of {hol_name : string, fact : Fact.fact} *
                             {clam_name : string, term : ClamAST.typ}

local

val clam_data = ref ([]:clam_data list);

in

fun reset_data () = (clam_data := []);

fun add_data (check_connected : unit -> unit) d =
   (check_connected (); clam_data := !clam_data @ [d]);

fun delete_data (check_connected : unit -> unit) select =
   (check_connected ();
    clam_data := CLaReTPortable.filter (not o select) (!clam_data));

val clam_data = fn () => !clam_data;

end;

fun delete_scheme_data check_connected name =
   let fun select (Scheme ({hol_name,...},_)) = (hol_name = name)
         | select _ = false
   in  delete_data check_connected select
   end;

fun delete_definition_data check_connected name =
   let fun select (Definition ({hol_name,...},_)) = (hol_name = name)
         | select _ = false
   in  delete_data check_connected select
   end;

fun delete_rule_data check_connected name =
   let fun select (Rule ({hol_name,...},_)) = (hol_name = name)
         | select _ = false
   in  delete_data check_connected select
   end;

fun delete_transitivity_data check_connected name =
   let fun select (Trans ({hol_name,...},_)) = (hol_name = name)
         | select _ = false
   in  delete_data check_connected select
   end;

fun delete_goal_data check_connected name =
   let fun select (Goal ({hol_name,...},_)) = (hol_name = name)
         | select _ = false
   in  delete_data check_connected select
   end;

fun delete_fact_data check_connected name =
   let fun select (Fact ({hol_name,...},_)) = (hol_name = name)
         | select _ = false
   in  delete_data check_connected select
   end;

local

val translations :HOLtoClam.translations ref = 
   ref (HOLtoClam.null_translations);

in

fun add_translations ts =
   translations := HOLtoClam.merge_translations ts (!translations);

fun find_goal name =
   let fun extract (Goal ({hol_name,goal},_) :: rest) =
          if (hol_name = name) then (goal,!translations) else extract rest
         | extract (_ :: rest) = extract rest
         | extract [] = error ("Goal `" ^ name ^ "' not found")
   in  extract (clam_data ())
   end;

end;

fun find_theorem name =
   let fun extract (Rule({hol_name,thm,rule=(s,th)},_)::rst) =
              if (s=name) then th else extract rst
         | extract (Definition({defs,...},_)::rst) =
             (case (Lib.assoc1 name defs)
                of NONE => extract rst
                 | SOME(_,th) => th)
         | extract (_ :: rest) = extract rest
         | extract [] = error ("Theorem `" ^ name ^ "' not found")
   in  extract (clam_data ())
   end;

fun find_scheme name =
   let fun extract (Scheme({thm,...},{clam_name,term}) :: rest) =
          if (clam_name = name) then thm else extract rest
         | extract (_ :: rest) = extract rest
         | extract [] = error ("Scheme `" ^ name ^ "' not found")
   in  extract (clam_data ())
   end;

fun known_hol_constants () =
   CLaReTPortable.flat
      (map (fn Definition ({hol_constant_names,...},_) => hol_constant_names
             | _ => []) (clam_data ()));

fun known_hol_types () =
   (CLaReTPortable.setify o CLaReTPortable.flat)
      (map (fn Scheme ({hol_types,...},_) => hol_types
             | _ => []) (clam_data ()));

end;

end; (* ClamDatabase *)
