/*
 * @(#)$Id: reduction.pl,v 1.14.2.2 1999/03/26 16:03:15 rjb Exp $
 *
 * $Log: reduction.pl,v $
 * Revision 1.14.2.2  1999/03/26 16:03:15  rjb
 * Some reduction operations were not being recorded in the plan. Applied a fix.
 *
 * Revision 1.14.2.1  1999/02/01 10:55:19  rjb
 * Modified to convert `f of x' to `f(x)' when f is an atom.
 *
 * Revision 1.14  1997/09/26 14:49:20  img
 * nf/[2,4], nf_plus/4 -> reduction_rtc/[2,4], reduction_tc/4
 *
 * Revision 1.13  1997/04/07 11:41:19  img
 * rationalize code for equal length lists.
 *
 * Revision 1.12  1997/01/14 10:44:23  img
 * Generalized conditional for multifile declaration.
 *
 * Revision 1.11  1996/12/12 12:41:18  img
 * Error message predicates.
 *
 * Revision 1.10  1996/12/04 12:30:30  img
 * rt/7.1: Labels on the RHS of reduction rules were not correct in the
 * case of rules like f(X) :=> X, where X was labelled inside f.  If the
 * label is variable, make it a tick.
 *
 * Revision 1.9  1996/06/18  17:16:00  img
 * more informative error message
 *
 * Revision 1.8  1996/06/11  16:30:57  img
 * cosmetic changes
 *
 * Revision 1.7  1996/05/31  12:42:34  img
 * Check polarity on each rule application.
 *
 * Revision 1.6  1996/05/24  09:52:26  img
 * New reduction rule machinery.  Uses simplification ordering
 * (extensible version of RPOS) to prove that a rewrite relation is
 * well-founded.  (See MRG Blue Book notes 1073 and 1084 for more
 * details.)  A terminating TRS is kept in reduction_rule/6 (these are
 * the rewrite rules), and the pair of quasi-precedence (a binary
 * relation over the set of function symbols in the theory) and status
 * function is kept in registry/3.  add_rule/3 is used to add new rules
 * to the TRS.  (See also the reduction/2 method.)
 *
 * New term rewriting machinery.  Uses labelled term rewriting (see MRG
 * Blue Book note 1095) to normalize terms wrt reduction rules.  Labelled
 * term rewriting support is in place for use in rippling too.
 *
 * nf/4 and nf_plus/4 compute the ref. trans. clos. and trans. clos. of
 * the congruence closure of the reduction rule rewrite relation.  Use
 * nf_plus/4 to put a term into normal form (eg method normalize_term).
 *
 * Equal rules removed (now these are reduction rules).  Cancellation
 * rules kept since they can have a special status during ripple then
 * cancel.
 *
 * Revision 1.5  1995/11/28  20:20:38  img
 * Capture <=> as equivalence
 *
 * Revision 1.4  1995/10/24  14:53:15  img
 * removed old parsing code
 *
 * Revision 1.3  1995/05/17  02:17:52  img
 * 	* Conditional multifile declaration (for Quintus).
 *
 * Revision 1.2  1995/03/01  04:14:32  img
 * 	* Added file_version/1 and associated multifile declaration
 *
 * Revision 1.1  1994/09/16  09:18:22  dream
 * Initial revision
 */

?-if(multifile_needed). ?-multifile file_version/1. ?-endif.
file_version('$Id: reduction.pl,v 1.14.2.2 1999/03/26 16:03:15 rjb Exp $').

/* This file contains all the code needed to deal with reduction
   rules.  Reduction rules are rewrite rules which are particularly
   well behaved, and which are used in the symbolic evaluation method
   for doing simple rewrites on expressions. 

   Here we use a simplification ordering to demonstrate termination of
   the rewrite set.  A simplification ordering is based on EPOS,
   which is in turn based on RPOS (recursive path ordering with
   status).  */

/* reduction_rule(LHS, RHS, Cond, Dir, Rulename,Ref): Rulename is the
   name of a rewrite Cond=>LHS:=>RHS that has been proven to be
   measure-decreasing under some well-founded termination order.  Dir
   describes the polarity restrictions in using the rule.  (Ref is the
   reference of the rule in the database.)

   Clam supports two different reduction rule sets in order to permit
   implicative rewrites to be used in either direction.  This is
   terminating because the polarity restriction on the use of the
   rules ensures that cycles are prevented.  Equality rules must be
   oriented left-to-right (or right-to-left) since they may be applied
   in positions of either polarity.  Thus, there are two registries,
   labelled "positive" and "negative", establishing termination of all
   reduction rules whose Dir is imp(left) and imp(right),
   respectively.  Equality rules are accounted for in both registries,
   since this simplifies the implementation.

   Equivalence rules L <=> R may be used in either direction, although
   they must be proven termination as L => R and R => L, and stored in
   separate TRSs.  For this reason they can no longer be seen as
   "equality" rules.

   The ground case is faster when no occurs check is required.  */
reduction_rule(LHS, RHS, Cond, Dir, Rulename,Ref) :-
    ground(LHS),!,
    recorded(reduction,reduction(LHS,_,RHS,_,Cond,Dir,Rulename),Ref).
reduction_rule(LHS, RHS, Cond, Dir, Rulename,Ref) :-
    recorded(reduction,reduction(LHSproper,_,RHS,_,Cond,Dir,Rulename),Ref),
    unify(LHSproper,LHS).

/* reduction_rule/8 is as reduction_rule/6, but labelling of LHS and
   RHS is returned in the additional arguments.  */
reduction_rule(LHS, LHSlabel,RHS,RHSlabel, C, D, Rulename,Ref) :-
    ground(LHS),!,
    recorded(reduction,
	     reduction(LHS,LHSlabel, RHS,RHSlabel,C,D,Rulename),Ref).
reduction_rule(LHS, LHSlabel,RHS,RHSlabel, C, D, Rulename,Ref) :-
    recorded(reduction,
	     reduction(LHSproper,LHSlabel,RHS,RHSlabel,C,D,Rulename),Ref),
    unify(LHSproper,LHS).


/* Prepare Thm for processing by add_rule/2;  Give user info if no
   rule was derived from some thm.  */
add_reduction_rules([]).
add_reduction_rules([Thm|Rest]) :-
    recorded(theorem,theorem(_,_,Goal,Thm),_),
    matrix(Vars,Matrix,Goal),
    replace_universal_vars(Vars,Matrix,LiftedGoal),
    add_rule(reduction,Thm,LiftedGoal),
    if(\+ reduction_rule(_,_,_,_,Thm,_),
       clam_warning('No reduction rules were derived from %t.\n',[Thm])),     
    !,
    add_reduction_rules(Rest).

/* The current TRS is kept in the recorded database.  The rewrite
   rules proper are those described by reduction_rule/6, and the
   registry is the pair (Tau,Prec).  Typically, Prec will not be a
   terminated list, since that allows it to be extended.  The
   extension of Tau is done is a different way.  It might be useful to
   have multiple TRSs, but for the moment there is only one.  */
registry(TRS,Tau,Prec,Ref) :-
    recorded(registry,registry(TRS,Tau,Prec),Ref).

/* This flag determines whether the registry will be extended
   dynamically during the application of the reduction method.  */ 
extending_registry :-
    fail.

                   /* TERM REWRITING STUFF */

/* Repeated application of reduction rules; uses labelled trees to
   speed up rewriting.  

   A labelling is a term built with constructors "node" and Prolog
   lists.  nodes are labelled with atoms "tick" or "cross".  A
   compelete labelling (CL) for a term T is one with the same term
   structure: "cross" means that the corresponding node in T is in
   normal form; "tick" means that this is not known to be the case.
   Hence a CL determines which subterms of T to examine for redexes.
   An incomplete labelling (IL) is like a CL but there are some
   (ie. at least one) subterms labelled with a variable (ie and
   incomplete tree).  These nodes are considered "tick"ed, that is,
   they are examined for redexes.  (Two labelling well-formedness
   properties are assumed: all nodes below a crossed node are crossed
   and all nodes above a ticked node are ticked.)

   (A term in normal form is labelled with a tree full of crosses.
   This seems wastful, but the uniformity allows unification to be
   used to propagate labellings.)

   Labellings are used to skip (crossed) subterms which are known not
   to contain redexes.  The result is a much faster rewriting process.
   Labellings are propagated during rewriting (the reduction rule
   parser stores propagation information at load-time; see lib_load/_
   and add_rule/_).

   rt(Pos,T,Tl,Ref,S,Sl,Hyps): There are two possibilities:

   (i) T can be rewritten into S using a reduction rule described by
   Ref.  Sl will be a complete labelling of S.  Tl may be an
   incomplete labelling (cf ii), but the path to the the rewritten
   subterm is ticked.

   (ii) T and S are identical and in normal form. Tl is an CL
   containing ticks or is an IL. Sl is a CL containing only crosses,
   reflecting the fact that S is in normal form.

   Pos is the position in T (S) of the redex (reduct).  Hyps is passed
   around in case conditions need to be established.  */

rt([],T,node(tick,Ts),[Thm,Dir],S,node(Sl,Ss),Hyps) :-
    %% This base case clause has to do some manipulation to ensure
    %% that labelling information on terms bound to variables in the
    %% reduction rule is propagated from T into S.  For example,
    %% rewriting s(x*)=s(y+) into x=y, using the cancellation rule for
    %% s, should copy the annotation (* and +) on the terms x and y
    %% into the reduct x=y.  Otherwise it will be lost.  See
    %% processing of reduction rules in lib_load/_ and add_rule/_.
    reduction_rule(T,node(tick,Ts),S,node(Sl,Ss), Cond,Dir,Thm,_),
    elementary(Hyps==>Cond,_),
    if(var(Sl),Sl=tick).
    %% No cut here since other conditions may be imposed on the rule
    %% applied (e.g., polarity).
    
rt(Pos,Term,_,_,Term,node(cross,[]),_Hyps):-
    atomic(Term),!,
    Pos = [].

rt([N|Pos],Term,node(tick,ArgsL),Ref, NewTerm,node(Applied,NewArgsL),Hyps) :-
    Term =.. [F|Args],
    rt_(Args,1,N,Pos,ArgsL,NewArgs,NewArgsL,Applied,Ref,Hyps),
    %% tail call seems no more efficient
    NewTerm =.. [F|NewArgs].


/* Try to rewrite one of the As.  Succeed when one has been rewritten,
   for elements of As that cannot be rewritten, recover the complete,
   fully crossed labelling.  */
rt_([],_M,_N,_Pos,[],[],[],cross,_,_Hyps).
rt_([A|As],M,N,PosFinal,[node(tick,L)|Ls],
    [NewA|NewAs],[NewALabel|NewLs],FlagFinal,Ref,Hyps) :-
    %% There is no point in trying to rewrite A if it is crossed, so
    %% in the head we explicitly ensure that it has the form tick(L)
    !,						% to allow use of clam_error
    rt(Pos,A,node(tick,L),LocalRef,NewA,NewALabel,Hyps),
    (A==NewA
	 -> (MM is M + 1,
	     rt_(As,MM,N,PosFinal,Ls,NewAs,NewLs,FlagFinal,Ref,Hyps))
	 ;  (PosFinal = Pos, M = N, NewLs = Ls,
	     Ref = LocalRef,
	     NewAs = As, FlagFinal = tick)).
rt_([A|As],M,N,Pos,[node(cross,Crossed)|Ls],[A|NewAs],
    [node(cross,Crossed)|NewLs],Applied,Ref,Hyps) :-
    %% A is crossed, so skip it and search for another term.  Notice
    %% that if A is labelled with tick the clause above will
    %% succeed since rt will, with NewALabel = cross.
    !,						%only for clam_error
    MM is M + 1,
    rt_(As,MM,N,Pos,Ls,NewAs,NewLs,Applied,Ref,Hyps).
rt_([_|_],_,_,_,[_|_],[_|_],[node(_,_)|_],_,_,_) :-
    clam_internal_error('rt_/10.4',[]).


/* Reflexive transitive closure of rt.  For a terminating rewrite
   relation this can be used to compute the normal form, S, of T.  */
reduction_rtc(T,S,Tactic,Hyps) :-
    rt_tc(T-_,S-_,_,Tactic,Hyps),!.

/* Ignore conditional rules */
reduction_rtc(T,S) :-
    reduction_rtc(T,S,_,[]).

/* Transitive closure of rt.  As reduction_rtc/3, but requires that S is not
   already in normal form.  This is useful when used inside a method
   which is iterated with other methods (e.g., symbolic evaluation).  */ 
reduction_tc(T,NormS,Tactic,Hyps) :-
    rt_tc(T-_,S-_,Rewritten,Tactic,Hyps),!,
    normalise_application(S,NormS),
    Rewritten == rewritten.

/* +U and ?S are related under the reflexive transitive closure of rt.
   Rewritten is set to 'rewritten' if one or more rewrites are
   applied, for a terminating rewrite relation, this is when S and U
   are different.

   ?Sl and ?Ul are the labellings for S and U.  */
rt_tc(S-Sl,U-Ul,Rewritten,Tactic,Hyps) :-
    %% compute S ->* U via S -> T and then recurse for T ->* U
    rt(Pos, S,Sl, [Rule,TypeDir], T,Tl,Hyps),
    (Tl = node(cross,_)
      -> (U = T, Ul = Tl, Tactic=[])
      ;  (reverse(Pos,RPos),			%more work to do
	  polarity_compatible(S, RPos, TypeDir),
	  Rewritten=rewritten,
	  Tactic = [reduction(RPos,[Rule,TypeDir])|RestTac],
	  rt_tc(T-Tl,U-Ul,_,RestTac,Hyps))). 

/* Slabel and Tlabel are labellings for terms S and T, such that
   variables in S and T are uniformly labelled with unique variables
   in Slabel and Tlabel, and function symbols in S and T are labelled
   with renamings of Label. Such labellings are used during labelled 
   term rewriting to propagate labelling annotations across rewrite
   rules.  */ 
rule_labelling(S,Sl,T,Tl) :-
    term_labels(S,[],LS),
    term_labels(T,LS,LST),
    subterm_label(S-Slabel,LST),
    subterm_label(T-Tlabel,LST),
    copy_term(Slabel-Tlabel,Sl-Tl).
term_labels(T,LS,LS) :-
    subterm_label(T-_,LS),!.
term_labels(T,LS,[T-node(_FreshVar,[])|LS]) :-
    atomic(T),!.
term_labels(V,LS,[V-V|LS]) :-
    var(V),!.
term_labels(T,LS, [T-node(_FreshVar,Labelling) | NewLS]) :-
    T =.. [_|As],
    term_labels_(As,Labelling,LS,NewLS).

term_labels_([],[],LS,LS).
term_labels_([T|As],[Tl|Tls],LS,NewLS) :-
    term_labels(T,LS,TmpLS),
    subterm_label(T-Tl,TmpLS),
    term_labels_(As,Tls,TmpLS,NewLS).

subterm_label(T-LL,[TT-LL|_LS]) :-
    T == TT,!.
subterm_label(T-LL,[_|LS]) :-
    subterm_label(T-LL,LS).

