(*---------------------------------------------------------------------------*
 * FILE: vistactic.sml                                                       *
 *                                                                           *
 * This structure provides implementations of "visible" HOL tactics - ones   *
 * that come with a prettyprinter attached.                                  *
 *---------------------------------------------------------------------------*)

functor VisibleTactic (val clam_theorem : string -> thm
                       val scheme : string -> thm) : VisibleTactic_sig =

struct

type term    = Term.term
type vtactic = (PrettyPrint.ppstream -> unit) * tactic

open PrettyPrint;
open Tactic;
open Rsyntax;

val scheme       = scheme
val clam_theorem = clam_theorem ;

fun tac_err(func,mesg) = HOL_ERR{origin_structure="PlantoTactic",
                                 origin_function = func,
                                 message = mesg};

datatype direction = LEFT | RIGHT;


(*---------------------------------------------------------------------------*
 * Miscellaneous.                                                            *
 *---------------------------------------------------------------------------*)
fun W f x = f x x;

fun paren s = String.concat["(",s,")"];

fun tzip (h1::rst1) (h2::rst2) = (h1,h2)::tzip rst1 rst2
  | tzip _ _ = [];
  
fun remove1 x = 
  let fun del [] = []
        | del (h::t) = if (x=h) then t else h::del t
  in del end;
 

(*---------------------------------------------------------------------------*
 * Projection functions for vtactics.                                        *
 *---------------------------------------------------------------------------*)
fun pp_of (pp,t) = pp
fun tactic_of (pp,t) = t;

(*---------------------------------------------------------------------------
 * Prettyprinting support.
 *---------------------------------------------------------------------------*)

fun pr_list pfun dfun bfun =
  let fun pr [] = ()
        | pr [i] = pfun i
        | pr (i::rst) = ( pfun i ; dfun() ; bfun() ; pr rst )
  in  pr   end;

fun pp_list pp_item ppstrm L =
  ( begin_block ppstrm CONSISTENT 0;
    add_string ppstrm "["; 
      begin_block ppstrm INCONSISTENT 0;
        pr_list (pp_item ppstrm)
                (fn () => add_string ppstrm ",") 
                (fn () => add_break ppstrm (1,0)) L;
      end_block ppstrm;
    add_string ppstrm "]";
    end_block ppstrm);

fun quote_pp strm s =
   (add_string strm "`"; add_string strm s; add_string strm "`");

fun string_literal_pp strm s = add_string strm (Lib.quote s);

(*---------------------------------------------------------------------------
 * Prettyprint a term so that it is reparsable. Uses SML/NJ quotations.
 *---------------------------------------------------------------------------*)
fun parsable_pp ppstrm tm = 
  let val stypes = !Globals.show_types
      val _ = Globals.show_types := true
  in 
    begin_block ppstrm INCONSISTENT 2;
    add_string ppstrm "`";
    pp_term ppstrm tm;
    add_string ppstrm ("`");
    end_block ppstrm;
    Globals.show_types := stypes
   end;


(*---------------------------------------------------------------------------
 * Free variables, in order of occurrence, from left to right in the 
 * syntax tree. 
 *---------------------------------------------------------------------------*)
fun free_vars_lr M =
  let fun fvs tm A =
      case (dest_term tm)
      of CONST _ => A
       | VAR _             => if (mem tm A) then A else tm::A
       | COMB{Rator, Rand} => fvs Rator (fvs Rand A)
       | LAMB{Bvar,Body}   => remove1 Bvar (fvs Body A)
  in fvs M []
  end;


(*---------------------------------------------------------------------------*
 *                                                                           *
 *                        INDUCTION SCHEMES                                  *
 *                                                                           *
 *---------------------------------------------------------------------------*)

(*---------------------------------------------------------------------------*
 * Apply an induction theorem to a goal.                                     *
 *---------------------------------------------------------------------------*)
fun RINDUCT_TAC thm =
  let val {Bvar=prop,Body} = dest_forall(concl thm)
      val n = (length o #1 o strip_forall o #2 o strip_imp) Body
      fun ndest_forall trm = 
          let fun dest (0,tm,V) = (rev V,tm)
                | dest (n,tm,V) = 
                   let val {Bvar,Body} = dest_forall tm
                   in dest(n-1,Body, Bvar::V)
                   end
          in dest(n,trm,[])
          end
      fun tac (asl,w) =
       let val (V,body) = ndest_forall w
           val P = list_mk_abs(V,body)
           val thm' = CONV_RULE(DEPTH_CONV GEN_BETA_CONV) (ISPEC P thm)
       in MATCH_MP_TAC thm' (asl,w)
       end handle _ => raise tac_err("RINDUCT_TAC","")
  in tac
  end;


(*---------------------------------------------------------------------------
 * Generate concrete syntax for obtaining the induction scheme.
 *---------------------------------------------------------------------------*)
fun pp_scheme name ppstrm 
    = (begin_block ppstrm INCONSISTENT 0;
        add_string ppstrm "scheme";
        add_break ppstrm (1,0);
        add_string ppstrm (Lib.quote name);
       end_block ppstrm)

(*---------------------------------------------------------------------------
 * IND_TAC (RENAME (scheme <name>) <names>) <variables> 
 *---------------------------------------------------------------------------*)
fun pp_induct name names vars ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "QIND_TAC";
        add_break ppstrm (1,0);
        add_string ppstrm "(";
        begin_block ppstrm CONSISTENT 2;
          add_string ppstrm "RENAME";
          add_break ppstrm (1,0);
          add_string ppstrm "("; pp_scheme name ppstrm; add_string ppstrm ")";
          add_break ppstrm (1,0);
          pp_list (pp_list string_literal_pp) ppstrm names;
        end_block ppstrm;
        add_string ppstrm ")";
        add_break ppstrm (1,0);
        pp_list parsable_pp ppstrm vars;
       end_block ppstrm);


(* Identify the clauses that have inductive hypotheses *)
fun step_cases thm = 
  let val {Bvar,Body} = dest_forall (concl thm)
      val (ants,_) = strip_imp Body
  in
      map (is_imp o snd o strip_forall) ants
  end;


fun IND_TAC thm V (asl,w) =
   let val (prefix, body) = strip_forall w
       val unused = subtract prefix V
       val thm' = Rewrite.PURE_REWRITE_RULE [AND_IMP_INTRO] thm
   in (REPEAT GEN_TAC 
        THEN MAP_EVERY Tactic.SPEC_TAC (map (fn x => (x,x)) unused)
        THEN MAP_EVERY Tactic.SPEC_TAC (map (fn x => (x,x)) (rev V))
        THEN RINDUCT_TAC thm'
        THEN REPEAT Tactic.CONJ_TAC
        THENL (map (fn true  => REPEAT GEN_TAC THEN DISCH_TAC
                     | false => ALL_TAC) (step_cases thm))
        THEN REPEAT GEN_TAC) (* Strip quantifiers as Clam does. *)
       (asl,w)   
   end;

fun QIND_TAC scheme V = IND_TAC scheme (map Parse.term_parser V);

fun mk_namings l = map (map (#Name o dest_var) o free_vars_lr) l;

fun rename_var v s = mk_var{Name=s, Ty=type_of v};

fun rename S tm = 
  let val (V,body) = strip_forall tm
  in if (is_imp body)  (* inductive case *)
     then let val V' = map2 rename_var V (Lib.flatten S)
              val theta = map2 (curry (op|->)) V V'
          in list_mk_forall(V',subst theta body) 
          end
     else let val (_,args) = strip_comb body     (* body = P M0 ... Mn *)
              val argsV = map free_vars_lr args
              val maps  = map (fn (l1,l2) => tzip l1 l2) (zip argsV (map rev S))
              val maps1 = flatten maps
              val theta = map (fn (v,s) => v |-> rename_var v s) maps1
          in
           list_mk_forall(map (subst theta) V, subst theta body)
          end
  end;

fun RENAME thm namesl =
  let val {Bvar=P,Body} = dest_forall (concl thm)
      val (ants,c) = strip_imp Body
      val ants' = map (rename (rev namesl)) ants
      val tm = mk_forall{Bvar=P, Body=list_mk_imp(ants',c)}
  in EQ_MP (ALPHA (concl thm) tm) thm
  end;

(*---------------------------------------------------------------------------*
 * Apply an induction scheme. There are a few steps in this.                
 *
 *    1. Use the first projections to permute the universal prefix 
 *       of the goal so that it matches the binding structure of the 
 *       scheme. We assuem that the order the induction variables come in 
 *       is the same as the order they should be in the new goal.
 *
 *    2. Extract the induction scheme from the database. 
 *
 *    3. Rename variables in the scheme into those that Clam used in 
 *       planning. This is necessary, because explicit terms are given when
 *       generalizing goals, and the goals that Clam and HOL are working on
 *       must agree on their constants and free variables.
 * 
 *    4. To enable fertilization to succeed as often as possible, especially
 *       in nested inductions, it may be necessary to maintain track of 
 *       inductive hypotheses so that there is no (or at least less) confusion 
 *       about which i.h. is to be used. (This has not been implemented yet.)
 *---------------------------------------------------------------------------*)

fun IND s plist = 
  let val V = map fst plist
      val ivar_map = mk_namings (map snd plist)
      val scheme' = RENAME (scheme s) ivar_map
  in 
     (pp_induct s ivar_map V,  IND_TAC scheme' V)
  end;


(*---------------------------------------------------------------------------
 * Dealing with universally quantified variables.
 *---------------------------------------------------------------------------*)
fun GENL_TAC L = 
   let fun pp ppstrm 
    = (begin_block ppstrm INCONSISTENT 2;
        add_string ppstrm "MAP_EVERY";
        add_break ppstrm (1,0);
        add_string ppstrm "Q.X_GEN_TAC";
        add_break ppstrm (1,0);
        add_string ppstrm "[";
          begin_block ppstrm CONSISTENT 2;
           pr_list (quote_pp ppstrm)
                   (fn () => add_string ppstrm ",") 
                   (fn () => add_break ppstrm (0,0)) L;
          end_block ppstrm;
        add_string ppstrm "]"; 
       end_block ppstrm)
   in
    (pp, MAP_EVERY Q.X_GEN_TAC (map (fn s => `s`) L))
   end;



(*---------------------------------------------------------------------------
 * Blow away trivial subgoals with ELEM_TAC. Finishing tactics are invoked 
 * in order of (roughly) increasing cost.
 *---------------------------------------------------------------------------*)
val rws = [theorem"num" "NOT_SUC", GSYM (theorem"num" "NOT_SUC"),
           theorem"prim_rec" "SUC_ID", GSYM(theorem"prim_rec" "SUC_ID"),
           theorem"prim_rec" "INV_SUC_EQ",
           theorem"list" "CONS_11",
           theorem"list""NOT_CONS_NIL", theorem"list" "NOT_NIL_CONS",
           mk_thm([],Parse.term_parser`!h t. (CONS h t = t) = F`),
           mk_thm([],Parse.term_parser`!h t. (t = CONS h t) = F`)];

fun BRING (asl,w) = 
  let val (_,body) = strip_forall w
      fun occurs M tm = (free_in tm M orelse 
                         (free_in (dest_neg tm) M handle _ => false))
      fun disch [] = ALL_TAC
        | disch [x] = UNDISCH_TAC x
        | disch (x::rst) = UNDISCH_TAC x THEN disch rst
  in 
    disch (filter (occurs body) asl)
  end;

(*---------------------------------------------------------------------------
 * Each tactic in the ORELSE chain must either finish off the proof, or fail.
 *---------------------------------------------------------------------------*)
fun ELEM_TAC g = 
  (REPEAT GEN_TAC 
   THEN (REFL_TAC
        ORELSE (BRING g
                THEN ((Rewrite.REWRITE_TAC rws THEN NO_TAC)
                      ORELSE
                      tautLib.TAUT_TAC 
                      ORELSE 
                      decisionLib.DECIDE_TAC)))) g;


val ELEMENTARY = (C add_string "ELEM_TAC", ELEM_TAC);


(*---------------------------------------------------------------------------
 * Term rewriting with explicit occurrences.
 *
 * The code assumes that the term is in some sort of standard form, i.e., no 
 * beta-redices, binders get applied to lambdas, etc. Paired bindings 
 * currently are not dealt with properly. Quantifiers magically disappear.
 * Equality has a special extra argument.
 *---------------------------------------------------------------------------*)

val LIST_MK_COMB = rev_itlist (Lib.C AP_THM);
val LIST_MK_ABS = itlist ABS;

fun remake_comb (h::t, th, R) = LIST_MK_COMB R (AP_TERM (list_mk_comb(h,t)) th)
  | remake_comb ([], _, _) = raise tac_err("remake_comb","");

fun breakup i L =
  let fun break 0 (h::t) L = (rev L, h, t)
        | break n (h::t) L = break(n-1) t (h::L)
        | break _ [] _ = raise tac_err("break", "");
  in break i L []
  end;

(*---------------------------------------------------------------------------
 * Occurrence lists have an extra entry for equalities, so we skip past the
 * entry, unless the equality was translated to a bi-implication (iff).
 *---------------------------------------------------------------------------*)
fun skip_eq f p L = 
   if (let val {Name,Ty} = dest_const f
       in  (Name = "=") andalso not (hd (#Args (dest_type Ty)) = bool)
       end
       handle _ => false)
   then case L 
        of (_::h1::t) => (h1,t) 
         | others     => raise tac_err("skip_eq", "")
   else p;

(*---------------------------------------------------------------------------
 * This may well be wrong for descending into lambdas. I don't know what
 * Clam does for that.
 *---------------------------------------------------------------------------*)
fun OCC_CONV conv = 
 let fun trav [] tm = conv tm
       | trav (path as (occ::rst)) tm =
          case (dest_term tm)
           of LAMB{Bvar,Body} (* there's only one way to go into a lambda *)
              => ABS Bvar (trav path Body) 
            | COMB{Rator,Rand}
              => if (is_binder (#Name(dest_const(Rator))) handle _ => false)
                 then remake_comb([Rator], trav path Rand, []) 
                 else let val (f,args) = strip_comb tm
                          val (occ', rst') = skip_eq f (occ,rst) path
                          val (L,a,R) = breakup occ' (f::args)
                      in remake_comb(L, trav rst' a, R)    end
            | CONST _ => trav rst tm (* Could make more checks *)
            | VAR   _ => raise tac_err("OCC_CONV", "VAR")
 in trav o rev
 end;

fun exp_at tm occlist = 
 let fun trav [] tm = tm
       | trav (path as (occ::rst)) tm =
          case (dest_term tm)
           of LAMB{Bvar,Body} (* there's only one way to go into a lambda *)
              => trav path Body
            | COMB{Rator,Rand}
              => if (is_binder (#Name(dest_const(Rator))) handle _ => false)
                 then trav path Rand
                 else let val (f,args) = strip_comb tm
                          val (occ', rst') = skip_eq f (occ,rst) path
                          val (L,a,R) = breakup occ' (f::args)
                      in trav rst' a end
            | CONST _ => trav rst tm (* Could make more checks *)
            | VAR   _ => raise tac_err("exp_at", "VAR")
 in trav (rev occlist) tm
 end;

(*---------------------------------------------------------------------------
 * tests for occurrence finding.

     set_goal(dest_thm (theorem "arithmetic" "ADD_ASSOC"));
     e INDUCT_TAC;
     val [plus1,plus2] = CONJUNCTS (definition"arithmetic" "ADD");
     val tm = snd(top_goal());

     (OCC_CONV (REWR_CONV plus1) [1,1] 
     THENC 
     OCC_CONV (REWR_CONV plus1) [1,2,1]) tm;

     Prolog.
     exp_at(s(plus(m,plus(n,p))) = s(plus(plus(m,n),p)) [1,1], X).

 *---------------------------------------------------------------------------*)

(*---------------------------------------------------------------------------*
 * We need a slightly tweaked  rewriter to handle weak fertilizations where  *
 * the variables of the rhs are not a subset of those of the lhs. The only   *
 * difference between this implementation and the standard one is that       *
 * the rewrite rule is not fully generalized, as is standard; rather, only   *
 * those variables occuring on the left hand side are generalized. In effect,*
 * we are choosing (self-)instantiations for the variables that won't get    *
 * matched.                                                                  *
 *---------------------------------------------------------------------------*)
(* This needs to be joined with CREWR_CONV
local fun GSPEC th =
        let val w = concl th 
        in if (is_forall w)
           then GSPEC (SPEC(genvar(type_of(#Bvar(dest_forall w)))) th)
           else th
        end

       fun trim th = 
         let val (asl,c) = dest_thm th
             val (V,body) = strip_forall c
             val {lhs,rhs} = dest_eq body
             val lhsvars = free_vars lhs
             val rhsvars = free_vars rhs
             val trouble = Lib.set_diff rhsvars lhsvars
             val aslvars = free_varsl asl
         in 
          if null(trouble) then th   (* std. rewrite rule *)
          else GENL (Lib.set_diff lhsvars aslvars) (SPEC_ALL th)
         end
in
fun rewrCONV th tm =
  let val pth = GSPEC (trim th)
      val eqn = INST_TY_TERM (Match.match_term (lhs(concl pth)) tm) pth
      val l = lhs(concl eqn)
  in 
    if (l = tm) then eqn else TRANS (ALPHA tm l) eqn
  end 
  handle _ => raise tac_err("REWR_CONV","")
end;
*)

(*---------------------------------------------------------------------------
 * Given a rule P ==> ...==> Z ==> (Q = R) and a term theta(Q), we return
 * 
 *   theta(P), ..., theta(Z) |- theta(Q) = theta(R)
 *
 * It is up to the caller to eliminate the assumptions.
 *---------------------------------------------------------------------------*)
fun RUNDISCH_ALL th =
   if (is_imp (concl th))
   then RUNDISCH_ALL 
         (let val {ant,conseq} = dest_imp (concl th)
          in if is_eq ant
             then let val {lhs,rhs} = dest_eq ant
                  in if (lhs = rhs)
                     then MP th (REFL lhs)
                     else UNDISCH th
                  end
             else UNDISCH th
          end)
   else th;


fun CREWR_CONV thl th tm =
  let val pth = GSPEC th
      val (_,eq) = strip_imp (concl pth)
      val ith = RUNDISCH_ALL(INST_TY_TERM (Match.match_term (lhs eq) tm) pth)
      val ith' = itlist PROVE_HYP thl ith
      val l = lhs(concl ith')
  in 
    if (l = tm) then ith' else TRANS (ALPHA tm l) ith'
  end 
  handle _ => raise tac_err ("CREWR_CONV", "");



fun ORW_TAC thm path lr (g as (asl,_))= 
 let val thm' = case lr of LEFT => thm 
                        | RIGHT => GSYM thm
     val thl = map ASSUME asl
 in CONV_TAC (OCC_CONV (CREWR_CONV thl thm') path) g
 end;

fun OCC_RW_TAC s = ORW_TAC (clam_theorem s);

fun OCC_RW (name, path, lr) = 
   let fun pp ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "OCC_RW_TAC";
        add_break ppstrm (1,0);
        add_string ppstrm (Lib.quote name);
        add_break ppstrm (1,0);
        add_string ppstrm "[";
          begin_block ppstrm CONSISTENT 2;
           pr_list (add_string ppstrm o Lib.int_to_string)
                   (fn () => add_string ppstrm ",") 
                   (fn () => add_break ppstrm (0,0)) path;
          end_block ppstrm;
        add_string ppstrm "]"; 
        add_break ppstrm (1,0);
        add_string ppstrm (case lr of LEFT => "LEFT" | RIGHT => "RIGHT");
       end_block ppstrm)
   in
   (pp, OCC_RW_TAC name path lr)
   end;

(*---------------------------------------------------------------------------
 * Weak fertilization. The goal is an equality. This "FIRST_ASSUM" business 
 * will eventually fail when more inductive hypotheses (or ordinary 
 * assumptions) are present. What is needed is to have some aspects of Clam's
 * indexing of inductive hypotheses communicated to HOL.
 *---------------------------------------------------------------------------*)

fun WFERT_EQ_TAC(path,dir) = 
   FIRST_ASSUM (fn th => ORW_TAC th path dir 
          THEN UNDISCH_TAC (concl th) THEN DISCH_THEN (K ALL_TAC));

fun WFERT_EQ (path,dir) = 
   let fun pp ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "WFERT_EQ_TAC";
        add_break ppstrm (0,0);
        add_string ppstrm "(";
        begin_block ppstrm INCONSISTENT 2;
          pp_list (C(C add_string o Lib.int_to_string)) ppstrm path;
          add_string ppstrm ","; 
          add_break ppstrm (0,0);
          add_string ppstrm (case dir of LEFT => "LEFT" | RIGHT => "RIGHT"); 
        end_block ppstrm;
        add_string ppstrm ")"; 
       end_block ppstrm)
   in
   (pp, WFERT_EQ_TAC(path,dir))
   end;


(*---------------------------------------------------------------------------
 * Strong fertilization.
 *---------------------------------------------------------------------------*)
val SFERT_TAC = FIRST_ASSUM MATCH_ACCEPT_TAC;
val SFERT = (Lib.C add_string "SFERT_TAC", SFERT_TAC);


(*---------------------------------------------------------------------------
 * Generalization. There is some hokey-pokey so that universal quantifiers 
 * binding the term to be generalized are removed.
 *---------------------------------------------------------------------------*)
fun QSPEC_TAC (tm,x) (asl,w) =
  let val tm' = Parse.term_parser tm
      val x' = Parse.term_parser x
  in
  ((if (free_in tm' w) then ALL_TAC 
   else let val (V,body) = strip_forall w
           val W = intersect (free_vars tm') V
           val Y = set_diff V W
        in REPEAT GEN_TAC THEN
           MAP_EVERY SPEC_TAC (map (fn x => (x,x)) (rev Y))
        end)
   THEN SPEC_TAC (tm',x')) (asl,w)
   end;

  
fun SPEC_TAC (tm,x) = 
  let fun pp ppstrm = 
          ( begin_block ppstrm CONSISTENT 2;
            add_string ppstrm "QSPEC_TAC";
            add_break ppstrm (1,0);
            add_string ppstrm "(";
              begin_block ppstrm CONSISTENT 1;
                parsable_pp ppstrm tm;
                add_string ppstrm ",";
                add_break ppstrm (0,0);
                parsable_pp ppstrm x;
              end_block ppstrm;
              add_string ppstrm ")"; 
            end_block ppstrm)
  in
     (pp, QSPEC_TAC (`^tm`,`^x`)) 
  end;


(*---------------------------------------------------------------------------
 * Case splitting.
 *---------------------------------------------------------------------------*)
val CASES_THM = 
 ONCE_REWRITE_RULE [decisionLib.DECIDE(Parse.term_parser`x \/ y = y \/ x`)]
                                  (REWRITE_RULE [] BOOL_CASES_AX);
fun QCASE_TAC qtm =
  let val tm' = Parse.term_parser qtm
      val th = ISPEC tm' CASES_THM
  in 
   DISJ_CASES_THEN ASSUME_TAC th
  end;


fun CASE tm = 
  let fun pp ppstrm = 
          ( begin_block ppstrm CONSISTENT 2;
            add_string ppstrm "QCASE_TAC";
            add_break ppstrm (1,0);
            parsable_pp ppstrm tm;
            end_block ppstrm)
  in
     (pp, QCASE_TAC `^tm`) 
  end;

(*---------------------------------------------------------------------------
 * Shows when unknown methods are encountered in the plan.
 *---------------------------------------------------------------------------*)
fun HUH_TAC s = ALL_TAC;

fun HUH s = 
  let fun pp ppstrm = 
          ( begin_block ppstrm CONSISTENT 0;
              add_string ppstrm "HUH_TAC";
              add_break ppstrm (1,0);
              add_string ppstrm (Lib.quote s); end_block ppstrm)
  in
     (pp, HUH_TAC s) 
  end;


(*---------------------------------------------------------------------------
 * Trivial methods.
 *---------------------------------------------------------------------------*)
val MK_GEN_TAC   = 
  let fun pp ppstrm = 
          ( begin_block ppstrm CONSISTENT 0;
              add_string ppstrm "TRY"; add_break ppstrm (1,0);
              add_string ppstrm "GEN_TAC"; end_block ppstrm)
  in 
     (pp, Tactical.TRY Tactic.GEN_TAC)
  end;

val MK_DISCH_TAC =  (C add_string "DISCH_TAC", Tactic.DISCH_TAC) ;
val MK_ALL_TAC   =  (C add_string "ALL_TAC",   Tactical.ALL_TAC);


(*---------------------------------------------------------------------------
 * Need more info for this to be solid, i.e., which assumption is getting
 * broken apart. As it is, I just do all of them. The info is sort of there
 * in the plan, because names are given for this tactic. However, plans in
 * general are insufficiently well-annotated to determine the names 
 * attached to assumptions.
 *---------------------------------------------------------------------------*)
val ASM_CONJ_TAC = ASSUM_LIST (fn athl => 
 MAP_EVERY (fn tm => UNDISCH_TAC tm THEN STRIP_TAC)
           (map concl (filter (can dest_conj o concl) athl)));

val ASM_CONJ =  (C add_string "ASM_CONJ_TAC", ASM_CONJ_TAC);


(*---------------------------------------------------------------------------
 * Using hypotheses.
 *---------------------------------------------------------------------------*)
fun var_eq tm = 
   let val {lhs,rhs} = dest_eq tm
   in 
     (is_var lhs andalso not (mem lhs (free_vars rhs)))
     orelse 
     (is_var rhs andalso not (mem rhs (free_vars lhs)))
   end
   handle _ => false;

(*---------------------------------------------------------------------------
 * When invoked, we know that th is an equality, at least one side of which
 * is a var.
 *---------------------------------------------------------------------------*)
fun orient th = 
  let val c = concl th
      val {lhs,rhs} = dest_eq c
  in if (is_var lhs)
     then if (is_var rhs) 
          then if term_lt lhs rhs (* both vars, rewrite to textually smaller *)
               then SYM th 
               else th
          else th
     else SYM th
  end;

fun VSUBST_TAC tm = 
   UNDISCH_TAC tm THEN 
   DISCH_THEN (SUBST_ALL_TAC o orient);

val ASM_EQ_TAC = 
  W (fn (asl,_) => MAP_EVERY VSUBST_TAC (filter var_eq asl));

val ASM_EQ =  (C add_string ("ASM_EQ_TAC"), ASM_EQ_TAC) ;

(*---------------------------------------------------------------------------
 * Do nonstandard conditional rewriting in the goal.
 *---------------------------------------------------------------------------*)
 
(*---------------------------------------------------------------------------
 * Follow a path into a theorem. Once at the occurrence, perform the operation
 * and sew everything back up again on the way out.
 *---------------------------------------------------------------------------*)
fun on_thm impth [] th = MATCH_MP impth th
  | on_thm impth (oc::rst) th = 
      let val tm = concl th
      in if (is_conj tm)
         then let val (th1,th2) = (CONJUNCT1 th, CONJUNCT2 th)
              in case oc of 
                  1 => CONJ (on_thm impth rst th1) th2
                | 2 => CONJ th1 (on_thm impth rst th2)
                | _ => raise tac_err ("on_thm:is_conj","bad path")
              end 
         else if (is_neg tm)
              then on_thm 
                    (CONV_RULE (ONCE_DEPTH_CONV CONTRAPOS_CONV) impth) rst th
         else if (is_imp tm)
              then raise tac_err ("on_thm:is_imp","not top level")
         else raise tac_err ("on_thm", "fall through")
      end;

(*---------------------------------------------------------------------------
 * Backchaining at depth. This uses "on_thm" to perform the justification.
 *---------------------------------------------------------------------------*)
fun on_goal th = 
  let val (_,body) = strip_forall(concl th)
      val {ant,conseq} = dest_imp body
      fun go [] tm = let val (tmtheta,tytheta) = match_term conseq tm
                         val ant' = subst tmtheta (inst tytheta ant)
                     in (ant', on_thm th [])
                     end
        | go (oc::rst) tm =
           if (is_conj tm)
           then let val {conj1,conj2} = dest_conj tm
                in case oc of 
                   1 => let val (M, just) = go rst conj1
                        in (mk_conj{conj1=M, conj2=conj2},
                            on_thm th (oc::rst))       end
                 | 2 => let val (M, just) = go rst conj2
                        in (mk_conj{conj1=conj1, conj2=M},
                            on_thm th (oc::rst))       end
                 | _ => raise tac_err ("on_goal:is_conj","bad path")
                end
(*           else if (is_disj tm)
                then let val {disj1,disj2} = dest_disj tm
                     in case oc of 
                        1 => DISJ1_TAC THEN go rst disj1
                      | 2 => DISJ2_TAC THEN go rst disj2
                      | _ => raise tac_err ("on_goal:is_disj","bad path")
                     end
*)
           else if (is_neg tm)
                then raise tac_err ("on_goal:is_neg","path inside negation") 
           else if (is_imp tm)
                then raise tac_err ("on_goal:is_imp","not top level")
           else raise tac_err ("on_goal", "fall through")
  in 
    fn occs => fn g => 
         let val (M,j) = go occs (snd g)
         in ([(fst g,M)],fn [th] => j th)
         end
  end;


fun OCC_ANT th0 =
 let fun chase(1::occ) = DISCH_THEN (MP_TAC o on_thm th0 occ)
       | chase(2::occ) = DISCH_THEN (fn th => on_goal th0 occ THEN MP_TAC th)
       | chase(_::occ) = raise tac_err("chase", "occ. not in {1,2}")
       | chase []      = MATCH_MP_TAC th0
 in chase o rev
 end;

fun ARW_TAC th occ LEFT  g = OCC_ANT th occ g
  | ARW_TAC th occ RIGHT g = OCC_ANT th occ g

fun ANT_RW_TAC s occ lr g = ARW_TAC (clam_theorem s) occ lr g;

fun WFERT_IMP_TAC (path,dir) = 
   FIRST_ASSUM (fn th => ARW_TAC th path dir
          THEN UNDISCH_TAC (concl th) THEN DISCH_THEN (K ALL_TAC));



fun WFERT_IMP (path,dir) = 
   let fun pp ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "WFERT_IMP_TAC";
        add_break ppstrm (0,0);
        add_string ppstrm "(";
        begin_block ppstrm INCONSISTENT 2;
          pp_list (C(C add_string o Lib.int_to_string)) ppstrm path;
          add_string ppstrm ","; 
          add_break ppstrm (0,0);
          add_string ppstrm (case dir of LEFT => "LEFT" | RIGHT => "RIGHT"); 
        end_block ppstrm;
        add_string ppstrm ")"; 
       end_block ppstrm)
   in
   (pp, WFERT_IMP_TAC(path,dir))
 end;


fun ANT_RW (thname, path, lr) =
   let fun pp ppstrm 
    = (begin_block ppstrm CONSISTENT 2;
        add_string ppstrm "ANT_RW_TAC";
        add_break ppstrm (1,0);
        add_string ppstrm (Lib.quote thname);
        add_break ppstrm (1,0);
        add_string ppstrm "[";
          begin_block ppstrm CONSISTENT 2;
           pr_list (add_string ppstrm o Lib.int_to_string)
                   (fn () => add_string ppstrm ",") 
                   (fn () => add_break ppstrm (0,0)) path;
          end_block ppstrm;
        add_string ppstrm "]"; 
        add_break ppstrm (1,0);
        add_string ppstrm (case lr of LEFT => "LEFT" | RIGHT => "RIGHT");
       end_block ppstrm)
   in
   (pp, ANT_RW_TAC thname path lr)
   end;

(*---------------------------------------------------------------------------*
 *  Support for extended interaction                                         *
 *---------------------------------------------------------------------------*)

fun DECIDE q = decisionLib.DECIDE (Parse.term_parser q);

type vdecider = (ppstream -> unit) * (term -> thm)

fun ppPROVED s ppstrm = 
     ( begin_block ppstrm CONSISTENT 0;
       add_string ppstrm "PROVED_TAC"; add_break ppstrm (1,0);
       add_string ppstrm s; end_block ppstrm);

fun ppCONJECTURE M ppstrm = 
          ( begin_block ppstrm CONSISTENT 0;
              add_string ppstrm "CONJECTURE_TAC";
              add_break ppstrm (1,0);
              add_string ppstrm "`"; 
              pp_term ppstrm M;
              add_string ppstrm "`";
              end_block ppstrm);

fun ppREFUTED ppstrm = 
    ( begin_block ppstrm CONSISTENT 0;
      add_string ppstrm "FAIL_TAC \"REFUTED!!\""; end_block ppstrm);

fun PROVED_TAC thm = MATCH_ACCEPT_TAC thm;
fun CONJECTURE_TAC M = ALL_TAC;


(*---------------------------------------------------------------------------*
 * When at a theorem node in the plan, look up the theorem in the Fact base, *
 * and add it to the reference cell.                                         *
 *---------------------------------------------------------------------------*)
local open Fact
in
fun PROVED r s = 
  case (get s)
  of SOME(f as (_,Fact.Proved (thm, _))) => (r := (OLD f :: !r);
                                             (ppPROVED s, PROVED_TAC thm))
   | SOME _ => raise tac_err ("PROVED", "unproved")
   | NONE   => raise tac_err ("PROVED", "no such fact")
  end;

(*---------------------------------------------------------------------------*
 * When at a conjecture, classify it. If it's proved then generate a name    *
 * for it, and attach that with it in the fact dbase. The system will find   * 
 * it on the next go-round.                                                  *
 *---------------------------------------------------------------------------*)

type classifier = term -> string * Fact.fact

fun CONJECTURE classify r M = 
  let val (factoid as (s,fact)) = classify M
      open Fact
  in 
     r := (NEW factoid :: !r);
     case fact
       of Proved (thm,_) => (ppPROVED s,     PROVED_TAC thm)
        | CantProve M    => (ppCONJECTURE M, CONJECTURE_TAC M)
        | Refuted M      => (ppREFUTED,      FAIL_TAC"REFUTED!!")
  end;

end; (* functor VisibleTactic *)
