(****************************************************************************)
(*                                                                          *)
(*    Copyright 1998 University of Cambridge and University of Edinburgh    *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : schemes.sml                                              *)
(* DESCRIPTION   : Finding and sending induction schemes.                   *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 4th August 1998                                          *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 4th August 1998                                          *)
(****************************************************************************)

structure Schemes =
struct

local

open CLaReTPortable;

val built_in_types =
   ["fun","bool","prod","one","sum","ind"];

fun type_operators_in_type ops ty =
   let val {Args,Tyop} = dest_type ty
       val ops' = if (member Tyop ops) orelse (member Tyop built_in_types)
                  then ops
                  else Tyop :: ops
   in  type_operators_in_types ops' Args
   end
   handle HOL_ERR _ => ops
and type_operators_in_types ops [] = ops
  | type_operators_in_types ops (ty::tys) =
   type_operators_in_types (type_operators_in_type ops ty) tys;

fun type_operators_in_term types tm =
   (case (dest_term tm)
    of CONST {Name,Ty} => type_operators_in_type types Ty
     | VAR {Name,Ty} => type_operators_in_type types Ty
     | COMB {Rator,Rand} =>
          type_operators_in_term (type_operators_in_term types Rator) Rand
     | LAMB {Bvar,Body} => type_operators_in_term types Body);

fun type_operators_in_terms types [] = types
  | type_operators_in_terms types (tm::tms) =
   type_operators_in_terms (type_operators_in_term types tm) tms;

in

fun theory_of_type name =
   find (member name o map #Name o types)
      (current_theory () :: ancestry "-");

fun possible_schemes theory =
   let val induct = explode "induct"
       and Induct = explode "Induct"
       and INDUCT = explode "INDUCT"
       fun select name =
          let val chars = explode name
          in  (is_subsequence induct chars) orelse
              (is_subsequence Induct chars) orelse
              (is_subsequence INDUCT chars)
          end
   in  filter (select o fst) (theorems theory)
   end;

fun subjects_of_scheme th =
   type_operators_in_types [] (CallClam.subjects_of_scheme (concl th));

fun subjects_and_schemes names theory =
   let fun sub [] = []
         | sub ((name,th)::schemes) =
          let fun filter [] = sub schemes
                | filter (s::ss) =
                 if (member s names)
                 then (s,(name,th)) :: filter ss
                 else filter ss
          in  filter (subjects_of_scheme th)
          end
   in  sub (possible_schemes theory)
   end;

fun schemes_for_type_operators names =
   let val names_and_theories = map (fn n => (n,theory_of_type n)) names
       val theories = setify (map snd names_and_theories)
       val names_and_schemes =
          flat (map (fn t => subjects_and_schemes
                                (map fst (filter (fn (n,t') => t' = t)
                                             names_and_theories)) t)
                   theories)
   in  map (fn n =>
              (n,map snd (filter (fn (name,_) => name = n) names_and_schemes)))
          names
   end;

fun send_schemes_for_terms ignore tms =
   let fun process_scheme (tyop,[]) =
          Lib.say ("No scheme found for type operator `" ^ tyop ^ "'\n")
         | process_scheme (tyop,schemes) =
          (map (fn (name,th) =>
                   (Lib.say ("Sending scheme `" ^ name ^
                             "' for type operator `" ^ tyop ^ "'\n");
                    CallClam.send_scheme (name,th)
                    handle _ => Lib.say "... send failed\n")) schemes;
           ())
       val known = type_operators_in_types [] (CallClam.known_hol_types ())
       val tyops = type_operators_in_terms [] tms
       val unknown_tyops = subtract (subtract tyops known) ignore
       val schemes = schemes_for_type_operators unknown_tyops
   in  map process_scheme schemes; ()
   end;

val auto_send = ref true;

fun send_schemes gl =
   if !auto_send
   then let val (tms,tm) = gl
        in  send_schemes_for_terms [] (tm :: tms)
        end
   else ();

end;

end; (* Schemes *)
