(*---------------------------------------------------------------------------
 * FILE: plan2tactic.sml
 *
 * This structure provides code for mapping Clam proof plans into Client
 * tactics.
 *---------------------------------------------------------------------------*)

functor PLAN2TAC(structure VisibleTactic : VisibleTactic_sig
                 structure Clam2Client   : Clam2Client
                 sharing type VisibleTactic.term = Clam2Client.client_term) 

  : Plan2Tactic_sig =

struct

structure VisibleTactic = VisibleTactic;

open VisibleTactic;
open Clam2Client;
open PrettyPrint;
open Rsyntax;

fun tac_err(func,mesg) = Fail (func^": "^mesg);

fun quote s = String.concat ["\"", s, "\""];

fun string_to_direction "left"  = LEFT
  | string_to_direction "right" = RIGHT
  | string_to_direction  s      = raise tac_err("string_to_direction",quote s);


(*---------------------------------------------------------------------------*
 * Abstract syntax trees for compound tactics. A tactic representation       *
 * carries a prettyprinter at each leaf node, so that concrete syntax can    *
 * be generated easily.                                                      *
 *---------------------------------------------------------------------------*)
nonfix THEN THENL

datatype tacticAST = NO_TAC
                   | ALL_TAC
                   | PRIM  of VisibleTactic.vtactic
                   | THEN  of tacticAST * tacticAST
                   | THENL of tacticAST * tacticAST list


(*---------------------------------------------------------------------------
 * We translate methods to tacticASTs; from these we can generate both 
 * concrete syntax and code. 
 *---------------------------------------------------------------------------*)
local open ClamConstructors
      fun unarg(Method m) = m
        | unarg x = raise tac_err("mk_tac","unarg")
in
fun make_tac classifier ts X = 
 let val r = (ref []: (string * Fact.fact) Fact.age list ref)
     fun mk_tac X =
case X of
  (MethodApp("ind_strat",
    [Method(MethodThen(Induction (x,ivl), Methods L))])) =>
      let val (Method(MethodApp("lemma",[Method(MethodApp(s,[]))]))) = x
          fun convert (InductionVariable(Inhabit (id,ty),tm)) =
              let val hty = mk_client_type ts ty
                  val v = Clam2Client.mk_client_var
                              {Name=client_var_name ts id, Ty=hty}
              in 
                (v,mk_client_term ts tm hty)
              end
      in 
         THENL(PRIM (IND s (map convert ivl)), map mk_tac L)
      end
  | (MethodApp("elementary",       [Method m]))          => PRIM ELEMENTARY
  | (MethodApp("base_case",        [Method(Methods L)])) => mk_tacl L
  | (MethodApp("base_case",        [Method m]))          => mk_tac m
  | (MethodApp("sym_eval",         [Method(Methods L)])) => mk_tacl L
  | (MethodApp("sym_eval",         [Method m]))          => mk_tac m
  | (MethodApp("intro",            [Method(Methods L)])) => mk_tacl L
  | (MethodApp("intro",            [Method m]))          => mk_tac m
  | (MethodApp("step_case",        [Method(Methods L)])) => mk_tacl L
  | (MethodApp("step_case",        [Method m]))          => mk_tac m
  | (MethodApp("normalize",        [Method(Methods L)])) => mk_tacl L
  | (MethodApp("normalize",        [Method m]))          => mk_tac m
  | (MethodApp("normalize_term",   [Method(Methods L)])) => mk_tacl L
  | (MethodApp("normalize_term",   [Method m]))          => mk_tac m
  | (MethodApp("ripple",           [dir, Method m]))     => mk_tac m
  | (MethodApp("ripple_and_cancel",[Method(Methods L)])) => mk_tacl L

  | (New idlist) => PRIM (GENL_TAC idlist)
  | (Generalise (term,Inhabit (id,typ))) => 
       let val hty = mk_client_type ts typ
       in PRIM(SPEC_TAC(mk_client_term ts term hty, 
                        Clam2Client.mk_client_var{Name=id, Ty=hty})) 
       end
  (* Reduction: the next two differ only in "equ" vs. "equiv" *)
  | (MethodApp("reduction",[NumList path, 
       Method(Methods[MethodApp(s,[]),
         MethodApp("equ",[_,Method(MethodApp(lr,[]))])])])) 
     => PRIM(OCC_RW (s, path, string_to_direction lr))
  | (MethodApp("reduction",[NumList path, 
       Method(Methods[MethodApp(s,[]),
         MethodApp("equiv",[Method(MethodApp(lr,[]))])])]))
     => PRIM(OCC_RW (s, path, string_to_direction lr))
  | (MethodApp("reduction",[NumList path, 
       Method(Methods[MethodApp(s,[]),
         MethodApp("imp",[Method(MethodApp(lr,[]))])])])) 
     => PRIM(ANT_RW (s, path, string_to_direction lr))
  | (MethodApp("wave", 
      [dir, NumList path,
       Method(Methods[MethodApp(s,[]), 
           MethodApp("equ",[_,Method(MethodApp(lr,[]))])]), _])) 
     => PRIM(OCC_RW (s, path, string_to_direction lr))
  | (MethodApp("wave", 
      [dir, NumList path,
       Method(Methods[MethodApp(s,[]), MethodApp complementary,
              MethodApp("equ",[_,Method(MethodApp(lr,[]))])]), _])) 
      => PRIM(OCC_RW (s, path, string_to_direction lr))
  | (MethodApp("wave", 
      [dir, NumList path,
       Method(Methods[MethodApp(s,[]), 
          MethodApp("equiv",[Method(MethodApp(lr,[]))])]), _])) 
      => PRIM(OCC_RW (s, path, string_to_direction lr))
  | (MethodApp("wave", 
       [dir, NumList path,
        Method(Methods[MethodApp(s,[]), MethodApp complementary,
            MethodApp("equiv",[Method(MethodApp(lr,[]))])]), _]))
      => PRIM(OCC_RW (s, path, string_to_direction lr))
  | (MethodApp("wave", 
       [dir, NumList path,
        Method(Methods[MethodApp(s,[]), 
           MethodApp("imp",[Method(MethodApp(lr,[]))])]), _]))
      => PRIM(ANT_RW (s, path, string_to_direction lr))

  | (MethodApp("unblock_then_fertilize",[str,Method m])) => mk_tac m
  | (MethodApp("unblock_then_wave",[str,Method m]))      => mk_tac m
  | (MethodApp("unblock_fertilize_lazy", _))             => ALL_TAC
  | (MethodApp("unblock_lazy", _))                       => ALL_TAC
  | (MethodApp("unblock",
       [Method (MethodApp ("wave_front",[])),
        NumList path,
         Method(Methods[MethodApp (s,[]),
              MethodApp("equiv",[Method (MethodApp (lr,[]))])])]))
      => PRIM(OCC_RW (s, path, string_to_direction lr))

  | (MethodApp("fertilize",
         [Method(MethodApp("strong",[])),_])) => PRIM SFERT
  | (MethodApp("fertilize",[str,Method m]))   => mk_tac m
  | (MethodApp("fertilize_then_ripple", l))   => mk_tacl (map unarg l)
  | (MethodApp("fertilize_left_or_right",
                 [dir, Method(Methods [m])])) => mk_tac m

  | (MethodApp("weak_fertilize",  (* fertilizing an "=" goal *)
         [Method(MethodApp(lr,[])),In,NumList path,var])) 
     => let val d = string_to_direction lr
           val (Method(MethodApp(v,[]))) = var  (* must pick up on later *)
           val index = case d of LEFT => 1 | RIGHT => 2
       in PRIM (WFERT_EQ(path@[index,1],d))
       end
  | (MethodApp("weak_fertilize", (* fertilizing an "==>" goal *)
        [Method(MethodApp(lr,[])),Connective "=>",NumList path,var])) 
     => let val d = string_to_direction lr
           val index = case d of LEFT => 1 | RIGHT => 2
       in 
        PRIM(WFERT_IMP (path@[index], string_to_direction lr))
       end
  | (MethodApp("normal",
        [Method (MethodApp("univ_intro",_))]))  => PRIM MK_GEN_TAC
  | (MethodApp("normal",
        [Method (MethodApp("imply_intro",_))])) => PRIM MK_DISCH_TAC
  | (MethodApp("normal",
      [Method (MethodApp("conjunct_elim",_))])) => PRIM ASM_CONJ
  | (MethodApp("equal", _))                     => PRIM ASM_EQ
  | (MethodApp("idtac", _))                     => PRIM MK_ALL_TAC

  | (MethodApp("casesplit", [Method(Disjunction[notp,p])])) 
     => PRIM (CASE (mk_client_term ts p Clam2Client.client_bool))
  | (MethodApp("casesplit", 
      [Method(Disjunction _)])) => PRIM (HUH"cases>2")
  | (MethodApp("external_decision",
         [Method(External(_, _, Conjecture tm))])) 
     => PRIM (CONJECTURE classifier r 
                (mk_client_term ts tm Clam2Client.client_bool))
  | (MethodApp("external_decision", [Method(External(_, _,Thm s))])) 
                                => PRIM(PROVED r s)
  | (MethodApp (id,_))          => PRIM (HUH id)
  | (Methods L)                 => mk_tacl L
  | (MethodThen(_, Methods[]))  => raise tac_err("mk_tac","MethodThen")
  | (MethodThen(m1,Methods[m])) => THEN(mk_tac m1, mk_tac m)
  | (MethodThen(m1,Methods L))  => THENL(mk_tac m1,map mk_tac L)
  | (MethodThen(m1,m2))         => THEN(mk_tac m1, mk_tac m2)
  |        x                    => PRIM (HUH"<not handled yet>")
and (* We should never get to these. *)
    mk_arg_tac ts (Method method) = mk_tac method
  | mk_arg_tac ts In              = PRIM (HUH"In")
  | mk_arg_tac ts (NumList L)     = PRIM (HUH"NumList")
  | mk_arg_tac _ _                = PRIM (HUH"<not handled yet>")
and 
    mk_tacl L = end_itlist (curry THEN) (map mk_tac L)
in
  (mk_tac X, !r)
end 
end (* make_tac *)


(*---------------------------------------------------------------------------
 * Top level function. Build a tactic from a plan.
 *---------------------------------------------------------------------------*)
fun tactic_of classifier (ClamConstructors.Plan(goal,_,_,method,_),translations)
   = make_tac classifier translations method;

end; (* structure PlanToTactic *)
