(****************************************************************************)
(*                                                                          *)
(* Copyright 1997, 1998 University of Cambridge and University of Edinburgh *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : load.sml                                                 *)
(* DESCRIPTION   : Main file for HOL/Clam system.                           *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 23rd January 1997                                        *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 4th August 1998                                          *)
(****************************************************************************)

load_library_in_place decision_lib;
load_library_in_place taut_lib;

val mlsyn_dir = "claret";

use "Q.sig"; use "Q.sml";

fun itlist f L base_value =
   let fun it [] = base_value
         | it (a::rst) = f a (it rst)
   in it L 
   end;

fun rev_itlist f =
   let fun rev_it [] base = base
         | rev_it (a::rst) base = rev_it rst (f a base)
   in rev_it
   end;

val open_in = TextIO.openIn;
val close_in = TextIO.closeIn;
val open_out = TextIO.openOut;
val close_out = TextIO.closeOut;
fun output (os,s) = TextIO.outputSubstr (os,Substring.all s);
val std_out = TextIO.stdOut;
fun inputc is n = TextIO.inputN (is,n);

(*
structure Option : sig datatype 'a option = NONE | SOME of 'a end = General;
*)

structure Integer = 
struct
  open Int
  fun print (i:int) = output (std_out,Int.toString i)
end;

val integer_of_string = Lib.string_to_int
and string_of_integer = Lib.int_to_string;

structure Array = 
struct
 open Array
 val arrayoflist = Array.fromList
end;

structure String = 
struct
  open Portable.String
  fun print (s:string) = output (std_out,s)
  exception Substring = Subscript
  val length = String.size
  fun ordof (s,i) = Char.ord (String.sub (s,i));
end;

val ordof = String.ordof;
fun ord s = String.ordof(s,0);
val chr = Char.toString o Char.chr;
fun curry f x y = f (x,y);
fun revfold f b l = rev_itlist (curry f) b l;
fun fold f b l = itlist (curry f) b l;

structure List = 
struct
 open List
 val fold = fold
 val revfold = revfold
 exception Nth = Subscript
 exception Hd = Empty
 exception Tl = Empty
 exception NthTail
 fun nthtail ([],0)  = []
   | nthtail ([],n) = raise NthTail
   | nthtail (h::t,n) = nthtail (t,n-1)
end;

structure Vector =
struct
 open Vector
 val vector = Vector.fromList
end;

structure Ref =
struct
   fun inc x = x := !x + 1;
   fun dec x = x := !x - 1;
end;

structure System =
struct
   open System;
   structure Print = Compiler.Control.Print;
end;
val getpid = Posix.ProcEnv.getpid;
val system = Portable.system;

val concatl = String.concat;
val explode = map String.str o String.explode;
val implode = String.concat;
structure HOL_Portable = Portable;
val makestring = Int.toString;
val max = Int.max;

val list_induct = prove(Parse.term_parser
`!P:'a list -> bool. P[] /\ (!h t. P t ==> P (CONS h t)) ==> !l. P l`,
GEN_TAC THEN STRIP_TAC
   THEN MATCH_MP_TAC (theorem "list" "list_INDUCT")
   THEN ASM_REWRITE_TAC[]
   THEN REPEAT STRIP_TAC THEN FIRST_ASSUM MATCH_MP_TAC
   THEN ASM_REWRITE_TAC[]);


(* Bring in CLaReT-generated stuff. *)
structure HOLBool = Bool;
structure Bool =
   struct type bool = bool;
   open Option;
end;
structure PrettyPrint = Compiler.PrettyPrint;
structure SMLChar = Char;
use (mlsyn_dir ^ "/runtime/mlsyn_runtime.sml");
structure Char = SMLChar;
use "clam.ast.sml";
nonfix THEN;
use "clam.parse.sml";
infix THEN;
use "clam.print.sml";
structure CLaReTPortable = Portable;
structure Portable = HOL_Portable;
val upto = CLaReTPortable.upto;

use "socket_server.sml";
use "hol_to_clam.sml";
use"Btree.sig"; use "Btree.sml";
use"Fact.sig";  use"Fact.sml";
use"vistactic.sig";   use"vistactic.sml";
use"clam_to_hol.sig"; use"clam_to_hol.sml";
use"clam2client.sig"; 
use"plan2tactic.sig"; use"plan2tactic.sml"; 
use"call_clam.sml";
use"ei.sig"; use "ei.sml";
use "definitions.sml";
use "schemes.sml";

(*---------------------------------------------------------------------------*
 * Generate a tactic from a tacticAST                                        *
 *---------------------------------------------------------------------------*)

local open CallClam.Plan2Tactic
     nonfix THEN THENL
    fun AST2Tactic NO_TAC         = Tactical.NO_TAC
      | AST2Tactic ALL_TAC        = Tactical.ALL_TAC
      | AST2Tactic (PRIM p)       = VisibleTactic.tactic_of p
      | AST2Tactic (THEN (t1,t2)) = Tactical.THEN(AST2Tactic t1,AST2Tactic t2)
      | AST2Tactic (THENL(t,L))   = Tactical.THENL(AST2Tactic t, 
                                                   map AST2Tactic L)
in
  val CLAM_TAC :tactic = fn g => 
        (Definitions.send_definitions g;
         Schemes.send_schemes g;
         CallClam.CLAM_TAC0 (EI.std_classifier()) AST2Tactic g)

  val EI_TAC = EI.EI (AST2Tactic o fst)

  (*-------------------------------------------------------------------------*
   * The `EI.alwaysFail' classifier always says "CantProve"; it should       *
   * thus force Clam to act the old way.                                     *
   *-------------------------------------------------------------------------*)
  val OLD_CLAM_TAC :tactic = EI.EI0 EI.alwaysFail (AST2Tactic o fst)
end;

(*---------------------------------------------------------------------------*
 * Generate concrete syntax for tactics                                      *
 *---------------------------------------------------------------------------*)
fun with_ppstream ppstrm = 
  let open PrettyPrint
  in
          {add_break=add_break ppstrm, 
         add_string=add_string ppstrm, 
        begin_block=begin_block ppstrm, 
          end_block=fn () => end_block ppstrm, 
        add_newline=fn () => add_newline ppstrm, 
     clear_ppstream=fn () => clear_ppstream ppstrm, 
     flush_ppstream=fn () => flush_ppstream ppstrm}
  end;


fun pr_list pfun dfun bfun =
  let fun pr [] = ()
        | pr [i] = pfun i
        | pr (i::rst) = ( pfun i ; dfun() ; bfun() ; pr rst )
  in  pr   end;


fun pp_tacticAST ppstrm = 
  let nonfix THEN THENL
      open CallClam.Plan2Tactic
      val {add_break, add_newline, add_string,
           begin_block, end_block, ...} = with_ppstream ppstrm
      val CB = PrettyPrint.CONSISTENT
      val IB = PrettyPrint.INCONSISTENT
      fun pp NO_TAC        = add_string"NO_TAC"
        | pp ALL_TAC       = add_string"ALL_TAC"
        | pp (PRIM p)      = VisibleTactic.pp_of p ppstrm
        | pp (THEN(t1,t2)) = (begin_block IB 0;
                              pp t1; add_break (1,0);
                              add_string "THEN"; add_break(1,0);
                              pp t2;
                              end_block())
        | pp (THENL(t,L)) = 
            (begin_block IB 0;
             pp t; add_break (1,0);
             add_string "THENL"; add_break(1,0);
             add_string"[";
               begin_block CB 0;
                    pr_list pp (fn () => add_string",") 
                               (fn () => add_break(0,0)) L;
               end_block();
             add_string"]";
             end_block())
  in fn x => (begin_block CB 0; pp x; end_block())
  end;

(*---------------------------------------------------------------------------*
 * Shows the tactic arising from the plan.                                   *
 *---------------------------------------------------------------------------*)
fun tac_of0 f () = 
  let val (tacAST,facts) = f () (top_goal())
      open PrettyPrint
      val ppstrm = mk_ppstream
             {consumer = fn s => output (Portable.std_out,s),
              flush = fn () => Portable.flush_out Portable.std_out,
              linewidth = 79}
  in
   (case facts
     of [] => pp_tacticAST ppstrm tacAST
      |  _ => let val {add_break, add_newline, add_string,
             begin_block, end_block,...} = with_ppstream ppstrm
           in
              begin_block CONSISTENT 0;
              add_string "let"; add_break (1,0);
              begin_block CONSISTENT 0;
              pr_list 
                (fn (s,Fact.Proved(_,f)) =>
                       ( begin_block INCONSISTENT 2;
                         add_string "val"; add_break (1,0);
                         add_string  s;    add_break (1,0);
                         add_string "=";   add_break (1,0);
                         f ppstrm; end_block())
                    | _ => ())
                (fn () => ())
                (fn () => add_break(0,0)) (rev facts);
              end_block(); 
              add_break (1,0);
              add_string "in"; add_break(1,2); pp_tacticAST ppstrm tacAST;
              add_break(1,0); add_string "end"; 
              end_block()
            end);
      flush_ppstream ppstrm;
      output(Portable.std_out,"\n")
     end;

(* Analogue of CLAM_TAC -- prints out tactic in ML *)
val tac_of = 
  let val g = CallClam.CLAM_TAC_AST0 o EI.std_classifier
      fun h x = (I ## map Fact.dest_age) o g x
  in tac_of0 h
  end;

(* Analogue of OLD_CLAM_TAC -- prints out tactic in ML *)
val old_tac_of = 
  let fun g0 x y = EI.EI0 EI.alwaysFail x y
      fun h () = g0 K
  in tac_of0 h
  end;

(* Analogue of EI_TAC -- prints out tactic in ML *)
val ei_tac_of = 
  let fun g0 x y = EI.EI0 EI.std_classifier x y
      fun h () = g0 K
  in tac_of0 h
  end;

fun plan_of () = CallClam.CLAM_PLAN (top_goal());

open CallClam;
open VisibleTactic;

fun SPEC_ALL_THEN tac (h,g) =
   foldr (fn (v,tac) => Tactic.SPEC_TAC (v,v) THEN tac)
      tac (free_vars g) (h,g);

val _ = Compiler.PPTable.install_pp 
          ["CallClam","Plan2Tactic","tacticAST"]
           pp_tacticAST;

val _ = Compiler.PPTable.install_pp  
          ["ClamAST","plan"]
           (Lib.C (ClamPP.PP.PPBoxes.write_ppbox 
                   o ClamPP.print_plan DefaultStringTable.string_table));

(* 
val _ = Compiler.PPTable.install_pp 
          ["ClamAST","method"]
           (Lib.C (ClamPP.PP.PPBoxes.write_ppbox 
                   o ClamPP.print_method DefaultStringTable.string_table));

val _ = Compiler.PPTable.install_pp  
          ["ClamAST","argument"]
           (Lib.C (ClamPP.PP.PPBoxes.write_ppbox 
                   o ClamPP.print_argument DefaultStringTable.string_table));
*)
