/****************************************************************************/
/*                                                                          */
/* Copyright 1997, 1998 University of Cambridge and University of Edinburgh */
/*                                                                          */
/*                           All rights reserved.                           */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/* FILE          : hol-clam.pl                                              */
/* DESCRIPTION   : File for creating a Clam executable for use with HOL.    */
/*                 (Load into a Clam executable.)                           */
/*                                                                          */
/* WRITES FILES  : hol-clam                                                 */
/*                                                                          */
/* AUTHOR        : R.J.Boulton                                              */
/* DATE          : 4th March 1997                                           */
/*                                                                          */
/* LAST MODIFIED : R.J.Boulton                                              */
/* DATE          : 31st July 1998                                           */
/****************************************************************************/

/* Prolog should not interactively query when predicates are redefined */
:- prolog_flag(redefine_warnings,_,off).

/* Redefine guess_type/3 to work with functions that haven't been defined in
 * terms of p_ind, etc.
 */
guess_type(App,_,Type) :-
    functor(App,F,N),
    functor(MetaApp,F,N),
    recorded(theorem,theorem(_,eqn,Eqn,_),_),
    matrix(_,MetaApp=_ in Type,Eqn).
guess_type(X, H,XType) :-
    memberchk(X:XType,H).
guess_type( lambda(V,Term),H, Vt => TermType) :-
    guess_type(Term, [V:Vt|H],TermType).
guess_type( F of Arg,H,CoDomain) :-
    % Account for dependant and independant types
    guess_type(F,  H,FType),
    once((FType = (_:Domain=>CoDomain);
          FType =   (Domain=>CoDomain))),!,
    meta_try guess_type(Arg, H, Domain),
    meta_try guess_type(Arg, H, CoDomain).
guess_type(F of Arg,H,XType) :-
    \+ noprove =: _,
    eval(F of Arg,Y), (F of Arg) \= Y,
    guess_type(Y,  H,XType).


/* Introduce new logical objects */

new_scheme(Name,G):-
    ctheorem(Name):=problem([]==>G,_,_,_),cpos(Name):=[],
    record_thm(Name,scheme),
/*
    construct_scheme(G,Skels,Scheme),
    asserta(scheme(Skels,Name,Scheme)).
*/
    add_induction_scheme(Name).

new_eqn(Name,G):-
    ctheorem(Name):=problem([]==>G,_,_,_),cpos(Name):=[],
    record_thm(Name,eqn).

new_eqns(Name,G,Names):-
    new_eqns(Name,G,1,Names,_).
new_eqns(Name,G1#G2,M,Names,N):-
    new_eqns(Name,G1,M,Names1,P),
    new_eqns(Name,G2,P,Names2,N),
    append(Names1,Names2,Names).
new_eqns(Name,G,M,[NameM],N):-
    concat_atom([Name,M],NameM),
    new_eqn(NameM,G),
    N is M + 1.

new_def(Name,G):-
    new_eqns(Name,G,Names),
    add_rewrite_rules_list(Names),
    add_complementary_sets(Names),
    add_reduction_rules(Names).

new_rule(Name,G):-
    ctheorem(Name):=problem([]==>G,_,_,_),cpos(Name):=[],
    record_thm(Name,wave),
    add_rewrite_rules_list([Name]),
    add_reduction_rules([Name]),
    add_cancel_rule(Name).

new_goal(Name,H==>G):-
    ctheorem(Name):=problem(H==>G,_,_,_),cpos(Name):=[],
    record_thm(Name,thm).


/* Delete logical objects */

delete_scheme(Name):-
    if(lib_present(scheme(Name)),lib_delete(scheme(Name))).

delete_def(Name):-
    lib_delete_aux(Name,1).

delete_rule(Name):-
    if(lib_present(thm(Name)),lib_delete(thm(Name))),
    if(lib_present(wave(Name)),lib_delete(wave(Name))),
    if(lib_present(red(Name)),lib_delete(red(Name))).

delete_goal(Name):-
    if(lib_present(thm(Name)),lib_delete(thm(Name))).

quiet_delete_method(M/A):-
    lib_present(mthd(M/A)),
    delete_method(M/A),
    writef('Deleted method %t\n',[M/A]).
quiet_delete_method(_).


/* Calling the planner */

:- dynamic planner_time_limit/1.
planner_time_limit(60).

prove_goal(Name,proof_plan(H==>G,Name,TimeTaken,Plan,Planner)):-
    select(Name),
    planner_time_limit(TimeLimit),
    dplan(TimeLimit,Plan),
    print_plan(Plan),
    dplan:=Plan,
    recorded(proof_plan,proof_plan(H==>G,Name,TimeTaken,Plan,Planner),_).

prove_and_save(Name,File,Plan):-
    prove_goal(Name,Plan),
    tell(File),
    writeq(Plan),write('.'),nl,
    told.


/* Process a term that represents some operation */
process_term(new_scheme(Name,G),[]):-
    new_scheme(Name,G).
process_term(new_def(Name,G),[]):-
    new_def(Name,G).
process_term(new_rule(Name,G),[]):-
    new_rule(Name,G).
process_term(new_goal(Name,H==>G),[]):-
    new_goal(Name,H==>G).
process_term(new_fact(Name,Type,G),[]):-
    new_external_fact(Name,Type,G).
process_term(delete_scheme(Name),[]):-
    delete_scheme(Name).
process_term(delete_def(Name),[]):-
    delete_def(Name).
process_term(delete_rule(Name),[]):-
    delete_rule(Name).
process_term(delete_goal(Name),[]):-
    delete_goal(Name).
process_term(delete_fact(Name),[]):-
    delete_external_fact(Name).
process_term(delete_all_facts,[]):-
    delete_all_external_facts.
process_term(prove_goal(Name),Plan):-
    prove_goal(Name,Plan).
process_term(prove_and_save(Name,File),Plan):-
    prove_and_save(Name,File,Plan).
process_term(trace_plan(New),Current):-
    trace_plan(Current,New).
process_term(time_limit(New),Current):-
    planner_time_limit(Current),
    retract(planner_time_limit(_)),
    asserta(planner_time_limit(New)).
process_term(iteration(on),[]):-
    load_method(external_decision/1,first),
    load_method(external_lemma/1,first).
process_term(iteration(off),[]):-
    quiet_delete_method(external_decision/1),
    quiet_delete_method(external_lemma/1).


/* Load new methods */
/*
:- load_method(classic_elem/1,first).
:- load_method(classic_simp/1,last).
*/

/* Save a new Clam executable */
:- save('hol-clam'),
   ((prolog_flag(argv,[Type,Name,TimeOut,Debug]),
     atom_chars(TimeOut,Chars), number_chars(Secs,Chars),
     use_module(library(sockets)),
     server(Type,Name,Secs:0,Debug));
    true).
