(****************************************************************************)
(*                                                                          *)
(*            Copyright 1994, 1995, 1996 University of Cambridge            *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : sml-nj-093.sml                                           *)
(* DESCRIPTION   : General purpose functions (for SML/NJ version 0.93).     *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 31st January 1994                                        *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 30th July 1996                                           *)
(****************************************************************************)

structure Portable : PORTABLE =
struct

val chr = chr;
val ord = ord;
val explode = explode;
val implode = implode;


structure Option : sig datatype 'a option = NONE | SOME of 'a end = Bool;
open Option;

exception Fail = General.Fail;


fun can f x = (f x; true) handle _ => false;

fun cannot f x = not (can f x);

fun funpow f n x = if (n <= 0) then x else funpow f (n - 1) (f x);

fun curry f x y = f (x,y);

fun uncurry f (x,y) = f x y;


exception Hd = List.Hd;
val hd = List.hd;

exception Tl = List.Tl;
val tl = List.tl;

val null = List.null;

val length = List.length;

fun flat [] = []
  | flat (x::xs) = x @ (flat xs);


val fold = List.fold;

val revfold = List.revfold;


exception Nth = List.Nth;
val nth = List.nth;

exception NthHead;
fun nthhead (_,0) = []
  | nthhead ([],_) = raise NthHead
  | nthhead (x::xs,n) = x::(nthhead (xs,n - 1));

exception NthTail = List.NthTail;
val nthtail = List.nthtail;

fun last l = nth (l,length l - 1);

fun butlast l = nthhead (l,length l - 1);

fun rotate n [] = []
  | rotate n l =
   let val pos = n mod (length l)
   in  nthtail (l,pos) @ nthhead (l,pos)
   end;


val exists = List.exists;

fun forall p l = not (exists (not o p) l);

fun filter p [] = []
  | filter p (x::xs) = if p x then x::(filter p xs) else filter p xs;


exception Find;
fun find p [] = raise Find
  | find p (x::xs) = if (p x) then x else find p xs;

exception NOT_FOUND;

fun assoc x [] = raise NOT_FOUND
  | assoc x ((x',y)::rest) = if (x = x') then y else assoc x rest;

fun rev_assoc y [] = raise NOT_FOUND
  | rev_assoc y ((x,y')::rest) = if (y = y') then x else rev_assoc y rest;

fun gen_assoc key x [] = raise NOT_FOUND
  | gen_assoc key x (y::ys) = if (key y = x) then y else gen_assoc key x ys;


fun zip ([],[]) = []
  | zip (x::xs,y::ys) = (x,y)::(zip (xs,ys))
  | zip _ = raise Fail "zip";

fun unzip [] = ([],[])
  | unzip ((x,y)::xys) =
   let val (xs,ys) = unzip xys
   in  (x::xs,y::ys)
   end;


fun member_component f x l = exists (fn y => f y = x) l;

fun member x l = member_component (fn x => x) x l;

fun intersect l1 l2 = filter (fn x => member x l2) l1;

fun subtract l1 l2 = filter (fn x => not (member x l2)) l1;

fun union l1 l2 = l1 @ (subtract l2 l1);

fun set_equal l1 l2 =
   (forall (fn x => member x l2) l1) andalso
   (forall (fn x => member x l1) l2);

fun remove_duplicates p l =
   let fun remove [] keep = keep
         | remove (x::xs) keep =
          if (exists (fn x' => p (x,x')) keep)
          then remove xs keep
          else remove xs (x::keep)
   in  rev (remove l [])
   end;

fun setify l = remove_duplicates (op =) l;


fun mapcan f [] = []
  | mapcan f (x::xs) =
   let val y = f x in y::(mapcan f xs) end handle _ => mapcan f xs;

fun duplicate_components f l =
   let fun dupl ds [] = rev ds
         | dupl ds (x::xs) =
          if (member_component f (f x) ds) orelse (member_component f (f x) xs)
          then dupl (x::ds) xs
          else dupl ds xs
   in  dupl [] l
   end;

fun duplicates l =
   let fun dupl [] = []
         | dupl (x::xs) = if (member x xs) then (x :: dupl xs) else dupl xs
   in  setify (dupl l)
   end;

fun split_at x [] = ([],[])
  | split_at x (l as y::ys) =
   if (y = x)
   then ([],l)
   else let val (l1,l2) = split_at x ys
        in  (y::l1,l2)
        end;

fun upto from to =
   if (from > to)
   then []
   else from::(upto (from + 1) to);

fun map_thread f ([],env) = ([],env)
  | map_thread f (x::xs,env) =
   let val (y,env') = f (x,env)
       val (ys,env'') = map_thread f (xs,env')
   in  (y::ys,env'')
   end;

fun flat_map_thread f (xs,env) =
   let val (ys,env') = map_thread f (xs,env)
   in  (flat ys,env')
   end;

fun initial_length f [] = 0
  | initial_length f (x::xs) = if (f x) then 1 + initial_length f xs else 0;

fun common_head (lists as (_,[])) = ([],lists)
  | common_head (lists as ([],_)) = ([],lists)
  | common_head (lists as (x::xs,y::ys)) =
   if (x = y)
   then let val (commons,remainders) = common_head (xs,ys)
        in (x::commons,remainders)
        end
   else ([],lists);

fun is_subsequence [] _ = true
  | is_subsequence (_::_) [] = false
  | is_subsequence (sub as x::xs) (y::ys) =
   if (x = y)
   then is_subsequence xs ys
   else is_subsequence sub ys;

fun pairings f (xs,ys) =
   flat (map (fn x => map (fn y => f (x,y)) ys) xs);

fun replace_or_add combine id_of l1 l2 =
   let fun replace [] xs' = xs'
         | replace (x::xs) xs' =
          let val id = id_of x
              val x' = find (fn x' => id_of x' = id) l2
          in  replace xs ((combine x x')::xs')
          end
          handle Find => replace xs (x::xs')
       val ids = map id_of l1
   in  (rev (replace l1 [])) @ (filter (fn x => not (member (id_of x) ids)) l2)
   end;


fun string_of_int i =
   let fun digits 0 = ""
         | digits i = digits (i div 10) ^ chr (ord("0") + (i mod 10))
   in  if (i = 0)
       then "0"
       else if (i < 0)
            then "~" ^ digits (~i)
            else digits i
   end;

exception IntOfString;
fun int_of_string s =
   let val zero = ord("0")
       fun digit (c,result) =
          if ("0" <= c) andalso (c <= "9")
          then ord(c) - zero + 10 * result
          else raise IntOfString
   in  case (explode s)
       of ["~"] => raise IntOfString
        | [] => raise IntOfString
        | "~"::cs => ~(revfold digit cs 0)
        | cs => revfold digit cs 0
   end;

exception Substring = String.Substring;
val substring = String.substring;

val ordof = String.ordof;


val min = Integer.min;

val max = Integer.max;


val inc = Ref.inc;

val dec = Ref.dec;

end;
