(****************************************************************************)
(*                                                                          *)
(*            Copyright 1993, 1994, 1996 University of Cambridge            *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : ppout.sml                                                *)
(* DESCRIPTION   : Pretty-printing to output streams.                       *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 7th December 1993                                        *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 16th May 1996                                            *)
(****************************************************************************)

signature LABEL_PPOUT =
sig
   type 'a ppbox
   val display_ppbox : int -> 'a ppbox -> unit
   val output_ppbox  : string -> int -> 'a ppbox -> unit
   type 'a text
   val display_text  : 'a text -> unit
   val output_text   : string -> 'a text -> unit
end;

functor LabelPPOutFun
   (LabelPPBoxes :
      sig
         structure PPStream : PPSTREAM
         structure Scale : sig
                              val char_size : int
                              val num_of_chars : int -> int
                           end
         type 'a ppbox
         val write_ppbox : 'a ppbox -> PPStream.ppstream -> unit
         type 'a text
         val write_text : 'a text -> PPStream.ppstream -> unit
         val width_of_text : 'a text -> int
      end) : LABEL_PPOUT =
struct

structure PPStream = LabelPPBoxes.PPStream;

type 'a ppbox = 'a LabelPPBoxes.ppbox
and 'a text = 'a LabelPPBoxes.text;

(*--------------------------------------------------------------------------*)
(* Compute the width (in number of characters) required by a text.          *)
(*--------------------------------------------------------------------------*)

fun width_of_text text =
   LabelPPBoxes.Scale.num_of_chars (LabelPPBoxes.width_of_text text);

(*--------------------------------------------------------------------------*)
(* Wrappers for top-level pretty-printing.                                  *)
(* A final new line is added and the stream is flushed.                     *)
(*--------------------------------------------------------------------------*)

local

fun wrap ppf pps =
   let val {consumer,flush,...} = PPStream.dest_ppstream pps
   in  (ppf pps;
        PPStream.flush_ppstream pps;
        consumer "\n";
        flush ())
   end;

in

fun wrap_ppbox box = wrap (LabelPPBoxes.write_ppbox box);
fun wrap_text text = wrap (LabelPPBoxes.write_text text);

end;

(*--------------------------------------------------------------------------*)
(* Function to make a pretty-printer stream from an output stream.          *)
(*--------------------------------------------------------------------------*)

fun out_pps str width =
   PPStream.mk_ppstream
      {consumer = (fn s => output (str,s)),
       flush = (fn () => ()),
       linewidth = width};

(*--------------------------------------------------------------------------*)
(* A pretty-printer stream for standard output.                             *)
(*--------------------------------------------------------------------------*)

val std_pps = out_pps std_out;

(*--------------------------------------------------------------------------*)
(* Pretty-print a ppbox/text to standard output.                            *)
(*--------------------------------------------------------------------------*)

fun display_ppbox width box = wrap_ppbox box (std_pps width)
and display_text text = wrap_text text (std_pps (width_of_text text));

(*--------------------------------------------------------------------------*)
(* Pretty-print a ppbox/text to a file.                                     *)
(*--------------------------------------------------------------------------*)

local

fun output filename width ppf =
   let val str = open_out filename
       val pps = out_pps str width
   in  (ppf pps; close_out str)
   end;

in

fun output_ppbox filename width = output filename width o wrap_ppbox
and output_text filename text =
   output filename (width_of_text text) (wrap_text text);

end;

end; (* LabelPPOutFun *)


signature PPOUT =
sig
   type ppbox
   val display_ppbox : int -> ppbox -> unit
   val output_ppbox  : string -> int -> ppbox -> unit
end;

functor PPOutFun
   (PPBoxes :
      sig
         structure PPStream : PPSTREAM
         structure Scale : sig
                              val char_size : int
                              val num_of_chars : int -> int
                           end
         type ppbox
         val write_ppbox : ppbox -> PPStream.ppstream -> unit
      end) : PPOUT =
let structure LabelPPOut =
       LabelPPOutFun
          (struct
              structure PPStream = PPBoxes.PPStream;
              structure Scale = PPBoxes.Scale;
              type 'a ppbox = PPBoxes.ppbox;
              val write_ppbox = PPBoxes.write_ppbox;
              type 'a text = PPBoxes.ppbox;
              val write_text = PPBoxes.write_ppbox;
              val width_of_text = fn _ => 0;
           end)
in  struct
       type ppbox = PPBoxes.ppbox
       val display_ppbox = LabelPPOut.display_ppbox
       val output_ppbox = LabelPPOut.output_ppbox
    end
end;
