structure Clam2HOL : Clam2HOL =
struct

structure Const = ClamConstructors

fun map_err(func,mesg) = HOL_ERR{origin_structure="PlantoTactic",
                                 origin_function = func,
                                 message = mesg};

(*---------------------------------------------------------------------------
 * Reverse mapping from Clam types to HOL types.
 *---------------------------------------------------------------------------*)
fun clam_to_hol_var_type (tyvs,_,_,_) x =
       assoc x tyvs handle NOT_FOUND =>
       raise map_err("clam_to_hol_type",
                     "unknown Clam type variable"^Lib.quote x);

fun clam_to_hol_const_type (_,tycs,_,_) x =
       assoc x tycs handle NOT_FOUND =>
       raise map_err("clam_to_hol_type",
                     "unknown Clam type constant"^Lib.quote x);


(*---------------------------------------------------------------------------
 * Reverse mapping from Clam constants to HOL constants.
 *---------------------------------------------------------------------------*)
fun clam_to_hol_const (_,_,_,tmcs) x =
       assoc x tmcs handle NOT_FOUND =>
       raise map_err("clam_to_hol_const","unknown Clam constant "^Lib.quote x);

fun clam_to_hol_var (_,_,tmvs,_) x = assoc x tmvs handle NOT_FOUND => x;


fun mk_hol_type ts (Const.Identifier s) =
       (mk_type{Tyop=clam_to_hol_const_type ts s,Args=[]} handle HOL_ERR _ =>
        mk_vartype (clam_to_hol_var_type ts s))
  | mk_hol_type ts (Const.Prop (fty,"=>",aty)) =
       mk_type{Tyop="fun",Args=[mk_hol_type ts fty,mk_hol_type ts aty]}
  | mk_hol_type ts (Const.Apply (s,L)) =
       mk_type{Tyop=clam_to_hol_const_type ts s,Args=map (mk_hol_type ts) L};


(*---------------------------------------------------------------------------*
 * This uses prettyprinting followed by parsing and type inference to        *
 * build HOL terms from Clam terms. Slow (but not a bottleneck, I imagine)   *
 * and ugly. It will probably not do the right thing in terms involving      *
 * insufficiently constrained polymorphic constants.                         *
 *---------------------------------------------------------------------------*)
local open Const
in
fun clam2string ts True A = "T"::A
  | clam2string ts Void A = "F"::A
  | clam2string ts (Identifier s) A = 
      (clam_to_hol_const ts s handle _ => clam_to_hol_var ts s)::A
  | clam2string ts (Prop (Inhabit(v,ty),"=>",tm)) A = 
     "!" :: clam_to_hol_var ts v 
         :: " : " :: type_to_string (mk_hol_type ts ty)
         :: "." 
         :: clam2string ts tm A
  | clam2string ts (Prop (Inhabit(v,ty),"#",tm)) A = 
     "?" :: clam_to_hol_var ts v 
         :: " : " :: type_to_string (mk_hol_type ts ty)
         :: "." 
         :: clam2string ts tm A
  | clam2string ts (Prop (p1,"#",p2)) A = 
     "(" :: clam2string ts p1
               (" /\\ " :: clam2string ts p2 (")"::A))
  | clam2string ts (Prop (p1,"\\",p2)) A = 
     "(" :: clam2string ts p1
               (" \\/ " :: clam2string ts p2 (")"::A))
  | clam2string ts (Prop (p,"=>",Void)) A = 
     "~(" :: clam2string ts p (")"::A)
  | clam2string ts (Prop (p1,"=>",p2)) A = 
     "(" :: clam2string ts p1
               (" ==> " :: clam2string ts p2 (")"::A))
  | clam2string ts (Prop (p1,"<=>",p2)) A = 
     "(" :: clam2string ts p1
               (" : bool = " :: clam2string ts p2 (")"::A))
  | clam2string ts (Number i) A = Lib.int_to_string i::A
  | clam2string ts (Apply(id,L)) A = 
      let val str = clam_to_hol_const ts id
      in 
        if (is_infix str) 
        then "(" :: clam2string ts (hd L)
                     (" " :: str :: " " 
                      :: clam2string ts (hd(tl L)) (")"::A))
        else "(" :: str 
                 :: itlist (fn a => fn S => " "::clam2string ts a S) L 
             (")"::A)
        end
  | clam2string ts (Eq(lhs,rhs,ty)) A = 
       "(" :: clam2string ts lhs (" : " :: type_to_string (mk_hol_type ts ty)
           :: " = " 
           :: clam2string ts rhs (")"::A))
  | clam2string ts _ _ = raise map_err ("clam2string", "fell through!")
end;


fun mk_hol_term ts tm ty = Parse.string_to_term 
 (String.concat
     [String.concat(clam2string ts tm []), " : ", type_to_string ty]);

end;
