(*--------------------------------------------------------------------------*)
(* Load the LR parser and build a structure for use with the lexer          *)
(*--------------------------------------------------------------------------*)

use "clam.grm.sig";
use "clam.grm.sml";
structure ClamTablesTokens = 
   ClamLrValsFun (structure Token = LrParser.Token);

(*--------------------------------------------------------------------------*)
(* Load the lexer                                                           *)
(*--------------------------------------------------------------------------*)

use "clam.lex.sml";

(*--------------------------------------------------------------------------*)
(* Main structure for Clam                                                  *)
(*--------------------------------------------------------------------------*)

structure ClamParse =
struct

(*--------------------------------------------------------------------------*)
(* Instantiate the lexer                                                    *)
(*--------------------------------------------------------------------------*)

structure L = ClamLexFun (structure Tokens = ClamTablesTokens.Tokens);

(*--------------------------------------------------------------------------*)
(* Build structure for parser                                               *)
(*--------------------------------------------------------------------------*)

structure P =
   Join (structure ParserData = ClamTablesTokens.ParserData
         structure Lex = L
         structure LrParser = LrParser);

(*--------------------------------------------------------------------------*)
(* Parsing functions for Clam                                               *)
(*--------------------------------------------------------------------------*)

local

val error = L.UserDeclarations.error;

in

(*--------------------------------------------------------------------------*)
(* Parsing of strings                                                       *)
(*--------------------------------------------------------------------------*)

fun parse_strings {line_number} ss =
   let val ss_ref = ref ss
       fun get_chars _ =
          case !ss_ref
          of [] => ""
           | s::ss => (ss_ref := ss; s)
       val _ = (L.yylineno := line_number)
       val lexer = P.makeLexer get_chars
       val (res,_) = P.parse (0,lexer,error,())
   in  res
   end;

(*--------------------------------------------------------------------------*)
(* Parsing a file                                                           *)
(*--------------------------------------------------------------------------*)

fun parse_file filename =
   let val _ = (L.yylineno := 1)
       val instr = open_in filename
       val lexer = P.makeLexer (inputc instr)
       val (res,_) = P.parse (0,lexer,error,())
                     handle e => (close_in instr; raise e)
   in  (close_in instr; res)
   end;

end;

end; (* ClamParse *)
