#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <mlvalues.h>
#include <alloc.h>
#include <memory.h>
#include <fail.h>
#include <str.h>
#include <signals.h>

/*
#include "unix.h"

#ifdef HAS_SOCKETS
#include "socketaddr.h"
#endif
*/
#if defined(HAS_SOCKETS) && defined(MSG_OOB) && defined(MSG_DONTROUTE) && defined(MSG_PEEK)


/* ML type: unit -> int * int * int * int * int * int * int * int * int *int */
value mlsocket_constants(value dummy)	/* ML */
{
  value res = alloc_tuple(10);
  Field(res, 0) = Val_long(SOCK_STREAM);
  Field(res, 1) = Val_long(SOCK_DGRAM);
  Field(res, 2) = Val_long(PF_UNIX);
  Field(res, 3) = Val_long(PF_INET);
  Field(res, 4) = Val_long(0); /* NO_RECVS */
  Field(res, 5) = Val_long(1); /* NO_SENDS */
  Field(res, 6) = Val_long(2); /* NO_RECVS_OR_SENDS */
  Field(res, 7) = Val_long(MSG_OOB);
  Field(res, 8) = Val_long(MSG_PEEK);
  Field(res, 9) = Val_long(MSG_DONTROUTE);
  return res;
}

struct addr {
  value size;
  value nspace;
  value data;
};

union saddr {
  struct sockaddr sockaddr_gen;
  struct sockaddr_un sockaddr_unix;
  struct sockaddr_in sockaddr_inet;
};

#define Sock_val(x) ((int) Field(x,0))

value socketpeek(value sock)
{
  return Val_int(Sock_val(sock));
}

static value newsocket(int sock)
{
  value result;

  result = Max_young_wosize >= 2 ? alloc_shr(2, Abstract_tag)
    : alloc(2, Abstract_tag);


  Sock_val(result) = sock;
  return result;
}

typedef  unsigned long int s_addr_type;


static value newinaddr(s_addr_type s_addr)
{
  value result;
  int bsize, wsize;

  bsize = sizeof(s_addr);
  wsize = (bsize >> 2) + 3;
  
  result = Max_young_wosize >= wsize ? alloc_shr(wsize, Abstract_tag)
                                     : alloc(wsize, Abstract_tag);
  *((s_addr_type*) result) = s_addr;

  return result;
}

static void make_saddr(union saddr *s, struct addr *a) 
{
  int size = Int_val(a->size);

  switch(Int_val(a->nspace)) {
  case AF_UNIX:
    s->sockaddr_unix.sun_family = AF_UNIX;
    bcopy(String_val(a->data), s->sockaddr_unix.sun_path, size + 1);
    break;
  case AF_INET:
    s->sockaddr_inet.sin_family = AF_INET;
    s->sockaddr_inet.sin_addr.s_addr = *((s_addr_type*) Field(a->data, 0));
    s->sockaddr_inet.sin_port = htons(Int_val(Field(a->data, 1)));
    break;
  }
} 


static value newaddr(int len, int namespace, value addrdata)
{
  struct addr * res;
  Push_roots(roots,1)
#define saddr roots[0]
  saddr = addrdata;
  res = (struct addr *) alloc_tuple(3);
  res->data = saddr;
  res->size = Val_int(len);
  res->nspace = Val_int(namespace);
  Pop_roots();
  return (value) res;
#undef saddr
} 

static value from_saddr(union saddr *s, int len) 
{

  value res;

  switch(s->sockaddr_gen.sa_family) {
  case AF_UNIX:
    { Push_roots(name, 1);
      name[0] = copy_string(s->sockaddr_unix.sun_path);
      res = newaddr(len,AF_UNIX,name[0]);
      Pop_roots();
      break;
    }
  case AF_INET:
    {
      Push_roots(data,1);  
      data[0] = alloc_tuple(2);
  
      Field(data[0], 0) = newinaddr(s->sockaddr_inet.sin_addr.s_addr);
      Field(data[0], 1) = Val_int(ntohs(s->sockaddr_inet.sin_port));
      res = newaddr(sizeof(struct sockaddr_in), AF_INET, data[0]);
      Pop_roots();
    }
  }

  return res;
} 

/* ML type: string -> addr */
value mlsocket_newfileaddr(value name) /* ML */
{ 
  struct sockaddr_un dummy;

  mlsize_t len = string_length(name);
  int addr_len = (offsetof (struct sockaddr_un, sun_path)
		  + len + 1);

  if (len >= sizeof(dummy.sun_path)) {
    failwith("ENAMETOOLONG");
  }

  return newaddr(addr_len, AF_UNIX, name);
}

/* ML type: string -> int -> addr */
value mlsocket_newinetaddr(value name, value port) /* ML */
{
  struct sockaddr_in addr;
  value res;

  if (inet_aton(String_val(name), &addr.sin_addr)) 
    {
      Push_roots(data,1);  
      data[0] = alloc_tuple(2);
  

      Field(data[0], 0) = newinaddr(addr.sin_addr.s_addr);
      Field(data[0], 1) = port;
      res = newaddr(sizeof(struct sockaddr_in), AF_INET, data[0]);
      Pop_roots();
    }
  else
    failwith("Invalid address");

  return res;

}


/* ML type: int -> int -> sock */
value mlsocket_socket(namespace, style) /* ML */
     value namespace, style;
{
  int result;

  result = socket(Int_val(namespace), Int_val(style), 0);
  if (result < 0) { 
    switch (errno) {
    case EPROTONOSUPPORT : 
      failwith("EPROTONOSUPPORT");
      break;
    case EMFILE :
      failwith("EMFILE");
      break;
    case ENFILE : 
      failwith("ENFILE");
      break;
    case EACCES : 
      failwith("EACCES");
      break;
    case ENOBUFS : 
      failwith("ENOBUFS");
      break;
    default:
      failwith("An unspecified error happened (socket)");
    }
  }
 
  return newsocket(result);
}

/* ML type: sock -> sock * addr */
value mlsocket_accept(value sock) /* ML */
{
  int len, ret;
  union saddr addr;
  value res;

  Push_roots(roots,2);
  len = sizeof(addr);
  enter_blocking_section();
  ret = accept(Sock_val(sock), &addr.sockaddr_gen, &len);
  leave_blocking_section();

  if (ret == -1) {
    switch (errno) {
    case EBADF:
      failwith("EBADF");
      break;
    case ENOTSOCK:
      failwith("ENOTSOCK");
      break;
    case EOPNOTSUPP:
      failwith("EOPNOTSUPP");
      break;
    case EWOULDBLOCK:
      failwith("EWOULDBLOCK");
      break;
    }
  }   

  roots[0] = from_saddr(&addr, len);
  roots[1] = newsocket(ret);
  res = alloc_tuple(2);
  Field(res, 0) = roots[1];
  Field(res, 1) = roots[0];
  Pop_roots();

  return res;
}

/* ML type: sock -> addr -> unit */
value mlsocket_bind(value socket, struct addr * address)      /* ML */
{
  int ret, size;
  union saddr addr;

  make_saddr(&addr, address);

  size  = Int_val(address->size);

  ret = bind(Sock_val(socket), &addr.sockaddr_gen, size);
  if (ret == -1) {
    switch (errno) {
    case EBADF:
      failwith("EBADF");
      break;
    case ENOTSOCK:
      failwith("ENOTSOCK");
      break;
    case EADDRNOTAVAIL:
      failwith("EADDRNOTAVAIL");
      break;
    case EADDRINUSE:
      failwith("EADDRINUSE");
      break;
    case EINVAL:
      failwith("EINVAL");
      break;
    case EACCES:
      failwith("EACCES");
      break;
    default:
      failwith("An unspecified error happened (bind)");
    }
  }
  return Val_unit;
}

/* ML type: sock -> addr -> unit */
value mlsocket_connect(value socket, struct addr * address)      /* ML */
{
  int ret, size;
  union saddr addr;

 
  make_saddr(&addr, address);
  size  = Int_val(address->size);


  /* should enter_blocking_section() be inserted? */
  ret = connect(Sock_val(socket), &addr.sockaddr_gen, size);
  if (ret == -1) {
    switch (errno) {
    case EBADF:
      failwith("EBADF");
      break;
    case ENOTSOCK:
      failwith("ENOTSOCK");
      break;
    case EADDRNOTAVAIL:
      failwith("EADDRNOTAVAIL");
      break;
    case EAFNOSUPPORT:
      failwith("EAFNOSUPPORT");
      break;
    case EISCONN:
      failwith("EISCONN");
      break;
    case ETIMEDOUT:
      failwith("ETIMEDOUT");
      break;
    case ECONNREFUSED:
      failwith("ECONNREFUSED");
      break;
    case ENETUNREACH:
      failwith("ENETUNREACH");
      break;
    case EADDRINUSE:
      failwith("EADDRINUSE");
      break;
    case EINPROGRESS:
      failwith("EINPROGRESS");
      break;
    case EALREADY:
      failwith("EALREADY");
      break;
    case ENOENT:
      failwith("ENOENT");
      break;
    default:
      failwith("An unspecified error happened (connect)");
    }
  }
  return Val_unit;
}

/* ML type: socket -> int -> unit */
value mlsocket_listen(sock, queuelength) /* ML */
     value sock, queuelength;
{

  if (listen(Sock_val(sock), Int_val(queuelength)) == -1) {
    switch (errno) {
    case EBADF:
      failwith("EBADF");
      break;
    case ENOTSOCK:
      failwith("ENOTSOCK");
      break;
    case EOPNOTSUPP:
      failwith("EOPNOTSUPP");
      break;
    default:
      failwith("An unspecified error happened (listen)");
    }
  }
  return Val_unit;
}

/* ML type: socket -> unit */
value mlsocket_close(value sock) /* ML */
{
  if (close(Sock_val(sock)) == -1) 
      failwith("Error happened when trying to close the socket.");
  return Val_unit;
}

/* ML type: socket -> int -> unit */
value mlsocket_shutdown(sock, how) /* ML */
     value sock, how;
{
  if (shutdown(Sock_val(sock), Int_val(how)) == -1) {
    switch (errno) {
    case EBADF:
      failwith("EBADF");
      break;
      case ENOTSOCK:
	failwith("ENOTSOCK");
	break;
      case ENOTCONN:
	failwith("ENOTCONN");
	break;
      default:
      failwith("An unspecified error happened (shutdown)");
    }
  }
  return Val_unit;
}

/* ML type: socket -> string -> int -> int -> int -> int */
value mlsocket_send(sock, buff, offset, size, flags) /* ML */ 
     value sock, buff, offset, size, flags;
{
  int ret;

  enter_blocking_section();
  ret = send(Sock_val(sock), &Byte(buff, Long_val(offset)), Int_val(size), 
	     Int_val(flags));
  leave_blocking_section();
  if (ret == -1) {
    switch (errno) {
    case EBADF:
	failwith("EBADF");
	break;
    case EINTR:
      failwith("EINTR");
      break;
    case ENOTSOCK:
      failwith("ENOTSOCK");
      break;
    case EMSGSIZE:
      failwith("EMSGSIZE");
      break;
    case EWOULDBLOCK:
      failwith("EWOULDBLOCK");
      break;
    case ENOBUFS:
      failwith("ENOBUFS");
      break;
    case ENOTCONN:
      failwith("ENOTCONN");
      break;
    case EPIPE:
      failwith("EPIPE");
      break;
    default:
      failwith("An unspecified error happened (send)");
    }
  }
  return Val_int(ret);
}

/* ML type: sock -> Word8Vector.vector -> int * int -> int -> addr -> int */
value mlsocket_sendto(sock, buff, tup, flags, address) /* ML */
     value sock, buff, tup, flags; 
     struct addr * address;
{
  int ret;
  union saddr addr;

 
  make_saddr(&addr, address);
  enter_blocking_section();
  ret = sendto(Sock_val(sock), &Byte(buff, Long_val(Field(tup,0))), 
	       Int_val(Field(tup, 1)), Int_val(flags), 
	       &addr.sockaddr_gen, Int_val(address->size));
  leave_blocking_section();
  if (ret == -1) {
    switch (errno) {
    case EBADF:
      failwith("EBADF");
      break;
    case EINTR:
      failwith("EINTR");
      break;
    case ENOTSOCK:
      failwith("ENOTSOCK");
      break;
    case EMSGSIZE:
      failwith("EMSGSIZE");
      break;
    case EWOULDBLOCK:
      failwith("EWOULDBLOCK");
      break;
    case ENOBUFS:
      failwith("ENOBUFS");
      break;
    case ENOTCONN:
      failwith("ENOTCONN");
      break;
    case EPIPE:
      failwith("EPIPE");
      break;
    default:
      failwith("An unspecified error happened (sendto)");
    }
  }
  
  return Val_int(ret);
}

/* ML type: sock -> Word8Vector.vector -> int -> int -> int -> int */
value mlsocket_recv(sock, buff, offset, len, flags) /* ML */
     value sock, buff, offset, len, flags;
{
  int ret;

  enter_blocking_section();
  ret = recv(Sock_val(sock), &Byte(buff, Long_val(offset)), Int_val(len),
             Int_val(flags));
  leave_blocking_section();
  if (ret == -1) {
    switch (errno) {
    case EBADF:
      failwith("EBADF");
      break;
    case ENOTSOCK:
      failwith("ENOTSOCK");
      break;
    case EWOULDBLOCK:
      failwith("EWOULDBLOCK");
      break;
    case EINTR:
      failwith("EINTR");
      break;
    case ENOTCONN:
      failwith("ENOTCONN");
      break;
    default:
      failwith("An unspecified error happened (recv)");
    }
  }
  return Val_int(ret);
}

/* ML type: sock -> Word8Vector.vector -> int -> int -> int -> int * addr */
value mlsocket_recvfrom(sock, buff, offset, size, flags) /* ML */
     value sock, buff, offset, size, flags;
{
  int ret, len;
  value res;
  union saddr addr;

  Push_roots(roots, 2);
  roots[0] = buff;

  len = sizeof(addr);

  enter_blocking_section();
  ret = recvfrom(Sock_val(sock), &Byte(roots[0], Long_val(offset)), 
		 Int_val(size),
		 Int_val(flags), &addr.sockaddr_gen, &len);

  leave_blocking_section();
  if (ret == -1) {
    switch (errno) {
    case EBADF:
      failwith("EBADF");
      break;
    case ENOTSOCK:
      failwith("ENOTSOCK");
      break;
    case EWOULDBLOCK:
      failwith("EWOULDBLOCK");
      break;
    case EINTR:
      failwith("EINTR");
      break;
    case ENOTCONN:
      failwith("ENOTCONN");
      break;
    default:
      failwith("An unspecified error happened (recvfrom)");
    }
  }
  roots[1] = from_saddr(&addr, len);
 
  res = alloc_tuple(2);
  Field(res, 0) = Val_int(len);
  Field(res, 1) = roots[1];
  Pop_roots();

  return Val_int(res);
}

static void vec_to_fdset(value vec, fd_set *fds)
{
  int i, upper = Wosize_val(vec);

  FD_ZERO(fds);
  for(i = 0; i < upper; i++) {
    FD_SET(Sock_val(Field(vec, i)), fds);
  } 
}

static value fdset_to_list(fd_set *fds)
{
  int i;
  value res;

  Push_roots(ls, 2);
  ls[0] = Atom(0);
  for (i = 0; i < FD_SETSIZE; i++) {
    if (FD_ISSET(i, fds)) {
      ls[1] = alloc(2, 1);
      Field(ls[1], 0) = newsocket(i);
      Field(ls[1], 1) = ls[0];
      ls[0] = ls[1];
    }
  }
  res = ls[0];
  Pop_roots();

  return res;
}

value mlsocket_select(rsocks, wsocks, esocks, tsec, tusec)
     value  rsocks, wsocks, esocks;
     int tsec, tusec;
{
  int ret;
  fd_set rfd, wfd, efd;
  struct timeval timeout, *top;
  value res;
  Push_roots(ls, 3);

  vec_to_fdset(rsocks, &rfd);
  vec_to_fdset(wsocks, &wfd);
  vec_to_fdset(esocks, &efd);

  if (Int_val(tsec) < 0) {
    top = NULL;
  }
  else {    
    timeout.tv_sec = Int_val(tsec);
    timeout.tv_usec = Int_val(tusec);
    top = &timeout;
  }
  ret = select(FD_SETSIZE, &rfd, &wfd, &efd, top);

  if (ret == -1) {
    switch (errno) {
    case EBADF:
      failwith("EBADF");
      break;
    case EINTR:
      failwith("EINTR");
      break;
    case EINVAL:
      failwith("EINVAL");
      break;
    default:
      failwith("An unspecified error happened (select)");
    }
  }

  ls[0] = fdset_to_list(&rfd);
  ls[1] = fdset_to_list(&wfd);
  ls[2] = fdset_to_list(&efd);

  res = alloc_tuple(3);
  Field(res, 0) = ls[0];
  Field(res, 1) = ls[1];
  Field(res, 2) = ls[2];

  Pop_roots();
  return res;
}


#else

value mlsocket_recv() { invalid_argument("recv not implemented"); }

value mlsocket_recvfrom() { invalid_argument("recvfrom not implemented"); }

value mlsocket_send() { invalid_argument("send not implemented"); }

value mlsocket_sendto() { invalid_argument("sendto not implemented"); }

#endif



