(*---------------------------------------------------------------------------*
 * General utilities for running sockets in MoscowML.                        *
 *---------------------------------------------------------------------------*)

structure U = 
struct

datatype 'a result = None of string
                   | Some of 'a;

fun stdOutput s = 
   let open TextIO
   in
      output(stdOut, s);
      flushOut stdOut
   end;


(* Be persistent in attempting to do something *)
fun attempt f i =
 if i < 1 then raise Fail "attempt"
 else f () handle Interrupt => raise Interrupt
                | _ => attempt f (i-1);

fun file_exists s = FileSys.access(s,[]);

fun find_file s () = 
   if file_exists s then ()
    else (Process.system"sleep 1"; raise Fail "find_file");

(*---------------------------------------------------------------------------*
 * Start the server, and wait until it is up and running before pestering    *
 * it with requests. The server will create the tmpfile when it is ready     *
 * to go. After the server does a "listen", it will remove the tmpfile.      *
 * The "rounds" argument is how many times to go through the                 *
 * probe-then-sleep cycle before giving up.                                  *
 *---------------------------------------------------------------------------*)

fun start_server rounds server tmpfile sfile =
  let val c1 = "archOS=x86-linux; export archOS\n"
      val c2 = "HOLDIR=/home/kxs/hol98; export HOLDIR\n"
      val c3 = String.concat [server," ",tmpfile," ",sfile," > /dev/null &"]
  in
    if (Process.system (String.concat [c1,c2,c3]) = Process.success)
    then attempt (find_file tmpfile) rounds
             handle Fail "attempt" 
              => (stdOutput (String.concat 
                     ["Unable to find Core Proof Engine after ", 
                      Int.toString rounds, " probes.\n"])
                   ;
                  raise Fail "start_server")
    else raise Fail "Unable to start Core Proof Engine"
  end;

fun createServer (cpe,k) 
 :(Socket.pf_file, Socket.active Socket.stream)Socket.socket
  = let val  _   = stdOutput "Starting server: this may take awhile..\n"
        val tmp  = FileSys.tmpName()
        val file = FileSys.tmpName()
        val  _   = start_server k cpe tmp file
        val sock = Socket.fileStream ()
        val addr = Socket.fileAddr file
        val _    = Socket.connect(sock, addr)
    in
      sock
    end;

(*---------------------------------------------------------------------------*
 * Prepend the size of a string to a string. Complicated because the size    *
 * needs to take account of the size of the size!                            *
 * Examples.                                                                 *
 *                                                                           *
 *  sizedstring "12345678" = "11 12345678"                                   *
 *  sizedstring "1234567"  = "9 1234567"                                     *
 *---------------------------------------------------------------------------*)

fun sizedstring s =
 let val i = 1 + String.size s  (* account for a space *)
     val si = Int.toString i
     val j = String.size si
     val guess0 = i + j
     val strguess = Int.toString guess0
     val k = String.size strguess
     val n = (if j < k then k else j) + i
 in 
   String.concat [Int.toString n, " ", s]
 end;
   

fun sendString sock s = 
  Socket.sendVec(sock, Byte.stringToBytes (sizedstring s));


local val chunksize = 1024
      val numeric = Char.contains"0123456789"
in
fun readString sock =
 let fun grab togo A =
       let val str = Byte.bytesToString (Socket.recvVec (sock, chunksize))
           val strsize = String.size str
       in if strsize < togo
          then grab (togo - strsize) (str::A)
          else String.concat (List.rev (str::A))
       end
     val str0 = Byte.bytesToString (Socket.recvVec (sock, chunksize))
     val str0size = String.size str0
     val sstr0 = Substring.all str0
     val (x,y) = Substring.splitl numeric sstr0
     val howbig = Int.fromString (Substring.string x)
     val ystr = Substring.string (Substring.triml 1 y)
  in 
    case howbig
     of NONE => raise Fail "Badly formed message: not prefixed with size!"
      | SOME n =>
         (n, if str0size < n then grab (n - str0size) [ystr]
                             else ystr)
  end
end;

local val alpha = Char.contains"SomeNone"
      val space = Char.contains" "
in
fun sendRqt sock [QUOTE rqt] =
      let val _ = sendString sock rqt
          val (n,reply) = readString sock
          val (prefx, rst) = Substring.splitl alpha (Substring.all reply)
          val prefstring  = Substring.string prefx
          val replystring = Substring.string (#2 (Substring.splitl space rst))
      in
        case prefstring
         of "None" => None replystring
          | "Some" => Some replystring
          |    _   => raise Fail (String.concat 
                              ["Protocol violation: ", 
                               "expecting a result constructor ",
                               "(\"Some\" or \"None\")."])
      end
   | sendRqt _ _ = raise Fail "sendRqt <sock> ` ... `"
end;

end;