(*---------------------------------------------------------------------------*
 * Dispatch. The part of the CPE that parses and carries out requests.       *
 *---------------------------------------------------------------------------*)

structure Dispatch :> Dispatch =
struct

open Exception Utils;

fun CPE_ERR f s = 
  HOL_ERR{origin_structure = "Core Proof Engine Dispatcher",
          origin_function = f, message = s};

val alphanumeric = Char.contains
  "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789_'";
val numeric = Char.contains "0123456789";

fun pp_terms tml =
  let open Portable_PrettyPrint
      fun ppl ppstrm tml = 
           (begin_block ppstrm CONSISTENT 0;
            add_string ppstrm "[";
            begin_block ppstrm CONSISTENT 0;
            pr_list_to_ppstream ppstrm Term.pp_term (Lib.C add_string ",")
                                add_newline tml;
            add_string ppstrm "]"; end_block ppstrm; end_block ppstrm)
  in 
    pp_to_string 72 ppl tml
  end;
     
fun pp_gstack() = 
  let open goalstackLib Portable_PrettyPrint
  in 
     Some (pp_to_string 75 pp_goalstack (p()))
  end handle HOL_ERR _ => None ("No goalstacks.")

fun pp_thm th = 
  let open Portable_PrettyPrint
  in 
     pp_to_string 75 Thm.pp_thm th
  end;

fun HOL_ERRstring {origin_structure,origin_function,message} =
  String.concat [origin_structure,".",origin_function,":\n",message];


(* Terribly hacky in the calculation of the name attached to the definition *)
fun tryDefine q = 
   let val defsthms0 = map #1 (Theory.definitions()@Theory.theorems())
       val def = bossLib.Define q
       val defsthms1 = map #1 (Theory.definitions()@Theory.theorems())
   in 
   case Lib.set_diff defsthms1 defsthms0
    of [defname] => 
        (case Thm.hyp def
          of [] => Some defname
           | tcs => Some (String.concat [defname, " ",pp_terms tcs]))
     |    _      => None "Implementation error : can't attach name to defn."
   end
   handle HOL_ERR r => None (HOL_ERRstring r)

(* These are easy. Just return true or failure *)
fun tryTaut s =
  (tautLib.TAUT_PROVE (Parse.string_to_term s); Some "true")
  handle HOL_ERR r => None (HOL_ERRstring r);

fun tryFOL q =
  (bossLib.PROVE[] q; Some "true")
  handle HOL_ERR r => None (HOL_ERRstring r);

fun tryArith s =
  (arithLib.ARITH_CONV (Parse.string_to_term s); Some "true")
  handle HOL_ERR r => None (HOL_ERRstring r);

fun tryDecide q = 
  (bossLib.DECIDE q; Some "true")
  handle HOL_ERR r => None (HOL_ERRstring r);

fun tryConcType q = 
  (bossLib.Hol_datatype q; Some "Concrete type definition succeeded")
  handle HOL_ERR r => None (HOL_ERRstring r);

fun tryGoal s = 
  let val tm = Parse.string_to_term s
      open Portable_PrettyPrint goalstackLib
  in 
     set_goal([],tm);
     pp_gstack()
  end
  handle HOL_ERR r => None (HOL_ERRstring r);

fun unknown s = 
   raise CPE_ERR "unknown"
                  (String.concat ["Request ", Lib.quote s, " rejected."]);


(*---------------------------------------------------------------------------*
 * Find the theorem that corresponds to the string. Currently, we look in    *
 * the (fixed) database and then in the current theory.                      *
 *---------------------------------------------------------------------------*)

fun string_to_thm s = 
   case Binaryset.find (fn (s1,_) => (s1=s)) dBase.boss_thms
    of NONE => (Theory.theorem s handle HOL_ERR _ => Theory.definition s)
     | SOME (_,(thm,_)) => thm;


val sstrs_to_thms = map (string_to_thm o Substring.string);

fun resolve_ss sstr =
  case (Substring.string sstr)
   of "bool_ss"  => bossLib.bool_ss
    | "arith_ss" => bossLib.arith_ss
    | "list_ss"  => bossLib.list_ss
    |  s         => raise CPE_ERR "resolve_ss"
                            (String.concat["Unknown simpset: ", Lib.quote s]);

fun resolvePROVE_TAC sstr =
   let val L = Substring.tokens Char.isSpace sstr
   in bossLib.PROVE_TAC (sstrs_to_thms L)
   end;

fun resolveRW_TAC sstr =
  case Substring.tokens Char.isSpace sstr
   of [] => raise CPE_ERR "resolveRW_TAC" "No simpset supplied for RW_TAC."
    | ss_str::rst => bossLib.RW_TAC (resolve_ss ss_str) (sstrs_to_thms rst)

fun resolveZAP_TAC sstr =
 case Substring.tokens Char.isSpace sstr
  of [] => raise CPE_ERR "resolveZAP_TAC" "No simpset supplied for ZAP_TAC."
   | ss_str::rst => bossLib.ZAP_TAC (resolve_ss ss_str) (sstrs_to_thms rst);


fun resolve_tac sstr =
   let val (s1,s2) = Substring.splitl alphanumeric sstr
   in case Substring.string s1
       of "TAUT_TAC"   => tautLib.TAUT_TAC
        | "DECIDE_TAC" => bossLib.DECIDE_TAC
        | "PROVE_TAC"  => resolvePROVE_TAC s2
        | "RW_TAC"     => resolveRW_TAC s2
        | "ZAP_TAC"    => resolveZAP_TAC s2
        |   s          => unknown s
   end;

(*---------------------------------------------------------------------------*
 * Applying tactics to the top element on the goalstack.                     *
 *---------------------------------------------------------------------------*)
fun tryApplyTac s = 
 let open goalstackLib
     val (comm,rst) = Substring.splitl alphanumeric 
                         (Substring.dropl Char.isSpace s)
 in 
   case Substring.string comm
    of "TAUT_TAC"   => ((expand tautLib.TAUT_TAC;  pp_gstack())
                          handle HOL_ERR r => None (HOL_ERRstring r))

     | "DECIDE_TAC" => ((expand bossLib.DECIDE_TAC;  pp_gstack())
                         handle HOL_ERR r => None (HOL_ERRstring r))

     | "Cases"      => ((expand bossLib.Cases;  pp_gstack())
                         handle HOL_ERR r => None (HOL_ERRstring r))

     | "Induct"     => ((expand bossLib.Induct;  pp_gstack())
                         handle HOL_ERR r => None (HOL_ERRstring r))

     | "Induct_on"  => ((expand (bossLib.Induct_on 
                               [QUOTE (Substring.string rst)]); 
                         pp_gstack())
                         handle HOL_ERR r => None (HOL_ERRstring r))

     | "Cases_on"   => ((expand (bossLib.Cases_on 
                              [QUOTE (Substring.string rst)]); 
                        pp_gstack())
                          handle HOL_ERR r => None (HOL_ERRstring r))

     | "PROVE_TAC"  => ((expand (resolvePROVE_TAC rst); pp_gstack())
                         handle HOL_ERR r => None (HOL_ERRstring r))

     | "RW_TAC"     => ((expand (resolveRW_TAC rst); pp_gstack())
                         handle HOL_ERR r => None (HOL_ERRstring r))

     | "ZAP_TAC"    => ((expand (resolveZAP_TAC rst); pp_gstack())
                         handle HOL_ERR r => None (HOL_ERRstring r))

     | "Have"       => (let fun isQ #"\"" = true | isQ _ = false
                            fun P c = Char.isSpace c orelse isQ c
                            val rst' = Substring.dropl P rst
                            val (q,rst2) = Substring.splitl (not o isQ) rst'
                            val tacsstr = Substring.dropl P rst2
                            val tac = resolve_tac tacsstr
                        in
                          expand (bossLib.by
                             ([QUOTE (Substring.string q)], tac)); 
                          pp_gstack()
                        end handle HOL_ERR r => None (HOL_ERRstring r))
     | "SposeNot"   => (let open goalstackLib Tactic
                        in
                          expand (bossLib.SPOSE_NOT_THEN STRIP_ASSUME_TAC);
                          pp_gstack()
                        end
                        handle HOL_ERR r => None (HOL_ERRstring r))

     |   _          => unknown (Substring.string comm)
   end;

fun tryBackup()  = (goalstackLib.backup(); pp_gstack())
                    handle HOL_ERR r => None (HOL_ERRstring r);
fun tryRestart() = (goalstackLib.restart(); pp_gstack())
                    handle HOL_ERR r => None (HOL_ERRstring r);
fun tryDrop()    = (goalstackLib.drop(); Some ("Goalstack dropped."))
                     handle HOL_ERR r => None (HOL_ERRstring r);

fun tryTopGoals() = pp_gstack();

fun tryRotate sstr = 
  let val (s1,s2) = Substring.splitl numeric 
                      (Substring.dropl Char.isSpace sstr)
  in
   case Int.fromString (Substring.string s1)
    of NONE => raise CPE_ERR "tryRotate" "Unable to find number of rotations."
     | SOME i => (goalstackLib.rotate i; pp_gstack())
                  handle HOL_ERR r => None (HOL_ERRstring r)
  end;

fun tryNthProof sstr = 
  let val (s1,s2) = Substring.splitl numeric 
                      (Substring.dropl Char.isSpace sstr)
  in
   case Int.fromString (Substring.string s1)
    of NONE => raise CPE_ERR "tryNthProof" "Unable to find index of proof."
     | SOME i => (goalstackLib.rotate_proofs i; pp_gstack())
                  handle HOL_ERR r => None (HOL_ERRstring r)
  end;


fun tryStoreTop s =
  let val (s1,_) = Substring.splitl alphanumeric 
                      (Substring.dropl Char.isSpace s)
      val thm = goalstackLib.top_thm()
      val name = Substring.string s1
  in 
     Theory.save_thm(name, thm);
     Some (String.concat ["Stored theorem\n ", pp_thm thm, "\nunder ", 
                          Lib.quote name, " in current theory."])
  end
  handle HOL_ERR r => None (HOL_ERRstring r);

fun tryFixity s =
  let val (s1,s2) = Substring.splitl alphanumeric 
                      (Substring.dropl Char.isSpace s)
      val name = Substring.string s1
      val (s3,s4) = Substring.splitl alphanumeric 
                      (Substring.dropl Char.isSpace s2)
  in 
    case Substring.string s3
     of "Prefix" => (Theory.set_fixity name Term.Prefix; 
                     Some (String.concat ["Fixity of ", Lib.quote name,
                                  " set to Prefix."]))
      | "Binder" => (Theory.set_fixity name Term.Binder;
                     Some (String.concat ["Fixity of ", Lib.quote name,
                                  " set to Binder."]))
      | "Infix"  => 
           let val (s5,_) = Substring.splitl numeric
                              (Substring.dropl Char.isSpace s4)
           in case Int.fromString (Substring.string s5)
               of SOME i => (Theory.set_fixity name (Term.Infix i);
                             Some (String.concat ["Fixity of ", 
                                     Lib.quote name,
                                    " set to Infix with precedence ",
                                    Substring.string s5]))
                | NONE   => raise CPE_ERR "tryFixity" 
                                "Can't parse number for Infix declaration"
           end
      | sbad => raise CPE_ERR "tryFixity" 
                      (String.concat ["Unknown kind of fixity: ", sbad])
  end 
  handle HOL_ERR r => None (HOL_ERRstring r);

fun tryMLname s =
   case Substring.tokens Char.isSpace s
    of [oldname, newname] => 
           (Theory.set_MLname (Substring.string oldname)
                              (Substring.string newname);
            Some (String.concat ["Theory-level binding of ", 
                                 Lib.quote (Substring.string oldname),
                                 " changed to ", 
                                 Lib.quote (Substring.string newname)]))
     | _ => None ("tryMLname: unable to parse names.");


fun tryMkThm s =
  let val (s1,s2) = Substring.splitl alphanumeric
                      (Substring.dropl Char.isSpace s)
      val thm  = Thm.mk_thm([],Parse.string_to_term 
                                 (Substring.string s2))
  in 
      Theory.save_thm (Substring.string s1, thm);
      Some (String.concat ["Stored theorem \n", Thm.thm_to_string thm,
                           "\n in current theory under name ",
                           Lib.quote (Substring.string s1)])
  end 
  handle HOL_ERR r => None (HOL_ERRstring r);

(*---------------------------------------------------------------------------*
 * This is the switch on requests made by the client.                        *
 *---------------------------------------------------------------------------*)

fun dispatch ("Define", s)  = tryDefine [QUOTE (Substring.string s)]
  | dispatch ("Taut", s)    = tryTaut (Substring.string s)
  | dispatch ("FOL", s)     = tryFOL [QUOTE (Substring.string s)]
  | dispatch ("Arith", s)   = tryArith (Substring.string s)
  | dispatch ("Decide", s)  = tryDecide [QUOTE (Substring.string s)]
  | dispatch ("ConcType",s) = tryConcType [QUOTE (Substring.string s)]
  | dispatch ("Goal",s)     = tryGoal (Substring.string s)
  | dispatch ("ApplyTac",s) = tryApplyTac s
  | dispatch ("Backup",_)   = tryBackup()
  | dispatch ("Restart",_)  = tryRestart()
  | dispatch ("Drop",_)     = tryDrop()
  | dispatch ("TopGoals",_) = tryTopGoals ()
  | dispatch ("Rotate",s)   = tryRotate s
  | dispatch ("NthProof",s) = tryNthProof s
  | dispatch ("StoreTop",s) = tryStoreTop s
  | dispatch ("Fixity",s)   = tryFixity s
  | dispatch ("MLname",s)   = tryMLname s
  | dispatch ("MkThm",s)    = tryMkThm s
  | dispatch (s, _)         = unknown s

fun to_vec (None s) = String.concat["None ", s]
  | to_vec (Some s) = String.concat["Some ", s];

fun comm_to_answerVector line =
  let val sstr = Substring.all line
      val (x,y) = Substring.splitl alphanumeric sstr
      val answer = dispatch (Substring.string x, 
                             Substring.dropl Char.isSpace y)
  in to_vec answer
  end
  handle HOL_ERR r => to_vec (None (HOL_ERRstring r))
       | Fail s    => to_vec (None s)
       | General.Interrupt => raise General.Interrupt
       |   _       => to_vec (None "Unknown error");

fun no_rqt() = to_vec (None "Empty request received.");
fun stop() = (Theory.new_theory"scratch";  (* reset theory *)
              to_vec (Some "Client terminating normally. CPE state reset."));

fun release() = 
 to_vec (Some "Client terminating normally. CPE state persists.");

end;
