(* Ken Larsen 26/10/1998 *)

structure Socket :> Socket =
struct

    prim_type sock
    prim_type addr 

    datatype ('sock, 'af) socket = SOCK of sock
    datatype 'addressfam sock_addr = ADDR of addr


    (* witness types for the socket parameter *)
    type dgram     = unit
    type 'a stream = unit
    type passive   = unit
    type active    = unit

    (* withness types fot the addressfam parameter *)
    type pf_file = unit
    type pf_inet = unit

    local 
	open Dynlib
	val hdl  = dlopen {lib = "/home/kxs/prosper/CPE/sockets/mlsocket.so",
			   flag = RTLD_LAZY, global = false}
	val symb = Dynlib.dlsym hdl
	fun app1 name = Dynlib.app1 (symb ("mlsocket_"^name))
	fun app2 name = Dynlib.app2 (symb ("mlsocket_"^name))
	fun app3 name = Dynlib.app3 (symb ("mlsocket_"^name))
	fun app4 name = Dynlib.app4 (symb ("mlsocket_"^name))
	fun app5 name = Dynlib.app5 (symb ("mlsocket_"^name))

	val constants_ : unit  -> int * int * int * int 
	    * int * int * int * int 
	    * int * int             
	           = app1 "constants" 

	val newfileaddr_ : string -> addr       = app1 "newfileaddr"
	val newinetaddr_ : string ->int -> addr = app2 "newinetaddr"

	            (* domain, type *)
	val socket_   : int -> int -> sock      = app2 "socket"

	val accept_   : sock -> sock * addr     = app1 "accept"
	val bind_     : sock -> addr -> unit    = app2 "bind"
	val connect_  : sock -> addr -> unit    = app2 "connect"
	val listen_   : sock -> int -> unit     = app2 "listen"
	val close_    : sock -> unit            = app1 "close"
	val shutdown_ : sock -> int -> unit     = app2 "shutdown"

	type buff = Word8Vector.vector

	                            (* offset, size, flags *)
	val send_     : sock -> buff -> int -> int -> int -> int = app5 "send"
	val sendto_   : sock -> buff -> int * int -> int -> addr -> int
	                                                      = app5 "sendto"
	val recv_     : sock -> buff -> int -> int -> int -> int = app5 "recv"
	val recvfrom_ : sock -> buff -> int -> int -> int -> int * addr
                                                              = app5 "recvfrom"

	val select_   : sock vector -> sock vector -> sock vector -> 
                                                              int -> int ->
	                sock list * sock list * sock list = app5 "select"

	val (SOCK_STREAM,
	     SOCK_DGRAM,
	     PF_UNIX,
	     PF_INET,
	     NO_RECVS_,
	     NO_SENDS_, 
	     NO_RECVS_OR_SENDS_,
	     MSG_OOB,
	     MSG_PEEK,
	     MSG_DONTROUTE) = constants_ ()
	
	val (MSG_OOB,
	     MSG_PEEK,
	     MSG_DONTROUTE) = (Word.fromInt MSG_OOB,
			       Word.fromInt MSG_PEEK,
			       Word.fromInt MSG_DONTROUTE)

	prim_val vector_ : int -> buff = 1 "create_string"

	fun extract vec len = Word8Vector.extract(vec, 0, SOME len)
    in
	fun fileAddr s = ADDR(newfileaddr_ s)
	fun inetAddr s port = ADDR(newinetaddr_ s port)

	fun fileStream () = SOCK(socket_ PF_UNIX SOCK_STREAM)
	fun fileDgram ()  = SOCK(socket_ PF_UNIX SOCK_DGRAM)
	fun inetStream () = SOCK(socket_ PF_INET SOCK_STREAM)
	fun inetDgram ()  = SOCK(socket_ PF_INET SOCK_DGRAM)

	fun accept (SOCK sock) = 
	    let val (s,a) = accept_ sock
	    in  (SOCK s, ADDR a)
	    end

	fun bind (SOCK sock, ADDR addr) = bind_ sock addr

	fun connect (SOCK sock, ADDR addr) = connect_ sock addr

	fun listen (SOCK sock, queuelen) = listen_ sock queuelen

	fun close (SOCK sock) = close_ sock

	

	datatype shutdown_mode = NO_RECVS | NO_SENDS | NO_RECVS_OR_SENDS
	
	fun shutdown (SOCK sock, NO_RECVS) = 
	    shutdown_ sock NO_RECVS_
	  | shutdown (SOCK sock, NO_SENDS) = 
	    shutdown_ sock NO_SENDS_
	  | shutdown (SOCK sock, NO_RECVS_OR_SENDS) = 
	    shutdown_ sock NO_RECVS_OR_SENDS_

	type out_flags = {don't_route : bool, oob : bool}
	type in_flags = {peek : bool, oob : bool}
	    
	type 'a buf = {buf : 'a, ofs : int, size : int option}
	
	fun sendVec (SOCK sock, vec) =
	    send_ sock vec 0 (Word8Vector.length vec) 0

	fun sendVec' (SOCK sock, {buf, ofs, size}, {don't_route, oob}) =
	    let	val len = Word8Vector.length buf
		val sz = case size of
                             SOME n => n
			   | NONE   => len - ofs
		val flags = let val f = if don't_route then MSG_DONTROUTE
					else 0w0
			    in  if oob then Word.orb(f, MSG_OOB)
				else f
			    end
	    in
		if ofs < 0 orelse ofs >= len orelse sz < 0 then 
		    raise Subscript
		else send_ sock buf ofs sz (Word.toInt flags)
	    end

	fun sendVecTo (SOCK sock, ADDR addr, vec) =
	    sendto_ sock vec (0, Word8Vector.length vec) 0 addr

	fun sendVecTo' (SOCK sock, ADDR addr,
			{buf, ofs, size}, {don't_route, oob}) =
	    let	val len = Word8Vector.length buf
		val sz = case size of
                             SOME n => n
			   | NONE   => len - ofs
		val flags = let val f = if don't_route then MSG_DONTROUTE
					else 0w0
			    in  if oob then Word.orb(f, MSG_OOB)
				else f
			    end
	    in
		if ofs < 0 orelse ofs >= len orelse sz < 0 then 
		    raise Subscript
		else sendto_ sock buf (ofs, sz) (Word.toInt flags) addr
	    end

	fun recvVec (SOCK sock, len) =
	    let val vec  = vector_ len 
		val size = recv_ sock vec 0 len 0; 
	    in	
		extract vec size
	    end

	fun recvVec' (SOCK sock, len, {peek, oob}) =
	    let val vec  = vector_ len 
		val flag = let val f = if peek then MSG_PEEK
				       else 0w0
			   in  if oob then Word.orb(f, MSG_OOB)
			       else f
			   end
		val size = recv_ sock vec 0 len (Word.toInt flag) 
	    in	
		extract vec size
	    end

	fun recvVecFrom (SOCK sock, len) =
	    let val vec  = vector_ len 
		val (size, addr) = recvfrom_ sock vec 0 len 0; 
	    in	
		(extract vec size, ADDR addr)
	    end

	fun recvVecFrom' (SOCK sock, len, {peek, oob}) =
	    let val vec  = vector_ len
		val flag = let val f = if peek then MSG_PEEK
				       else 0w0
			   in  if oob then Word.orb(f, MSG_OOB)
			       else f
			   end
		val (size, addr) = recvfrom_ sock vec 0 len (Word.toInt flag) 
	    in	
		(extract vec size, ADDR addr)
	    end

	fun desSOCK (SOCK s) = s

	fun select rls wls els timeout =
	    let val (tsec, tusec) = case timeout of
                                        NONE => (~1,0)
				      | SOME res => res
		val rvec = vector(List.map desSOCK rls)
		val wvec = vector(List.map desSOCK wls)
		val evec = vector(List.map desSOCK els)
		val (rls, wls, els) = select_ rvec wvec evec tsec tusec
	    in
		(map SOCK rls, map SOCK wls, map SOCK els)
	    end
    end
end
