 #########################################################################
 #                                                                       #
 # Copyright (C) 1994 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contract MDA972-92-C-0027.           #
 #                                                                       #
 #########################################################################

# File:	password.tcl
#
# Description:
#	`password' widget, that behaves like an entry box but does not
#	show the contents.
#
# Transient procedures:
#c	password:alias$w
#		The Tk widget procedure that implements the entry box $w,
#		renamed so that widget commands can be intercepted.
#
#c	password:update$w
#		Procedure that updates the user-visible contents of the
#		password widget $w when its text variable changes.
#
# Global variables:
#c	password_active($w)
#		This array element is set to 1 when a widget command is
#		in progress on a password widget.  It is checked when
#		the password's text variable is changed, to determine
#		whether the front-end display needs to be updated.

 # password.tcl,v 1.1.1.1 1994/12/07 10:17:30 donald Exp
 # /homes/drs1004/repository/tkaux/password.tcl,v
 # password.tcl,v
# Revision 1.1.1.1  1994/12/07  10:17:30  donald
# First import after some small chanegs.
#
# Revision 1.3  1994/10/27  18:29:42  kennykb
# Release 2.0 -- 10-27-94.  To be uploaded to archive sites.
#
# Revision 1.2  1994/03/09  18:51:01  kennykb
# Added code to update the string of bullets on display in the password
# box when the associated text variable changes, and to correctly report
# the text variable in response to `config' requests.
#
# Revision 1.1  1994/03/07  22:11:21  kennykb
# Initial revision
#

# Procedure:	password
#
# Synopsis:
#	`password' widget, that behaves like an entry box but does not
#	show the contents.
#
# Usage:
#c	password pathName ?-option value?...
#
# Parameters:
#c	pathName
#		Path name of the password widget
#
# Return value:
#	Path name of the password widget.  The widget procedure is renamed
#	and replaced with a Tk procedure that intercepts the widget commands
#	and routes them appropriately.  In addition, a child widget,
#	pathName.real, is created.  This widget's text maintains the actual
#	password text, but the widget is never mapped.
#
# Description:
#	`password' is an entry widget that displays its content as a string
#	of bullet characters, instead of in a readable form.

proc password {w args} {

	# `aliasargs' are the parameters belonging to the widget on display.
	# `realargs' belong to the $w.real widget that holds the password.
	# The -textvariable option goes to `realargs'; all the others go to
	# `aliasargs.'  The `exportselection' argument is not recommended,
	# since it will allow a malicious user to grab the password from the
	# X selection.

	set aliasargs {-exportselection 0 }
	set realargs {}
	while {$args != ""} {
		set flag [lindex $args 0]
		set value [lindex $args 1]
		set args [lrange $args 2 end]
		if {[string range $flag 0 1] == "-t"
		    && [string range "-textvariable" 0 \
				[expr [string length $flag]-1]] == $flag} {
			lappend realargs $flag $value
			set textvar $value
		} else {
			lappend aliasargs $flag $value
		}
	}

	eval entry $w $aliasargs
	eval entry $w.real $realargs

	# Swizzle names so that widget commands get directed to 
	# `password:alias'.  Set up to unswizzle the names when the
	# widget is destroyed.  Use `tkauxlib' binding mechanism if available.

	rename $w password:alias$w
	proc $w {command args} "
		password:command $w \$command \$args
	"
	set status [catch {
		widget_addBinding $w Destroy "password:deleteAlias $w"
	}]
	if {$status != 0} {
		bind $w <Destroy> "password:deleteAlias $w"
	}

	# Set the widget to show the correct number of bullets for its
	# text variable

	if [info exists textvar] {
		password:configTextVariable $w $textvar
	}

	# Return the path name.

	return $w
}

# Procedure:	password:command
#
# Synopsis:
#	Process the widget command on a password widget.
#
# Usage:
#c	password:command pathName command arguments...
#
# Parameters:
#c	pathName
#		Path name of the password widget
#
#c	command
#		Widget command to execute
#
#c	arguments
#		Arguments to the individual widget command
#
# Return value:
#	Specific to the widget command.
#
# Description:
#	The `password:command' procedure intercepts the widget command
#	on password widgets.  It dispatches commands as follows:
#
#c	configure
#		Dispatched to the `password:configure' procedure for
#		further processing.
#
#c	delete
#		Deletes the requisite number of characters from both
#		front- and back-end widgets.
#
#c	icursor
#		Sets insertion cursor on both front- and back-end widgets.
#
#c	index
#		Retrieves the index from the front-end widget, which is
#		the one that scrolls.
#
#c	insert
#		Inserts the text into the backend widget, and
#		an appropriate number of bullets into the front-end widget.
#
#c	scan
#		Does the scan operation on the front-end widget.
#
#c	select
#		Does the selection operation on both the front- and
#		back-end widgets.
#
#c	view
#		Does the view operation on the front-end widget; the
#		back-end widget doesn't scroll.

proc password:command {w command argv} {
	global password_active
	set password_active($w) 1
	set result [\
	    case $command in {
		{c co con conf confi config configu configur configure} {
			password:configure $w $argv
		}
		{d de del dele delet delete} {
			eval password:alias$w delete $argv
			eval $w.real delete $argv
		}
		{g ge get} {
			eval $w.real get $argv
		}
		{ic icu icur icurs icurso icursor} {
			eval password:alias$w icursor $argv
			eval $w.real icursor $argv
		}
		{ind inde index} {
			eval password:alias$w index $argv
		}
		{ins inse inser insert} {
			set index [lindex $argv 0]
			set string [lindex $argv 1]
			set xxx {}
			set l [string length $string]
			for {set i 0} {$i < $l} {incr i} {
				append xxx \267
			}
			$w.real insert $index $string
			password:alias$w insert $index $xxx
		}
		{sc sca scan} {
			eval password:alias$w scan $argv
		}
		{se sel sele selec select} {
			eval $w.real select $argv
			eval password:alias$w select $argv
		}
		{v vi vie view} {
			eval password:alias$w view $argv
		}
		default {
			eval password:alias$w $command $argv
		}
	}]
	unset password_active($w)
	return $result
	    
}

# Procedure:	password:configure
#
# Synopsis:
#	Configure the `password' widget.
#
# Usage:
#c	password:configure pathName ?-option? ?value? ?-option value...?
#
# Parameters:
#c	pathName
#		Path name of the password widget
#
# Options:
#	Same as for the `entry' widget.
#
# Return value:
#	Same as for configuring the `entry' widget.
#
# Description:
#	`password:configure' routes configuration options between the
#	two Tk widgets that make up a password widget.  The `-textvariable'
#	option goes to the back-end widget; everything else goes to the
#	front-end widget.  The `-exportselection' option causes the X selection
#	to return the string of bullets, not the password; this is a feature,
#	not a bug.

proc password:configure {w argv} {
	case [llength $argv] in {
		0 {
			set list [password:alias$w configure]
			set replitem [$w.real configure -textvariable]
			set retval {}
			foreach item $list {
				if {[lindex $item 0] == "-textvariable} {
					lappend retval $replitem
				} else {
					lappend retval $item
				}
			}
			return $retval
		}
		1 {
			set flag [lindex $argv 0]
			if {[string range $flag 0 1] == "-t"
			    && [string range "-textvariable" 0 \
				    [expr [string length $flag]-1]] == $flag} {
				return [$w.real configure $flag]
			} else {
				return [password:alias$w configure $flag]
			}
		}
	}

	set aliasargs {-exportselection 0}
	set realargs {}

	while {$argv != ""} {
		set flag [lindex $argv 0]
		set value [lindex $argv 1]
		set argv [lrange $argv 2 end]
		if {[string range $flag 0 1] == "-t"
		    && [string range "-textvariable" 0 \
				[expr [string length $flag]-1]] == $flag} {
			set textvar $value
			lappend realargs $flag $value
			password:cleanup $w
		} else {
			lappend aliasargs $flag $value
		}
	}

	eval password:alias$w configure $aliasargs
	eval $w.real configure $realargs

	# Set the widget to show the correct number of bullets for its
	# text variable

	if [info exists textvar] {
		password:configTextVariable $w $textvar
	}
}

# Procedure: password:configTextVariable
#
# Synopsis:
#	Configure the text variable of a password widget
#
# Usage:
#c	password:configTextVariable pathName varName
#
# Parameters:
#c	pathName
#		Path name of a password widget
#
#c	varName
#		Name of the widget's text variable
#
# Return value:
#	Not specified.
#
# Description:
#	password:configTextVariable establishes a trace on
#	a password widget's text variable so that the correct number
#	of bullets is always displayed in the front end.

proc password:configTextVariable {w v} {
	if {$v == ""} return
	proc password:update$w args "password:updateTextVariable $w"
	uplevel #0 trace variable $v w password:update$w
	if {[catch {
		widget_addBinding $w.real Destroy "password:cleanup $w"
	}] != 0} {
		bind $w.real <Destroy> "password:cleanup $w"
	}
	password:updateTextVariable $w
}

# Procedure:	password:cleanup
#
# Synopsis:
#	Clean up after a password widget
#
# Usage:
#c	password:cleanup pathName
#
# Parameters:
#c	pathName
#		Path name of a password widget
#
# Return value:
#	Not specified
#
# Description:
#	password:cleanup deletes all traces on the text variable associated
#	with a password widget.

proc password:cleanup w {
	set status [catch {lindex [$w.real config -textvariable] 4} v]
	if {$status == 0 && $v != ""} {
		uplevel #0 trace vdelete $v w password:update$w
	}
	catch { rename password:update$w {} }
}

# Procedure:	password:deleteAlias
#
# Synopsis:
#	Delete the widget alias from a password widget.
#
# Usage:
#c	password:deleteAlias pathName
#
# Parameters:
#c	pathName
#		Path name of a password widget.
#
# Return value:
#	Not specified.
#
# Description:
#	password:deleteAlias is called in response to the <Destroy> event
#	on a password widget.  It removes the Tcl-level alias to the widget
#	command, reverting the widget to the Tk widget command.  Tk will
#	then destroy the widget command.

proc password:deleteAlias w {
	catch {rename password:alias$w $w}
	catch {rename $w {}}
}

# Procedure: password:updateTextVariable
#
# Synopsis:
#	Update the content of a password box when the associated text variable
#	changes.
#
# Usage:
#c	password:updateTextVariable pathName
#
# Parameters:
#c	pathName
#		Path name of the password box.
#
# Return value:
#	Not specified.
#
# Description:
#	password:updateTextVariable is called whenever the text
#	variable associated with a password widget changes its value.
#	It updates the content of the displayed portion of the password
#	widget to contain the appropriate number of bullets.

proc password:updateTextVariable w {
	global password_active
	if [info exists password_active($w)] return
	set v [lindex [$w.real configure -textvariable] 4]
	upvar #0 $v content
	set l [string length $content]
	password:alias$w delete 0 end
	for {set i 0} {$i < $l} {incr i} {
		password:alias$w insert end \267
	}
}
