# pref.tcl
#
# User pref.  This uses a table-driven scheme to set a bunch
# of variables in the rest of the application.  The results are
# written out to a Xresources-style file that is read by Preferences_Init
# at startup.
#
# Copyright (c) 1993 Xerox Corporation.
# Use and copying of this software and preparation of derivative works based
# upon this software are permitted. Any distribution of this software or
# derivative works must comply with all applicable United States export
# control laws. This software is made available AS IS, and Xerox Corporation
# makes no warranty about the software, its performance or its conformity to
# any specification.
#
#   preferences.tcl,v 1.6 1995/04/04 16:39:38 drs1004 Exp
#    Copyright (C) 1994  Donald Syme
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 1, or (at your option)
#    any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#    Contact Details:
#	Donald Syme
#	The Computer Laboratory
#	New Musuems Site
#	Pembroke St.
#	Cambridge U.K. CB2 3QG
#
#	email: Donald.Syme@cl.cam.ac.uk
#


#----------------------------------------------------------------------------
#
# ABSTRACT DATA OBJECT (Preferences Data Base)
# 
# CONFIGURATION OPTIONS
#
#	-group
#		A group of preferences from the preferences database.
#
#		
# WIDGET COMMAND
#
# EXAMPLE
#
#----------------------------------------------------------------------------

proc PrefVar { item } { lindex $item 0 }
proc PrefXres { item } { lindex $item 1 }
proc PrefDefault { item } { lindex $item 2 }
proc PrefComment { item } { lindex $item 3 }
proc PrefHelp { item } { lindex $item 4 }

proc preferences::InitialisePackage { } {
    global pref 
    global gui_flags

    set userDefaults ~/.tkholwb-defaults

    set pref(uid) 0
    set pref(panes) {}
    set pref(userDefaults) $userDefaults

    if [file exists $userDefaults] {
	PreferencesReadFile $userDefaults user
    }


    Preferences_Add "Windows & Scrolling" \
"Window placement and scroll-related preferences are set here.
Contrained scrolling keeps the last line of text in a window stuck
to the bottom of the window." [list \
	[list gui_flags(scrollbarSide) scrollbarSide {CHOICE right left} {Scrollbar side} "Which side the scrollbars appear on.  This only takes effect after you restart $gui_flags(title)."] \
	[list gui_flags(scrollSpeed) scrollSpeed 1 {Drag-Scroll speed} "How fast things scroll when you drag a text widget around with the (by default) middle button."] \
	[list gui_flags(scrollAccel) scrollAccel 4 {Drag-Scroll acceleration} "How fast things scroll when you drag a text widget around with the (by default) middle button with Shift depressed."] \
    	[list widgetText(constrained) textConstrainScroll OFF "Constrained Scrolling" "Constrained scrolling clamps the last line of text to the bottom of the text widget."] \
    	[list widgetText(scrollContext) textContextLines 2 "Scroll Context" "Scroll context is the number of lines that pages overlap when paging up and down through text."] \
    	[list widgetText(selectDelay) textSelectDelay 50 "Scroll/Select Time Constant" "When you drag a selection off the top or bottom of a text widget, the widget automatically scrolls and extends the selection.  This parameter is a polling time period that affects the behavior.  Too big and it is not responsive enough.  Too small and it may consume too many cycles.  Units are milliseconds."] \
	[list gui_flags(placeToplevel)	placeToplevel ON	{Remember window placement} "With this enabled, $gui_flags(title) will remember the placement of the various popup windows between sessions.  This means you can position them once manually and they will always appear there.  However, if you use a virtual root window manager and run $gui_flags(title) in different \"rooms\"then a remembered placement might be in the wrong room.  You can nuke all the placement memory from the end of the .tkholwb-defaults file and always run $gui_flags(title) from the same room, or just disable this feature.."] \
    ]
    


    Preferences_Add "General Fonts" "Fonts for buttons, lists and labels may be specified below.  Changes to these values generally only take effect after you restart $gui_flags(title)." [list \
	[list gui_flags(font,labels) labelsFont [list FONT -Adobe-Helvetica-Bold-R-Normal-*-*-100-*] "Labels" "The normal font used for labels."] \
	[list gui_flags(font,menus) menusFont [list FONT -Adobe-Helvetica-Bold-R-Normal-*-*-120-*] "Menus" "The normal font used for menus."] \
	[list gui_flags(font,buttons) buttonsFont [list FONT -Adobe-Helvetica-Bold-R-Normal-*-*-100-*] "Buttons" "The normal font used for buttons."] \
	[list gui_flags(font,feedback) feedbackFont [list FONT -Adobe-Helvetica-Bold-R-Normal-*-*-100-*] "Feedback" "The normal font used for the feedback line."] \
	[list gui_flags(font,listboxes) listboxesFont [list FONT -Adobe-Helvetica-Medium-R-Normal-*-*-100-*] "Lists" "The normal font used to display lists of items."] \
	[list gui_flags(font,textentry) textEntryFont [list FONT -Adobe-Helvetica-Medium-R-Normal-*-*-100-*] "Text Entry" "The normal font used for entry and text widgets."] \
	[list gui_flags(font,codeentry) codeEntryFont [list FONT -Adobe-Courier-Medium-R-Normal-*-*-100-*] "Code Entry" "The normal font used for widgets where code is entered."] \
	[list gui_flags(font,helptext) helpTextFont [list FONT -Adobe-Helvetica-Medium-R-Normal-*-*-100-*] "Help Text" "The normal font used for help text"] \
    ]
    

}


proc PreferencesReadFile { basename level } {
    if [catch {option readfile $basename $level} err] {
	errormessage .error "Error in $basename: $err"
    }
    if {[tk colormodel .] == "color"} {
	if [file exists $basename-color] {
	    if [catch {option readfile $basename-color $level} err] {
		errormessage .error "Error in $basename-color: $err"
	    }
	}
    } else {
	if [file exists $basename-mono] {
	    if [catch {option readfile $basename-mono $level} err] {
		errormessage .error "Error in $basename-mono: $err"
	    }
	}
    }
}
proc Preferences_Add { id text prefs } {
    global pref

    # Set up the table that drives the UI layout
    set ix [lsearch pref(panes) $id]
    if {$ix < 0} {
	lappend pref(panes) $id
	set pref($id,prefs) $prefs
	set pref($id,text) $text
    } else {
	lappend pref($id,prefs) $prefs
	append pref($id,text) \n$text
    }

    # Initialize the global variable from the option database,
    # else the default value supplied.

    foreach item $prefs {
	set varName [PrefVar $item]
	set xresName [PrefXres $item]
	set value [PrefValue $varName $xresName]
	set default [PrefDefault $item]
	if {$value == {}} {
	    # Set variables that are still not set
	    switch -- [lindex $default 0] CHOICE {
	        PrefValueSet $varName [lindex $default 1]
	    } FONT {
	        PrefValueSet $varName [lindex $default 1]
	    } COLOR {
	        PrefValueSet $varName [lindex $default 1]
	    } LIST {
	        PrefValueSet $varName [lrange $default 1 end]
	    } OFF {
		PrefValueSet $varName 0
	    } ON {
		PrefValueSet $varName 1
	    } default {
		PrefValueSet $varName $default
	    }
	} else {
	    # Warp booleans to 0 or 1
	    if {$default == "OFF" || $default == "ON"} {
		case $value {
		    {0 1} { # ok as is }
		    {true True TRUE} { PrefValueSet $varName 1}
		    {false False FALSE} {PrefValueSet $varName 0}
		    default {
			catch {puts stderr "Bogus boolean value $value for Xresource $xresName"}
			PrefValueSet $varName 0
		    }
		}
	    }
	}
    }
}
# Return the value of the given variable,
# or the value from the xresource database,
# or {} if neither exist
proc PrefValue { _var _xres } {
    upvar #0 $_var var
    if [info exists var] {
	return $var
    }
    set var [option get . $_xres {}]
}
# set the value of the variable
proc PrefValueSet { _var _value } {
    upvar #0 $_var var
    set var $_value
}
proc PrefEntrySet { entry varName } {
    PrefValueSet $varName [$entry get]
}

proc preferences::groups { } {
	global pref
	return [lsort $pref(panes)]
}

proc PreferencesSave { args } {
    global pref PrefEntry
    set newstuff {}
    foreach id $pref(panes) {
	foreach item $pref($id,prefs) {
	    set varName [PrefVar $item]
	    set xresName [PrefXres $item]
	    if {[info exists PrefEntry($varName)] && [llength [info commands $PrefEntry($varName)]] >= 1} {
		PrefValueSet $varName [$PrefEntry($varName) get]
	    }
	    set value [PrefValue $varName $xresName]
	    lappend newstuff [format "%s\t%s" *${xresName}: $value]
	}
    }
    Preferences_RewriteSection "Lines below here automatically added" "End Preferences State" $newstuff
    PreferencesReset
}
proc Preferences_RewriteSection { boundary1 boundary2 newstuff } {
    global pref
    if [catch {
	set old [open $pref(userDefaults) r]
	set oldValues [split [string trimright [read $old] \n] \n]
	close $old
    }] {
	set oldValues {}
    }
    if [catch {open $pref(userDefaults).new w} out] {
	errormessage .error "Cannot save in $pref(userDefaults).new: $out" warn
	return
    }
    set state "before"
    foreach line $oldValues {
	case $state {
	    "before" {
		if {[string compare $line "!!! $boundary1"] == 0} {
		    set state "inside"
		    puts $out "!!! $boundary1"
		    puts $out "!!! [exec date]"
		    puts $out "!!! Do not edit below here"
		    foreach item $newstuff {
			puts $out $item
		    }
		    puts $out "!!! $boundary2"
		} else {
		    puts $out $line
		}
	    }
	    "inside" {
		if {[string compare $line "!!! $boundary2"] == 0} {
		    set state "after"
		}
	    }
	    "after" {
		puts $out $line
	    }
	}
    }
    if {$state == "before"} {
	puts $out "!!! $boundary1"
	puts $out "!!! [exec date]"
	puts $out "!!! Do not edit below here"
	foreach item $newstuff {
	    puts $out $item
	}
	puts $out "!!! $boundary2"
    }
    close $out
    set new [glob $pref(userDefaults).new]
    set old [file root $new]
    if [catch {exec mv $new $old} err] {
	errormessage .error "Cannot install $new: $err"
	return
    }
}
proc Preferences_ReadSection { boundary1 boundary2 } {
    global pref
    if [catch {
	set old [open $pref(userDefaults) r]
	set oldValues [split [string trimright [read $old] \n] \n]
	close $old
    }] {
	set oldValues {}
    }
    set state "before"
    set results {}
    foreach line $oldValues {
	case $state {
	    "before" {
		if {[string compare $line "!!! $boundary1"] == 0} {
		    set state "inside"
		}
	    }
	    "inside" {
		if {![regexp {^!!!} $line]} {
		    lappend results $line
		}
		if {[string compare $line "!!! $boundary2"] == 0} {
		    break
		}
	    }
	}
    }
    return $results
}
proc PreferencesReset { {id_in {}} } {
    global pref

# Re-read user defaults.  Don't clear the option database since
# so many widget defaults are stored there!!
#    option clear

    PreferencesReadFile $pref(userDefaults) user
    # Now set variables
    if {$id_in == {}} {
	set id_in $pref(panes)
    } else {
	set id_in [list $id_in]
    }
    foreach id $id_in {
	foreach item $pref($id,prefs) {
	    set varName [PrefVar $item]
	    set xresName [PrefXres $item]
	    set xresval [option get . $xresName {}]
	    if {$xresval != {}} {
		set default $xresval
	    } else {
		set default [PrefDefault $item]
	    }
	    switch -- [lindex $default 0] CHOICE {
		PrefValueSet $varName [lindex $default 1]
	    } FONT {
		PrefValueSet $varName [lindex $default 1]
	    } COLOR {
		PrefValueSet $varName [lindex $default 1]
	    } LIST {
		PrefValueSet $varName [lrange $default 1 end]
	    } OFF {
		PrefValueSet $varName 0
	    } ON {
		PrefValueSet $varName 1
	    } default {
		global PrefEntry
		if {[info exists PrefEntry($varName)] && [llength [info commands $PrefEntry($varName)]]>=1} {
		    set entry $PrefEntry($varName)
		    $entry delete 0 end
		    $entry insert 0 $default
		}
		PrefValueSet $varName $default
	    }
	}
    }
}
proc Preferences_Resource { _varName _rname _default } {
    set _rval [option get . $_rname {}]
    if {$_rval != {}} {
	PrefValueSet $_varName $_rval
    } else {
	PrefValueSet $_varName $_default
    }
}



#----------------------------------------------------------------------------
#
# WIDGET CLASS preferences
# 
# CONFIGURATION OPTIONS
#
#	-group
#		A group of preferences from the preferences database.
#
#		
# WIDGET COMMAND
#
# EXAMPLE
#
#----------------------------------------------------------------------------

proc preferences { args} {
    	global pref
    	global gui_flags

	for {set i 0} {$i<[llength $args]} {incr i} {
	    case [lindex $args $i] -group {
	        incr i
	        set group [lindex $args $i]
	    } default {
	    	error "unrecognized arg [lindex $args $i]"
	    }
	}
	
	set id $group
	
    global pref env
    set ix [lsearch $pref(panes) $id]
    if {$ix < 0} {
	return
    }
    if [catch {toplevel .pref$ix}] {
	raise .pref$ix
    } else {
	global gui_flags
	wm title .pref$ix "$id Preferences"
	pack [frame .pref$ix.but] -side bottom -padx 10 -pady 10
        set buttons .pref$ix.but
	wm protocol .pref$ix WM_DELETE_WINDOW [list preferences::dismiss $ix]
    	pack [fontcheck focusable button $buttons.reset \
		-text Reset \
		-command [list PreferencesReset $id] \
		-font $gui_flags(font,buttons)] \
	    -side left
    	pack [fontcheck focusable button $buttons.save \
		-text "Save All" \
		-command "[list PreferencesSave -group $id] ; preferences::dismiss" \
		-font $gui_flags(font,buttons)] \
	    -side left
    	pack [fontcheck focusable button $buttons.close \
		-text Close \
		-command [list preferences::dismiss $ix] \
		-font $gui_flags(font,buttons)] \
	    -side left
	wm minsize .pref$ix 25 2
	pack [frame .pref$ix.help \
		-borderwidth 2 \
		-relief sunken] \
	    -expand yes -fill both
		      
	pack [fontcheck scrollable text .pref$ix.help.text \
		-height 8 \
		-font $gui_flags(font,helptext) \
		-wrap word] -expand yes -fill y
	set txt .pref$ix.help.text.b
	$txt insert 1.0 $pref($id,text)
	set maxWidth 0
	foreach item $pref($id,prefs) {
	    set len [string length [PrefComment $item]]
	    if {$len > $maxWidth} {
		set maxWidth $len
	    }
	}
	foreach item $pref($id,prefs) {
	    preferences::dialog_item .pref$ix $item $maxWidth
	    $txt insert end "\n\n[PrefComment $item]\n\t[PrefHelp $item]"
	}
	$txt configure -state disabled
    }
    
    focus_goToFirst .pref$ix
}

proc preferences::next { ix {i 1}} {
    global pref
    set geo [string trimleft [wm geometry .pref$ix] -x0123456789]
    destroy .pref$ix
    catch {preferences::nuke_item_help .prefitemhelp}
    incr ix $i
    set id [lindex $pref(panes) $ix]
    if {$id != {}} {
	PreferencesSectionDialog $id
	wm geometry .pref$ix $geo
    }
}

proc preferences::dialog_item { frame item width } {
	global gui_flags
    	global pref
    	incr pref(uid)
    	set f [frame $frame.p$pref(uid)]
    	pack $f -side top -fill both
    	pack [fontcheck label $f.label \
		-text [PrefComment $item] \
		-width $width \
		-font $gui_flags(font,labels)] \
            -side left -fill both
    	bind $f.label <1> "preferences::item_help  %X %Y \[PrefHelp [list $item]\]"

    	set default [PrefDefault $item]
    switch -regexp --  [lindex $default 0] {
	CHOICE {
	    foreach choice [lreplace $default 0 0] {
	        incr pref(uid)
	        pack [fontcheck focusable radiobutton $f.c$pref(uid) \
			-text $choice \
			-value $choice \
			-font $gui_flags(font,buttons) \
			-variable [PrefVar $item] \
			-relief flat] \
		    -side left
	    }
	}
	LIST {
	    pack [fontcheck listeditor $f.listedit \
			-variable [PrefVar $item] \
			-geometry 40x5] \
		    -side left -fill both -expand yes -pady 10
	}
	^(OFF|ON)\$ { 
	    # This is a boolean
	    set varName [PrefVar $item]
	    pack [fontcheck focusable checkbutton $f.check \
		    -text "On" \
		    -variable $varName \
		    -relief flat \
		    -font $gui_flags(font,buttons) \
		    -command [list preferences::bool_fixup $f.check.b $varName]] \
	        -side left
	    preferences::bool_fixup $f.check.b $varName
	} 
	default {
	    # This is a string or numeric
	    global PrefEntry
	    pack [fontcheck entry $f.entry \
		    -width 10 -relief sunken \
		        -font $gui_flags(font,textentry)] \
	        -side left -fill both -expand yes
	    set PrefEntry([PrefVar $item]) $f.entry

	    set varName [PrefVar $item]
	    $f.entry insert 0 [uplevel #0 [list set $varName]]
	    bind $f.entry <Return> [list PrefEntrySet %W $varName]
	}
    }
}
proc preferences::bool_fixup { check varName } {
    upvar #0 $varName var
    if {$var} {
	$check config -text On
    } else {
	$check config -text Off
    }
}
proc preferences::item_help { x y text } {
    global pref
    catch {destroy .prefitemhelp}
    if {$text == {}} {
	return
    }
    set self [Widget_Toplevel .prefitemhelp "Item help" Itemhelp [expr $x+10] [expr $y+10]]
    wm transient .prefitemhelp
    pack [message $self.msg -text $text -aspect 1500]
    bind $self.msg <1> {preferences::nuke_item_help .prefitemhelp}
    tkwait visibility .prefitemhelp
}
proc preferences::nuke_item_help { t } {
    global pref
    destroy $t
}


proc preferences::dismiss {{ix {}}} {
    global pref
    catch {destroy .pref$ix}
    catch {preferences::nuke_item_help .prefitemhelp}
    if {$ix == {}} {
	catch {destroy .prefhelp}
	set ix 0
	foreach id $pref(panes) {
	    catch {destroy .pref$ix}
	    incr ix
	}
    }
}
	

