#############################################################################
#   TkTheoryViewer.tcl,v 1.26 1995/04/04 16:39:25 drs1004 Exp
#    Copyright (C) 1994  Donald Syme
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 1, or (at your option)
#    any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#    Contact Details:
#	Donald Syme
#	The Computer Laboratory
#	New Musuems Site
#	Pembroke St.
#	Cambridge U.K. CB2 3QG
#
#	email: Donald.Syme@cl.cam.ac.uk
#
#############################################################################



#----------------------------------------------------------------------------
#
# WIDGET CLASS TkTheoryViewer
#
# OPTIONS
#	-theory
#		The initial theory to display
#
#	-withfeedback
#		Whether a feedback line should be displayed at the bottom
#		of the TkTheoryViewer.
# COMMANDS
#
#----------------------------------------------------------------------------

option add *TkTheoryViewer.withfeedback 1 widgetDefault

proc TkTheoryViewer { w args } {
	global feedback
	global TkTheoryViewer_flags
	global gui_flags

	for {set i 0} {$i<[llength $args]} {incr i} {
	    case [lindex $args $i] -theory {
	        incr i
	        set theory [lindex $args $i]
	    } -withfeedback {
	        incr i
	        set withfeedback [lindex $args $i]
	    } default {
	    	error "unrecognized arg [lindex $args $i]"
	    }
	}

	global busy
	incr busy 1

	toplevel $w -class TkTheoryViewer
	wm minsize $w 1 1
	catch {wm withdraw $w} 

	if ![info exists withfeedback] { set withfeedback [option get $w withfeedback WithFeedback] }
	if ![info exists theory] { set theory [ML current_theory()] }
	
	global vals
	set vals($w,richtext) [hol88]
	set vals($w,theory) $theory
							      
	set feedback [list {} "Displaying theory $theory"]

	# 3. Make the internals. First make a whole lot of frame
	# to put the different subcomponents in.

	if {$withfeedback} {
	    fontcheck label $w.feedback \
		-height 1 \
		-width 40 \
		-anchor w \
		-relief sunken \
		-font $gui_flags(font,feedback)
	    pack $w.feedback -side bottom -expand no -fill x
	}

	pack [frame $w.subframes] -expand yes -fill both -side bottom

	# 3a.  Make the Parents frame.

	pack [collapsible ParentsFrame $w.parents \
		-title Parents \
		-visible 1 \
		-multiselect 1 \
		-height 150 \
		-width 650 \
		-relief sunken \
		-command "TkTheoryViewer::change_theory $w" \
		-collapseCommand "pack config $w.parents -expand "] \
	    -side top -in $w.subframes -expand yes -fill both

	# 3b. Make the Theorems, Axioms and Definitions Frames.
	#
	# When the frames are expanded/collapsed we must alter the
	# packing on the frame so they don't take up extra space.  Hence
	# the -collapseCommand argument.

	foreach pr [list [list axiom Axioms] [list theorem Theorems] [list definition Definitions]] {
	    set objtype [lindex $pr 0]
	    set label [lindex $pr 1]
	    pack [collapsible TheoremsFrame $w.$objtype \
			-title "[set label]:" \
			-visible 0 \
			-richtextvar vals($w,richtext) \
			-height 6 \
			-width 65 \
			-multiselect 1 \
			-collapseCommand "pack config $w.$objtype -expand "] \
	        -fill both -expand no -in $w.subframes
	}

	# 4. Display the theory into these internals.

	TkTheoryViewer::display_theory $w

	# 5. Destruction stuff.

	widget_addBinding $w Destroy "TkTheoryViewer::upon_destroy $w"
	wm protocol $w WM_DELETE_WINDOW "TkTheoryViewer::verify_destroy $w"


	# 6. 

	catch {wm deiconify $w}
 	tkwait visibility $w
	update
	focus_goToFirst $w.parents
	catch "focus_skip $w.menu 1"
	catch "focus_skip $w.theorem.b.box.list 1"
	catch "focus_skip $w.axiom.b.box.list 1"
	catch "focus_skip $w.definition.b.box.list 1"

        if [hol88] {
	    ML -type void "current_theory_add_tcl_client ([ml_string TkTheoryViewer],[ml_string $w])"
	}
	if [hol90] {
	    ML -type void "TclCurrentTheoryNotification.add_client \n(TclCurrentTheoryNotification.mk_client([ml_string TkTheoryViewer],\n[ml_string $w]))"
	}
	set feedback [list {} "Please wait..."]

	incr busy -1
	return .
}


#----------------------------------------------------------------------------
# TkTheoryViewer::change_theory
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::change_theory { w theories } {
	global vals
	global busy
	global feedback
	incr busy 1
	if {[llength $theories]==1} {
	    set theory [lindex $theories 0]
	    if {$vals($w,theory)!=$theory} {
	        set feedback [list $w "Opening theory $theory..."]
	    	TkTheoryViewer::undisplay_theory $w
	    	set vals($w,theory) $theory
	    	TkTheoryViewer::display_theory $w
	    }
	}
	incr busy -1
}


#----------------------------------------------------------------------------
# TkTheoryViewer::verify_destroy 
#
# Called when the user tries to destroy the TkTheoryViewer via
# WM_DELETE_WINDOW window manager protocol or by the "Close" menu option.
#----------------------------------------------------------------------------

proc TkTheoryViewer::verify_destroy { w } {
	global vals
	global busy
	incr busy
	global feedback
	set feedback [list $w "Please wait..."]
	destroy $w
	incr busy -1
}


#----------------------------------------------------------------------------
# TkTheoryViewer::upon_destroy - called just as the TkTheoryViewer is destroyed.
#----------------------------------------------------------------------------

proc TkTheoryViewer::upon_destroy { w } {
        if [hol88] {
	    ML -type void "current_theory_remove_tcl_client ([ml_string TkTheoryViewer],[ml_string $w])"
	}
	if [hol90] {
	    ML -type void "TclCurrentTheoryNotification.remove_client \n(TclCurrentTheoryNotification.mk_client([ml_string TkTheoryViewer],\n[ml_string $w]))"
	}
	unset_vals_for_widget $w
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::entries_for_packages { w packages } {
	set entries ""
	foreach package $packages {
	    set p [lindex $package 0]
	    set text [lindex $package 1]
	    lappend entries [list command "$text" "newwin [set p]" 0]
	}
	return $entries

}


#----------------------------------------------------------------------------
# remake_menus
#
#
# 1. Determine the mode and whether we need to do anything or not.
# Also find the list of theories to show within
# Open Other =>.
#
# Nb. do not call "ancestors".  It takes ages.
#
# 
# 2. Set up the menus using MakeMenus
#
# 3. Create the Open Other cascade menu.  There is a bug here, as
# when new parents and libraries are added they are not reflected
# in this menu.  They will, however, always be available from the
# menu for the current theory.
# 
# 4. Adjust the disabled/enabled states of the menu entries under the View
# menu according to the values of vals($w,exist_<objtype>)
#
# Note these menu options are in existence in all modes, hence we do not
# have to check if the menu options exist.
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::remake_menus { w } {
	global vals 
	global TkTheoryViewer_flags     
	global gui_flags
	global feedback
	global TkTheoryViewer_version

	if [ML -type bool "draft_mode()"] {
	    set mode draft
	} else {
	    set mode proof
	}
	set vals($w,mode) $mode


	foreach win [info commands $w.menu.*] {
	    catch {destroy $win}
	}
	set viewmenu                                                   	\
	[list view "View" 0 left				       	\
	    [list 						       	\
		[list command "Expand All" "TkTheoryViewer::expandall $w" 0] \
		[list command "Expand Theorems" "TkTheoryViewer::expandkind $w theorem" 7] \
		[list command "Expand Definitions" "TkTheoryViewer::expandkind $w definition" 7] \
		[list command "Expand Axioms" "TkTheoryViewer::expandkind $w axiom" 7] \
		[list command "Expand Types" "TkTheoryViewer::expandkind $w type" 8] \
		[list command "Expand Constants" "TkTheoryViewer::expandkind $w constant" 7] \
	    	[list sep]						\
		[list command "Rescan Theory Tree" "ParentsFrame::remake_hierarchy $w.parents.b" 0] \
	    	[list sep]						\
		[list checkbutton "Rich Text" vals($w,richtext) 	\
			"TkTheoryViewer::RichTextToggled $w" 0] 	\
	    ]								\
	]

	
	set helpmenu 							\
	[list help "Help" 0 left					\
	    [HolHelp::menu_entries]					\
	]

	set theoremsmenu [list theorems "Theorems" 1 left [TkTheoryViewer::entries_for_packages $w $TkTheoryViewer_flags(proof_packages)]]
	set definitionsmenu [list definitions "Definitions" 0 left [TkTheoryViewer::entries_for_packages $w $TkTheoryViewer_flags(definition_packages)]]
	set typesmenu [list types "Types" 0 left [TkTheoryViewer::entries_for_packages $w $TkTheoryViewer_flags(type_packages)]]
	set axiomsmenu [list axioms "Axioms" 0 left [TkTheoryViewer::entries_for_packages $w $TkTheoryViewer_flags(axiom_packages)]]

	set searchmenu 							\
	[list search "Search" 0 left					\
	    [list 							\
		[list command "Selection..." "newwin TkTRS -sources \[hierarchy::selection $w.parents.b.hier\]" 0]			\
		[list command "Ancestors of selection..." "newwin TkTRS -sources \[hierarchy::selection $w.parents.b.hier\] -ancestors 1" 0]	\
		[list command "Descendants of selection..." "newwin TkTRS -sources \[hierarchy::selection $w.parents.b.hier\] -descendants 1" 0] \
		[list command "All Theories..." "incr busy; newwin TkTRS -sources \[hol_ancestry\]; incr busy -1" 0]	\
	    ]								\
	]

	set entries ""
	foreach preference_group [preferences::groups] {
	    lappend entries [list command "$preference_group..." "preferences -group [list $preference_group]" 0]
	}
	set preferencesmenu [list preferences "Options" 0 left $entries ]

	set entries [TkTheoryViewer::entries_for_packages $w $TkTheoryViewer_flags(other_packages)]
	set packagesmenu [list other "Packages" 0 left $entries ]

	set file_entry_new [list command "New Theory..." "TkTheoryViewer::new_theory $w" 0]
	set file_entry_load [list command "Load Theory..." "TkTheoryViewer::load_theory $w" 5]
	set file_entry_export [list command "Export Theory..." "TkTheoryViewer::export_theory $w" 0]
	set file_entry_save_hol [list command "Save HOL Image..." "TkTheoryViewer::save_hol $w" 0]
	set file_entry_add_parent [list command "Add Parent..." "TkTheoryViewer::new_parent $w" 0]
	set file_entry_load_mlfile [list command "Load ML File..." "TkTheoryViewer::load_mlfile $w" 5]
	set file_entry_load_library [list command "Load Library..." "TkTheoryViewer::load_library $w" 0]
	set file_entry_view [list cascade "View Other" $w.menu.file.ancestors 0]
	set file_entry_close [list command "Close Window" "TkTheoryViewer::verify_destroy $w" 0]
	set file_entry_exit [list command "Exit" "after 1 exit" 1]


	set draft_file_menu                                           \
	[list file "File" 0 left 					\
	    [list 							\
		$file_entry_new                         		\
	        $file_entry_load                        		\
	        $file_entry_export                        		\
	        $file_entry_add_parent                        		\
	    	[list sep]						\
	        $file_entry_load_library                        	\
	        $file_entry_load_mlfile	                        	\
	    	[list sep]						\
	        $file_entry_save_hol                        		\
	    	[list sep]						\
	        $file_entry_view                        		\
	    	[list command "Proof Mode" 				\
			"TkTheoryViewer::enter_mode $w proof" 0] 	\
	    	[list sep]						\
		$file_entry_close                       		\
		$file_entry_exit                       			\
	    ] 								\
	]						


	set proof_file_menu                                           \
	[list file "File" 0 left 					\
	    [list 							\
	        $file_entry_new                         		\
	        $file_entry_load                        		\
	        $file_entry_export                        		\
	    	[list sep]						\
	        $file_entry_load_library                        	\
	        $file_entry_load_mlfile	                        	\
	    	[list sep]						\
	        $file_entry_save_hol                        		\
	    	[list sep]						\
	        $file_entry_view                        		\
	    	[list command "Draft Mode" 				\
		"TkTheoryViewer::enter_mode $w draft" 0] 		\
	    	[list sep]						\
	    	$file_entry_close                       		\
	    	$file_entry_exit                       			\
	    ] 								\
	]  

	set title "$gui_flags(title): \[[set vals($w,theory)]\]"

	switch $mode draft {
	    wm title $w [concat $title "(draft mode)"]
	    set menus [list $draft_file_menu $viewmenu]
	} proof {
	    wm title $w [concat $title "(proof mode)"]
	    set menus [list $proof_file_menu $viewmenu]
	}

        if {[llength [lindex $searchmenu 4]]>0} { lappend menus $searchmenu }
        if {[llength [lindex $axiomsmenu 4]]>0} { lappend menus $axiomsmenu }
	if {[llength [lindex $definitionsmenu 4]]>0} { lappend menus $definitionsmenu }
	if {[llength [lindex $typesmenu 4]]>0} { lappend menus $typesmenu }
	if {[llength [lindex $theoremsmenu 4]]>0} { lappend menus $theoremsmenu }
	lappend menus $preferencesmenu
	lappend menus $packagesmenu $helpmenu
    	MakeMenus $w $menus

 	if !$vals($w,show_constants) { 
	    $w.menu.view.m entryconfigure "*Constants*" -state disabled 
	}
 	if !$vals($w,show_types) { 
 	    $w.menu.view.m entryconfigure "*Types*" -state disabled
	}
 	if !$vals($w,show_definitions) { 
 	    $w.menu.view.m entryconfigure "*Definitions*" -state disabled
	}
 	if !$vals($w,show_axioms) { 
 	    $w.menu.view.m entryconfigure "*Axioms*" -state disabled
	}
 	if !$vals($w,show_theorems) { 
 	    $w.menu.view.m entryconfigure "*Theorems*" -state disabled
	}


	menu $w.menu.file.ancestors
	set menu $w.menu.file.ancestors
	set ancestors [lsort [hol_ancestry]]
	while {[llength $ancestors]!=0} {
	    set head [lrange $ancestors 0 19]
	    set ancestors [lrange $ancestors 20 end]
	    foreach ancestor $head {
	        $menu add command -label $ancestor 	\
		    -command "TkTheoryViewer::change_theory $w $ancestor"	\
		    -underline 0
	    }
	    if {[llength $ancestors]!=0} {
	        $menu add cascade -label "More" 	\
		    -menu $menu.menu \
		    -underline 0
	        set menu $menu.menu
		menu $menu
	    }
	}
	

 	if [hol90] { 
 	    $w.menu.view.m entryconfigure "*Rich*" -state disabled
	}

}



proc TkTheoryViewer::expandall { w } {
	global busy
	incr busy
	foreach kind [list theorem definition axiom types constants] { 
	    TkTheoryViewer::expandkind $w $kind
	}
	incr busy -1
}



proc TkTheoryViewer::expandkind { w kind } {
	if {[winfo exists $w.$kind.b]} { 
	    global busy
	    incr busy
	    global feedback
	    set feedback [list $w "Expanding [set kind]s"]
  	    TheoremsFrame::expandall $w.$kind.b 
	    incr busy -1
	}
}


proc TkTheoryViewer::undisplay_theory { w } {
	global vals 
	TheoremsFrame::settheorems $w.theorem.b {}
	TheoremsFrame::settheorems $w.axiom.b {}
	TheoremsFrame::settheorems $w.definition.b {}
}


proc TkTheoryViewer::display_theory { w } {
	global vals 
	global feedback
	global TkTheoryViewer_flags

	#
	# 1. Lookup what to display about the theory. Always display
	# definitions by default.
	#
	set theory $vals($w,theory)

	set vals($w,show_theorems) [ML -type bool "not (null (theorems [ml_string $theory]))"]
	set vals($w,show_definitions) [ML -type bool "not (null (definitions [ml_string $theory]))"]
	set vals($w,show_axioms) [ML -type bool "not (null (axioms [ml_string $theory]))"]
	set vals($w,show_constants) [ML -type bool "not (null(constants [ml_string $theory]))"]
	set vals($w,show_types) [ML -type bool "not (null(types [ml_string $theory]))"]

	set vals($w,show_parents) 1
	
	# Configure each of the sub-frames with information about what
	# to display.  Pack each of them into the display.

	TkTheoryViewer::remake_menus $w

	ParentsFrame::setselection $w.parents.b $vals($w,theory)

	foreach pr [list [list axiom Axioms] [list theorem Theorems] [list definition Definitions]] {
	    set objtype [lindex $pr 0]
	    set label [lindex $pr 1]
		set theory $vals($w,theory)
		set mltheory [ml_string $theory]
	        set feedback [list $w "Finding [set label]..."]
		set names [lsort [ML -type stringlist "map fst ([set objtype]s $mltheory)"]]
		set specs ""
		foreach name $names {
		    lappend specs [list $name $objtype $theory]
		}
		eval [list TheoremsFrame::settheorems $w.$objtype.b] $specs
	    if {[set vals($w,show_[set objtype]s)]} {
	        collapsible_show $w.$objtype
	    } else {
	        collapsible_hide $w.$objtype
	    }
	    TheoremsFrame::configure $w.$objtype.b \
		-trace [list $theory $objtype]
	}

	set vals($w,theory_is_current) [expr {[ML current_theory()]==$vals($w,theory)}]


}



#----------------------------------------------------------------------------
# TkTheoryViewer::enter_mode
#
# Called when the user selects a new mode from the menu.
#----------------------------------------------------------------------------

proc TkTheoryViewer::enter_mode { w mode } {
  	global vals
	global busy
	incr busy 1
	global feedback
	set feedback [list $w "Entering $mode mode..."]
	if {$vals($w,mode)==$mode} {
	    return
	} else {
	    case $mode draft {
	    	ML -type void -log 1 "extend_theory (current_theory())"
	    } proof {
	       ML -type void -log 1 "close_theory()"
	    }
	}	
	incr busy -1
}


#----------------------------------------------------------------------------
# TkTheoryViewer::current_theory_mode_change_Notify
#
# Gets called when the mode of the current theory is changed in
# the HOL session, either via the interface or some other means.  
# See the HolTheoryNotification package for
# how.
#
# TkTheoryViewer::current_theory_change_Notify
#
# Gets called when the current theory changes.  We context-switch
# to the current theory if the old theory displayed used to be the current
# theory.
#
# TkTheoryViewer::current_theory_ancestry_change_Notify
#
# Gets called when the ancestry of the current theory changes.
#----------------------------------------------------------------------------


proc TkTheoryViewer::current_theory_mode_change_Notify { w isdraft } {
	global busy
	incr busy 1
	TkTheoryViewer::remake_menus $w
	incr busy -1
}

proc TkTheoryViewer::current_theory_change_Notify { w new_current } {
	global vals
	if {$vals($w,theory_is_current)} {
	    TkTheoryViewer::change_theory $w $new_current
	}
}
									 
proc TkTheoryViewer::current_theory_ancestry_change_Notify { w current } {
}
									 
#----------------------------------------------------------------------------
# TkTheoryViewer::load_theory
#
# Load a new theory as the current theory.  
# This will normally mean the current theory will have
# to enter "view" mode, hence recompute_mode is called for the
# current theory if everything else succeeds.
#----------------------------------------------------------------------------

						     
proc TkTheoryViewer::load_theory { w } {
	global vals
	if [ML -type bool "draft_mode()"] {
	    errormessage $w.error "You cannot load a theory while in draft mode.\nFirst change to proof mode using the option under the File menu."
	    return
 	}
	set theory_pathname [selectfile -pattern *.[hol_thryext] -message "Select a descendant theory:"]
	if {$theory_pathname!=""} {
	    set theory_segment [file rootname [file tail $theory_pathname]]
	    set theory_dir [file dirname $theory_pathname]/
	    if [hol88] {
	    	if {![ML -type bool -check 1 "mem [ml_string $theory_dir] (search_path())"]} {
		    ML -log 1 -type void -check 1 "set_search_path(([ml_string $theory_dir])\n.(search_path()))"
		}
	    }
	    if [hol90] {
	    	if {![ML -type bool "mem [ml_string $theory_dir] (!Globals.theory_path)"]} {
		    ML -log 1 -type any -check 1 "Globals.theory_path := ([ml_string $theory_dir])::(!Globals.theory_path)"
		}
	    }
	    if [catch {ML -type void -log 1 -check 1 "load_theory [ml_string $theory_segment]"} err] {
	        errormessage $w.err $err
	    }
	}
}

#----------------------------------------------------------------------------
# TkTheoryViewer::load_mlfile
#
#----------------------------------------------------------------------------

						     
proc TkTheoryViewer::load_mlfile { w } {
	global feedback
	set file [selectfile -pattern *.[ml_file_ext] -message "Select a ML file to load:"]
	if {$file!=""} {
	    set feedback [list $w "Loading $file..."]
	    ML_load $file -log 1 -trace 1
	}
}

#----------------------------------------------------------------------------
# TkTheoryViewer::new_parent
#
# Adds a parent to the existing current theory.
#
# Problem: Should add the new parent theory(ies) to the Open Other => 
# cascade menu of every theory.  This would be easier if the cascade
# menu was shared amongst all TkTheoryViewers.  This should happen
# via notifications on any new_parent calls.
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::new_parent { w } {
	global vals				
	set theory_pathname [selectfile -pattern *.[hol_thryext] -message "Select the new parent to add:"]
	if {$theory_pathname!=""} {
	    set theory_segment [file rootname [file tail $theory_pathname]]
	    ML -type void -log 1 -check 1 "new_parent [ml_string $theory_segment]"
	}
}


#----------------------------------------------------------------------------
# TkTheoryViewer::load_library
#
# Problem: Should add the new parent theory(ies) to the Open Other => 
# cascade menu of every theory.  This would be easier if the cascade
# menu was shared amongst all TkTheoryViewers.  This should happen
# via notifications on any new_parent calls.
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::load_library { w } {
	global vals				
	global feedback
	if [hol88] {
	    set vals($w,waiter) [GetString $w.get "Load Library" "Library: "]
	}
	if [hol90] {
	    set feedback [list $w "Looking for libraries..."]
	    modalDialog transient LoadLibraryBox $w.loadlib \
		-header "Select a library to load:" \
		-resultsvariable vals($w,waiter)
	    widget_waitVariable vals($w,waiter)
	    modalDialog.end $w.loadlib
	    update
	}
	global busy
	incr busy
	if {[lindex $vals($w,waiter) 0] == "Ok"} {
	    set library [lindex $vals($w,waiter) 1]
	    set feedback [list $w "Loading library $library..."]
	    if [catch {hol_load_library $library} err] {
	        errormessage $w.error $err; incr busy -1; return
	    }
	}
	incr busy -1
}



#----------------------------------------------------------------------------
# TkTheoryViewer::new_theory
#
# Creates a new theory.
#
# This will normally mean the current theory will have
# to enter "view" mode, hence recompute_mode is called for the
# current theory if everything else succeeds.
#----------------------------------------------------------------------------

proc TkTheoryViewer::new_theory { w } {
	global TkTheoryViewer_flags
	global busy
	set res [GetString $w.get "NewTheory" "Theory Name: "]
    	if {[lindex $res 0]=="Ok"} {
	    incr busy
	    set newtheory [lindex $res 1]
            if [file exists $newtheory.[hol_thryext]] {
		error "Theory $newtheory.[hol_thryext] already exists." 
	    } else {
	        if [catch {ML -type void -log 1 "new_theory [ml_string $newtheory]"} err] {
		    errormessage $w.err $err; incr busy -1; return
		}
	    }	
	    incr busy -1
        }
}


#----------------------------------------------------------------------------
# TkTheoryViewer::save_hol
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::save_hol { w } {
	global vals
	global feedback
	set image_pathname [selectfile -new 1 -pattern *.[hol_thryext] -message "Select a filename for the new executable:"]
	global busy
	incr busy
	set feedback [list $w "Saving the HOL image to disk..."]
	if {$image_pathname!=""} {
	    if [hol90] {
	        set function save_hol
	    }
	    if [hol88] {
	        set function save
	    }
	    if [catch {ML -type void -log 1 "$function [ml_string $image_pathname]"} err] {
	        errormessage $w.err $err; incr busy -1; return
	    }
	}
	incr busy -1
}


#----------------------------------------------------------------------------
# TkTheoryViewer::export_theory
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::export_theory { w } {
	global vals
	global feedback
	global busy
	incr busy
	set feedback [list $w "Exporting the current theory to disk..."]
	if [hol90] {
	    if [catch {ML -type void -log 1 "export_theory()"} err] {
	        errormessage $w.err $err; incr busy -1; return
	    }
	}
	if [hol88] {
	    if [catch {ML -type void -log 1 "close_theory()"} err] {
	        errormessage $w.err $err; incr busy -1; return
	    }
	    if [catch {ML -type void -log 1 "extend_theory (current_theory())"} err] {
	        errormessage $w.err $err; incr busy -1; return
	    }
	}
	incr busy -1
}

#----------------------------------------------------------------------------
# TkTheoryViewer::RichTextToggled
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::RichTextToggled { w  } {
  	global vals TkTheoryViewer_priv
	modalDialog transient choicebox $w.verify \
		-text "Reformat all displayed theorems?" \
		-buttons [list Yes No] \
		-icon think \
		-textvariable TkTheoryViewer_priv
	widget_waitVariable TkTheoryViewer_priv
	modalDialog.end $w.verify
	if {$TkTheoryViewer_priv=="Yes"} {
	    foreach wintype [list theorem definition axiom types constants] {
	        if [winfo exists $w.$wintype.b] { 
  		    TheoremsFrame::reformatall $w.$wintype.b 
		}
	    }
	}
	unset TkTheoryViewer_priv
}


#----------------------------------------------------------------------------
#
# ProcessArgs
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::ProcessArgs { argc argv } {
	global TkTheoryViewer_flags
	global gui_flags
	set TkTheoryViewer_flags(view_only) 0
	set TkTheoryViewer_flags(proof_packages) ""
	set TkTheoryViewer_flags(type_packages) ""
	set TkTheoryViewer_flags(definition_packages) ""
	set TkTheoryViewer_flags(axiom_packages) ""
	set TkTheoryViewer_flags(search_packages) ""
	set TkTheoryViewer_flags(other_packages) ""


	if {[llength [info commands Preferences_Add]]==1} {
    	    Preferences_Add "Theory Hierarchy Display" "Preferences related to the display of the theory hierarchy are set here." \
	 	[list \
		    [list TkTheoryViewer_flags(theoryTreeAnchor) theoryTreeAnchor [list CHOICE s se w nw n ne e sw] "Theory Tree Anchor" "This determines the manner in which the theory tree is drawn.  The value indicates a position on the compass from which the tree is drawn.  For instance \"s\" indicates that the tree should be drawn with the current theory at the bottom."] \
		    [list TkTheoryViewer_flags(theoryTreeInitialExpand) theoryTreeInitialExpand 2 "Initial Expansion Depth" "The depth to which the theory tree is initially expanded."] \
		    [list TkTheoryViewer_flags(theoriesToFlatten) theoriesToFlatten [list HOL BASIC-HOL BASIC_HOL] "Theories to Flatten" "This determines the theories at which the theory tree will be \"flattened\".  This is done by showing all the ancestors of these theories as parents, instead of just showing the immediate parents.  This is useful for collector theories such as HOL and BASIC-HOL (BASIC_HOL under hol90), and possibly for some of your own theories as well."] \
		    [list TkTheoryViewer_flags(theoriesToIgnore) theoriesToIgnore [list HOL BASIC-HOL BASIC_HOL] "Theories to Ignore" "These theories are ignored unless they are the sole parent of a theory.  This is useful for theories which are only parents of theories due to their status as the initial theory of a HOL session.  Theories such as HOL and BASIC-HOL (BASIC_HOL under hol90) fall into this category."] \
		    [list TkTheoryViewer_flags(theoriesToPrune) theoriesToPrune [list HOL] "The initial expansion of the tree is pruned at these theories.  This can be used to prevent blowout of the initial display of the theory tree."] \
	     ]
    	}

	for {set arg 0} {$arg < $argc} {incr arg} {
	    switch -- [lindex $argv $arg] -viewonly {
	        set TkTheoryViewer_flags(view_only) 1
	    } -theoryTreeInitialExpand {
		incr arg
	        set TkTheoryViewer_flags(theoryTreeInitialExpand) [lindex $argv $arg] 
	    } -theoryTreeAnchor {
		incr arg
	        set TkTheoryViewer_flags(theoryTreeAnchor) [lindex $argv $arg] 
	    } -proofpackage {
		incr arg
	        lappend TkTheoryViewer_flags(proof_packages) [lindex $argv $arg] 
	    } -typepackage {
		incr arg
	        lappend TkTheoryViewer_flags(type_packages) [lindex $argv $arg] 
	    } -definitionpackage {
		incr arg
	        lappend TkTheoryViewer_flags(definition_packages) [lindex $argv $arg] 
	    } -searchpackage {
		incr arg
	        lappend TkTheoryViewer_flags(search_packages) [lindex $argv $arg] 
	    } -axiompackage {
		incr arg
	        lappend TkTheoryViewer_flags(axiom_packages) [lindex $argv $arg] 
	    } -otherpackage {
		incr arg
	        lappend TkTheoryViewer_flags(other_packages) [lindex $argv $arg] 
	    }
	}
}
	
#----------------------------------------------------------------------------
# Test routine used from interactive tcl.
#----------------------------------------------------------------------------

proc te { } {
   after 1 "
   	source \$TkTheoryViewer_library/TkTheoryViewer.tcl
    	source \$TkTheoryViewer_library/ParentsFrame.tcl
    	source \$hierarchy_library/hierarchy.tcl
    	source \$HolRichText_library/TheoremsFrame.tcl
    	source \$HolRichText_library/RichText.tcl
   	newwin TkTheoryViewer -withfeedback 1
    "
}  



