#----------------------------------------------------------------------------
# WIDGET CLASS LoadLibraryBox
#
#	A box containing items specifying where to save a theorem to.
#
#	The box does not destroy itself.  It sets a global variable
#	which may be traced by trace or widget_waitVariable or a similar
#	mechanism.
#
# CONFIGURATION OPTIONS
#
#	-header
#		Some text printed at the top of the box as a message.
#
#	-resultsvariable
#
#		This variable is set when either the OK or Cancel
#		buttons are pressed.  It is set to a 4-tuple
#		containing the name of the button pressed, a boolean
#		value indicating whether the save checkbox was
#		selected, the text typed into the save-name field
#		a boolean value indicating whether the bind field
#		was selected.  If Cancel was pressed then the tuple
#		only contains one element.
#
#		The widget checks that a save name has been entered
#		if the save field has been selected, and that the name
#		is a valid ML identifier.
#		
# WIDGET COMMAND
# 
# EXAMPLE
#
#	modalDialog transient LoadLibraryBox $w.save \
#		-header "Select a library:" \
#		-resultsvariable vals($w,waiter)
#	widget_waitVariable vals($w,waiter)
#	modalDialog.end $w.save
#	switch -- [lindex $vals($w,waiter) 0] Ok { ... } Cancel { ... }
#----------------------------------------------------------------------------

option add *LoadLibraryBox.header "" widgetDefault
option add *LoadLibraryBox.resultsvariable results widgetDefault


proc LoadLibraryBox { w args } {
	global vals 
	global gui_flags


	for {set i 0} {$i<[llength $args]} {incr i} {
	    switch -- [lindex $args $i] -header {
	        incr i
	        set header [lindex $args $i]
	    } -resultsvariable {
	        incr i
	        set vals($w,resultsvariable) [lindex $args $i]
	    } default {
	    	error "unrecognized arg [lindex $args $i]"
	    }
	}
	frame $w -class LoadLibraryBox
	if ![info exists header] { set header [option get $w header Header] }
	if ![info exists resultsvariable] { set resultsvariable [option get $w resultsvariable ResultsVariable] }

	pack [label $w.header \
		-text $header \
		-font $gui_flags(font,labels) \
		-anchor w] \
	    -side top -padx 10 -fill x
	    

	pack [frame $w.options] -side top -pady 10
	pack [frame $w.options.buttons] \
	    -side right -pady 10
	    
	pack [entry $w.options.lib \
		-relief sunken \
		-borderwidth 2 \
		-font $gui_flags(font,textentry) \
		-textvariable vals($w,entry)] \
	    -side bottom -fill x -padx 10
	
	pack [scrollable listbox $w.options.list \
		-geometry 20x10 \
		-relief sunken \
		-borderwidth 2 \
		-font $gui_flags(font,listboxes)] \
	    -side left -padx 10
	tk_listboxSingleSelect $w.options.list.b
    	bind $w.options.list.b <Any-ButtonRelease-1> \
		[list LoadLibraryBox::select $w %y]
    	bind $w.options.list.b <Double-Button-1> \
		[list LoadLibraryBox::ok_pressed $w]
	pack [fontcheck focusable button $w.options.buttons.find_lib \
		-width 13 \
		-text "Find Library" \
		-command "LoadLibraryBox::find_library $w" \
	 	-font $gui_flags(font,buttons)] \
	    -side top
	pack [fontcheck focusable button $w.options.buttons.add_path \
		-width 13 \
		-text "Add Path..." \
		-command "LoadLibraryBox::add_path $w" \
	 	-font $gui_flags(font,buttons)] \
	    -side top
	    
	    
	pack [frame $w.controls] -side bottom -pady 10
	pack [fontcheck focusable button $w.controls.ok \
		-width 6 \
		-text Ok \
		-command "LoadLibraryBox::ok_pressed $w" \
	 	-font $gui_flags(font,buttons)] \
	    -side left \
	    -padx 10
	pack [fontcheck focusable button $w.controls.cancel \
		-width 6 \
		-text Cancel \
		-command "set $vals($w,resultsvariable) Cancel" \
		-font $gui_flags(font,buttons)] \
	    -side left \
	    -padx 10
	    
	LoadLibraryBox::refill_list $w
	    
	wm protocol [winfo toplevel $w] WM_DELETE_WINDOW "set $vals($w,resultsvariable) Cancel"
	button_setDefault $w.controls.ok
	focus_goToFirst $w
	bind $w <Destroy> "unset_vals_for_widget $w"

}



proc LoadLibraryBox::ok_pressed { w } {
	global vals
 	set libname [$w.options.lib get]
	upvar #0 $vals($w,resultsvariable) var
 	set var [list Ok $libname]
}

proc LoadLibraryBox::find_library { w } {
	global vals
	global busy
	incr busy
 	set libname [$w.options.lib get]
	if [catch {ML -check 1 -type any "find_library [ml_string $libname]"} err] {
	    errormessage $w.err $err; incr busy -1; return
	}
	LoadLibraryBox::refill_list $w
	incr busy -1
}


proc LoadLibraryBox::add_path { w } {
	global env
	set file [selectfile -pattern *.[hol_thryext] \
		-message "Select a directory to add, then press OK:" \
		-dironly 1 \
		-directory $env(HOME) lib_dir \
		-action "Add Path"]
	if {$file!=""} {
	    ML -toplevel 1 "Globals.library_path := \n([ml_string $file/] \n:: (!Globals.library_path))"
	}
}

proc LoadLibraryBox::refill_list { w } {
	global vals
 	set libname [$w.options.lib get]
	if [hol90] {
	    set known_libs [ML -type stringlist "map (#name o dest_library) (known_libraries ())"]
	    set loaded_libs [ML -type stringlist "map (#name o dest_library) (loaded_libraries ())"]
	}
	$w.options.list.b delete 0 end
	set unloaded_libs ""
	foreach lib $known_libs {
	        if {[lsearch -exact $loaded_libs $lib]==-1} {
		    lappend unloaded_libs $lib
		}
	}
	if {[llength $unloaded_libs]!=0} {
	    eval [list $w.options.list.b insert end] [lsort $unloaded_libs]
	}
}


proc LoadLibraryBox::select { w y } {
	global vals
    	upvar #0 vals($w,entry) entryvar
    	set i [$w.options.list.b nearest $y]
    	$w.options.list.b select from $i
    	set entryvar [$w.options.list.b get $i]
}

