#############################################################################
#   HolSlave.tcl,v 1.27 1995/04/04 16:39:14 drs1004 Exp
#    Copyright (C) 1994  Donald Syme
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 1, or (at your option)
#    any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#    Contact Details:
#	Donald Syme
#	The Computer Laboratory
#	New Musuems Site
#	Pembroke St.
#	Cambridge U.K. CB2 3QG
#
#	email: Donald.Syme@cl.cam.ac.uk
#
#############################################################################

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------


proc ML_int { mlcode } {
	ML -type int $mlcode
}
proc ML_bool { mlcode } {
	ML -type bool $mlcode
}
proc ML_void { mlcode } {
	ML -type void $mlcode
}



proc ML_load { file args } {
    if [hol88] { eval [list ML -toplevel 1] $args [list "loadt `$file`"] }
    if [hol90] { eval [list ML -toplevel 1] $args [list "use \"$file.sml\""] }
}


#----------------------------------------------------------------------------
# TkHolSlave::Create
#
# - Start the HOL session
# - Load the ML code to call tcl code from ML.
# - Load the theory specified by any -theory flags (TkHolSlave_flags(theory)).
# - Load any libraries from -library flags (TkHolSlave_flags(libraries)).
#----------------------------------------------------------------------------

proc TkHolSlave::Create { exe } {
    	global TkHolSlave_flags 
	global gui_flags 
	global TkHolSlave_library
	global feedback

	global spawn_id

	global hol_prompt
	if [info exists TkHolSlave_flags(initialprompt)] {
	    set hol_prompt $TkHolSlave_flags(initialprompt)
	} else {
	    if [hol88] { set hol_prompt "#" }
	    if [hol90] { set hol_prompt "-" }
	}

	# Echoing and return/new-line processing is set to the emacs
	# shell default to allow compatibility of output with what
	# emacs expects.  This does not seem to stuff up output
	# when TkHol is run from a shell.

	set feedback [list {} "Starting HOL..."]
	set stty_init "-echo -onlcr"
	set old_log_user [log_user]
	log_user 0
	spawn $exe
	match_max 20000	
	log_user $old_log_user

	set timeout 30
	expect -- "\n[hol_prompt]"
	if [catch {exp_send "\n"}] { 
		puts "\n\"$exe\" could not be executed.  Perhaps it is not a valid HOL executable or is not on your path.  Please specify a HOL executable to use with a -hol flag.\n\nExiting...\n\n"
		exit 
	}


        if [hol88] {
	    if {$TkHolSlave_flags(prompt_strategy)==""} {
		set hol_prompt "hol88 # "
	    } else {
		set hol_prompt $TkHolSlave_flags(prompt_strategy)
	    }
	    ML -toplevel 1 "set_prompt [ml_string $hol_prompt]"
	}
        if [hol90] {
	    global hol_secondary_prompt
	    if {$TkHolSlave_flags(prompt_strategy)==""} {
		set hol_prompt "hol90 - "
	    } else {
		set hol_prompt $TkHolSlave_flags(prompt_strategy)
	    }
	    if {$TkHolSlave_flags(secondaryprompt_strategy)==""} {
		set hol_secondary_prompt "hol90 = "
	    } else {
		set hol_secondary_prompt $TkHolSlave_flags(secondaryprompt)
	    }
	    ML -toplevel 1 "System.Control.primaryPrompt := [ml_string $hol_prompt]"
	    ML -toplevel 1 "System.Control.secondaryPrompt := [ml_string $hol_secondary_prompt]"
	}

	# Write out the new prompt so user's aren't confused by
	# the new prompt only appearing when they type something
	# for the first time.

	if [log_user] {
	    puts -nonewline "\n$hol_prompt"
	}


	# - Load each lisp extension file into hol88.  There is only
	# one file at the moment.  We make a check to see if the 
	# compiled versions of the files actually exist.

        if [hol88] {
	    set lisp_dir $TkHolSlave_library/lisp/$gui_flags(arch)
	    set lispload [file readable $lisp_dir/qconcat.o]
	    if $lispload {
	            ML -toplevel 1 "lisp [ml_string "(load \"$lisp_dir/qconcat.o\")"]"
	    } else {
		puts stderr "\nHmmm.. $gui_flags(title) couldn't find the hol88 lisp extension .o files in $lisp_dir.\nPerhaps you haven't compiled them for this architecture ($gui_flags(arch))?\n\nAborting..."
		exit
	    }
	} 
	
	# - Define the ML function concatl for fast concatenation of lists
	# ML, ML_int, ML_bool and ML_void make use of this function, and hence 
	# cannot be used before this.

        ML_load $TkHolSlave_library/fconcatl
		
	# - Define the "tcl" function to allow callbacks from the HOL session.
	# This works by writing the strings STARTTCL and ENDTCL 
	# on the output stream followed by the tcl to execute.
	#
	# - Redefine the quit() function in hol88.
	# - Redefine the exit() function in hol90.
	#
	# - Load the initial libraries specified
	#
	# - Load the initial theories specified.
	#

	ML -toplevel 1 "[ml_fun] tcl s = [ml_output] ([ml_string "[ml_eol]STARTTCL[ml_eol]{"] ^ s ^ [ml_string "}[ml_eol]"] ^ [ml_string "ENDTCL[ml_eol]"])"
	ML -toplevel 1 "[ml_fun] interface_on () = tcl [ml_string {newwin TkTheoryViewer}]"
	ML -toplevel 1 "[ml_fun] interface_off () = tcl [ml_string {foreach win [info commands .?*] { catch {if {[winfo toplevel $win]!={.}} { catch {destroy [winfo toplevel $win]} } }}}]"

	if [hol88] {
	    ML -toplevel 1 "let really_quit = quit"
	    ML -toplevel 1 "letrec quit () = tcl [ml_string {global busy; incr busy; exit}]" 
	}
	if [hol90] {
	    ML -toplevel 1 "val really_exit = exit"
	    ML -toplevel 1 "fun exit () = tcl [ml_string {global busy; incr busy; exit}]" 
	}

	foreach lib $TkHolSlave_flags(libraries) {
	    set feedback [list {} "Loading $lib library"]
	    hol_load_library $lib
	}	
        if {$TkHolSlave_flags(initial_theory)!=""} {
	    if {[ML "current_theory()"]!=$TkHolSlave_flags(initial_theory)} {
		ML -log 1 -type void "load_theory [ml_string $TkHolSlave_flags(initial_theory)]"
	    }
	}

     	TkHolSlave::SIGINT_effects_master

}



#----------------------------------------------------------------------------
# Slave::Destroy
#
#----------------------------------------------------------------------------

proc TkHolSlave::Destroy { } {
	    if [hol88] {
	       exp_send "really_quit();;\n"
	    }
	    close
}


#----------------------------------------------------------------------------
# Slave::Reset
#
# This function is usually called as a result of a SIGINT on the
# tcl process.
#
# Bring a HOL slave which is freaking out because of a SIGINT 
# back to normal state.  For HOL88 this is done by "typing" :q.  
#
# Often, this signal will happen whilst the Tcl master is waiting
# for input from the HolSlave.  With any luck the reading routine will
# correctly  pickup that an interrupt has occured by analyzing the
# output from the slave.
#
#----------------------------------------------------------------------------

proc TkHolSlave::Reset { } {
	exec kill -INT [exp_pid]
	if [hol88] {
	    exp_send "\n:q\n"
	}
}


#----------------------------------------------------------------------------
# Slave::SIGINTtrap
#
# Called every time a SIGINT signal (^C) occurs, except when we are
# executing code on the slave and hence allow the slave to be interrupted
# by sending us a SIGINT.
#
# It seems that ^C's from bash always effect the children too,
# Hence the slave will be behaving as if its been given a ^C, and we
# have to recover from that.
# This means reading through the input generated in the slave by the
# interrupt.
#----------------------------------------------------------------------------

proc TkHolSlave::SIGINTtrap { } {
	global gui_flags

	if ![info exists gui_flags(title)] { set gui_flags(title) TkHol }

	catch {grab release [grab current]}
	global busy
	if {[info exists busy] && $busy>0} {
	    set busy 0
	    puts "$gui_flags(title) was in a hung state.  This has now been cleared."
	    return
	}

	puts "\nPress Ctrl-C again to kill $gui_flags(title)."
    	puts "  (NOTE: You may also use Ctrl-C to interrupt HOL computations )"
    	puts "  ($gui_flags(title) just wants to make sure you aren't trying to do that)"

	trap {global busy ; set busy 1 ; exit } SIGINT
}


#----------------------------------------------------------------------------
# Slave::SIGINT_effects_slave
#
# Internal function. Called before executing code on the slave.
# Interprets Control-C interrupts (SIGTERMs) as break-the-slave's.
#
#----------------------------------------------------------------------------

proc TkHolSlave::SIGINT_effects_slave { } {
	trap {
	    puts "Interrupting HOL..."
	    TkHolSlave::Reset
	    TkHolSlave::SIGINT_effects_master
	} SIGINT
}


proc TkHolSlave::SIGINT_effects_master { } {
	trap TkHolSlave::SIGINTtrap SIGINT
}






#----------------------------------------------------------------------------
# Expect Patterns
#
# These patterns and actions are designed to sit inside a loop
# which is processing HOL output.  The loop should wait on the
# variables "return" (indicating the end of a sequence has happened)
# and "error" indicating an error has been detected in the HOL session.
#
# At the end of the loop a variable called "delayed_executions" will
# contain any Tcl code ordered to be executed by embedded STARTTCL/ENDTCL
# output.
#
# Each of the different patterns/actions can have different side effects.
# These are described just now:
#
# EXPECT_TCL_SEQUENCE
#	Pattern: Looks for STARTTCL's in the HOL output.  
#	Side effects: Appends the code to execute to "delayed_executions"
#
# EXPECT_MLDONE
#	Pattern: Looks for TCLML_DONE lines in the HOL output
#	Side effects: Does a break when one appears
#
# EXPECT_MLERROR
#	Pattern: Looks for common error strings such as interrupts and
# 		typing errors.
#	Side effects: Sets "error" to 1 and "errval" to a meaningful error
#		string
#
# EXPECT_CAPTURE_HOL_PROMPT
#	Pattern: Looks for a HOL prompt starting a line.
#	Side effects: Sets "return" to 1, adds hol prompt 
#		line to output accumulated in "retval"
#
# EXPECT_CAPTURE_LINE
#	Pattern: Captures a line
#	Side effects: Adds the line to output accumulated in "retval"
#
# EXPECT_PROCESS_HOL_PROMPT
#
# EXPECT_PROCESS_LINE
#
# EXPECT_ML_RESULT
#
# EXPECT_FULL_BUFFER
#
#
#----------------------------------------------------------------------------

proc EXPECT_TCL_SEQUENCE {} {
	list "STARTTCL\n" {
	    expect -re "(\[^\n\]*)\nENDTCL\n" {
	        lappend delayed_executions [lindex $expect_out(1,string) 0]
	        continue
	    }
	}
}

proc EXPECT_FEEDBACK {} {
	list -re "([join [list \
{\[opening [^\n]*\]} \
{\[Major collection... [^\n]*\]} \
{\[Increasing heap to [^\n]*\]} \
{Loading the library [^\n]*\.} \
	] {|} ])" {    
			global feedback
			set feedback [list {} $expect_out(1,string)]
	}
}

proc EXPECT_MLDONE { } { 
	list "\nTCLML_DONE" break
}


proc EXPECT_MLERROR { } {
	list -re "([join [list \
{unbound or non.*} \
{syntax error.*} \
{unbound variable or constructor.*} \
{operator and operand don't agree.*} \
{Interrupt.*} \
{Console interrupt.*} \
{ill\-typed phrase.*} \
{Indeterminate types.*} \
{unclosed string.*} \
{HOL Error.*} \
{illegal token.*} \
{evaluation failed.*} \
{skipping.*} \
{uncaught exception.*} \
{Error: operator is not a function.*} \
{Error: syntax error found at.*} \
{Error: unbound structure.*} \
{Error: expression and handler don't.*} \
{Unconstrained type variable in.*} \
{Type inference failure.*} \
{The library.*is already loaded.*} \
	] {|}])[hol_re_prompt]" {
			set error 1
			set errval $expect_out(1,string)
      		  }
}


proc EXPECT_CAPTURE_HOL_PROMPT { } {
	list [hol_prompt] { 
		    if ![info exists retval] { set retval "" }
		    set retval [set retval]$expect_out(0,string)
		    set return 1
	        }
}


proc EXPECT_CAPTURE_LINE { } {
    list -re "(\[^\n\]*\n)" { 
	if ![info exists retval] { set retval "" }
	set retval [set retval]$expect_out(1,string)

    }
}

proc EXPECT_PROCESS_HOL_PROMPT { } {
	list -re "(.*[hol_re_prompt])" {
	   	    set output $expect_out(1,string)
	   	    set error [catch {eval $output_processor [list $output]} errval]
		    set return [expr !$error]
		    set retval ""
	}
}

proc EXPECT_PROCESS_LINE { } {
	list -re (\[^\n\]*)\n { 
	   	    set output $expect_out(buffer)
	   	    set error [catch {eval $output_processor [list $output]} errval]
		    continue
	}
}


proc EXPECT_ML_RESULT { } {
    list -re "TCLML_STARTRESULT\n(\[^\n]*)\nTCLML_ENDRESULT\n" { 
        expect -- \n[hol_prompt]
	set retval $expect_out(1,string)
	set return 1
    }
}


proc EXPECT_FULL_BUFFER { } {
    list full_buffer {
	puts stderr "full buffer - increasing buffer size.  Repeat the operation..."
	match_max [expr [match_max]*2]
    }
}


#----------------------------------------------------------------------------
# ML
#
# TEST CASES
# ML -direct 1 -output_processor puts "current_theory();;"
# 
#----------------------------------------------------------------------------

proc ML {args} {

	global TkHolSlave_flags
	global TkHolSlave_library
	global busy


	set direct 0
	set toplevel 0
	set check 0
	set run 1
	set log 0
	set trace 0
	set type string
	set output_processor ""
	for {set i 0} {$i<[llength $args]} {incr i} {
	    switch -- [lindex $args $i] -type {
	        incr i
	        set type [lindex $args $i]
	    } -toplevel {
	        incr i
	        set toplevel [lindex $args $i]
	    } -run {
	        incr i
	        set run [lindex $args $i]
	    } -trace {
	        incr i
	        set trace [lindex $args $i]
	    } -direct {
	        incr i
	        set direct [lindex $args $i]
	    } -check {
	        incr i
	        set check [lindex $args $i]
	    } -log {
	        incr i
	        set log [lindex $args $i]
	    } -output_processor {
	        incr i
	        set output_processor [lindex $args $i]
	    } default {
	    	if [info exists mlcode] { error "too many arguments to ML" }
		set mlcode [lindex $args $i]
	    }
	}
	if {!$direct && !$toplevel} {
	    switch $type {
		bool { 
		    set val_code "(if ($mlcode) then [ml_string 1] else [ml_string 0])"
		}
		string {
		    set val_code "($mlcode)"
		}
		int {
    		    set val_code "([ml_int_to_string] ($mlcode))"
		}
		void {
		    set val_code "(($mlcode); [ml_string "{ }"])"
    		}
		stringlist {
		    set val_code "(string_list_to_tcl_list ($mlcode))"
		}
		any {
		    set val_code "(($mlcode); [ml_string "{ }"])"
    		}
	    }

	    set logged_code "$mlcode[ml_end_dec]"
	    if [hol90] {
		set val_code "($val_code) 
handle HOL_ERR \{message=s,origin_function=origf,origin_structure=os\} 
=> (output(std_out, \"\\nHOL Error: \" ^ s ^ \" (\" ^ os ^ \": \" ^ origf ^ \")\\n\"); raise Interrupt)"
	    }
            set full_code "[ml_output] \n(concatl [ml_list [list [ml_string [ml_eol]TCLML_DONE[ml_eol]TCLML_STARTRESULT[ml_eol]]  \n$val_code\n  [ml_string [ml_eol]TCLML_ENDRESULT[ml_eol]]]])[ml_end_dec]"
	    
	}

	if $toplevel {
    	    set full_code "$mlcode[ml_end_dec]\n [ml_output] [ml_string "[ml_eol]TCLML_DONE[ml_eol]TCLML_STARTRESULT[ml_eol]{ }[ml_eol]TCLML_ENDRESULT[ml_eol]"][ml_end_dec]"
	    set logged_code "$mlcode[ml_end_dec]"
	}


	if $direct {
	    set full_code $mlcode
	    set logged_code $mlcode
	}

	# Now do some checks on the code to see if it looks likely to
	# stuff to HOL session around.
	# The code checked is exactly what would go in the log file
	# since if that's OK then what we construct should be OK.

	if $check {
	    if [hol88] { set checker $TkHolSlave_library/../cmldecls/cmldecls }
	    if [hol90] { set checker $TkHolSlave_library/../cmldecls/smldecls }
	    if [catch {exec $checker << $logged_code} error] { 
		error "\nsyntax error: $error\n\n" 
	    }
	}


	# Part 2.  Run the code.

        set delayed_executions ""

	set retval ""
    if {$run} {
       	global expect_out
	set return 0
	set error 0
	TkHolSlave::SIGINT_effects_slave
	incr busy
	set old_log_user [log_user]
	log_user [expr $TkHolSlave_flags(trace) || $trace]
	if [expr $TkHolSlave_flags(trace) || $trace] {
	    puts $full_code
	}
	
	# perform a simple check - individual lines longer than 256
	# characters sieze up the input stream!
	#
	# 
	
	set pieces [split $full_code \n]
	foreach piece $pieces {
	    if {[string length $piece]>200} {
	        puts stderr "Warning: line of input to HOL longer than 200 characters\nWarning: (line = $piece).  Splitting..."
		set full_code [join [split $full_code "("] "(\n"]
		puts stderr "Warning: (new code = $full_code)"
		break
	    }
	} 
	
	
	exp_send "$full_code\n"
    	if {$direct && $output_processor==""} {
	      while {!$return && !$error} {
	        eval expect [EXPECT_TCL_SEQUENCE] [EXPECT_FEEDBACK] [EXPECT_CAPTURE_HOL_PROMPT] [EXPECT_CAPTURE_LINE]
	      }
	} 
    	if {$direct && $output_processor!=""} {
	      while {!$return && !$error} {
	        eval expect [EXPECT_TCL_SEQUENCE] [EXPECT_FEEDBACK] [EXPECT_PROCESS_HOL_PROMPT] [EXPECT_PROCESS_LINE]
	      }
	}
	if {!$direct} {
	      while {!$return && !$error} {
     		  eval expect [EXPECT_TCL_SEQUENCE] [EXPECT_FEEDBACK] [EXPECT_MLERROR] [EXPECT_MLDONE] 
	      }
	      while {!$return && !$error} {
	 	eval expect [EXPECT_ML_RESULT] [EXPECT_FEEDBACK] [EXPECT_MLERROR] [EXPECT_FULL_BUFFER]
	      }

 	}

    	TkHolSlave::SIGINT_effects_master
	log_user $old_log_user

	incr busy -1
	if {$error} { 
	    error $errval
	}
    }
	incr busy

	if {$TkHolSlave_flags(log) && $log} {
	    catch {
		set f [open $TkHolSlave_flags(logfile) a+]
	    	puts $f $logged_code
	    	close $f
	    }
	}

	foreach delayed_execution $delayed_executions {
	    if [expr $TkHolSlave_flags(trace) || $trace] {
		puts "Delayed TclExecution: $delayed_execution"
	    }
	    if [catch {uplevel #0 $delayed_execution} errval] { incr busy -1; error $errval }
	}

	incr busy -1
	return $retval

}


proc ML_bind { id mlcode args} {
	if [hol88] {
	    eval [list ML -toplevel 1 "[ml_val] $id = \n$mlcode"] $args
	} elseif [hol90] {
	    eval [list ML -toplevel 1] $args [list "
[ml_val] $id = ($mlcode) 
     	handle HOL_ERR \{message=s,origin_function=origf,origin_structure=os\} 
	=> (output(std_out, \"\\nHOL Error: \" ^ s ^ \" (\" ^ os ^ \": \" ^ origf ^ \")\\n\"); raise Interrupt);
		"]
	}
}



