#############################################################################
#   TkGoalProof.tcl,v 1.24 1995/04/04 16:39:11 drs1004 Exp
#    Copyright (C) 1994  Donald Syme
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 1, or (at your option)
#    any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#    Contact Details:
#	Donald Syme
#	The Computer Laboratory
#	New Musuems Site
#	Pembroke St.
#	Cambridge U.K. CB2 3QG
#
#	email: Donald.Syme@cl.cam.ac.uk
#
#############################################################################


proc TkGoalProof::delayedLoad { } {
	global TkGoalProof_flags TkGoalProof_library
	global feedback
	if ![info exists TkGoalProof_flags(loaded)] {
	    if [hol90] {
	        set feedback [list {} "Loading the replacement hol90 Goalstack package..."]
	    	ML -toplevel 1 "use [ml_string $TkGoalProof_library/repl_gstack.sig]"
	    	ML -toplevel 1 "use [ml_string $TkGoalProof_library/repl_gstack.sml]"
	    }
            set TkGoalProof_flags(loaded) 1
	}
}

#----------------------------------------------------------------------------
#
# WIDGET CLASS TkGoalProof
#
# OPTIONS
#
# COMMANDS
#
# NOTES
#
# We create a new goal stack for each TkGoalProof window.  This
# means creating a set of ML identifiers.
#
# The ML code is a straight copy out of stack.ml in 
# the hol88/hol90 source code.
#
# We redefine it once for each goalstack created - this seems like a
# good idea, to allow multiple goal proof sessions,
# but more of it could be abstracted.
#
#----------------------------------------------------------------------------

option add *TkGoalProof.withfeedback 1 widgetDefault

proc TkGoalProof { w args} {
	global TkGoalProof_version
	global TkGoalProof_flags
	global gui_flags
	global vals
	global busy
	incr busy

	TkGoalProof::delayedLoad
	if [hol90] {
	    if {[ML -type int "version_number"] < 7} {
	        errormessage $w "Sorry, TkGoalProof requires hol88 or hol90.7 or greater"
		incr busy -1; return
	    } 
	}

	set title "Proof Window"

	for {set i 0} {$i<[llength $args]} {incr i} {
	    case [lindex $args $i] -title {
	        incr i
	        set title [lindex $args $i]
	    } default {
	    	error "unrecognized arg [lindex $args $i]"
	    }
	}
	toplevel $w -class TkGoalProof
	if ![info exists withfeedback] { set withfeedback [option get $w withfeedback WithFeedback] }

	regsub -all {\.} $w "" tmp1
	set vals($w,unique_name) gdp$tmp1
	if [hol88] { set vals($w,rich_text) 0 }
	if [hol90] { set vals($w,rich_text) 0 }

	# 2. Do some windowing grunge
	#
	global feedback
	
	wm minsize $w 1 1
	wm title $w $title

	wm withdraw $w

	#
	# 3. Create some ML code to manipulate the goal stack.
	#
	set feedback [list $w "Creating HOL goal stacks..."]
	set unique_name [set vals($w,unique_name)]
	
	if [hol88] {
	
	    ML -direct 1 "
		letref [set unique_name]_goals = abs_goals \[\]	
	            and ([set unique_name]_backup_list :goalstack list) 
					=  \[\];;"

	    ML -direct 1 "
	        let [set unique_name]_change_state newgoals =				
		       do (let newbackup =					
              		fst (chop_list backup_limit [set unique_name]_backup_list)	
              		? [set unique_name]_backup_list			
       			in						
       		   	([set unique_name]_backup_list := 		
				[set unique_name]_goals . newbackup;	
     		   	[set unique_name]_goals := newgoals));;"

	    ML -direct 1 "
	        let [set unique_name]_set_goal =			
		  let bty = \":bool\" in				
		  let isbty tm = (type_of tm = bty) in			
		  \\(asl,w). 						
			if forall isbty (w.asl) then			
		          ([set unique_name]_change_state (abs_goals 	
				(new_stack (asl,w))))		
		        else failwith `Term in goal not of type \":bool\"`;;"
			
	    ML -direct 1 "
               let [set unique_name]_expandf tac =                	
                [set unique_name]_change_state (abs_goals              	
                  (push_fsubgoals (rep_goals [set unique_name]_goals)  	
                       tac));;"

	    ML -direct 1 "
	        let [set unique_name]_expand = 			
		    [set unique_name]_expandf o VALID;;"

	    ML -direct 1 "
	        let [set unique_name]_rotate n = 			
		[set unique_name]_change_state (abs_goals 		
		    (rotate_top n (rep_goals [set unique_name]_goals)));;"

	    ML -direct 1 "
	        let ([set unique_name]_backup : void->void) () =	
   		(let newgoals.newbackup = [set unique_name]_backup_list in 
    		if null (rep_goals newgoals) then failwith `no goals`		
		    else (do						
		     ([set unique_name]_goals := newgoals;		
		      [set unique_name]_backup_list := newbackup);	
			()))						
		 ? failwith `backup:  backup list is empty`;;"

	    ML -direct 1 "
	        let [set unique_name]_top_thm () = 			
		   top_proof(rep_goals [set unique_name]_goals);;"

	    ML -direct 1 "
	    	let [set unique_name]_top_goal () =			
  		   hd (fst (hd (rep_goals [set unique_name]_goals)));;"

	    ML -direct 1 "
	        let [set unique_name]_depth () =
			length (rep_goals [set vals($w,unique_name)]_goals) - 1;;"
			
	    ML -direct 1 "
		let [set unique_name]_numgoals () =
			length (fst (hd (rep_goals [set vals($w,unique_name)]_goals)));;"
	    # this is just a harmless execution for syncing...
	    ML -type bool "true"
	}
	
	if [hol90] {


	    # Konrad has a neat functional implementation
	    # of the goalstack stuff in HOL90.  
	    #
	    	    

	ML -direct 1 "
	    val [set unique_name]_goalstack = ref (ReplGoalstack.Functional.set_goal (\[\], (--`T`--)));"

	ML -direct 1 "
	    fun [set unique_name]_set_goal g = 
	       	([set unique_name]_goalstack := 
			ReplGoalstack.Functional.set_goal g;
		());"

	ML -direct 1 "
            fun [set unique_name]_backup () = 
		([set unique_name]_goalstack := 
			ReplGoalstack.Functional.backup (![set unique_name]_goalstack);
		());"
		
	ML -direct 1 "
            fun [set unique_name]_expand tac = 
		([set unique_name]_goalstack := 
			ReplGoalstack.Functional.expandf (Tactical.VALID tac) (![set unique_name]_goalstack);
		());"
		
	ML -direct 1 "
            fun [set unique_name]_top_thm () = 
		ReplGoalstack.Functional.top_thm (![set unique_name]_goalstack);"
		
	ML -direct 1 "
            fun [set unique_name]_top_goal () =
		ReplGoalstack.Functional.top_goal (![set unique_name]_goalstack);"

	ML -direct 1 "
            fun [set unique_name]_rotate n = 
		([set unique_name]_goalstack := 
			ReplGoalstack.Functional.rotate n (![set unique_name]_goalstack);
		());"
		
	ML -direct 1 "
	    fun [set unique_name]_numgoals () = 
		(length (ReplGoalstack.Functional.top_goals (![set unique_name]_goalstack)))
		handle HOL_ERR _ => 0;"
		
	ML -direct 1 "
            fun [set unique_name]_depth () = ReplGoalstack.Functional.depth (![set unique_name]_goalstack);"

	    # this is just a harmless execution for syncing...
	    ML -type bool "true"
	    
	}

	#
	# 4. now set up the menus for the TkGoalProof window.  Disable the "Save"
	# menus until the theorem is actually proven.
	#

	if {$withfeedback} {
	    fontcheck label $w.feedback \
		-height 1 \
		-width 40 \
		-anchor w \
		-relief sunken \
		-font $gui_flags(font,feedback)
	    pack $w.feedback -side bottom -expand no -fill x
	}

	set feedback [list $w "Creating TkGoalProof display..."]
	
	set menus ""
	lappend menus \
	    [list proof "Proof" 0 left 					\
	    	[list 							\
	            [list command "Clear" "TkGoalProof::clear_goal $w" 0] \
	            [list command "Set Goal" "TkGoalProof::set_goal $w" 4]  \
	            [list command "ApplyTactic" 			\
			"TkGoalProof::ApplyML $w 			\
				\[$w.nexttac.tac.b get 1.0 end\]" 0] 	\
	            [list command "Backup" "TkGoalProof::Backup $w" 0]  \
	    	    [list sep]						\
		    [list command "Save..." "TkGoalProof::save $w" 0]	\
	    	    [list sep]						\
	    	    [list command "Close" "destroy $w" 0] 		\
	    	] 							\
	    ]
	
	
	lappend menus                                                  \
	[list goal "Goal" 0 left				       	\
	    [list 						       	\
		[list checkbutton "Rich Text" vals($w,rich_text) 	\
			"TkGoalProof::RedisplayTopGoal $w" 0] 		\
	    ]								\
	]

	lappend menus [list help "Help" 0 left [HolHelp::menu_entries]]

	MakeMenus $w $menus

	if [hol90] {
	    $w.menu.goal.m entryconfigure "Rich*" -state disabled
	
	}
	$w.menu.proof.m entryconfigure "Save*" -state disabled

	# 
	# 5. Create the buttons and stuff for controlling and displaying
	# the goals.
	#

	

	pack [frame $w.goal] -fill both -expand yes -padx 10 -pady 10
	pack [fontcheck label $w.goal.lab \
		-text "Top Goal:" \
		-anchor w \
		-font $gui_flags(font,labels)] \
	    -fill x
	    
	pack [HolRichText $w.topgoal \
		-height 10 \
		-width 65 \
		-editable 1 \
		-richtextvar vals($w,rich_text)] \
	    -side left -fill both -expand yes -in $w.goal
	focus $w.topgoal.text

	
	
	pack [frame $w.info] -side top -padx 10 -pady 10
	pack [fontcheck label $w.numgoals \
		-text "Goals Remaining: 0" \
		-width 20 \
		-font $gui_flags(font,labels)] \
	    -side left -fill x -in $w.info
	pack [fontcheck label $w.depth \
		-text "Current Depth: 0" \
		-width 20 \
		-font $gui_flags(font,labels)] \
	    -side right -fill x -in $w.info

	pack [frame $w.controls] -side top -padx 10

	pack [fontcheck focusable button $w.controls.new -text "Clear" \
		-command "TkGoalProof::clear_goal $w" \
		-width 8 \
		-font $gui_flags(font,buttons)] \
	    -side left -padx 5
	pack [fontcheck focusable button $w.controls.setgoal -text "Set Goal" \
		-command "TkGoalProof::set_goal $w" \
		-width 8 \
		-font $gui_flags(font,buttons)] \
	    -side left -padx 5
	pack [fontcheck focusable button $w.controls.applyml -text "Apply Tactic" \
	   	-command "TkGoalProof::ApplyML $w \[$w.nexttac.tac.b get 1.0 end\]" \
	   	-width 11 \
		-font $gui_flags(font,buttons)] \
	    -side left -padx 5
	pack [fontcheck focusable button $w.controls.backup -text "Backup" \
		-command "TkGoalProof::Backup $w" \
		-width 8 \
		-font $gui_flags(font,buttons)] \
	    -side left -padx 5
	pack [fontcheck focusable button $w.controls.close -text "Close" \
	   	-command "TkGoalProof::verify_destroy $w" \
	   	-width 8 \
	    	-font $gui_flags(font,buttons)] \
	    -side left -padx 5

					     
	    
	    
	pack [frame $w.tacs] \
		-side top -fill both -padx 5 -pady 10
	
	pack [frame $w.nexttac] -padx 5 -side left -fill x -in $w.tacs
	pack [fontcheck label $w.lab2 \
		-text "Next Tactic:" \
		-anchor w \
		-font $gui_flags(font,labels)] \
	    -side top -fill x -in $w.nexttac
	pack [fontcheck scrollable text $w.nexttac.tac  \
		-borderwidth 2 \
		-relief sunken \
		-height 10 \
		-width 40 \
		-font $gui_flags(font,codeentry)] \
	    -side left -fill both
	
	pack [frame $w.entiretac] -padx 5 \
	    -side right -fill x -in $w.tacs
	
	pack [fontcheck label $w.lab3 \
		-text "Entire Tactic:" \
		-anchor w \
		-font $gui_flags(font,labels)] \
	    -side top -fill x -in $w.entiretac

    	
    	pack [scrollable text $w.entiretac.tac \
		-borderwidth 2 \
		-relief sunken \
		-height 10 \
		-width 40 \
		-font $gui_flags(font,codeentry) \
		-state disabled] \
    	    -side left -fill both


	pack [frame $w.tacbuttons] \
		-side right -padx 10 -pady 10 -in $w.controls

	    pack [menubutton $w.tacbuttons.tactic_history \
		    -text "Tactic History =>" \
	    	    -menu $w.tacbuttons.tactic_history.menu \
		    -font $gui_flags(font,buttons) \
		    -width 16 \
		    -relief raised \
		    -borderwidth 2] \
		-side top

	TkGoalProof::remake_tactic_history_menu $w


	set n 1
	while {[info exists TkGoalProof_flags(tactics$n)]} {
	    if {[llength $TkGoalProof_flags(tactics$n)] == 0} { incr n; continue }

	    pack [menubutton $w.tacbuttons.tactics$n \
		    -text "$TkGoalProof_flags(tactics$n,title) =>" \
	    	    -menu $w.tacbuttons.tactics$n.menu \
		    -font $gui_flags(font,buttons) \
		    -width 16 \
		    -relief raised \
		    -borderwidth 2] \
		-side top

	    menu $w.tacbuttons.tactics$n.menu
	    foreach tactic $TkGoalProof_flags(tactics$n) {
	    	set isdelayed [expr {[string first ... $tactic]!=-1}] 
		if $isdelayed {
		    set command [list TkGoalProof::PrepareML $w $tactic]
		} else {
		    set command [list TkGoalProof::ApplyML $w $tactic]
		} 
		$w.tacbuttons.tactics$n.menu add command \
		    -command $command \
	 	    -font $gui_flags(font,buttons) \
		    -label $tactic
	    }
	    
	    incr n
	}
	    	      
	set n 1
	while {[info exists TkGoalProof_flags(tacticals$n)]} {
	    if {[llength $TkGoalProof_flags(tacticals$n)] == 0} { incr n; continue }

	    pack [menubutton $w.tacbuttons.tacticals$n \
		    -text "$TkGoalProof_flags(tacticals$n,title) =>" \
	    	    -menu $w.tacbuttons.tacticals$n.menu \
		    -font $gui_flags(font,buttons) \
		    -width 16 \
		    -relief raised \
		    -borderwidth 2] \
		-side top

	    menu $w.tacbuttons.tacticals$n.menu
	    foreach tactical $TkGoalProof_flags(tacticals$n) {
		set tactical "$tactical ..."
		set command [list TkGoalProof::PrepareML $w $tactical]
		$w.tacbuttons.tacticals$n.menu add command \
		    -command $command \
	 	    -font $gui_flags(font,buttons) \
		    -label $tactical
	    }
	    
	    incr n
	}
	    	      
	set n 1
	while {[info exists TkGoalProof_flags(conversions$n)]} {
	    if {[llength $TkGoalProof_flags(conversions$n)] == 0} { incr n; continue }

	    pack [menubutton $w.tacbuttons.conversions$n \
		    -text "$TkGoalProof_flags(conversions$n,title) =>" \
	    	    -menu $w.tacbuttons.conversions$n.menu \
		    -font $gui_flags(font,buttons) \
		    -width 16 \
		    -relief raised \
		    -borderwidth 2] \
		-side top

	    menu $w.tacbuttons.conversions$n.menu
	    foreach conversion $TkGoalProof_flags(conversions$n) {
	    	set isdelayed [expr {[string first ... $conversion]!=-1}] 
		if $isdelayed {
		    set command [list TkGoalProof::PrepareML $w $conversion]
		} else {
		    set command [list TkGoalProof::ApplyML $w $conversion]
		} 
		$w.tacbuttons.conversions$n.menu add command \
		    -command $command \
	 	    -font $gui_flags(font,buttons) \
		    -label $conversion
	    }
	    
	    incr n
	}
	    	      

	# 
	# 6. Register the upon-destroy handler.
	#

	widget_addBinding $w Destroy "TkGoalProof::upon_destroy $w"
	wm protocol $w WM_DELETE_WINDOW "TkGoalProof::verify_destroy $w"

	TkGoalProof::clear_goal $w
	update idletasks
	wm deiconify $w	
	tkwait visibility $w
	update
	button_setDefault $w.controls.setgoal
	catch "focus_skip $w.menu 1"
	focus_goToFirst $w.topgoal.text
	update
	
	incr busy -1
	return $w
}
#----------------------------------------------------------------------------
# TkGoalProof::verify_destroy 
#
# Called when the user tries to destroy the window via
# WM_DELETE_WINDOW window manager action or by "Close" menu option.
#----------------------------------------------------------------------------

proc TkGoalProof::verify_destroy { w } {
	global vals
	global verify
	if {[info exists vals($w,saved)] 
	    && [info exists vals($w,savable)] 
            && [set vals($w,savable)] 
            && ![set vals($w,saved)]} {
	    modalDialog transient choicebox $w.verify \
		-text "You have not saved the theorem.\nDo you want to do this now?" \
		-buttons [list Yes No Cancel] \
		-icon think \
		-textvariable verify \
		-withfeedback 0
	    widget_waitVariable verify
	    modalDialog.end $w.verify
	    case $verify Yes {
		TkGoalProof::save $w
	    } No {
		destroy $w
	    } 
	} else {
	    destroy $w
	}
}

#----------------------------------------------------------------------------
# TkGoalProof::upon_destroy 
#
# Empty the goal stacks when the window is destroyed.
#
#----------------------------------------------------------------------------

proc TkGoalProof::upon_destroy { w } {
	global vals
	if [hol88] {
	    catch {ML -type void  "[set vals($w,unique_name)]_goals := abs_goals \[\]"}
	    catch {ML -type void  "[set vals($w,unique_name)]_backup_list := \[\]"}
	}
	unset_vals_for_widget $w

}


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc TkGoalProof::remake_tactic_history_menu { w } {
	global TkGoalProof_flags
	global gui_flags
	catch {$w.tacbuttons.tactic_history.menu delete 0 last}
	catch {menu $w.tacbuttons.tactic_history.menu}
	foreach history_tactic $TkGoalProof_flags(tactic_history) {
		set command [list TkGoalProof::ApplyML $w $history_tactic]
		$w.tacbuttons.tactic_history.menu add command \
		    -command $command \
	 	    -font $gui_flags(font,buttons) \
		    -label $history_tactic
	}
}



#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------


proc TkGoalProof::save { w args } {
	global vals

	modalDialog transient TheoremSaveBox $w.save \
		-header "Goal Proved!" \
		-resultsvariable vals($w,waiter) \
		-withfeedback 0
	widget_waitVariable vals($w,waiter)
	modalDialog.end $w.save
	switch -- [lindex $vals($w,waiter) 0] Ok {
	    set save [lindex $vals($w,waiter) 1]
	    set savename [lindex $vals($w,waiter) 2]
	    set bind [lindex $vals($w,waiter) 3]
	    set view_script [lindex $vals($w,waiter) 4]
	    set bindname $savename

	    global feedback
	    set feedback [list $w "Saving theorem..."]
	    set tactic [$w.entiretac.tac.b get 1.0 end]
	    set tactic [string range $tactic 0 [expr [string length $tactic]-2]]
	    set script "prove([hol_term $vals($w,goal)],\n\t[join [split $tactic \n] \n\t]\n)"
	    
	    if {$bind && $save} {
	 	ML -toplevel 1 -check 1 "[ml_val] $bindname = save_thm ( [ml_string $savename], ([set vals($w,unique_name)]_top_thm ()))"
	        set script "[ml_val] $bindname = \nsave_thm([ml_string $savename], $script)[ml_end_dec]\n"
	    } elseif {$bind} {
	 	ML -toplevel 1 -check 1 "[ml_val] $bindname = [set vals($w,unique_name)]_top_thm ()"
	        set script "[ml_val] $bindname = $script[ml_end_dec]\n"
	    } elseif {$save} {
	 	ML -toplevel 1 -check 1 "save_thm ( [ml_string $savename], ([set vals($w,unique_name)]_top_thm ()))"
	        set script "save_thm ([ml_string $savename], $script)[ml_end_dec]\n"
	    } else {
	        set script "$script[ml_end_dec]\n"
	    }
            if {$bind || $save} {
		ML -run 0 -log 1 -direct 1 $script
	    	set vals($w,saved) 1
	    }
	    if $view_script {	    	       
	    	global gui_flags
	    	set newwin [newwin toplevel]
	    	pack [fontcheck scrollable text $newwin.script \
			-font $gui_flags(font,codeentry) \
	       		-borderwidth 2 \
			-relief sunken \
			-height 10] \
	    	-side top \
	    	-expand yes \
	    	-fill both -padx 10 -pady 10
	    	$newwin.script.b insert 1.0 $script
	  	pack [frame $newwin.controls] -pady 5 -side bottom
		pack [fontcheck focusable button $newwin.controls.close \
			-command "destroy $newwin" \
			-text "Close" \
			-width 5 \
			-font $gui_flags(font,buttons)]
	    }
	}
}


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------
proc TkGoalProof::clear_goal { w } {
	global vals
	global busy
	incr busy
	$w.topgoal.text.b config -state normal
     	HolRichText::deleteall $w.topgoal
	set vals($w,savable) 0
	$w.entiretac.tac.b config -state normal
	$w.entiretac.tac.b delete 1.0 end
	$w.entiretac.tac.b config -state disabled
	$w.controls.applyml.b config -state disabled
	foreach tacbutton [info commands $w.tacbuttons.*] {
	    catch {$tacbutton config -state disabled}
	}
	$w.controls.backup.b config -state disabled
	$w.menu.proof.m entryconfigure "*Backup*" -state disabled
	$w.menu.proof.m entryconfigure "*Apply*" -state disabled
	$w.menu.proof.m entryconfigure "*Save*" -state disabled
	$w.menu.proof.m entryconfigure "*Clear*" -state disabled
	$w.menu.proof.m entryconfigure "*Set*" -state normal
	$w.controls.setgoal.b config -state normal
	$w.controls.new.b config -state disabled
	$w.nexttac.tac.b config -state disabled
	button_makeDefault $w.controls.setgoal.b
	update
	focus_goTo $w.topgoal.text.b
	incr busy -1
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------


proc TkGoalProof::set_goal { w } {
	global vals
	global busy
	incr busy
	global feedback
	set feedback [list $w "Setting goal..."]
	if [catch {ML -type void  -check 1 "[set vals($w,unique_name)]_set_goal (\[\],[hol_term [$w.topgoal.text.b get 1.0 end]])"} err] {
	    incr busy -1
	    errormessage $w.error $err
	    return
	}
	set vals($w,goal) [$w.topgoal.text.b get 1.0 end]
	TkGoalProof::RedisplayTopGoal $w
	set vals($w,tacdepth) 0
	set vals($w,olddepth) 0
	$w.entiretac.tac.b config -state normal
	$w.entiretac.tac.b delete 1.0 end
	$w.entiretac.tac.b config -state disabled
	set vals($w,saved) 0
	set vals($w,savable) 0
	set vals($w,entire_tac_stack) ""
	set vals($w,next_tac_stack) ""
	$w.topgoal.text.b config -state disabled
	$w.nexttac.tac.b config -state normal
	$w.controls.setgoal.b config -state disabled
	$w.controls.applyml.b config -state normal
	foreach tacbutton [info commands $w.tacbuttons.*] {
	    catch {$tacbutton config -state normal}
	}
	$w.controls.backup.b config -state normal
	$w.controls.new.b config -state normal
	button_makeDefault $w.controls.applyml.b
	update
	focus_goTo $w.nexttac.tac.b
	$w.menu.proof.m entryconfigure "*Set*" -state disabled
	$w.menu.proof.m entryconfigure "*Backup*" -state normal
	$w.menu.proof.m entryconfigure "*Apply*" -state normal
	$w.menu.proof.m entryconfigure "*Clear*" -state normal

	incr busy -1
}



#----------------------------------------------------------------------------
# TkGoalProof::Backup
#
#----------------------------------------------------------------------------

proc TkGoalProof::Backup { w } {
	global vals
	global busy
	incr busy 
	if {($vals($w,depth) == 0) && ($vals($w,numgoals) != 0)} {
	    TkGoalProof::clear_goal $w
	    $w.topgoal.text.b delete 1.0 end
	    $w.topgoal.text.b insert 1.0 $vals($w,goal)
	    incr busy -1
	    return
	}
        if [catch {ML -type void  "[set vals($w,unique_name)]_backup ()"} err] {
            errormessage $w.error $err; incr busy -1; return
        }
	$w.entiretac.tac.b config -state normal
	TkGoalProof::RedisplayTopGoal $w
	$w.entiretac.tac.b delete 1.0 end
	upvar 0 vals($w,entire_tac_stack) entire_tac_stack
	set len [llength $entire_tac_stack]
	$w.entiretac.tac.b insert end [lindex $entire_tac_stack [expr $len-1]]
	set vals($w,entire_tac_stack) [lrange $entire_tac_stack 0 [expr $len-2]]
	$w.entiretac.tac.b config -state disabled
	
	$w.nexttac.tac.b delete 1.0 end
	upvar 0 vals($w,next_tac_stack) next_tac_stack
	set len [llength $next_tac_stack]
	$w.nexttac.tac.b insert end [lindex $next_tac_stack [expr $len-1]]
	set vals($w,next_tac_stack) [lrange $next_tac_stack 0 [expr $len-2]]
	
	$w.nexttac.tac.b tag add sel 1.0 end
	incr busy -1
}


#----------------------------------------------------------------------------
# TkGoalProof::Rotate
#
# BUGS
#   Does not adjust cumulative tactic.  This seems hard to implement.
# Hence it is not allowed at the moment.
#----------------------------------------------------------------------------

proc TkGoalProof::Rotate { w } {
	global vals
	global busy
	incr busy
	ML -type void  "[set vals($w,unique_name)]_rotate 1"
	TkGoalProof::RedisplayTopGoal $w
	incr busy -1
}



#----------------------------------------------------------------------------
# TkGoalProof::RedisplayTopGoal
#
#----------------------------------------------------------------------------

proc TkGoalProof::RedisplayTopGoal { w } {
	global vals
	global feedback
	global busy
	global TkGoalProof_flags
	incr busy

	set numgoals [ML -type int "[set vals($w,unique_name)]_numgoals ()"]
	set depth [ML -type int "[set vals($w,unique_name)]_depth ()"]
	set vals($w,numgoals) $numgoals
	set vals($w,depth) $depth
        HolRichText::deleteall $w.topgoal
	$w.depth config -text "Depth: $depth"
	if {$numgoals==0} {

	    $w.numgoals config -text "Goal Proved!"
	    set feedback [list $w "Displaying theorem..."]
	    HolRichText::insertthm $w.topgoal \
		-header "Top theorem\n" \
		-index "topthm" \
		[list "([set vals($w,unique_name)]_top_thm ())"]
	    set vals($w,savable) 1
	    $w.menu.proof.m entryconfigure "Save*" -state normal

	} else {
	    set feedback [list $w "Displaying goal..."]
	    HolRichText::insertgoal $w.topgoal \
		-firstOnTop [expr {$TkGoalProof_flags(assumptionOrdering) == "FirstOnTop"}] \
		-numberAssums $TkGoalProof_flags(numberAssums) \
		[list "([set vals($w,unique_name)]_top_goal ())"]
	    HolRichText::yview $w.topgoal 1.0
	    $w.numgoals config -text "Goals Remaining: $numgoals"
	    $w.menu.proof.m entryconfigure "Save*" -state disabled


	}
	incr busy -1
}



#----------------------------------------------------------------------------
# TkGoalProof::ApplyML
#
# Expands the given ml code as a tactic.  Adds the code to the
# cumulative tactic.
#
# Tactic indenting is always in 4 space increments at themoment.
#
# BUGS
#   There is no limit on the depth to which the entire tactic is recorded
# in the backup stack, though there is a limit on how far back 
# the goalstacks are kept.  The limit should be the same in each case to
# be efficient with memory.  The memory will be reclaime when the
# proof window gets claimed though, so this seems OK.
#----------------------------------------------------------------------------

proc TkGoalProof::spaces { w depth } {
    set sp ""
    for {set i 0} {$i<$depth} {incr i} {
        append sp "    "
    }
    $w.entiretac.tac.b insert end $sp
}

proc TkGoalProof::ApplyML { w ml } {
	if [focus_update] return

	global vals
	global TkGoalProof_flags
	global feedback
	global busy
	incr busy
	
	#
	# 1. Work out the depth of the stack to help in construction of
	# the entire tactic.
	#
	
	set olddepth [ML -type int "[set vals($w,unique_name)]_depth ()"]
	     							 
	#
	# 2. Apply the tactic.  Don't do anything permanent before
	# here because evaluating the tactic may cause an exception.
	# Wrap the tactic up in a CHANGED_TAC if necessary.
	#

	if {$TkGoalProof_flags(automaticChangedTac)} {
	    if [hol88] {
		set realml "[set vals($w,unique_name)]_expand (CHANGED_TAC ($ml)) ??\[`fail`\] failwith `no change to goal`"
	    } else {
	        set realml "[set vals($w,unique_name)]_expand (CHANGED_TAC ($ml))"
	    }
	} else {
	    set realml "$ml"
	}
	set feedback [list $w "Applying tactic..."]
        if [catch {ML -type void -check 1  $realml} message] {
	    errormessage $w.error $message
	    incr busy -1
	    return
	}
	
	#
	# 3. Save the entire tactic to enable "backup" to restore
	# the previous entire tactic.
	
	lappend vals($w,entire_tac_stack) [$w.entiretac.tac.b get 1.0 end]
	$w.entiretac.tac.b config -state normal
	
	lappend vals($w,next_tac_stack) $ml
	$w.nexttac.tac.b delete 1.0 end
	$w.nexttac.tac.b insert end $ml
		   
	# 3b. Adjust the tactic history list
	# Only keep th list to depth $TkGoalProof_flags(tactic_history,depth).
	# Also remake the tactic history list.

	set where [lsearch -exact $TkGoalProof_flags(tactic_history) $ml]
#	puts "where = $where, ml = $ml, TkGoalProof_flags(tactic_history) = $TkGoalProof_flags(tactic_history)"
	if {$where==-1} {
	    set TkGoalProof_flags(tactic_history) [lrange $TkGoalProof_flags(tactic_history) 0 [expr {$TkGoalProof_flags(tactic_history,depth) - 2}]]
	} else {
	    set TkGoalProof_flags(tactic_history) [lreplace $TkGoalProof_flags(tactic_history) $where $where]
	}
	set TkGoalProof_flags(tactic_history) [linsert $TkGoalProof_flags(tactic_history) 0 $ml]
	TkGoalProof::remake_tactic_history_menu $w
	
	#
	# 4. Redisplay the new goal/proven theorem.  This also computes
	# the new depth and current number of top-level goals.
	#
	
        TkGoalProof::RedisplayTopGoal $w

	#
	# 5. Now adjust the "entire" tactic by adding on the latest tactic
	# and branching with a THENL or THEN depending on the number
	# of new goals.  Alternatively, we might have solved some
	# of the goals, in which case we might have to
	# close off some THENL branches.
	#
	 
	    
	set numgoals [set vals($w,numgoals)]
	set newdepth [set vals($w,depth)]
	    
	TkGoalProof::spaces $w [set vals($w,tacdepth)]
	$w.entiretac.tac.b insert end $ml
	if {$newdepth > $olddepth} {
	    
	    	# Nb. expansion caused new goal(s) (i.e. didn't solve any goals)
		# Look at the number of new goals and use either
		# THENL or THEN.  tacdepth only increases if we use THENL.
		# Keep track of whether THEN or THENL was used at each
		# depth in order to correctly adjust tacdepth as
		# goals get solved.
		#
		# nb. newdepth = olddepth + 1 always
		
		if {$numgoals==1} {
		    $w.entiretac.tac.b insert end " THEN"
		    set vals($w,depth,$newdepth,thenl) 0
		} else {
		    $w.entiretac.tac.b insert end " THENL \["
		    set vals($w,depth,$newdepth,thenl) 1
	   	    incr vals($w,tacdepth)
		}
		$w.entiretac.tac.b insert end "\n"
		
	} 
	if {$newdepth <= $olddepth} {
	    
	        # going to the same depth, hence the tactic
		# just applied solved some goals.
		#
		
	    	$w.entiretac.tac.b insert end "\n"
	        for {set i $olddepth} {$i > $newdepth} {incr i -1} {
		    if [set vals($w,depth,$i,thenl)] {
			incr vals($w,tacdepth) -1
			TkGoalProof::spaces $w $vals($w,tacdepth)
		        $w.entiretac.tac.b insert end "\]"
	    	        $w.entiretac.tac.b insert end "\n"
		    }
	        }
		
		if {$numgoals != 0} {
		    TkGoalProof::spaces $w [expr $vals($w,tacdepth)-2]
	    	    $w.entiretac.tac.b insert end [ml_listsep]
		}
		
	}
	$w.entiretac.tac.b config -state disabled
	$w.nexttac.tac.b tag add sel 1.0 end
	
	if {$numgoals==0} {
	    TkGoalProof::save $w
	}

	incr busy -1
}

#----------------------------------------------------------------------------
# TkGoalProof::PrepareML
#
#----------------------------------------------------------------------------

proc TkGoalProof::PrepareML { w ml } {
	global vals
	if [focus_update] return
	global busy
	incr busy
	
	    regsub -all {\.\.\.} $ml "" realml
	    set adjust [expr {[string length $realml] - [string first ... $ml]}]
	    catch {$w.nexttac.tac.b delete sel.first sel.last}
	    $w.nexttac.tac.b insert insert $realml
	    $w.nexttac.tac.b mark set insert "insert - $adjust chars"
	
	incr busy -1	
}


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc gdp { } {
   after 1 {
   	source $TkGoalProof_library/TkGoalProof.tcl
   	newwin TkGoalProof
   }
}
