structure TheoryNotification : TheoryNotification_sig = 
struct
	type client =
		string * 				(* the theory being watched *)
		(string -> string -> thm -> unit) * 	(* called when a new theorem is added to theory *)
		(string -> string -> thm -> unit) * 	(* called when a new definition is added to theory *)
		(string -> string -> thm -> unit) *	(* called when a new axiom is added to theory *)
		(string -> term -> unit) *		(* called when a new constant is added to theory *)
		(string -> {Arity:int, Name:string} -> unit);		(* called when a new type added to theory *)

	fun mk_client (x:client) = x;
		
	val notification_list = ref []: ((string * client) list) ref; 

	fun add_client theory client =
		(notification_list := 
			(append [(theory,client)] (! notification_list));
		());

	fun remove_client theory clientname =
		(notification_list := 
		    filter ( fn (th,(cl,_,_,_,_,_)) => not(th = theory andalso cl = clientname)) (! notification_list);
		());

	fun relevant_client_functions theory selector =
		map selector (filter ( fn client => fst(client) = theory) (! notification_list));


	fun most_general_wrapper operation =
	    let val constants_before = constants "-"
		and types_before = types "-"
		and theorems_before = theorems "-"
		and definitions_before = definitions "-"
	    	and axioms_before = axioms "-" in
		(fn res => (
		        map (fn new_theorem => map (fn f => f (current_theory()) (fst new_theorem) (snd new_theorem))
			      (relevant_client_functions (current_theory()) (fn (_,(_,x,_,_,_,_)) => x))
			    ) (subtract (theorems "-") theorems_before)
		      ; map (fn new_definition => map (fn f => f (current_theory()) (fst new_definition) (snd new_definition))
			      (relevant_client_functions (current_theory()) (fn (_,(_,_,x,_,_,_)) => x))   
			    ) (subtract (definitions "-") definitions_before)
		      ; map (fn new_axiom => map (fn f => f (current_theory()) (fst new_axiom) (snd new_axiom))
			      (relevant_client_functions (current_theory()) (fn (_,(_,_,_,x,_,_)) => x))   
			    ) (subtract (axioms "-") axioms_before)
		      ; map (fn new_constant => map (fn f => f (current_theory()) new_constant)
			      (relevant_client_functions (current_theory()) (fn (_,(_,_,_,_,x,_)) => x))   
			    ) (subtract (constants "-") constants_before)
		      ; map (fn new_type => map (fn f => f (current_theory()) new_type)
			      (relevant_client_functions (current_theory()) (fn (_,(_,_,_,_,_,x)) => x))
			    ) (subtract (types "-") types_before)
		      ; res))
	 	(operation ())
	    end;;

	
	fun save_thm (name,thm) =
		( fn thm => ((map ( fn f => f (current_theory()) name thm) 
			(relevant_client_functions (current_theory()) ( fn (_,(_,x,_,_,_,_)) => x)))
		      ; thm))
	 	(Theory.save_thm(name,thm));
	
	fun store_thm (name,term,tactic) =
		( fn thm => ((map ( fn f => f (current_theory()) name thm) 
			(relevant_client_functions (current_theory()) ( fn (_,(_,x,_,_,_,_)) => x)))
		      ; thm))
	 	(Tactical.store_thm(name,term,tactic));
	
	fun new_specification spec =
	    	most_general_wrapper (fn () => Const_spec.new_specification spec);
				
	fun new_definition (name,term) =
	    	most_general_wrapper (fn () => Const_def.new_definition (name,term));
				
	fun define_type spec =
	    	most_general_wrapper (fn () => Define_type.define_type spec);
				
	fun test () = 
	    let exception NEW_THEOREM
		exception NEW_DEFINITION
		exception NEW_AXIOM 
		exception NEW_CONSTANT 
		exception NEW_TYPE in (
		add_client "test4" (mk_client(
			"client1", 
			(fn x => raise NEW_THEOREM),
			(fn x => raise NEW_DEFINITION),
			(fn x => raise NEW_AXIOM),
			(fn x => raise NEW_CONSTANT),
			(fn x => raise NEW_TYPE)
		));
		(Theory.new_theory "test4"; true)
		andalso ((save_thm("TRUTH",TRUTH); false) handle NEW_THEOREM => true)
	   ) end;

end;



(*--------------------------------------------------------------       	*)
(* theory tcl clients		 					*)
(*        						       		*)
(* For testing:       						       	*)
(* fun tcl x= tty_write x; 						*)
(* 								       	*)
(*--------------------------------------------------------------       	*)

structure TclTheoryNotification : TclTheoryNotification_sig = 
struct
	datatype selection = 
		on_new_thm | 
		on_new_definition | 
		on_new_axiom | 
		on_new_const |
		on_new_type;

	type client = (string * string * (selection) list);
	fun mk_client (x:client) = x;

	fun add_client theory (tclclass,tclobj,selection) =
		TheoryNotification.add_client theory (
		   (tclclass ^ tclobj),
	   (if (mem on_new_thm selection) 
	       then ( fn thr =>  fn thmname =>  fn thm => tcl (tclclass ^ "::new_thm_Notify "  ^ tclobj ^ " " ^ thr ^ " " ^ thmname))
	       else ( fn thr =>  fn thmname =>  fn thm => ())),
	   (if (mem on_new_definition selection) 
	       then ( fn thr =>  fn thmname =>  fn thm => tcl (tclclass ^ "::new_definition_Notify "  ^ tclobj ^ " " ^ thr ^ " " ^ thmname))
	       else ( fn thr =>  fn thmname =>  fn thm => ())),
	   (if (mem on_new_axiom selection) 
	       then ( fn thr =>  fn thmname =>  fn thm => tcl (tclclass ^ "::new_axiom_Notify "  ^ tclobj ^ " " ^ thr ^ " " ^ thmname))
	       else ( fn thr =>  fn thmname =>  fn thm => ())),
	   (if (mem on_new_const selection) 
	       then ( fn thr =>  fn const => tcl (tclclass ^ "::new_const_Notify "  ^ tclobj ^ " " ^ thr ^ " " ^ (#Name (dest_const const))))
	       else ( fn thr =>  fn const => ())),
	   (if (mem on_new_type selection) 
	       then ( fn thr =>  fn newtype => tcl (tclclass ^ "::new_type_Notify "  ^ tclobj ^ " " ^ thr ^ " " ^ (#Name newtype)))
	       else ( fn thr =>  fn newtype => ()))
	);


	fun remove_client theory (tclclass,tclobj,selection) =
		TheoryNotification.remove_client theory (tclclass ^ tclobj);
end;


(*--------------------------------------------------------------       *)
(* Test Fragments      						       *)
(* 								       *)
(*--------------------------------------------------------------       *)

(*
fun tcl x = output(std_out,x);

use "notify.sml";
add_client ("Class","Obj");
new "junk2";
add_client "junk2" ("Class","Obj",[on_new_thm]);
save_thm ("blah",ETA_AX);

load_library {lib=find_library "set", theory = "-"}

*)
