#############################################################################
#   TheoremsFrame.tcl,v 1.14 1995/04/04 16:38:46 drs1004 Exp
#    Copyright (C) 1994  Donald Syme
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 1, or (at your option)
#    any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#    Contact Details:
#	Donald Syme
#	The Computer Laboratory
#	New Musuems Site
#	Pembroke St.
#	Cambridge U.K. CB2 3QG
#
#	email: Donald.Syme@cl.cam.ac.uk
#
#############################################################################

#----------------------------------------------------------------------------
#
#
# WIDGET CLASS TheoremsFrame
#
# A listbox of theorem names and a place to display selected
# theorems.  Also with facilities to automatically trace
# the theorems coming and going from a theory segment.
#
# CONFIGURATION OPTIONS
#	All options valid for text widgets may also be specified
#	for TheoremsFrame widgets, in addition to the following:
#
#	-trace {<theory> <type>}  
#		Trace the given set of theorems,
#		definitions or axioms in the given theory.  The argument
#		Must be a list containing both a theory name and either
#		"theorem", "definition" or "axiom".
#		Defaults to no tracing.  Only one trace
# 		parameters may be specified.  If either is
#		specified both must be specified.
#
#	-multiselect bool
#		Whether the list box is single or multi
#		select.  
#
#		Defaults to 1.
#
#	-bindbuttons bool
#		Whether the buttons allowing the results to be bound
#		to ML identifiers should be shown.
#
#		Defaults to 1.
#
#	-expandbuttons bool
#		Whether the buttons allowing the results to be expanded
#		should be shown.  You might want to disable these if
#		the theorems are automatically expanded.
#
#		Defaults to 1.
#
# THEOREM SPECIFICATIONS
#
#	As used by HolRichText.
#
# COMMANDS
#	TheoremsFrame::addtheorems widget [thmspec ...]
#		Add the given theorems to the list of theorems
#		displayed in the listbox of the widget.
#
#	TheoremsFrame::deletetheorems widget [thmspec ...]
#		Remove the given theorems from the list of theorems
#		displayed in the listbox of the widget.
#
#	TheoremsFrame::settheorems widget [thmspec ...]
#		Set the theorems displayed in the listbox of
#		the widget to the given set of theorems.
#
#	TheoremsFrame::theorems widget
#		Return the list of all theorems.
#
# 	TheoremsFrames::selection widget
# 		Returns a list of the set of currently
#		selected theorems in the listbox of the widget.
#
#	TheoremsFrame::expand widget [thmspec ...]
#		Expand the given theorems in the richtext frame.  If the
#		theorems are already expanded the just "yview" to the
#		last theorem in the list.
#
#	TheoremsFrame::expandall widget
#		Expand all the theorems displayed in the widget
#		into richtext frame.
#
#	TheoremsFrame::reformatall widget
#		Redisplay all expanded theorems.
#
#	TheoremsFrame::expandedtheorems widget
#		Return a list of all theorems which have been expanded
#		into the richtext frame.
#
#	TheoremsFrame::bind widget [thmspec ...]
#		Bind the given theorems to their corresponding ML identifiers
#		in the HOL session.
#
#	TheoremsFrame::bindall widget
#		Bind all the theorems displayed in the widget.
#
#	TheoremsFrame::bindselection widget
#		Bind all the theorems displayed in the widget.
#
# Components of the widget:
#	$w		The containing frame
#	$w.box		The containg frame for the listbox and scrollbar
#	$w.box.scroll
#	$w.box.list
#	$w.richtext		The theorems are displayed here.  This is
#			a "HolRichText".
#----------------------------------------------------------------------------


option add *TheoremsFrame.multiselect 1 widgetDefault
option add *TheoremsFrame.trace "" widgetDefault
option add *TheoremsFrame.bindbuttons 1 widgetDefault
option add *TheoremsFrame.expandbuttons 1 widgetDefault


proc TheoremsFrame { w args } {
	global vals 
	global gui_flags
	global HolRichText_library

	frame $w -class TheoremsFrame
	set passon_args [eval [list TheoremsFrame::configure $w] $args]
	if ![info exists vals($w,multiselect)] { set vals($w,multiselect) [option get $w multiselect MultiSelect] }
	if ![info exists vals($w,trace)] { set vals($w,trace) [option get $w trace Trace] }
	if ![info exists vals($w,expandbuttons)] { set vals($w,expandbuttons) [option get $w expandbuttons ExpandButtons] }
	if ![info exists vals($w,bindbuttons)] { set vals($w,bindbuttons) [option get $w bindbuttons BindButtons] }
	
	set vals($w,displayed_theorems) ""
	
	pack [frame $w.controls -relief sunken -borderwidth 2] -side left -fill y
	if {$vals($w,expandbuttons)} {
	    pack [button $w.controls.expandall -bitmap @$HolRichText_library/expandall.xbm -command "TheoremsFrame::expandall $w"]
	}
	if $vals($w,bindbuttons) {
	    pack [button $w.controls.bind1 -bitmap @$HolRichText_library/bind1.xbm -command "TheoremsFrame::bindselection $w"]
            pack [button $w.controls.bindall -bitmap @$HolRichText_library/bindall.xbm -command "TheoremsFrame::bindall $w"]
	}

	pack [frame $w.box] -side left -fill y
    	pack [scrollable listbox $w.box.list \
		-relief sunken \
		-export true \
            	-setgrid 1 \
		-font $gui_flags(font,listboxes)] \
    	    -side left -fill y
	if {!$vals($w,multiselect)} {
	    tk_listboxSingleSelect $w.box.list.b
	}

	bind $w.box.list.b <ButtonRelease-1> "TheoremsFrame::expandselection $w"

	pack [eval [list HolRichText $w.richtext] $passon_args] \
		-expand yes -fill both -side right

	TheoremsFrame::register_trace $w

	widget_addBinding $w Destroy "TheoremsFrame::upon_destroy $w"

	if $gui_flags(debug) { puts "Leaving TheoremsFrame"}
	return $w
}


#----------------------------------------------------------------------------
#
#
#----------------------------------------------------------------------------

proc TheoremsFrame::configure { w args } {
	global vals
	set passon_args ""
	set firsttime [expr ![winfo exists $w.richtext]]
	for {set i 0} {$i<[llength $args]} {incr i} {
	    case [lindex $args $i] -multiselect {
		if !$firsttime { set oldvals(multiselect) $vals($w,multiselect) }
	        incr i
	        set vals($w,multiselect) [lindex $args $i]
	    } -bindbuttons {
		if !$firsttime { set oldvals(bindbuttons) $vals($w,bindbuttons) }
	        incr i
	        set vals($w,bindbuttons) [lindex $args $i]
	    } -expandbuttons {
		if !$firsttime { set oldvals(expandbuttons) $vals($w,expandbuttons) }
	        incr i
	        set vals($w,expandbuttons) [lindex $args $i]
	    } -trace {
		if !$firsttime { set oldvals(trace) $vals($w,trace) }
	        incr i
	        set vals($w,trace) [lindex $args $i]
	    } default {
	    	lappend passon_args [lindex $args $i]
	  	incr i
	    	lappend passon_args [lindex $args $i]
	    }
	}

	if $firsttime {
	    return $passon_args
	} else {
	    eval [list HolRichText::configure $w.richtext] $passon_args
	    if [info exists oldvals(multiselect)] {
	    	puts stderr "Warning: (TheoremsFrame:configure) Reconfiguration of -multiselect ignored"
	    }
	    if [info exists oldvals(trace)] {
	        TheoremsFrame::deregister_trace $w
	        TheoremsFrame::register_trace $w
	    }
	    if [info exists oldvals(bindbuttons)] {
	    	puts stderr "Warning: (TheoremsFrame:configure) Reconfiguration of -bindbuttons ignored"
	    }
	    if [info exists oldvals(expandbuttons)] {
	    	puts stderr "Warning: (TheoremsFrame:configure) Reconfiguration of -exapndbuttons ignored"
	    }
	}
	

}


#----------------------------------------------------------------------------
#
# Register this tcl "object" as a notification client
# for this theory.  This means the tcl routines
# below will get called every time a theorem is saved
# via save_thm_and_notify.
#----------------------------------------------------------------------------

proc TheoremsFrame::register_trace { w } {	
	global vals	
	if {$vals($w,trace)!=""} {
	    set theory [lindex $vals($w,trace) 0]
 	    set type [lindex $vals($w,trace) 1]
	    case $type th* {
	    	set vals($w,trace_spec) on_new_thm
	    } def* {
	    	set vals($w,trace_spec) on_new_definition
	    } ax* {
	    	set vals($w,trace_spec) on_new_axiom
	    } default {
	        error "unexpected trace type $type"
	    }
	    if [hol88] {
		ML -type void "theory_add_tcl_client [ml_string $theory] ([ml_string TheoremsFrame],[ml_string $w],\[$vals($w,trace_spec)\])"
	    }
	    if [hol90] {
		ML -type void "TclTheoryNotification.add_client [ml_string $theory] \n(TclTheoryNotification.mk_client([ml_string TheoremsFrame],\n[ml_string $w],\[TclTheoryNotification.$vals($w,trace_spec)\]))"
	    }
	    set vals($w,traced_theory) $theory
	}
}


#----------------------------------------------------------------------------
#
#
#----------------------------------------------------------------------------


proc TheoremsFrame::deregister_trace { w  } {	
	global vals
	if {[info exists vals($w,traced_theory)] && [info exists vals($w,trace_spec)]} {
	    set theory $vals($w,traced_theory)
	    if [hol88] {
		ML -type void "theory_remove_tcl_client [ml_string $theory] ([ml_string TheoremsFrame],[ml_string $w],\[$vals($w,trace_spec)\])"
	    }
	    if [hol90] {
		ML -type void "TclTheoryNotification.remove_client [ml_string $theory] \n(TclTheoryNotification.mk_client([ml_string TheoremsFrame],\n[ml_string $w],\[TclTheoryNotification.$vals($w,trace_spec)\]))"
	    }
	    unset vals($w,trace_spec)
	}
}



#----------------------------------------------------------------------------
# TheoremsFrame::upon_destroy
#
# It is particularly important to free this space since we store a big
# name->spec translation table.
#
#----------------------------------------------------------------------------

proc TheoremsFrame::upon_destroy { w } {
	global vals
	TheoremsFrame::deregister_trace $w
	unset_vals_for_widget $w
}

#----------------------------------------------------------------------------
# TheoremsFrame::new_thm_Notify
#
# Called by ML when a theorem is saved via save_thm_and_notify and tracing
# for theorems is on.
#
#----------------------------------------------------------------------------

proc TheoremsFrame::new_thm_Notify { w theory thmname } {
	global vals
	TheoremsFrame::addtheorems $w [list $thmname theorem [lindex $vals($w,trace) 0]]
}

proc TheoremsFrame::new_definition_Notify { w theory thmname } {
	global vals
	TheoremsFrame::addtheorems $w [list $thmname definition [lindex $vals($w,trace) 0]]
}

proc TheoremsFrame::new_axiom_Notify { w theory thmname } {
	global vals
	TheoremsFrame::addtheorems $w [list $thmname axiom [lindex $vals($w,trace) 0]]
}

#----------------------------------------------------------------------------
# TheoremsFrames::addtheorems
#
#----------------------------------------------------------------------------

proc TheoremsFrame::addtheorems { w args } {
	global vals
	foreach thmspec $args {
	    set thmname [lindex $thmspec 0]
	    set size [$w.box.list.b size]
	    set inserted 0
	    for {set i 0} {$i < $size} {incr i} {
		if {[$w.box.list.b get $i] > $thmname} {
		    $w.box.list.b insert $i $thmname
		    set inserted 1
		    break
		}
	    }
	    if {!$inserted} {
		$w.box.list.b insert end $thmname
	    }
	    set vals($w,thms,$thmname) $thmspec
	}
}

#----------------------------------------------------------------------------
# TheoremsFrames::settheorems
#
#----------------------------------------------------------------------------

proc TheoremsFrame::settheorems { w args } {
	global vals
	$w.box.list.b delete 0 end
	set vals($w,displayed_theorems) ""
	HolRichText::deleteall $w.richtext
	eval [list TheoremsFrame::addtheorems $w] $args
}


#----------------------------------------------------------------------------
# TheoremsFrames::deletetheorems
#
#----------------------------------------------------------------------------


proc TheoremsFrame::deletetheorems { w args } {
	global vals
	global busy
	incr busy
	foreach thmspec $args {
 	    if {[lsearch -exact [set vals($w,displayed_theorems)] $thmspec] != -1} {
	        HolRichText::deletethm $w.richtext $thmspec
	    }
	    set vals($w,displayed_theorems) [lremove [set vals($w,displayed_theorems)] $thmspec]

	    # search through the listbox for the theorem to delete

	    set size [$w.box.list.b size]
	    for {set i 0} {$i < $size} {incr i} {
		if {[$w.box.list.b get $i]==[lindex $thmspec 0]} {
		    $w.box.list.b delete $i
		    break
		}
	    }
	}
	incr busy -1
}






#----------------------------------------------------------------------------
# TheoremsFrames::expand
#
#----------------------------------------------------------------------------


proc TheoremsFrame::expand { w args } {
	global vals
	global feedback
#	global busy
#	incr busy
#	set feedback [list $w "Expanding..."]
	foreach thmspec $args {
	    if {[lsearch -exact [set vals($w,displayed_theorems)] $thmspec] == -1} {
	        HolRichText::insertthm $w.richtext $thmspec
	        lappend vals($w,displayed_theorems) $thmspec
	    } else {
	        HolRichText::yviewthm $w.richtext $thmspec 
	    }
	}
#	incr busy -1
}

#----------------------------------------------------------------------------
# TheoremsFrames::selection
#
# Returns the theorem specs of the selected theorems
#----------------------------------------------------------------------------

proc TheoremsFrame::selection { w } {
	global vals
	set sel [$w.box.list.b curselection]
	set thmspecs ""
	foreach index $sel {
	    set thmname [$w.box.list.b get $index]
	    lappend thmspecs [set vals($w,thms,$thmname)]
	}
	return $thmspecs
}


proc TheoremsFrame::expandselection { w } {
	global busy
	incr busy
	global feedback
	set feedback [list $w "Expanding theorem..."]
    	eval [list TheoremsFrame::expand $w] [TheoremsFrame::selection $w]
	incr busy -1
}



#----------------------------------------------------------------------------
# TheoremsFrame::expandall
#
#----------------------------------------------------------------------------

proc TheoremsFrame::expandall { w } {
	global vals		     
	global busy
	incr busy
	set size [$w.box.list.b size]
	
	for {set i 0} {$i<$size} {incr i} {
	    set thmname [$w.box.list.b get $i]
	    if {$thmname==""} break
	    set thmspec [set vals($w,thms,$thmname)]
	    if {[lsearch -exact [set vals($w,displayed_theorems)] $thmspec] == -1} {
		HolRichText::insertthm $w.richtext $thmspec
	        HolRichText::yviewthm $w.richtext $thmspec
	        lappend vals($w,displayed_theorems) $thmspec
	        update idletasks
	    }
	}
	incr busy -1
}

#----------------------------------------------------------------------------
# TheoremsFrame::expandedtheorems
#
#----------------------------------------------------------------------------

proc TheoremsFrame::expandedtheorems { w } {
	global vals		     
	return $vals($w,displayed_theorems)
}

#----------------------------------------------------------------------------
# TheoremsFrame::theorems
#
#----------------------------------------------------------------------------

proc TheoremsFrame::theorems { w } {
	global vals		     
	set size [$w.box.list.b size]
	set thmspecs ""
	for {set i 0} {$i<$size} {incr i} {
	    set thmname [$w.box.list.b get $i]
	    set thmspec [set vals($w,thms,$thmname)]
	    lappend thmspecs $thmspec
	}
	return $thmspecs
}


#----------------------------------------------------------------------------
# TheoremsFrame::reformatall
#
#----------------------------------------------------------------------------

proc TheoremsFrame::reformatall { w } {
	global vals
	global busy
	incr busy
	set size [$w.box.list.b size]
	HolRichText::deleteall $w.richtext
	set oldlist [set vals($w,displayed_theorems)]
	set vals($w,displayed_theorems) ""
	foreach thmspec $oldlist {
	    HolRichText::insertthm $w.richtext $thmspec
	        lappend vals($w,displayed_theorems) $thmspec
	    HolRichText::yviewthm $w.richtext $thmspec
	    update idletasks
	}
	incr busy -1
}

#----------------------------------------------------------------------------
# TheoremsFrame::bind
#
#----------------------------------------------------------------------------


proc TheoremsFrame::bindselection { w } {
	global busy
	incr busy
    	set thmspecs [TheoremsFrame::selection $w]
	if {[llength $thmspecs]==0} { 
	    errormessage $w.error "Select a theorem to bind first"
	    return
	}
	foreach thmspec $thmspecs {
	    TheoremsFrame::bind $w $thmspec
	}
	incr busy -1
}

proc TheoremsFrame::bindall { w } {
	global busy
	incr busy
    	set thmspecs [TheoremsFrame::theorems $w]
	foreach thmspec $thmspecs {
	    TheoremsFrame::bind $w $thmspec
	}
	incr busy -1
}

proc TheoremsFrame::bind { w thmspec } {
	global feedback
	set thmname [lindex $thmspec 0]
	set thmtype [lindex $thmspec 1]
	set theory [lindex $thmspec 2]
	set feedback [list $w "Binding ML identifier $thmname"]
	ML -toplevel 1 -log 1 "[ml_val] $thmname = $thmtype [ml_string $theory] [ml_string $thmname]"
}

