#############################################################################
#   RichText.tcl,v 1.14 1995/04/04 16:38:44 drs1004 Exp
#    Copyright (C) 1994  Donald Syme
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 1, or (at your option)
#    any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#    Contact Details:
#	Donald Syme
#	The Computer Laboratory
#	New Musuems Site
#	Pembroke St.
#	Cambridge U.K. CB2 3QG
#
#	email: Donald.Syme@cl.cam.ac.uk
#
#############################################################################


#----------------------------------------------------------------------------
#
# delayedLoad
#
# Load all the ML code to do rich text formatting.
#
# Set up the initial interface map.
#
#----------------------------------------------------------------------------

proc HolRichText::delayedLoad { } {
	global interface_map   
	global HolRichText_library HolRichText_flags gui_flags

	if {$HolRichText_flags(loaded)} { return }
	set HolRichText_flags(loaded) 1
        if {[hol88] && $HolRichText_flags(richtext)} {
	    set lisp_dir $HolRichText_library/lisp/$gui_flags(arch)
	    set files [list $lisp_dir/f-tcl-format.o $lisp_dir/hol-tcl-writ.o]
	    set norichtext 0
	    foreach file $files {
	        if ![file readable $file] {
		    set norichtext 1
		}
	    }
	    if {$norichtext} {
		puts stderr "Hmmm.. Couldn't find the extension .o files in $lisp_dir"
		puts stderr "       Perhaps you haven't compiled them for this architecture ($gui_flags(arch))?"
		puts stderr "       Trying to proceed without the richtext extensions."
		puts stderr "       This will almost certainly fail."
	    } else {
	        foreach file $files {
	            ML -type any "lisp`(load \"$file\")`"
	        }
	    }
	} 
	
	# We always load this in, rich text or not, HOL88 or HOL90.
	# These are functions
	# to convert terms, theorems and goals to strings.
	
        ML_load $HolRichText_library/conv_to_text
	
	# Basic HOL interface mapping.
	
	set interface_map(tmconst,!) [list tmsym \"]
	set interface_map(tmconst,?) [list tmsym \$]
#	set interface_map(tmconst,@) [list tmsym \']
	set interface_map(tmconst,~) [list tmsym "\x07e"]
	set interface_map(tmconst,/\\) [list tmsym "\x0d9"]
	set interface_map(tmconst,\\/) [list tmsym "\x0da"]
	set interface_map(tmconst,==>) [list tmsym "\x0de"]
	set interface_map(tmconst,<==) [list tmsym "\x0dc"]
	set interface_map(tmconst,->) [list tmsym "\x0ae"]
	
	# Arithmetic interface mapping.
	
	set interface_map(tmconst,>=) [list tmsym "\x0b3"]
	set interface_map(tmconst,<=) [list tmsym "\x0a3"]
	set interface_map(tmconst,*) [list tmsym "\x0b4"]
	set interface_map(tmconst,lambda) [list tmsym l]
	
	# Set Theory interface mapping.
	
	set interface_map(tmconst,IN) [list tmsym "\x0ce"]
	set interface_map(tmconst,SUBSET) [list tmsym "\x0cd"]
	set interface_map(tmconst,PSUBSET) [list tmsym "\x0cc"]
	set interface_map(tmconst,INTER) [list tmsym "\x0c7"]
	set interface_map(tmconst,UNION) [list tmsym "\x0c8"]
	set interface_map(tmconst,{}) [list tmsym "\x0c6"]
	set interface_map(tmconst,setlambda) [list tmsym |]
	
	# John Harrison's real numbers
	
	set interface_map(tmconst,real_le) [list tmsym "\x0a3"]
	set interface_map(tmconst,real_lt) [list tmsym <]
	set interface_map(tmconst,real_gt) [list tmsym >]
	set interface_map(tmconst,real_ge) [list tmsym "\x0b3"]
	set interface_map(tmconst,real_neg) [list tmsym -]
	set interface_map(tmconst,real_sub) [list tmsym -]
	set interface_map(tmconst,real_add) [list tmsym +]
	set interface_map(tmconst,Sum) [list tmbigsym S]
	set interface_map(tmconst,real_mul) [list tmsym "\x0b4"]
	set interface_map(tmconst,/) [list tmsym "\x0b8"]
	set interface_map(tmconst,real_of_num) [list tmconst &]
	set interface_map(tmconst,pi) [list tmsym p]

}



#----------------------------------------------------------------------------
#
# WIDGET CLASS HolRichText
#
# A window+scrollbar used to display theorems, terms and goals.
#
# The default state of $w.text.b is disabled.  It is temporarily put into
# state "normal" when any of the Insert functions below are called.
#
# OPTIONS
#	-editable bool	
#		If true, also adds bindings
# 		for term entry.  At the moment this is just a 
#		warning against entering hol88 quotes around a term.
#
#		Defaults to false.
#
#	-richtextvar variable
#		A global variable holding 0/1 determining how
#		the theorems should be displayed - in plain
#		text or rich text format.  Changes in the
#		variable are *not* automatically detected via tracing.
#
#		Defaults to the global variable "richtext".
#
# THEOREM SPECIFICATIONS
#	Theorems are specified by triplets.  Each triplet is made by
#		[list theoremname thmtype theory]
#	Here theoremname is the name of the theorem as it is stored in a
# 	theory file, thmtype is axiom, definition or theorem, and 
#	theory is the name of the theory segment it is stored in.
#
#	Theorems which are simply bound to ML identifiers may also be
#	specified.  theoremname should be the name of the ML identifier,
#	and thmtype and theory should be empty.  The value 
#       of the last member of the
# 	triplet is ignored, and need not even be given.
#
# GOAL SPECIFICATIONS
#	Goals are presently specified only by a list with a single
#	element - the ML code which will evaluate to the goal
#	itself.
#
# COMMANDS
#	HolRichText::configure widget [option value]...
#		Unlike nearly every other Tcl defined widget in TkHol, this
#		widget does support some reconfiguration.
#
# 	HolRichText::insertthm widget thmspec
#
#		Insert the given theorem at the end of the frame.
#
# 	HolRichText::insertgoal widget goalspec
#
#		Insert the given goal at the end of the frame.
#
# 	HolRichText::deletethm widget thmspec
#
# 	HolRichText::deleteall widget
#
# 	HolRichText::yviewthm widget thmspec
#
#		Adjust $w.text.b so the given theorem is in view.
#
# 	HolRichText::yview widget index
#
#		Adjust $w.text.b so the given text index is in view.  Useful
#		for scrolling to the top of the window.
#
#----------------------------------------------------------------------------

option add *HolRichText.richtextvar richtext widgetDefault
option add *HolRichText.editable 0 widgetDefault

proc HolRichText { w args} {
	global HolRichText_flags
	global gui_flags
	global vals

	frame $w -class HolRichText
	set passon_args [eval [list HolRichText::configure $w] $args]
	if ![info exists vals($w,editable)] { set vals($w,editable) [option get $w editable Editable] }
	if ![info exists vals($w,richtextvar)] { set vals($w,richtextvar) [option get $w richtextvar RichTextVar] }
	upvar $vals($w,richtextvar) richtext
	if ![info exists richtext] {
	    set richtext [hol88]
	}
	
	
	pack [eval [list scrollable text $w.text -relief sunken \
		-borderwidth 2 \
		-font $HolRichText_flags(font,teletype) \
		-foreground $HolRichText_flags(color,teletype) \
		-state disabled] $passon_args] \
	    -side left -expand yes -fill both

	HolRichText::reconfigure_tags $w
	HolRichText::adjust_editable $w

	foreach var [list font,teletype font,richtext,var font,richtext,tmsym font,richtext,tmbigsym font,richtext,tmconst font,richtext,tm font,thmname font,thmkind color,teletype color,richtext,var color,richtext,tmsym color,richtext,tmconst color,richtext,tm color,thmname color,thmkind] {
	    trace variable HolRichText_flags($var) w "HolRichText::tag_config_change_notify $w"
	}

	bind $w <Destroy> "HolRichText::upon_destroy $w"

	return $w
}

proc HolRichText::upon_destroy { w } {
	global HolRichText_flags
	global vals
	foreach var [list font,teletype font,richtext,var font,richtext,tmsym font,richtext,tmbigsym font,richtext,tmconst font,richtext,tm font,thmname font,thmkind color,teletype color,richtext,var color,richtext,tmsym color,richtext,tmconst color,richtext,tm color,thmname color,thmkind] {
	    trace vdelete HolRichText_flags($var) w "HolRichText::tag_config_change_notify $w"
	}
	unset_vals_for_widget $w
}


proc HolRichText::tag_config_change_notify { w args } {
	HolRichText::reconfigure_tags $w
}

proc HolRichText::reconfigure_tags { w } {
	global HolRichText_flags
	$w.text.b tag configure sp -foreground $HolRichText_flags(color,richtext,tm) -font "-*-courier-medium-r-*-*-*-120-*"
	$w.text.b tag configure tm -foreground $HolRichText_flags(color,richtext,tm) -font $HolRichText_flags(font,richtext,tm)
	$w.text.b tag configure var -foreground $HolRichText_flags(color,richtext,var) -font $HolRichText_flags(font,richtext,var)
	$w.text.b tag configure tmsym -foreground $HolRichText_flags(color,richtext,tmsym) -font $HolRichText_flags(font,richtext,tmsym)
	$w.text.b tag configure tmbigsym -foreground $HolRichText_flags(color,richtext,tmsym) -font $HolRichText_flags(font,richtext,tmbigsym)
	$w.text.b tag configure tmconst -foreground $HolRichText_flags(color,richtext,tmconst) -font $HolRichText_flags(font,richtext,tmconst)
	$w.text.b tag configure thmname -foreground $HolRichText_flags(color,thmname) -font $HolRichText_flags(font,thmname) -underline 1
	$w.text.b tag configure thmkind -foreground $HolRichText_flags(color,thmkind) -font $HolRichText_flags(font,thmkind) 
	
	# lower tags below "sel"
	
	$w.text.b tag configure sel -relief flat
	$w.text.b tag raise sel
}


proc HolRichText::configure { w args } {
	global vals
	set passon_args ""
	set firsttime [expr ![winfo exists $w.text.b]]
	for {set i 0} {$i<[llength $args]} {incr i} {
	    switch -- [lindex $args $i]  -editable {
		if !$firsttime { set oldvals(editable) $vals($w,editable) }
	        incr i
	        set editable [lindex $args $i]
	    } -richtextvar {
		if !$firsttime { set oldvals(richtextvar) $vals($w,richtextvar) }
	        incr i
	        set vals($w,richtextvar) [lindex $args $i]
	    } default {
	    	lappend passon_args [lindex $args $i]
	        incr i
	    	lappend passon_args [lindex $args $i]
	    }
	}

	if $firsttime {
	    return $passon_args
	} else {
	    if {[llength $passon_args]!=0} {
	        eval [list $w.text.b config] $passon_args
	    }
	    if [info exists oldvals(editable)] {
	    	puts stderr "Warning: (HolRichText::configure) Reconfiguration of -editable ignored"
	    }
	}
}


proc HolRichText::adjust_editable { w } {
	global vals
	if $vals($w,editable) {
	        bind $w.text.b {"} "errormessage $w.note {Note: quotes should not be entered around terms}"
	        $w.text.b config -state normal
	}
}


#----------------------------------------------------------------------------
# HolRichText::index_from_thmspec
#
#----------------------------------------------------------------------------

proc HolRichText::index_from_thmspec { thmspec } {
	regsub -all -- {(\-|\ |\+|\(|\))} [lindex $thmspec 0] "" thmname
	regsub -all -- {(\-|\ |\+|\(|\))}  [lindex $thmspec 1] "" thmtype
	regsub -all -- {(\-|\ |\+|\(|\))}  [lindex $thmspec 2] "" theory
	if {$thmtype==""} {
	    return $thmname
	} else {
	    return $thmname$thmtype$theory
	}
}	      

#----------------------------------------------------------------------------
# HolRichText::insertthm widget [-header text] [-index index] thmspec
#	    
#
#----------------------------------------------------------------------------


proc HolRichText::insertthm { w args } {
	global vals
	global HolRichText_flags
	for {set i 0} {$i<[llength $args]} {incr i} {
	    switch -- [lindex $args $i] -header {
	        incr i
	        set header [lindex $args $i]
	    } -index {
	        incr i
	        set index [lindex $args $i]
	    } default {
	    	if [info exists thmspec] { error "too many arguments to ML" }
		set thmspec [lindex $args $i]
	    }
	}
	HolRichText::delayedLoad
	set oldstate [lindex [$w.text.b config -state] 4]
	$w.text.b config -state normal
	set thmname [lindex $thmspec 0]
	set thmtype [lindex $thmspec 1]
	set theory [lindex $thmspec 2]
	if ![info exists index] {
	    set index [HolRichText::index_from_thmspec $thmspec] 
	}
	if [info exists header] {
	    HolRichText::InsertPlainText $w $header [list $index thmname]
	} else {
	    HolRichText::InsertPlainText $w "$thmname " [list $index thmname]
	    if {$thmtype==""} {
	        HolRichText::InsertPlainText $w "  (theorem bound to ML identifier)\n" [list $index thmkind]
	    } else {
	        HolRichText::InsertPlainText $w "  ($thmtype in $theory.th)\n" [list $index thmkind]
	    }
	}
	    
	        # When calculating the margins to use for HOL, we allow
	        # more leeway for richtext as it tends to take up more
	        # room.
		#
		# Hmmm.. as it turns out it doesn't seem we need to.  This
		# will depend alot on font selection vis a vis the default
		# font for the text window.  Even more so on the interface
		# mapping that occurs!
		#
		# This is irritating.  It doesn't seem
		# possible to find out the current width of the window
		# after resizing except via winfo.  This means we have
		# to recalculate the number of pixels/character by
		# looking at the config settings and the requested width
		# via winfo.
		# 
		
	set pixwidth [winfo width $w.text.b]
	set reqwidth [winfo reqwidth $w.text.b]
    	set reqchars [lindex [$w.text.b config -width] 4]
	set width [expr {$pixwidth*$reqchars/$reqwidth - 2}]
		
        upvar #0 $vals($w,richtextvar) richtext
	if {$thmtype==""} {
	    set thmcode $thmname
	} else {
	    set thmcode "($thmtype [ml_string $theory] [ml_string $thmname])"
	}
        if $richtext {
	    if {$width < 15} { set width 15 }
       	    if [hol88] {
       	        ML -type any "set_margin $width"
	    }
	    ML -type string "thm_to_hol_rich_text_via_file ($thmcode) [ml_string $HolRichText_flags(tmpfile)]"
	    HolRichText::InsertHolRichTextFromFile  $w $HolRichText_flags(tmpfile) $index
	 } else {
	    if {$width < 15} { set width 15 }
       	    if [hol88] {
		ML -type any "set_margin $width"
	    }
	    ML -type string "thm_to_plain_text_via_file ($thmcode) [ml_string $HolRichText_flags(tmpfile)]"
	    HolRichText::InsertPlainTextFromFile $w $HolRichText_flags(tmpfile) $index
	}
	HolRichText::InsertPlainText $w "\n" $index
	$w.text.b yview "$index.first -1 lines"
	$w.text.b config -state $oldstate
	catch {exec rm -f $HolRichText_flags(tmpfile)}
}



proc HolRichText::insertgoal { w args } {
	global vals
	global HolRichText_flags
	set firstOnTop 1
	set numberAssums 1
	for {set i 0} {$i<[llength $args]} {incr i} {
	    switch -- [lindex $args $i] -firstOnTop {
	        incr i
	        set firstOnTop [lindex $args $i]
	    } -numberAssums {
	        incr i
	        set numberAssums [lindex $args $i]
	    } default {
	    	if [info exists goalspec] { error "too many arguments to HolRichText::insertgoal" }
		set goalspec [lindex $args $i]
	    }
	}
	HolRichText::delayedLoad
        upvar #0 $vals($w,richtextvar) richtext
	set goalcode [lindex $goalspec 0]
	if $richtext {
	        ML "goal_to_hol_rich_text_via_file ([ml_bool $numberAssums],[ml_bool $firstOnTop]) ($goalcode) [ml_string $HolRichText_flags(tmpfile)]"
                HolRichText::InsertHolRichTextFromFile $w $HolRichText_flags(tmpfile) goal
	} else {
	        ML "goal_to_plain_text_via_file ([ml_bool $numberAssums],[ml_bool $firstOnTop]) ($goalcode) [ml_string $HolRichText_flags(tmpfile)]"
                HolRichText::InsertPlainTextFromFile $w $HolRichText_flags(tmpfile) goal
	}
	catch {exec rm -f $HolRichText_flags(tmpfile)}
}


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc HolRichText::yviewthm { w thmspec } {
   	$w.text.b yview "[HolRichText::index_from_thmspec $thmspec].first -1 lines"
}

proc HolRichText::yview { w index } {
   	$w.text.b yview $index
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc HolRichText::deletethm { w thmspec } {
	set oldstate [lindex [$w.text.b config -state] 4]
	$w.text.b config -state normal
	set tag [HolRichText::index_from_thmspec $thmspec]
	$w.text.b delete $tag.first $tag.last
	$w.text.b config -state $oldstate
}
proc HolRichText::deleteall { w  } {
	set oldstate [lindex [$w.text.b config -state] 4]
	$w.text.b config -state normal
	$w.text.b delete 0.0 end
	$w.text.b config -state $oldstate
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc HolRichText::InsertHolRichTextFromFile { w file extra_tags } {
  	global HolRichText_flags
        if {[hol90] || !$HolRichText_flags(richtext)} { 
	    return [HolRichText::InsertPlainTextFromFile $w $file $extra_tags]
	}
	set f [open $file r]
	set tagged_text_list [read $f]
	close $f
	$w.text.b yview -pickplace insert
	HolRichText::InsertHolRichText $w $tagged_text_list $extra_tags
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc HolRichText::InsertHolRichText { w tagged_text_list extra_tags } {
  	global HolRichText_flags

        if {[hol90] || !$HolRichText_flags(richtext)} { 
	    return [HolRichText::InsertPlainText $w $tagged_text_list $extra_tags]
	}
	global interface_map
	set oldstate [lindex [$w.text.b config -state] 4]
	$w.text.b config -state normal
	set len 0

	foreach pr $tagged_text_list {
	    set tag [lindex $pr 0]
	    set text [lindex $pr 1]
	    if {$HolRichText_flags(interfaceMapping) && [info exists interface_map($tag,$text)]} {
 		set new $interface_map($tag,$text)
		set tag [lindex $new 0]
		set text [lindex $new 1]
	    }
	    $w.text.b insert end $text
	    incr len [string length $text]
	    lappend real_text [list $tag $text]
	}
	set sofar 0				   
	
	# Strange Tk 3.6 bugs in tagging - removing all tags before
	# taking another pass to actually tag seems to fix them most
	# of the time.				 
	#
	# How inefficient is this?
	
	foreach tag {var tm tmsym tmconst sp} {
	    catch {$w.text.b tag remove $tag "end -$len c" end}
   	} 
 
	# Now tag the text
	
	foreach pr $real_text {
	    set tag [lindex $pr 0]
	    set strlen [string length [lindex $pr 1]]


	    $w.text.b tag add $tag "end -$len c +$sofar c" "end -$len c +$sofar c +$strlen c"
	    incr sofar $strlen
	}
#	    puts "tag ranges ETA_AXaxiombool = [$w.text.b tag ranges ETA_AXaxiombool]"
#	    puts "tag ranges ELdefinitionlist = [$w.text.b tag ranges ELdefinitionlist]"
#	    puts "tag ranges var = [$w.text.b tag ranges var]"
#	    puts "tag ranges tm = [$w.text.b tag ranges tm]"
#	    puts "tag ranges tmsym = [$w.text.b tag ranges tmsym]"
#	    puts "tag ranges tmconst = [$w.text.b tag ranges tmconst]"
#	    puts "tag ranges thm = [$w.text.b tag ranges thm]"
#	    puts "tag ranges sp = [$w.text.b tag ranges sp]"
	$w.text.b insert end "\n"
	foreach extra_tag $extra_tags {
	    $w.text.b tag add $extra_tag "end -$len c -1 lines" "end -1 c"
	}
	$w.text.b yview -pickplace insert
	$w.text.b config -state $oldstate
}
	


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc HolRichText::InsertPlainTextFromFile { w file extra_tags } {
	set oldstate [lindex [$w.text.b config -state] 4]
	$w.text.b config -state normal
        set f [open $file r]
	while {[gets $f line] >= 0} {
	    $w.text.b insert end $line
	    $w.text.b insert end "\n"
	    foreach extra_tag $extra_tags {
	        $w.text.b tag add $extra_tag "end - [string length $line] chars -1 c" "end -1 c" 
	    }
	}
	close $f
	$w.text.b yview -pickplace insert
	$w.text.b config -state $oldstate
	
}
	


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc HolRichText::InsertPlainText { w text extra_tags } {
	set oldstate [lindex [$w.text.b config -state] 4]
	$w.text.b config -state normal
	$w.text.b insert end $text
	foreach extra_tag $extra_tags {
	    $w.text.b tag add $extra_tag "end - [string length $text] chars -1 c" "end -1 c" 
	}
	
	# Strange Tk 3.6 bug in tagging - removing all tags 
	# seems to fix it most
	# of the time.				 
	#
	# Is this ultra inefficient?
	
	set len [string length $text]
	foreach tag {var tm tmsym tmconst sp} {
	    catch {$w.text.b tag remove $tag "end -$len c" end}
   	} 
	$w.text.b yview -pickplace insert
	$w.text.b config -state $oldstate
 
	
}



