#---------------------------------------------------------------
#
#
# install.tcl - Installation support
#
# Author: Donald Syme, Based on code originally found in exmh
# by Brent Welch.
#
# This file and "setup.tcl" provide a configurable software
# installation utility.  
#
# Most of the different routines here are for registering
# information about the installation process.  The [[install_dialog]]
# routine begins the dialog box that the user interacts with, and
# which displays all the relevant information and options.
#---------------------------------------------------------------


option add *Entry.background white startup
option add *Entry.foreground black startup

proc install_init { appName dotFile dotFileShell } {
    global install
    set install(appName) $appName
    set install(dotFile) $dotFile
    set install(dotFileShell) $dotFileShell
    if [file readable $dotFile] {
	if [catch {uplevel #0 source $dotFile} msg] {
	    puts stderr "source $dotFile: $msg"
	}
    } else {
	if {[catch {glob ../$appName*/$dotFile} files] == 0} {
	    installAlternates $files 1
	}
    }
    set install(dirlist) ""
    set install(sed) ""
}


proc install_var { priority tclvar shellvar value {comment {}} } {
    global install
    lappend install(sequence,$priority) $tclvar
    lappend install(sequence) $tclvar
    set install(shellvar,$tclvar) $shellvar
    set install(field,$tclvar) [list $tclvar $value $comment]
}


proc install_version { priority tclvar shellvar version {comment {}} } {
    global install
    if {$comment == {}} {
	if [info exists install(appName)] {
	    set comment "$install(appName) version stamp"
	} else {
	    set comment {Version stamp}
	}
    }
    set install(versionVar) $tclvar
    install_var $priority $tclvar $shellvar $version $comment
}

proc install_dirVar { priority tclvar shellvar pathname comment } {
    install_var $priority $tclvar $shellvar $pathname $comment
    global install
    lappend install(dircheck) $tclvar
    set install(dircheck,$tclvar) $comment
}
proc install_fileVar { priority tclvar shellvar pathname comment } {
    install_var $priority $tclvar $shellvar $pathname $comment
    global install
    lappend install(filecheck) $tclvar
    set install(filecheck,$tclvar) $comment
}
proc install_progVar { priority tclvar shellvar pathname comment } {
    install_var $priority $tclvar $shellvar $pathname $comment
    global install
    lappend install(progcheck) $tclvar
    set install(progcheck,$tclvar) $comment
}
proc install_sed { suffix args } {
    global install
    lappend install(sed) [list $suffix $args]
}
proc install_wish { priority tclvar shellvar pathname comment } {
    install_var $priority $tclvar $shellvar $pathname $comment
    global install
    set install(wishVar) $var
    lappend install(progcheck) $var
    set install(progcheck,$var) $comment
}




proc installFieldVar { item } { lindex $item 0 }
proc installFieldComment { item } { lindex $item 2 }
proc installFieldDefault { item {override 0} } {
    set default [lindex $item 1]
    if {$override} {
	return $default
    }
    set varName [installFieldVar $item]
    if [catch {installGetValue $varName} value] {
	return $default
    } else {
	return $value
    }
}

proc install_help { text } {
    global install
    set install(helpText) $text
}

proc installFeedback { text } {
    global install
    catch {
	$install(msg) configure -text $text
	update
    }
}
proc installError { text } {
    puts stderr $text
    installFeedback $text
}
proc installFieldInit {} {
    global install
    set install(lastentry) {}
}
proc installDoField { item {override 0}} {
    global install
    if ![info exists install(wuid)] { set install(wuid) 0 }
    incr install(wuid)
    set f [frame .rim.import$install(wuid) -relief raised]
    set var [installFieldVar $item]
    label $f.label -text [format "%-37s:" [installFieldComment $item]] \
	    -font fixed
    entry $f.entry -font fixed -width 55 -relief sunken
    $f.entry insert 0 [installFieldDefault $item $override]
    bind $f.entry <Return> [list installSetValue $var]
    if {$install(lastentry) != {}} {
	bind $install(lastentry) <Tab> [list focus $f.entry]
    } else {
	set install(firstentry) $f.entry
    }
    set install(lastentry) $f.entry
    set install(entry,$var) $f.entry
    lappend install(allEntries) $f.entry

    pack append .rim $f {top expand fill}
    pack append $f $f.label {left padx 3}
    pack append $f $f.entry {right expand fill}
}
proc installUpdateField { item {override 0} } {
    global install
    set var [installFieldVar $item]
    set entry $install(entry,$var)
    $entry delete 0 end
    $entry insert 0 [installFieldDefault $item $override]
}
proc installFieldDone {} {
    global install
    if {[info exists install(firstentry)] && \
	[info exists install(lastentry)]} {
	bind $install(lastentry) <Tab> [list focus $install(firstentry)]
    }
}

proc installSetValue { _var } {
    global install
    if [info exists install(entry,$_var)] {
	set _value [$install(entry,$_var) get]
	if [string match *(* $_var] {
	    set _arrayName [lindex [split $_var (] 0]
	    global $_arrayName
	} else {
	    global $_var
	}
	set $_var $_value
#	installFeedback "$_var $_value"
    }
}
proc installGetValue { var } {
    if [string match *(* $var] {
	set arrayName [lindex [split $var (] 0]
	global $arrayName
    } else {
	global $var
    }
    return [set $var]
}
proc installShowValue { var } {
    global install
    if [info exists install(entry,$var)] {
	installSetValue $var
	set entry $install(entry,$var)
	$entry select from 0
	$entry select to end
	focus $entry
    }
}
proc installVerify {} {
    global install
    installFeedback "Checking Pathnames..."
    set errors {}
    if [info exists install(dircheck)] {
	foreach var $install(dircheck) {
	    installSetValue $var
	    set path [installGetValue $var]
	    if {[string length $path] == 0} {
		continue
	    }
	    if ![file isdirectory $path] {
		set willMakeDir 0
		foreach dirType $install(dirlist) {
		    set newdir $install(dir,$dirType)
		    if {$newdir == $path} {
			set willMakeDir 1
		    }
		}
		if {! $willMakeDir} {
		    lappend errors [format "%-30s <%s> %s" \
			$install(dircheck,$var) $path "not a directory"]
		}
	    }
	    if ![regexp ^/ $path] {
		lappend errors [format "%-30s Warning: <%s> %s" \
		    $install(dircheck,$var) $path "is not an absolute pathname"]
	    }
	}
    }
    if [info exists install(filecheck)] {
	foreach var $install(filecheck) {
	    installSetValue $var
	    set path [installGetValue $var]
	    if {[string length $path] == 0} {
		continue
	    }
	    if ![file exists $path] {
		lappend errors [format "%-30s <%s> %s" \
		    $install(filecheck,$var) $path "does not exist"]
	    }
	    if ![regexp ^/ $path] {
		lappend errors [format "%-30s Warning: <%s> %s" \
		    $install(filecheck,$var) $path "is not an absolute pathname"]
	    }
	}
    }
    if [info exists install(progcheck)] {
	foreach var $install(progcheck) {
	    installSetValue $var	;# Snarf current value from entry
	    set path [installGetValue $var]
	    if {[string length $path] == 0} {
		continue
	    }
	    if ![file executable $path] {
		lappend errors [format "%-30s <%s> %s" \
		    $install(progcheck,$var) $path "is not executable"]
	    }
	    if ![regexp ^/ $path] {
		lappend errors [format "%-30s Warning: <%s> %s" \
		    $install(progcheck,$var) $path "is not an absolute pathname"]
	    }
	}
    }
    if {$errors != {}} {
	installFeedback "Verify errors"
    } else {
	installFeedback "Verify OK"
	return
    }
    toplevel .verify
    frame .verify.top
    button .verify.top.quit -text "Dismiss" -command {destroy .verify}
    label .verify.top.label -text "  Verify Errors "
    pack append .verify .verify.top {top fill expand}
    pack append .verify.top .verify.top.quit left .verify.top.label {left fill}

    set numLines [llength $errors]
    if {$numLines < 30} {
	text .verify.t -width 80 -height $numLines -font fixed
	pack append .verify .verify.t {bottom expand fill}
    } else {
	text .verify.t -width 80 -height 30 -yscrollcommand {.verify.s set} -font fixed
	scrollbar .verify.s -orient vert -command {.verify.t yview}
	pack append .verify \
	    .verify.s {right filly} \
	    .verify.t {left expand fill}
    }
    foreach line $errors {
	.verify.t insert end $line\n
    }

}

proc installSed { } {
    global install
    global applychanges
    set id 0
    while {[catch {open /tmp/sed.$id w} script]} {
	incr id
	if {$id > 100} {
	    installFeedback "installSed: Cannot create sed script in /tmp"
	    return
	}
    }
    foreach var $install(sequence) {
	set shellvar $install(shellvar,$var)
	if {[installGetValue $var] == ""} {
	    puts $script "s,\^$shellvar=\$,# $shellvar=,"
	    puts $script "s,\^export $shellvar\$,# export $shellvar,"
        } else {
	    puts $script "s,\^$shellvar=\$,$shellvar=[installGetValue $var],"
	    puts $script "s,\^# $shellvar=\$,$shellvar=[installGetValue $var],"
	    puts $script "s,\^# export $shellvar\$,export $shellvar,"
	}
    }

    # Set up for helper wish scripts, if needed and if possible
    if [info exists install(wishVar)] {
	installSetValue $install(wishVar)
	set pathname [installGetValue $install(wishVar)]
	if {$pathname != {}} {
	    puts $script "s,#!wish,#!$pathname,"
	}
    }

    # Insert configuration information
    puts $script /^#CONFIGURATION/a\\
    foreach v $install(sequence) {
	set item $install(field,$v)
	set var [installFieldVar $item]
	installSetValue $var
	global $var
	puts $script [list set $var [installGetValue $var]] nonewline
	puts $script \\
    }
    puts $script ""
    close $script
    foreach progsuf $install(sed) {
        set suf [lindex $progsuf 0]
        foreach prog [lindex $progsuf 1] {
	    puts "creating $prog"
	    if [catch {
	        exec sed -f /tmp/sed.$id < ${prog}$suf > $prog
	        exec chmod +x $prog
	    } msg] {
	        set sed_error "sed error on $prog: $msg"
	        break
            }
	}
    }


    if [info exists sed_error] {
	if $applychanges {
	    puts stderr "$sed_error"
        } else {
            toplevel .sed_errors
	    frame .sed_errors.top
    	    button .sed_errors.top.quit -text "Dismiss" -command {destroy .sed_errors}
    	    label .sed_errors.top.label -text "  Errors while applying changes"
    	    pack append .sed_errors .sed_errors.top {top fill expand}
    	    pack append .sed_errors.top .sed_errors.top.quit left .sed_errors.top.label {left fill}
	    
	    text .sed_errors.t -width 80 -height 10 -font fixed
	    pack append .sed_errors .sed_errors.t {bottom expand fill}
	    .sed_errors.t insert end $sed_error\n
        }
    }
}
proc installPatch {} {
    global install
    installSave
    installVerify
    installFeedback "Applying changes..."
    installSed
    installFeedback "Finished"
}
proc install_test { args } {
    global install
    set install(test) $args
}
proc installTest {} {
    global install
    # Run patch again with testing library, if it is defined
    if {[info exists install(testLib)] && [info exists install(libDirVar)]} {
	set var $install(libDirVar)
	if [info exists install(entry,$var)] {
	    set realValue [$install(entry,$var) get]
	    $install(entry,$var) delete 0 end
	    $install(entry,$var) insert 0 $install(testLib)
	    installSed
	}
    }
    if [info exists install(test)] {
	installFeedback $install(test)
	eval $install(test)
    } else {
	installFeedback "No install_test command"
    }
    if [info exists realValue] {
	$install(entry,$var) delete 0 end
	$install(entry,$var) insert 0 $realValue
    }
}

proc installSave { } {
    global install argv0
    # Save it
    installSetValue install(dotFile)
    if [catch {open $install(dotFile) w} out] {
	installFeedback "Cannot write $install(dotFile)"
	return
    }
    if ![info exists argv0] {
	set argv0 $install(appName).install
    }
    puts $out "# Saved state from $argv0"
    puts $out "# [exec date]"
    foreach v $install(sequence) {
	set item $install(field,$v)
	set varName [installFieldVar $item]
	installSetValue $varName
	set value [installGetValue $varName]
	puts $out [list set $varName $value]
    }
    close $out
    if [catch {open $install(dotFileShell) w} out] {
	installFeedback "Cannot write $install(dotFileShell)"
	return
    }
#    puts $out "#!/bin/sh"
    foreach varName $install(sequence) {
	set shellvar $install(shellvar,$varName)
	installSetValue $varName
	set value [installGetValue $varName]
	puts $out "$shellvar=$value;"
	puts $out "export $shellvar;"
    }
    close $out
    exec chmod ug+x $install(dotFileShell)

    installFeedback "Saved settings in $install(dotFile) and $install(dotFileShell)"
}
proc installCancel {} {
    after 10 {
	destroy .rim.buttons.yes ; destroy .rim.buttons.no
	pack before .rim.buttons.quit .rim.buttons.install left
    }
}
proc installCmd { logProc unixCmd } {
    if {$logProc != "nolog"} {
	$logProc $unixCmd
    } else {
	eval exec $unixCmd
    }
}
proc installInner { {logProc nolog} } {
    global install
    installVerify
    installSed
    foreach dirType $install(dirlist) {
	#
	# Install directory - make sure it exists
	#
	set dir $install(dir,$dirType)
	if ![file isdirectory $dir] {
	    installCmd $logProc [list mkdir $dir]
	    installCmd $logProc [list chmod a+rx $dir]
	}
	if {($logProc == "nolog") && ![file isdirectory $dir]} {
	    installError "LibDir $dir is not a directory"
	    continue
	}
	if [info exists install(glob,$dirType)] {
	    #
	    # Install glob pattern - copy the files in
	    #
	    foreach f [eval glob $install(glob,$dirType)] {
		if [catch {
		    set t [file tail $f]
		    if {$dirType == "man"} {
			# Hack to tweak file suffix
			set end [expr [string length $dir]-1]
			set suffix [string index $dir $end]
			set newf [file root $t].$suffix
		    } else {
			set newf $t
		    }
		    installCmd $logProc [list rm -f $dir/$newf]
		    installCmd $logProc [list cp $f $dir/$newf]
		    installCmd $logProc [list chmod a+r $dir/$newf]
		} msg] {
		    installFeedback "Dir install error: $msg"
		    return
		} else {
		    if {$logProc == "nolog"} {
			installFeedback "Installed $newf"
		    }
		}
	    }
	}
    }
    if {$logProc == "nolog"} {
	installCancel
	installFeedback "Install complete"
    }
}
proc installFake {} {
    global exmh install
    toplevel .fake
    frame .fake.top
    button .fake.top.quit -text "Dismiss" -command {destroy .fake}
    label .fake.top.label -text "  Pending install actions"
    pack append .fake .fake.top {top fill expand}
    pack append .fake.top .fake.top.quit left .fake.top.label {left fill}

    text .fake.t -width 80 -height 20 -yscrollcommand {.fake.s set} -font fixed
    scrollbar .fake.s -orient vert -command {.fake.t yview}
    pack append .fake \
	.fake.s {right filly} \
	.fake.t {left expand fill}

    proc log { text } {
	.fake.t insert end $text\n
    }
    installInner log
}

proc install_dialog {} {
    global install
    global applychanges

    wm minsize . 100 100
    
    if !$applychanges {
        toplevel .msg -borderwidth 2 -relief sunken
        wm title .msg "$install(appName) - Setup Instructions"
        wm title . "$install(appName) - Setup Parameters"
        # pack .msg -side top -expand yes -fill both -padx 10 -pady 10
        text .msg.t -font fixed -height 20 -yscrollcommand {.msg.s set}
        scrollbar .msg.s -orient vert -command {.msg.t yview}
        .msg.t insert end $install(helpText)
        pack .msg.t -side left -expand yes -fill both
        pack .msg.s -side right -fill y
    }

    
    frame .rim -bd 5 -relief flat
    pack append . .rim {top expand fill}

    installFieldInit
    for {set i 0} {$i < 10} {incr i} {
        if ![info exists install(sequence,$i)] continue
        foreach v $install(sequence,$i) {
            installDoField $install(field,$v)
        }
    }
    installFieldDone

    set install(msg) [label .rim.feedback -text ""]
    pack append .rim $install(msg) {top expand fill}
    
    frame .rim.buttons -relief raised
    pack append .rim .rim.buttons {top expand fill}
    
    button .rim.buttons.quit -width 12 -text "Quit" -command {exit}
    button .rim.buttons.patch -width 12 -text "Apply Changes" -command {installPatch}
#    button .rim.buttons.test -width 12 -text "Test" -command {installTest}
    frame .rim.buttons.space -width 10 -height 10

    	pack .rim.buttons
    	pack .rim.buttons.patch -side left -padx 5
#    	pack .rim.buttons.test -padx 5 -side left
    	pack .rim.buttons.space -padx 5 -side left
    	pack .rim.buttons.quit -padx 5 -side right
}
proc installUpdateAll {} {
    global install
    foreach v $install(sequence) {
#	if {$v == $install(versionVar)} {
#	    set override 1	;# over-ride saved value with new version num.
#	} else {
#	    set override 0
#	}
	    set override 0
	installUpdateField $install(field,$v) $override
    }
}

