(*--------------------------------------------------------------------------*)
(*                  Copyright (c) Donald Syme 1992                          *)
(*                  All rights reserved                                     *)
(*                                                                          *)
(* Donald Syme, hereafter referred to as `the Author', retains the copyright*)
(* and all other legal rights to the Software contained in this file,       *)
(* hereafter referred to as `the Software'.                                 *)
(*                                                                          *)
(* The Software is made available free of charge on an `as is' basis. No    *)
(* guarantee, either express or implied, of maintenance, reliability,       *)
(* merchantability or suitability for any purpose is made by the Author.    *)
(*                                                                          *)
(* The user is granted the right to make personal or internal use of the    *)
(* Software provided that both:                                             *)
(* 1. The Software is not used for commercial gain.                         *)
(* 2. The user shall not hold the Author liable for any consequences        *)
(*    arising from use of the Software.                                     *)
(*                                                                          *)
(* The user is granted the right to further distribute the Software         *)
(* provided that both:                                                      *)
(* 1. The Software and this statement of rights are not modified.           *)
(* 2. The Software does not form part or the whole of a system distributed  *)
(*    for commercial gain.                                                  *)
(*                                                                          *)
(* The user is granted the right to modify the Software for personal or     *)
(* internal use provided that all of the following conditions are observed: *)
(* 1. The user does not distribute the modified software.                   *)
(* 2. The modified software is not used for commercial gain.                *)
(* 3. The Author retains all rights to the modified software.               *)
(*                                                                          *)
(* Anyone seeking a licence to use this software for commercial purposes is *)
(* invited to contact the Author.                                           *)
(*--------------------------------------------------------------------------*)



signature Tcl_sig =
sig
    val tksml_compat_marker : int ref
    val tcl_pid : int ref
    val tcl_tmpdir : string ref
    val tcl : string -> string
    val tcl_ref : (string -> string) ref

(* different implementations of ML-Tcl communication. tcl_ref is one
   of these. *)
    val tcl_async : string -> string
    val tcl_nop : string -> string

    val exit : unit -> unit
    val Tcl_Merge : string list -> string
end;


(* compile "tksml/packages/tksml_slave/src/tcl.sig"; *)
(* compile "tksml/packages/tksml_slave/src/tcl.sml"; *)

structure Tcl : Tcl_sig =
struct
   val tksml_compat_marker = ref 100
   val tcl_pid = ref 0
   val tcl_tmpdir = ref "/tmp"
   fun tcl_async s = 
      let val intref = ref 1
          val filename = ref (!tcl_tmpdir ^ "/tksml_tcl" ^ makestring (!tcl_pid) ^ "." ^ makestring (!intref))
          val _ = 
             while System.Unsafe.SysIO.access(!filename,[]) do 
              (intref := (!intref)+1; 
               filename := !tcl_tmpdir ^ "/tksml_tcl" ^ makestring (!tcl_pid) ^ "." ^ makestring (!intref))
          val f = open_out (!filename);
          val _ = output (f,s);
          val _ = close_out f
          val _ = System.system("kill -USR1 " ^ makestring (!tcl_pid))
      in "" end
   fun tcl_nop (s:string) = ""
   val tcl_ref = ref tcl_nop; (* tcl_async is installed by Tcl when it's ready *)
   fun tcl s = (!tcl_ref) s
   fun exit () = (tcl "global busy; incr busy; exit"; ())


(*---------------------------------------------------------------------- 
 * Tcl_Merge
 *
 * DESCRIPTION
 *
 * Merge a list of strings into one string representing the list.
 * Use the Tcl algorithm *exactly*.  This means using bytearrays
 * and other gross operations.
 *
 * TESTS
 *
 * val rf = ref false;  val rt = ref true;
 *  use "tksml/packages/tksml_slave/src/tcl.sml";
 * compile "tksml/packages/tksml_slave/src/tcl.sml";
 * Tcl_ConvertElement "abc" (ByteArray.array(10,0),0) (rf,rf,rf);
 * Tcl_ConvertElement "abc" (ByteArray.array(10,0),0) (rt,rf,rf);
 * Tcl_ConvertElement "abc" (ByteArray.array(10,0),0) (rt,rt,rf);
 * Tcl_ConvertElement "abc" (ByteArray.array(10,0),0) (rt,rt,rt);
 * 
 * Tcl_Merge ["abc","def"];
 * Tcl_Merge ["ab cd","de df"];
 * Tcl_Merge ["{","de df"];
 * Tcl_Merge ["\\","/\\"];
 * Tcl_Merge ["\\"];
 * Tcl_Merge ["\\ ","a /\\ b"];
 * Tcl_Merge ["\\\n];
 * scan ["\\"];
 * Tcl_ScanElement "\\";
 *-----------------------------------------------------------------------*)



local
      open ByteArray
in
   fun string_to_bytearray (s:string) = 
        if (String.length s = 1) 
        then (array(1,ord s))
        else ((System.Unsafe.cast s) : ByteArray.bytearray)

   fun Tcl_ScanElement s =
      let val (USE_BRACES,TCL_DONT_USE_BRACES,BRACES_UNMATCHED) = (ref false, ref false, ref false)
          val sb = string_to_bytearray s
          val nestingLevel = ref 0
          val p = ref 0
          val size = ByteArray.length sb
          val _ = if (size = 0 orelse 
                      sub(sb,!p) = ord "{" orelse
                      sub(sb,!p) = ord "\"") 
                  then USE_BRACES := true
                  else ()
          val _ = while (!p < size) do (
(*                    output (std_out,"!p = " ^ makestring(!p) ^ "\n"); *)
                    (case sub(sb,!p) of 
                        123 (* "{" *) => inc nestingLevel
                      | 125 (* "}" *) => (
                           dec nestingLevel;
                           if (!nestingLevel < 0) 
                           then (TCL_DONT_USE_BRACES := true; BRACES_UNMATCHED := true)
                           else ()
                        )
                      | 91 (* "[" *) => USE_BRACES := true
                      | 36 (* "$" *) => USE_BRACES := true
                      | 59 (* ";" *) => USE_BRACES := true
                      | 32 (* " " *) => USE_BRACES := true
                      | 9  (* "\t" *) => USE_BRACES := true
                      | 10 (* "\n" *) => USE_BRACES := true
                      | 92 (* "\\" *) => (
                         if ((!p+1) >= size orelse sub(sb,!p+1) = 10 (* "\n" *))
                         then TCL_DONT_USE_BRACES := true
                         else (USE_BRACES := true; inc p))
                      | _ => ())
                     ; inc p
                  )
         val _ = if (!nestingLevel <> 0) 
                 then (TCL_DONT_USE_BRACES := true; BRACES_UNMATCHED := true)
                 else ()

      in (2 * size + 3,(USE_BRACES,TCL_DONT_USE_BRACES,BRACES_UNMATCHED))
      end

   fun Tcl_ConvertElement srcstring (dstarray,dstindex) (USE_BRACES,TCL_DONT_USE_BRACES,BRACES_UNMATCHED) =
      let val srcarray = string_to_bytearray srcstring
          val src = ref 0
          val p = ref dstindex
          val srcsize = length srcarray
          val addself = ref false
          val _ = if (!USE_BRACES andalso not (!TCL_DONT_USE_BRACES)) then (
                     update(dstarray,!p,ord "{");
                     inc p;
                     while (!src < srcsize) do (
                        update(dstarray,!p,sub(srcarray,!src)); 
                        inc src;
                        inc p
                     );
                     update(dstarray,!p,ord "}");
                     inc p
                  ) else if (!src = srcsize) then (
                     update(dstarray,!p,ord "\\");
                     inc p
                  ) else (
                     (if (sub(srcarray,!src) = ord "{") then (
                        update(dstarray,!p,ord "\\");
                        inc p;
                        update(dstarray,!p,ord "{");
                        inc p;
                        inc src;
                        BRACES_UNMATCHED := true
                      ) else ());
                     while (!src < srcsize) do (
                       addself := true;
                       (case sub(srcarray,!src) of
                        91 (* "[" *) => (update(dstarray,!p,ord "\\"); inc p)
                      | 36 (* "$" *) => (update(dstarray,!p,ord "\\"); inc p)
                      | 59 (* ";" *) => (update(dstarray,!p,ord "\\"); inc p)
                      | 32 (* " " *) => (update(dstarray,!p,ord "\\"); inc p)
                      | 92 (* "\\" *) => (update(dstarray,!p,ord "\\"); inc p)
                      | 34 (* "\"" *) => (update(dstarray,!p,ord "\\"); inc p)
                      | 123 (* "{" *) => 
                         if (!BRACES_UNMATCHED) 
                         then (update(dstarray,!p,ord "\\"); inc p)
                         else ()  
                      | 125 (* "}" *) => 
                         if (!BRACES_UNMATCHED) 
                         then (update(dstarray,!p,ord "\\"); inc p)
                         else ()  
                      | 10 (* "\n" *) => (update(dstarray,!p,ord "\\"); inc p;
                                          update(dstarray,!p,ord "n"); inc p; addself := false)
                      | 9 (* "\t" *) => (update(dstarray,!p,ord "\\"); inc p;
                                          update(dstarray,!p,ord "n"); inc p; addself := false)
                      | _ => ());
                      if (!addself) 
                      then (update(dstarray,!p,sub(srcarray,!src)); inc p)
                      else ();
                      inc src
                     )
                   )
      in
         (!p - dstindex)
      end



   fun scan (h::t) = 
             let val (size,flags) = Tcl_ScanElement h
                 val (tsize,tflags) = scan t
             in (size+tsize,(h,flags)::tflags)
             end
     | scan [] = (0,[])

   fun copy (dstarray,dstindex) [(srcstring,flags)] = 
        dstindex + Tcl_ConvertElement srcstring (dstarray,dstindex) flags
     | copy (dstarray,dstindex) ((srcstring,flags)::t) = 
      let val size = Tcl_ConvertElement srcstring (dstarray,dstindex) flags
          val _ = update(dstarray,dstindex+size,ord " ")
      in copy (dstarray,dstindex+size+1) t
      end
     | copy (dstarray,dstindex) [] = dstindex
          
   fun Tcl_Merge strings =
      let val (numChars,flagsl) = scan strings
          val dstarray = array (numChars,0)
          val size = copy (dstarray,0) flagsl
      in
          extract (dstarray,0,size)
      end
end;




end;

open TclResult;
open Tcl;

