#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------





 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File: progress.tcl
#
# Description:
#       Management of a widget containing a panel representing progress
#       of a long-running operation.
#
#
# Global variables:
#c      progress_priv(variable,$w)
#               Variable name that holds the fraction of progress made
#               on the operation displayed in window $w.


# Procedure:    progress
#
# Synopsis:
#       Widget that displays a panel representing the progress of a
#       long-running operation.
#
# Usage:
#c      progress pathName ?-option value?...
#
# Parameters:
#c      pathName
#               Path name of the window that will hold the progress indication.
#
# Options:
#       Name:                   quitCommand
#       Class:                  QuitCommand
#       Command-line string:    -quit, -quitcommand
#       Default:                "destroy <pathName>"
#               Command to be invoked when the user presses the QUIT button.
#
#       Name:                   text
#       Class:                  Text
#       Command-line string:    -text
#       Default:                The empty string.
#               Text to be displayed in the `progress' dialog box.
#
#       Name:                   variable
#       Class:                  Variable
#       Command-line string:    -var, -variable
#       Default:                progress<pathName>
#               Global variable that holds the fraction of the operation that
#               has completed.
#
#       Other options are as for `frame.'
#
# Return value:
#       The path name of the widget is returned.
#
# Description:
#       The `progress' command constructs a simple dialog box consisting of:
#       - A message
#       - A bar showing the fraction of progress made in an operation
#       - A `quit' button.
#
#       The action taken by the `quit' button, and the message are
#       customizable.  The bar showing the amount of progress is tied to a
#       user-specified global variable.
#
# Bugs:
#       The `progress' widget is not a first-class widget.  It should at least
#       respond correctly to the `config' command. 

        # Default look and feel:

option add *Progress.quitCommand {} widgetDefault
option add *Progress.text {} widgetDefault
option add *Progress.variable {} widgetDefault

proc progress {w args} {

        frame $w -class Progress

        set quit [option get $w quitCommand QuitCommand]
        if {$quit == ""} {
                set quit "destroy $w"
        }
        set text [option get $w text Text]
        set variable [option get $w variable Variable]
        if {$variable == ""} {
                set variable progress$w
        }
        set fa {}

        # Parse command line

        while {[llength $args] > 0} {
                set string [lindex $args 0]
                set args [lrange $args 1 end]
                case $string in {
                        {-quit -quitcommand} {
                                set quit [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        -text {
                                set text [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-v -var -variable} {
                                set variable [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        default {
                                lappend fa $string
                                lappend fa [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                }
        }

        eval [list $w config] $fa

        if {$text != ""} {
                pack append $w \
                        [label $w.l -text $text] \
                                {top padx 10 frame center}
        }
        pack [progress:2 $w.p $variable] 
            -side top -anchor w

        pack [eval fraction [list $w.f -variable $variable] $args] \
                -side top -expand yes -padx 10 -pady 10 -anchor center
        pack [button $w.b -text "Quit" \
                        -command "after 1 [list $quit]"] \
                -side top -expand yes -pady 5 -anchor n


        return $w
}

# Procedure:    progress:2
#
# Synopsis:
#       Internal procedure to aid in creating a `progress' widget.
#
# Usage:
#c      progress:2 pathName variable
#
# Parameters:
#c      pathName
#               Path name of a window
#c      variable
#               Name of a global variable containing the fraction of
#               progress to be displayed.
#
# Return value:
#       pathName is returned.
#
# Description:
#       progress:2 is an internal procedure that creates a label widget that
#       gets packed into the frame created by a `progress' window, and
#       establishes a trace on the variable to update the label.

proc progress:2 {w var} {
        global progress_priv
        set progress_priv(variable,$w) $var
        label $w -text " 0.0% complete"
        proc update$w {v1 v2 op} [format {
                trace_action $v1 $v2 $op {
                        progress:2:update %s $value
                }
        } $w]
        uplevel #0 trace variable $var w update$w
        bind $w <Destroy> "+progress:2:destroy $w"
        return $w
}

# Procedure:    progress:2:update
#
# Synopsis:
#       Internal procedure to update part of a progress widget.
#
# Usage:
#       progress:2:update pathName fraction
#
# Parameters:
#c      pathName
#               Path name of a label widget.
#c      fraction
#               Fraction of the operation that has completed.
#
# Return value:
#       None.
#
# Description:
#       progress:2:update is called by a `trace' operation any time the value
#       on display in a `progress' window changes.  It updates the label widget
#       and synchronizes the display.
#
# Possible improvement:
#       Bunch the updates by use of an `after' handler.  This might be a big
#       performance win if a task runs quickly.

proc progress:2:update {w value} {
        $w config -text [format "%4.1f%% complete" [expr 100.0*$value]]
        update
}

# Procedure:    progress:2:destroy
#
# Synopsis:
#       Internal procedure to clean up when a progress widget is destroyed.
#
# Usage:
#c      progress:2:destroy pathName
#
# Parameters:
#c      pathName
#               Path name of a label widget.
#
# Return value:
#       None.
#
# Description:
#       `progress:2:destroy' is an internal procedure that is called when
#       a `progress' widget is destroyed.

proc progress:2:destroy w {
        global progress_priv
        uplevel #0 trace vdelete $progress_priv(variable,$w) w update$w
        rename update$w {}
        unset progress_priv(variable,$w)
}
