(*--------------------------------------------------------------------------*)
(*                  Copyright (c) Donald Syme 1992                          *)
(*                  All rights reserved                                     *)
(*                                                                          *)
(* Donald Syme, hereafter referred to as `the Author', retains the copyright*)
(* and all other legal rights to the Software contained in this file,       *)
(* hereafter referred to as `the Software'.                                 *)
(*                                                                          *)
(* The Software is made available free of charge on an `as is' basis. No    *)
(* guarantee, either express or implied, of maintenance, reliability,       *)
(* merchantability or suitability for any purpose is made by the Author.    *)
(*                                                                          *)
(* The user is granted the right to make personal or internal use of the    *)
(* Software provided that both:                                             *)
(* 1. The Software is not used for commercial gain.                         *)
(* 2. The user shall not hold the Author liable for any consequences        *)
(*    arising from use of the Software.                                     *)
(*                                                                          *)
(* The user is granted the right to further distribute the Software         *)
(* provided that both:                                                      *)
(* 1. The Software and this statement of rights are not modified.           *)
(* 2. The Software does not form part or the whole of a system distributed  *)
(*    for commercial gain.                                                  *)
(*                                                                          *)
(* The user is granted the right to modify the Software for personal or     *)
(* internal use provided that all of the following conditions are observed: *)
(* 1. The user does not distribute the modified software.                   *)
(* 2. The modified software is not used for commercial gain.                *)
(* 3. The Author retains all rights to the modified software.               *)
(*                                                                          *)
(* Anyone seeking a licence to use this software for commercial purposes is *)
(* invited to contact the Author.                                           *)
(*--------------------------------------------------------------------------*)




structure ZPretermOps: ZPretermOps_sig =
struct

open PretermOps;
open Z;
open Parse_support.Preterm;
open Parse_support;
type schema_context = string list;
type varstruct = string list;

fun is_input v =
 let val cl = rev(explode v)
 in
 (length cl > 2) andalso (hd cl = "I") end;

fun is_output v =
 let val cl = rev(explode v)
 in
 (length cl > 2) andalso (hd cl = "O") end;

fun is_plain v = 
   not(isprime_Var v orelse is_input v orelse is_output v);

fun forall f (h::t) = f h andalso forall f t
  | forall f [] = true;

(*----------------------------------------------------------------------------*)
(* dest_dec `v IN S`  --->  (`v`,`S`)                                         *)
(* is_dec pm tests whether pm has the form `v IN S`                           *)
(*----------------------------------------------------------------------------*)

fun dest_Dec pm =
 let val (oper,args) = strip_Comb pm
 in
   if is_Const oper andalso
    (dest_Const oper = "IN") andalso
    (length args = 2)        andalso
    is_Var(hd args)
   then (dest_Var (hd args), hd(tl args))
   else raise (Fail "dest_Dec")
 end;

fun mk_Dec (v,p) =
   list_mk_Comb(mk_Const "IN",[mk_Var v, p])

fun dest_input dec = 
 let val (v,_) = dest_Dec dec
 in
   if is_input v then v else raise (Fail "dest_input")
 end;

fun dest_output dec = 
 let val (v,_) = dest_Dec dec
 in
   if is_output v then v else raise (Fail "dest_output")
 end;

fun dest_plain dec = 
 let val (v,_) = dest_Dec dec
 in
   if is_plain v then v else raise (Fail "dest_input")
 end;

val get_dec_var = #1 o dest_Dec;
val get_dec_set = #2 o dest_Dec;

val is_Dec = can dest_Dec;

(*----------------------------------------------------------------------------
 * dest_Decs `[SC1; SC2; ...; v1 IN S1; ... ; vn IN Sn]`                      
 * --->                                                                       
 * ([SC1,SC2,...],[(`v1`,`S1`);...;(`vn`,`Sn`)])
 *----------------------------------------------------------------------------*)

fun dest_Decs' (h::t)  = 
    let val (sc,decs) = dest_Decs' t in
       (sc,(dest_Dec h::decs))
       handle _ => (h::sc,decs)
    end
  | dest_Decs' [] = ([],[])
val dest_Decs = dest_Decs' o dest_List

(*----------------------------------------------------------------------------*)
(* is_Decs pm tests whether pm has the form `[SC1,SC2,...,v1 IN S1, ... , vn IN Sn]`      *)
(*----------------------------------------------------------------------------*)

val is_Decs = can dest_Decs;

infix subset;
fun s1 subset s2 = (subtract s1 s2 = []);;
fun appendl l = itlist (op append) l [];;


(*----------------------------------------------------------------------------
 * Schema Recognisers/Destructors
 * 
 * Need to be mutually recursive as schemas can contain schemas, and
 * we have to find the varstruct for the schema at destruction time.
 *
 * SC -> SC / (v1,v2,...) 
 * SC' -> SC / (v1',v2',...) etc.
 * DELTA SC -> SC / (v1,v2...,v1',v2',...)
 * XI SC -> SC / (v1,v2...,v1',v2',...)
 * S1 AND S2 -> (S1,S2) / (v1,v2... varstruct_union v1',v2',...)
 * S1 OR S2 -> (S1,S2) / (v1,v2... varstruct_union v1',v2',...)
 * S1 IMP S2 -> (S1,S2) / (v1,v2... varstruct_union v1',v2',...)
 * NOT SC -> S1 / (v1,v2...)
 *                                                                            
 * SCHEMA decs bdy -> (decs,body) / (frees decs union frees body)
 * check that:                                                           
 *                                                                       
 *    1. decs is a list of declarations and schemas
 *                                                                       
 *    2. free variables in bdy are declared in decs or in schemas in decs
 *                                                                       
 *    3. no declared variable occurs in the rhs of a declaration
 *
 *
 *----------------------------------------------------------------------------*)

exception NOT_ONE_OF_THOSE
fun dest_ConstComb name numargs p =
 let val (oper,args) = strip_Comb p
     val opcon = dest_Const oper handle _ => raise NOT_ONE_OF_THOSE
     val _ = if (opcon = name) then () else raise NOT_ONE_OF_THOSE
     val _ = if (length args = numargs) then () else raise (Fail (name ^ ":" ^ (makestring numargs) ^ "argument(s) required"))
 in 
     args
 end;


fun Varstruct_for_defined_schema name = 
   map (#Name o dest_var) (dest_varstruct (varstruct_for_defined_schema name));

(*----------------------------------------------------------------------------
 * 
 *----------------------------------------------------------------------------*)

fun dest_SchemaConstant p = 
   let val name = (dest_Var p handle _ => dest_Const p)
       val _ = (force_schema_constant name handle _ => raise NOT_ONE_OF_THOSE)
   in
       if is_defined_schema name
       then name
       else raise NOT_ONE_OF_THOSE
   end
fun is_SchemaConstant p = can dest_SchemaConstant p
fun is_PlainSchemaConstant p = 
    is_SchemaConstant p andalso (#2 (unprime (dest_SchemaConstant p)) = 0)
fun varstruct_SchemaConstant name = 
   let val _ = force_schema_constant name
   in
     Varstruct_for_defined_schema name
   end

(*----------------------------------------------------------------------------
 * 
 *----------------------------------------------------------------------------*)

fun dest_Delta p = 
    let val [schema] = dest_ConstComb "DELTA" 1 p handle _ => raise NOT_ONE_OF_THOSE
        val _ = assert is_PlainSchemaConstant schema handle _ => raise (Fail "DELTA: Undecorated schema constant required as argument")
        val name = dest_SchemaConstant schema 
    in
       name
    end
and is_Delta p = can dest_Delta p
and varstruct_Delta name = 
   let val vs = Varstruct_for_defined_schema name
   in vs@prime_Varstruct (vs,1) 
   end

and dest_Xi p = 
    let val [schema] = dest_ConstComb "XI" 1 p handle _ => raise NOT_ONE_OF_THOSE
        val _ = assert is_PlainSchemaConstant schema handle _ => raise (Fail "XI: Undecorated schema name required as argument")
        val name = dest_SchemaConstant schema 
    in
       name
    end
and is_Xi p = can dest_Xi p
and varstruct_Xi name = 
   let val vs = Varstruct_for_defined_schema name
   in vs@prime_Varstruct (vs,1) 
   end

and dest_SCHEMA' p =
  let val [decs,bdy] = dest_ConstComb "SCHEMA" 2 p
      val bodyl = dest_List bdy
              handle _ => raise (Fail "SCHEMA: bad body section in schema")
      val (schemas,decl) = dest_Decs decs handle _ => raise (Fail "SCHEMA: bad declaration section")
      val (dec_vars,dec_sets) = split decl
      val _ = if (forall is_Schema schemas) then () else raise (Fail "SCHEMA: bad entry in declaration section")
      val schema_vs = map varstruct_Schema schemas
      val schema_vars = itlist union schema_vs []
      val useable_vars = union schema_vars dec_vars
      val _ = (assert (op subset) (Frees bdy,useable_vars) 
               handle _ => raise (Fail "SCHEMA: undeclared free variable in body"))
      val _ = (assert null (intersect dec_vars (appendl(map Frees dec_sets)))
               handle _ => raise (Fail "SCHEMA: variable occurs on both lhs and rhs of declarations"))
 in
   {schemas=combine (schemas,schema_vs),decs=decl,body=bodyl}
 end

and is_SCHEMA' p = can dest_SCHEMA' p
and varstruct_SCHEMA' {schemas,decs,body} = 
    let
      val (dec_vars,_) = split decs
      val schema_vars = itlist union (map #2 schemas) []
    in
      union schema_vars dec_vars
    end

and dest_Schema2op  name p = 
    let val [schema1,schema2] = dest_ConstComb name 2 p handle _ => raise NOT_ONE_OF_THOSE
        val (s1,s2) = (is_Schema schema1, is_Schema schema2) 
        val _ = if (s1 andalso (not s2))
                then raise (Fail (name ^ ": Either both arguments or neither argument should be a schema"))
                else if (s2 andalso (not s1))
                then raise (Fail (name ^ ": Either both arguments or neither argument should be a schema"))
                else ()
    in
       ((schema1,varstruct_Schema schema1),(schema2,varstruct_Schema schema2))
    end
and is_Schema2op name = can (dest_Schema2op name)
and varstruct_Schema2op ((schema1,vs1),(schema2,vs2)) = 
       union vs1 vs2
and dest_SchemaDisj p = dest_Schema2op "\\/" p
and is_SchemaDisj p = is_Schema2op "\\/" p
and varstruct_SchemaDisj p = varstruct_Schema2op p

and dest_SchemaConj p = dest_Schema2op "/\\" p
and is_SchemaConj p = is_Schema2op "/\\" p
and varstruct_SchemaConj p = varstruct_Schema2op p

and dest_SchemaImplies p = dest_Schema2op "==>" p
and is_SchemaImplies p = is_Schema2op "==>" p
and varstruct_SchemaImplies p = varstruct_Schema2op p

and dest_SchemaNot p = 
    let val [schema] = dest_ConstComb "~" 1 p
        val _ = (assert is_Schema schema handle _ => raise NOT_ONE_OF_THOSE) 
    in
       (schema,varstruct_Schema schema)
    end
and is_SchemaNot p = can dest_SchemaNot p
and varstruct_SchemaNot (schema,vs) = vs

and is_Schema p = 
        is_SchemaConstant p orelse
        is_SCHEMA' p orelse 
        is_Delta p orelse 
        is_Xi p orelse 
        is_SchemaDisj p orelse
        is_SchemaConj p orelse
        is_SchemaImplies p orelse
        is_SchemaNot p orelse
        false
and varstruct_Schema p =
        varstruct_SchemaConstant (dest_SchemaConstant p)
        handle _ => varstruct_SCHEMA' (dest_SCHEMA' p) 
        handle _ => varstruct_Delta (dest_Delta p) 
        handle _ => varstruct_Xi (dest_Xi p)
        handle _ => varstruct_SchemaDisj (dest_SchemaDisj p) 
        handle _ => varstruct_SchemaConj (dest_SchemaConj p) 
        handle _ => varstruct_SchemaImplies (dest_SchemaImplies p) 
        handle _ => varstruct_SchemaNot (dest_SchemaNot p) 

fun dest_SchemaPredForall p = 
    let val (name,body) = dest_Forall p handle _ => raise NOT_ONE_OF_THOSE
        val _ = force_schema_constant name handle _ => raise NOT_ONE_OF_THOSE
        val _ = assert is_defined_schema name handle _ => raise NOT_ONE_OF_THOSE
        val schema = Const {Name=name,Ty=type_of (#const (const_decl name))}
    in
       ((schema,varstruct_Schema schema),body)
    end
fun is_SchemaPredForall p = can dest_SchemaPredForall p

fun dest_SchemaPredExists p = 
    let val (name,body) = dest_Exists p handle _ => raise NOT_ONE_OF_THOSE
        val _ = force_schema_constant name handle _ => raise NOT_ONE_OF_THOSE
        val _ = assert is_defined_schema name handle _ => raise NOT_ONE_OF_THOSE
        val schema = Const {Name=name,Ty=type_of (#const (const_decl name))}
    in
       ((schema,varstruct_Schema schema),body)
    end
fun is_SchemaPredExists p = can dest_SchemaPredExists p

(*----------------------------------------------------------------------------
 * (decs`, `bdy`) ---> (`SCHEMA decs bdy`,varstruct)
 *                                                                           
 * and 
 *                                                                           
 *----------------------------------------------------------------------------*)

fun Varstruct_canon l = sort (curry  String.<) (mk_set l);

fun mk_SchemaApplication (schema,vs) =
        mk_Comb (schema,mk_Varstruct (Varstruct_canon vs))

fun mk_SchemaUsage (schema,vs) =
        mk_Comb(mk_Const "BOTH",mk_SchemaApplication (schema,vs))

fun mk_SchemaConstant name = 
    (force_schema_constant name ; mk_Const name)

fun mk_SCHEMA' {schemas,decs,body} =
   let val schema_vars = itlist union (map #2 schemas) []
       val vs = union schema_vars (map #1 decs)
       val schemas' = mk_List (map mk_SchemaApplication schemas)
       val decs' = mk_List (map mk_Dec decs)
       val body' = mk_List body
   in
      mk_Pabs(Varstruct_canon vs,list_mk_Comb(mk_Const "SCHEMA",[schemas',decs',body']))
   end

fun mk_Schema2op oper ((schema1,vs1),(schema2,vs2)) =
   mk_Pabs(Varstruct_canon (vs1@vs2),
           list_mk_Comb(mk_Const oper,[mk_SchemaApplication(schema1,vs1),mk_SchemaApplication(schema2,vs2)]))
val mk_SchemaDisj = mk_Schema2op "ZOR";
val mk_SchemaConj = mk_Schema2op "ZAND";
val mk_SchemaImplies = mk_Schema2op "ZIMPLIES";
fun mk_SchemaNot (schema,vs) = 
   mk_Pabs(Varstruct_canon vs,list_mk_Comb(mk_Const "ZNOT",[mk_SchemaApplication(schema,vs)]))

fun mk_SchemaPredForall ((schema,vs),body) =
   mk_Pforall(Varstruct_canon vs,mk_Imp(schema,body))
fun mk_SchemaPredExists ((schema,vs),body) =
   mk_Pexists(Varstruct_canon vs,mk_Imp(schema,body))

fun mk_Delta name =
   let val vs = Varstruct_for_defined_schema name
   in
       mk_SchemaConj((mk_Const name, vs),
                     (mk_Const (name ^ "'"), prime_Varstruct (vs,1)))
   end
fun mk_Xi name =
   let val vs = Varstruct_for_defined_schema name
   in
       mk_SCHEMA'{
           schemas=[(mk_Const name,vs), 
                    (mk_Const (name ^ "'"),prime_Varstruct(vs,1))],
           decs=[],
           body=[mk_Eq (mk_Varstruct vs,mk_Varstruct(prime_Varstruct(vs,1)))]
       }
   end

(*==========================================================================
 * Preprocess applications `f x` to `f ZAPPLY x`, if `f` is a schema
 * variable
 *==========================================================================*)

(*----------------------------------------------------------------------------
 * is_set_fun (Var "f") returns true if f
 * sctxt, otherwise it returns false. 
 *
 * EXAMPLE:
 *
(-<`SCHEMA 
   [birthday IN (num -+> bool)]
   (*---------------------------*)
   [birthday 1]`>-);
 *----------------------------------------------------------------------------*)

fun is_set_fun (SOME sctxt) (Var {Name,Ty}) = mem Name sctxt
  | is_set_fun _ _ = false;

fun mk_set_fun p1 p2 =
 mk_Comb (mk_Comb((mk_Const "ZAPPLY"), p1), p2);


(*----------------------------------------------------------------------------*)
(* Macroexpand schema operations.                                             *)
(*----------------------------------------------------------------------------*)

fun use_schema in_schema sctxt schema =
        if in_schema
        then schema
        else mk_SchemaUsage (schema,sctxt)

fun expand_Z' sctxt p =
   let val in_schema = case sctxt of SOME _ => true | NONE => false in
      if (is_Var p) orelse (is_Const p) then (
         let val name = dest_SchemaConstant p
             val sctxt' = varstruct_SchemaConstant name in
           use_schema in_schema sctxt' (mk_SchemaConstant name)
         end
         handle NOT_ONE_OF_THOSE =>
         preterm_to_preterm_in_env p
      ) else if is_Comb p then (
        let val (p1,p2) = dest_Comb p in
          if is_set_fun sctxt p1
          then mk_set_fun (preterm_to_preterm_in_env p1) (expand_Z' sctxt p2)
          else raise NOT_ONE_OF_THOSE
        end
        handle NOT_ONE_OF_THOSE =>
        let val {schemas,decs,body} = dest_SCHEMA' p
            val sctxt' = varstruct_SCHEMA' {schemas=schemas,decs=decs,body=body} 
        in
          use_schema in_schema sctxt'
                       (mk_SCHEMA' {
                             schemas=map (expand_Z' (SOME sctxt') ## I) schemas,
                             decs=map (I ## expand_Z' (SOME sctxt')) decs,
                             body=map (expand_Z' (SOME sctxt')) body
                       })
        end
        handle NOT_ONE_OF_THOSE =>
        let val name = dest_Delta p
            val sctxt' = varstruct_Delta name in
            use_schema in_schema sctxt' (mk_Delta name)
        end
        handle NOT_ONE_OF_THOSE =>
        let val name = dest_Xi p
            val sctxt' = varstruct_Xi name in
            use_schema in_schema sctxt' (mk_Xi name)
        end
        handle NOT_ONE_OF_THOSE =>
        let val ((schema1,vs1),(schema2,vs2)) = dest_SchemaDisj p
            val sctxt' = varstruct_SchemaDisj ((schema1,vs1),(schema2,vs2)) in
            use_schema in_schema sctxt'
                         (mk_SchemaDisj ((expand_Z' (SOME sctxt') schema1,vs1),(expand_Z' (SOME sctxt') schema2,vs2)))
        end
        handle NOT_ONE_OF_THOSE =>
        let val ((schema1,vs1),(schema2,vs2)) = dest_SchemaConj p
            val sctxt' = varstruct_SchemaConj ((schema1,vs1),(schema2,vs2)) in
            use_schema in_schema sctxt'
                         (mk_SchemaConj ((expand_Z' (SOME sctxt') schema1,vs1),(expand_Z' (SOME sctxt') schema2,vs2)))
        end
        handle NOT_ONE_OF_THOSE =>
        let val ((schema1,vs1),(schema2,vs2)) = dest_SchemaImplies p
            val sctxt' = varstruct_SchemaImplies ((schema1,vs1),(schema2,vs2)) in
            use_schema in_schema sctxt'
                         (mk_SchemaImplies ((expand_Z' (SOME sctxt') schema1,vs1),(expand_Z' (SOME sctxt') schema2,vs2)))
        end
        handle NOT_ONE_OF_THOSE =>
        let val (schema,vs) = dest_SchemaNot p
            val sctxt' = varstruct_SchemaNot (schema,vs) in
            use_schema in_schema sctxt'
                         (mk_SchemaNot (expand_Z' (SOME sctxt') schema,vs))
        end
        handle NOT_ONE_OF_THOSE =>
        let val ((schema,vs),body) = dest_SchemaPredForall p
        in
             mk_SchemaPredForall ((expand_Z' sctxt schema,vs),expand_Z' sctxt body)
        end
        handle NOT_ONE_OF_THOSE =>
        let val ((schema,vs),body) = dest_SchemaPredExists p
        in
             mk_SchemaPredExists ((expand_Z' sctxt schema,vs),expand_Z' sctxt body)
        end
        handle NOT_ONE_OF_THOSE =>
        mk_Comb(expand_Z' sctxt (preterm_rator p),expand_Z' sctxt (preterm_rand p))

(*         else if is_SchemaForall p1e p2e
           then mk_SchemaForall(preterm_rand p1e,p2e)
           else if is_SchemaExists p1e p2e
           then mk_Schema_exists(preterm_rand p1e,p2e)
           else if is_SchemaSeq p1e p2e
           then mk_Schema_seq(preterm_rand p1e,p2e)
           else if is_SchemaPre p1e p2e
           then mk_SchemaPre p2e
           else if is_Sig p1e p2e
           then mk_Sig p2e
           else if is_Theta p1e p2e
           then mk_Theta p2e
           else if is_Pred p1e p2e
           then mk_Pred p2e 
           else if is_SchemaHide p1e p2e
           then mk_SchemaHide p1e p2e *)
     ) else if is_Abs p then (
         let val (p1,p2) = dest_Abs p in
            mk_Abs(p1, expand_Z' sctxt p2)
         end
     ) else if is_Constrained p then (
         let val (p1,ty) = dest_Constrained p in
            mk_Constrained(expand_Z' sctxt p1, ty)
         end
     ) else raise (Fail "expand_Z")
  end;

fun expand_Z p = expand_Z' NONE p;
fun expand_Z_schema p = expand_Z' (SOME []) p;

fun -- fl a = typecheck_cleanup (typecheck ((fn PTM x => x) (make_preterm (expand_Z (preterm_parser fl)))));
fun -< fl a = typecheck_cleanup (typecheck ((fn PTM x => x) (make_preterm (expand_Z_schema (preterm_parser fl)))));
val >- = 1;

end;

open ZPretermOps;











