#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




proc richtext {} {}

composite_define RichText {
        {-structuredtextvar structuredtextVar StructuredtextVar "" composite_configPrivateNop}
        {-richtextvar richtextVar RichtextVar "" composite_configPrivateNop}
        {-selectiontypesvar selectiontypesVar SelectiontypesVar selectiontypes composite_configPrivateNop}
        {-focusvar focusVar FocusVar "" composite_configPrivateNop}
        {-cursorvar cursorVar CursorVar "" composite_configPrivateNop}
        {-interactionmode interactionMode InteractionMode subobjectselection composite_configPrivateNop}
} RichText::initProc


proc RichText::initProc { w } {
        global RichText_flags
        global TheoremWidgets_flags
        global gui_flags
        upvar #0 config$w config
        upvar #0 $w data

        set data(numobjects) 0
        set data(flagvars) ""
        if {$config(richtextVar)==""} { set config(richtextVar) [set w]richtext }
        if {$config(structuredtextVar)==""} { set config(structuredtextVar) [set w]structuredtext }
        upvar #0 $config(structuredtextVar) structuredtext
        upvar #0 $config(selectiontypesVar) selectiontypes
        upvar #0 $config(richtextVar) richtext
        if ![info exists richtext] { set richtext $RichText_flags(richtextOnByDefault) }
        if ![info exists structuredtext] { set structuredtext $RichText_flags(structuredtextOnByDefault) }
        set selectiontypes ""
        
        pack [scrollable text $w.text \
                -font $gui_flags(font,codeentry) \
                -relief sunken \
                -borderwidth 2 \
                -state disabled] \
            -side left -expand yes -fill both
        set data(looks) [list $RichText_flags(default_look)]

        # command not available in constructor...
        RichText::reconfiguretags $w

#       trace variable structuredtext w "$w reformatall"
#       trace variable richtext w "$w reformatall"
        trace variable RichText_flags w "RichText::tag_flag_change_trigger $w"

        bind $w <Destroy> "+RichText::upon_destroy $w %W"

        return $w
}

composite_configFlag RichText selectiontypesVar { w arg value } {
        upvar #0 config$w config
        upvar #0 $config(selectiontypesVar) old_selectiontypes
        upvar #0 $value new_selectiontypes
        set new_selectiontypes $old_selectiontypes
        return private
}

composite_configFlag RichText richtextVar { w arg value } {
        upvar #0 config$w config
        upvar #0 $config(richtextVar) old_richtext
        upvar #0 $value new_richtext
        if [info exists old_richtext] {
#            trace vdelete old_richtext w "$w reformatall"
             if [catch {set new_richtext $old_richtext} error] {
                puts stderr "warning: $error\n\ttrace vinfo new_richtext = [trace vinfo new_richtext]"
                global errorInfo
                error $error $errorInfo
             }
        }
#       trace variable new_richtext w "$w reformatall"
        return private
}

composite_configFlag RichText structuredtextVar { w arg value } {
        upvar #0 config$w config
        upvar #0 $config(structuredtextVar) old_structuredtext
        upvar #0 $value new_structuredtext
        if [info exists old_structuredtext] {
#           trace vdelete old_structuredtext w "$w reformatall"
             if [catch {set new_structuredtext $old_structuredtext} error] {
                puts stderr "warning: $error\n\ttrace vinfo new_structuredtext = [trace vinfo new_structuredtext]"
                global errorInfo
                error $error $errorInfo
             }
        }
#       trace variable new_structuredtext w "$w reformatall"
        return private
}
composite_configFlag RichText interactionMode { w arg value } {
        upvar #0 config$w config
# puts "Setting interactionMode, w = $w, value = $value"
        switch -- $value {
            freeselect {
               $w.text.b config -state disabled
               bindtags $w.text.b Text
            }
            freeedit {
               $w.text.b config -state normal
               RichText::bind_for_normal_selection $w
            }
            subobjectselection {
               $w.text.b config -state disabled
               RichText::bind_for_subobject_selection $w
            }
            leaffocus {
               $w.text.b config -state disabled
               RichText::bind_for_leaffocus $w
            }
            leafedit {
               $w.text.b config -state normal
               RichText::bind_for_leafedit $w
            }
            default { puts "unrecognised interaction mode \"$value\"" }
        }
        return private
}


proc RichText::upon_destroy { w realw } {
        global RichText_flags
        if {$w==$realw} {
#           upvar #0 config$w config
#           upvar #0 $config(structuredtextVar) structuredtext
#           upvar #0 $config(richtextVar) richtext
#           trace vdelete RichText_flags w "RichText::tag_flag_change_trigger $w"
#           trace vdelete structuredtext w "after 1 \{$w reformatall\}"
#           trace vdelete richtext w "after 1 \{$w reformatall\}"
            catch {trace vdelete focus w "RichText::focus_change $w"}
            catch {trace vdelete cursor w "RichText::cursor_change $w"}
        }
}           


proc RichText::tag_flag_change_trigger { w args } {
        RichText::reconfiguretags $w
}

proc RichText::reconfiguretags { w } {
        global RichText_flags
        global TheoremWidgets_flags
        upvar #0 $w data
        if ![info exists data(looks)] { return }
        set l $data(looks)
        set data(looks) ""
        foreach look $l {
            RichText::configure_look_tag $w $look
        }
        set textw [$w.text text]

        # a hack - RichText widgets should be completely independent
        # of references to theorems etc.  Decoration tags should
        # be added / manipulated via some other interface.
#       $textw tag configure DECORATE:thmname -foreground $TheoremWidgets_flags(color,thmname) -font $TheoremWidgets_flags(font,thmname) -underline 1
#       $textw tag configure DECORATE:thmorigin -foreground $TheoremWidgets_flags(color,thmkind) -font $TheoremWidgets_flags(font,thmkind) 
        $textw tag configure DECORATE:sel \
                -background #b2dfee
        
        
}


proc RichText::configure_look_tag { w look } {
        global RichText_flags
        # puts "w = $w, look = \"$look\""
        if ![regexp {(.*)\|(.*)} $look dummy parta partb] { error "Implementation error: Badly constructed look $look (didn't match ..|..)" }
        # puts "parta = $parta, partb = $partb"
        if ![regexp (.*)-(.*)-(.*)-(.*)-(.*) $parta \
                dummy fontgroup wght slant sWdth sizegroup] { error "Implementation error: Badly constructed look $look (part A didn't match)" }
        if ![regexp (.*)-(.*)-(.*)-(.*)-(.*)-(.*)-(.*)-(.*)-(.*) $partb \
                dummy background bgstipple \
                borderwidth fgstipple foreground relief \
                underline offset justify] { error "Implementation error: Badly constructed look $look (part B didn't match)" }
        lappend data(looks) $look
        set family $RichText_flags(families,$fontgroup)
        set ptSz $RichText_flags(pointsizes,$sizegroup)
        set font "$family-$wght-$slant-$sWdth-*-*-$ptSz-*-*-*-*-*-*"
        # puts "font = $font"
        set args ""
        lappend args -font $font
        if {$background!="*"} { lappend args -background $background }
        if {$bgstipple!="*"} { lappend args -bgstipple $bgstipple }
        if {$borderwidth!="*"} { lappend args -borderwidth $borderwidth }
        if {$fgstipple!="*"} { lappend args -fgstipple $fgstipple }
        if {$foreground!="*"} { lappend args -foreground $foreground }
        if {$relief!="*"} { lappend args -relief $relief }
        if {$underline!="*"} { lappend args -underline $underline }
        if {$offset!="*"} { 
           if {$offset > 100} { 
               lappend args -offset [expr {100-$offset}] 
           } else {
               lappend args -offset $offset
           }
        }
        if {$justify!="*"} { lappend args -justify $justify }
        eval [list $w.text.b tag configure TMLOOK:$look] $args
}



#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------


proc RichText::new_objectid { w } {
        upvar #0 data$w data
        if ![info exists data(objectid_count)] {
            set data(objectid_count) 0
        }
        incr data(objectid_count)
        return OBJECT:$data(objectid_count)
}             


#----------------------------------------------------------------------------
# $w insertobject widget objspec
#           
# When calculating the margins to use for HOL, we allow
# more leeway for richtext as it tends to take up more
# room.
#
# Hmmm.. as it turns out it doesn't seem we need to.  This
# will depend alot on font selection vis a vis the default
# font for the text window.  Even more so on the interface
# mapping that occurs!
#
# This is irritating.  It doesn't seem
# possible to find out the current width of the window
# after resizing except via winfo.  This means we have
# to recalculate the number of pixels/character by
# looking at the config settings and the requested width
# via winfo.
# 
#----------------------------------------------------------------------------

proc RichText::calculate_width { w } {
        set pixwidth [winfo width [$w.text text]]
        set reqwidth [winfo reqwidth [$w.text text]]
        set reqchars [lindex [$w.text.b config -width] 4]
        set width [expr {$pixwidth*$reqchars/$reqwidth - 5}]
        if {$width < 15} { set width 15 }
        return $width
}

composite_subcommand RichText insertobject { w args } {
        upvar #0 $w data
        upvar #0 config$w config
        global RichText_flags
        global TheoremWidgets_flags
        set textw [$w.text text]
        set oldstate [$textw cget -state]
        $textw config -state normal
        for {set i 0} {$i<[llength $args]} {incr i} {
            switch -- [lindex $args $i] -after {
                incr i
                set after [lindex $args $i]
            } default {
                if [info exists objspec] { error "too many arguments to insertobject" }
                set objspec [lindex $args $i]
            }
        }
        if ![info exists objspec] {
            error "too few arguments to insertobject - objspec must be given"
        }
        set objectid [RichText::new_objectid $w]
        # puts "new objectid = $objectid"
        if [info exists after] {
            # puts "after = $after"
            set firstindex [$textw index "$after.last + 2 c"]
        } else {
            set firstindex 1.0
        }


        # insert a newline and get rid of any spurious tags which will 
        # hang aroound when we insert new text.  Then position insert
        # cursor just before the newline character we've just added.

        $textw mark set insert $firstindex
        $textw insert insert \n
        set afternewline [$textw index insert]
        $textw mark set insert $firstindex

        upvar #0 $config(richtextVar) richtext
        upvar #0 $config(structuredtextVar) structuredtext

        set flags(width) [RichText::calculate_width $w]
        set flags(richtext) $richtext
        set flags(structuredtext) $structuredtext
        foreach flagvar $data(flagvars) {
            upvar #0 [lindex $flagvar 1] var
            set flags([lindex $flagvar 0]) $var
        }
        
        # try and get the header - this need not be defined
        if ![catch {set text [[lindex $objspec 0]::header [lindex $objspec 1] flags]}] {
            RichText::InsertStructuredText $w insert text $objectid
        }
        set text [[lindex $objspec 0]::generate [lindex $objspec 1] flags]
        lappend text [list "\n"]
        RichText::InsertStructuredText $w insert text $objectid

        $w yviewobject $objectid
        set data($objectid,spec) $objspec
        incr data(numobjects)
        $textw config -state $oldstate
        return $objectid
}


#----------------------------------------------------------------------------
# Adding flags
#----------------------------------------------------------------------------

composite_subcommand RichText addflagvar { w flagname flagvar } {
        upvar #0 $w data
        lappend data(flagvars) [list $flagname $flagvar]
}

#----------------------------------------------------------------------------
# Object editing
#----------------------------------------------------------------------------

composite_subcommand RichText getall { w } {
        set textw [$w.text text]
        return [string trim [$textw get 1.0 end]]
}

composite_subcommand RichText getobject { w objectid } {
        set textw [$w.text text]
        set ranges [$textw tag ranges $objectid]
        if {[llength $ranges]==0} { error "$object does not exist" }
        set first [lindex $ranges 0]
        puts "getobject, ranges = $ranges"
        set last [lindex $ranges [expr [llength $ranges ]-1]]
        return [string trim [$textw get $first $last]]
}

composite_subcommand RichText getsubobject { w subobject  } {
        set textw [$w.text text]
        set object [lindex $subobject 0]
        set path [lindex $subobject 1]
        puts "getsubobject, object = $object, path = $path"
        set ranges [RichText::range_of_subobject $branchlist]
        set first [lindex $ranges 0]
        puts "getsubobject, ranges = $ranges"
        set last [lindex $ranges 1]
        return [string trim [$textw get $first $last]]
}


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

composite_subcommand RichText yviewobject { w tag } {
        set text [$w.text text]
        # objects need not have *any* characters, hence this will sometimes fail
        catch {$text yview "$tag.first -1 lines"}
}


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

composite_subcommand RichText reformatall { w args } {
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

composite_subcommand RichText deleteobject { w tag } {
        set text [$w.text text]
        set oldstate [$text cget -state]
        $text config -state normal
        $text delete "$tag.first - 1 c" "$tag.last + 1 c"
        $text config -state $oldstate
        selection clear $w
}

composite_subcommand RichText deleteall { w  } {
        upvar #0 $w data
        set text [$w.text text]
        set oldstate [$text cget -state]
        set data(numobjects) 0
        $text config -state normal
        selection clear $w
        $text delete 1.0 end
        $text config -state $oldstate
}

composite_subcommand RichText text { w  args } {
        if {[llength $args] > 0} {
            eval $w.text text $args
        } else {
            return [$w.text text]
        }
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc RichText::InsertStructuredText { w atindex textitemsvar extra_tags } {
        global RichText_flags
        upvar #0 config$w config
        upvar #0 $config(structuredtextVar) structuredtext
        upvar #0 $config(richtextVar) richtext
        upvar 1 $textitemsvar textitems
        # puts "RichText::InsertStructuredText, textitems = $textitems"

        set textw [$w.text text]

        $textw mark set insert $atindex
        $textw yview -pickplace "insert - 3 lines"
        set start [$textw index insert]
        foreach textitem $textitems {
            set text [lindex $textitem 0]
            if {$text==""} continue
            set where [lindex $textitem 2]
            set tags ""
            set look [lindex $textitem 1]
            if {$look != ""} {
                if [catch {set dummy [$textw tag cget TMLOOK:$look -font]}] { set dummy "" }
                # puts "look = $look, dummy = $dummy"
                if {$dummy==""} { RichText::configure_look_tag $w $look }
                lappend tags TMLOOK:$look 
            }
            if $structuredtext {
               if {$where!=""} { lappend tags TMWHERE:$where }
            }
            $textw insert insert $text $tags
            if $structuredtext {
               if {$where != ""} {  $textw tag lower TMWHERE:$where }
            }
        }
        foreach extra_tag $extra_tags {
            $textw tag add $extra_tag $start "insert -1 c"
        }
}
        
#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc RichText::InsertPlainText { w atindex text extra_tags } {
        global RichText_flags
        set textw [$w.text text]
        set oldstate [$textw cget -state]
        $textw mark set insert $atindex
        $textw yview -pickplace "insert - 3 lines"
        $textw config -state normal
        lappend extra_tags $RichText_flags(default_look)
        $textw insert insert $text $extra_tags
        $textw config -state $oldstate
}


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc RichText::tag_range { w tag first last } {
        # puts "first = $first, last = $last"
        set textw [$w.text text]

        $textw tag remove $tag 1.0 end
        regexp {([^\.]*)\.(.*)} $first dummy froml fromc
        regexp {([^\.]*)\.(.*)} $last dummy tol toc
        #puts "tol = $tol, toc = $toc"
        for {set line $froml} {$line <= $tol} {incr line; } {
            if {($line == $froml)} {
                set from $line.$fromc
            } else {
                set linetext [$textw get $line.0 "$line.0 lineend"]
                regexp "^(\[ \]*).*" $linetext dummy whitespace
                set from "$line.[string length $whitespace]"
            }
            if {$line==$tol} {
                set to "$line.$toc + 1 c"
            } else {
                set to "$line.0 lineend"
            }
        #puts "from = $from, to = $to"
            $textw tag add $tag $from $to
        }
}

composite_subcommand RichText subobject_at { w from to  } {

        set textw [$w.text text]
        global vals
        upvar #0 $w data
        upvar #0 config$w config
        if [$textw compare $from > $to] {
            set temp $from
            set from $to
            set to $temp
        } 
        regexp {([^\.]*)\.(.*)} $from dummy froml fromc
        regexp {([^\.]*)\.(.*)} $to dummy tol toc
        set paths ""
        # puts "froml = $froml, fromc = $fromc, tol = $tol, toc = $toc"
        for {set line $froml} {$line <= $tol} {incr line; } {
           set numchars [lindex [split [$textw index "$line.0 lineend"] .] 1]
           for {set char [expr "($line == $froml)?$fromc:0"] } \
               {($line == $tol)?($char <= $toc):($char < $numchars)} \
               {incr char} {
              set tags [$textw tag names $line.$char]
              # puts "tags at $line.$char = $tags"
              foreach tag $tags {
                 if [info exists tagdone($tag)] continue
                 set tagdone($tag) 1
                 if [regexp ^OBJECT:.*\$ $tag] {
                          if {[info exists object] && $object!=$tag} {
                              # puts "crosses object boundaries"
                              return ""
                          }
                          set object $tag
                 }
                 if [regexp ^TMWHERE:(.*)\$ $tag blah path] {
                     lappend paths $path
                          if {![info exists shortest_path] || [string length $shortest_path]>[string length $path]} {
                             set shortest_path $path
                          }
                 }
              }
           }
        }
        
        #------------------------------ 
        # We now have all the relevant paths, and the shortest of
        # all these as well.  shortest_path is a good upper-bound
        # on the longest common path.  
        # Starting with the shortest_path we keep making the path
        # one shorter until we find one that matches with all
        # the paths, which will be the longest common path.
        #------------------------------ 
        
        if ![info exists object] {
            return ""
        }
        # puts "paths = $paths, shortest_path = $shortest_path"
        if {$paths == ""} { return "" }
        set shortest_path_list [split $shortest_path -]
        for {set i [llength $shortest_path_list]} {$i > 0} {incr i} {
           set longest_common_path [join [lrange $shortest_path_list 0 [expr $i-1]] -]
           set ok 1
           foreach path $paths {
               # these string matches check to see if the current longest_common_path candidate 
               # matches the head of "path"
               if {$longest_common_path!=$path && ![string match [set longest_common_path]-* $path]} {
                  set ok 0
                  break
               }
           }
           if $ok break
        }
        if {$i==0} {
           set longest_common_path ""
        }
        return [list $object [split $longest_common_path -]]
}

#------------------------------ 
# compute the smallest range of text over the object
# which is covered by longest_common_path.
#------------------------------ 
composite_subcommand RichText range_of_subobject { w object branchlist } {
        upvar #0 $w data
        upvar #0 config$w config
        set branchtag [join $branchlist -]
        set textw [$w.text text]

        # optimization when 1 object is shown, and using focus interaction
        # model.  This is a hack.  It doesn't work for non-focus
        # interaction as sub-objects are not leaves in other cases,
        # whereas in focus interaction sub-objects are always
        # leaves.  This stuff needs to be reorganised anyway, but now I
        # understand what's going on at least.

        if {$data(numobjects)==1 && $config(focusVar)!=""} {
            #puts "branchtag = $branchtag"
            set ranges [$textw tag ranges TMWHERE:$branchtag] 
            if {[llength $ranges]==0} { error "sub-object $branchtag of $object does not exist (1)" }
            set first [lindex $ranges 0] 
            #puts "ranges = $ranges"
            set last [$textw index "[lindex $ranges [expr [llength $ranges ]-1]] - 1 c"]
            return [list $first $last]
        }

        set object_ranges [$textw tag ranges $object]
        # puts "object_ranges = $object_ranges"
        set on_first 1
        set first_set 0
        set last_set 0
        foreach object_range $object_ranges {
            if $on_first {
               set object_from $object_range
               set on_first 0
               continue
            } else {
               set object_to $object_range
               set on_first 1
            } 
            
            regexp {([^\.]*)\.(.*)} $object_from dummy froml fromc
            regexp {([^\.]*)\.(.*)} $object_to dummy tol toc
            
            # puts "object_from = $object_from, object_to = $object_to"
            for {set line $froml} {$line <= $tol} {incr line; } {
               set numchars [lindex [split [$textw index "$line.0 lineend"] .] 1]
               # puts "line = $line, numchars = $numchars"
               for {set char [expr {($line == $froml)?$fromc:0}] } \
                   {($line == $tol)?($char < $toc):($char < $numchars)} \
                   {incr char} {
                  set tags [$textw tag names $line.$char]
                  # puts "line = $line, char = $char, tags = $tags, branchtag = $branchtag"
                  if {[lsearch -exact $tags TMWHERE:[set branchtag]]!=-1 ||
                      ($branchtag!="" && [lsearch -glob $tags TMWHERE:[set branchtag]-*]!=-1) ||
                      ($branchtag=="" && [lsearch -glob $tags TMWHERE:*]!=-1)} {
                      if {!$first_set || [$textw compare $line.$char < $first]} {
                          set first $line.$char
                          set first_set 1
                      }
                      if {!$last_set || [$textw compare $line.$char > $last]} {
                          set last $line.$char
                          set last_set 1
                      }
                  }
               }
            }
        }    
        if ![info exists first] { error "sub-object $branchtag does not exist" }
        # puts "first = $first, last = $last"
        
        set res [list $first $last]
        return $res

}

composite_subcommand RichText tag_subobject { w subobject tag } {
        $w text tag raise $tag
        set object [lindex $subobject 0]
        set path [lindex $subobject 1]
        set firstlast [$w range_of_subobject $object $path]
        RichText::tag_range $w $tag [lindex $firstlast 0] [lindex $firstlast 1]
}



#--------------------------------------------------------------------------
# normal-selection interaction model
#--------------------------------------------------------------------------
proc RichText::bind_for_normal_selection { w } {
        set textw [$w.text text]
        tk_bindForTraversal $textw
        bindtags $textw [list Text $textw [winfo toplevel $w] all]

        $textw tag remove DECORATE:cursor 1.0 end
        $textw tag remove DECORATE:focus 1.0 end
        $textw tag configure DECORATE:sel -underline 0 -relief raised -borderwidth 1
}


#--------------------------------------------------------------------------
# subobject-selection interaction model
#
# double clicking selects the smallest subobject.
#
#--------------------------------------------------------------------------

proc RichText::select_smallest_subobject { textw from to } {
        set w [winfo parent [winfo parent $textw]]
        global vals
        upvar #0 $w data
        upvar #0 config$w config

        #------------------------------ 
        # find the smallest subobject covering the selection
        #------------------------------ 
        set subobject [$w subobject_at $from $to]
        if {$subobject==""} { return "" }
        set object [lindex $subobject 0]
        set path [lindex $subobject 1]
        
        #------------------------------ 
        # we allow arbitrary selection types to be created by
        # object-class functions "select_subobject".
        #------------------------------ 
        
        selection own $w "catch \{$textw tag remove DECORATE:sel 1.0 end\}; set $config(selectiontypesVar) {}"
        $w tag_subobject $subobject DECORATE:sel
        upvar #0 $config(selectiontypesVar) selectiontypes
        set selectiontypes [[lindex $data($object,spec) 0]::select_subobject $w [lindex $data($object,spec) 1] $path]
}



bind RichTextSubObjectSelection <Double-1> {
    RichText::select_smallest_subobject %W \
        [%W index @%x,%y] [%W index @%x,%y]
}
bind RichTextSubObjectSelection <ButtonPress-1> {
    %W mark set insert @%x,%y
    %W mark set anchor insert
    if {[lindex [%W config -state] 4] == "normal"} {
        focus_goTo %W
    }
}
bind RichTextSubObjectSelection <B1-Motion> {
    RichText::select_smallest_subobject %W \
        [%W index anchor] [%W index @%x,%y]
}

proc RichText::bind_for_subobject_selection { w } {
        upvar #0 $w data
        set textw [$w.text text]
        bindtags $textw [list RichTextSubObjectSelection TextScan $textw [winfo toplevel $w] all]
        $textw tag configure DECORATE:sel -underline 1 \
                -relief flat
        $textw tag remove DECORATE:cursor 1.0 end
        $textw tag remove DECORATE:focus 1.0 end
        $textw tag remove sel 1.0 end
}




#--------------------------------------------------------------------------
# subobject-focus interaction model
# 
# Only leaf nodes may be highlighted by the focus.  We find the
# next node in a particular direction by looking through the text in that
# direction for the first different TMWHERE:* tag.
#
# This kind of assumes only one object is being displayed... It could
# be written for more.
#
# Assumes each node in the text is only tagged with one TMWHERE:* tag,
# which should *always* be the case!
#--------------------------------------------------------------------------

proc tag_is_leaf { tag } {
        set path [split $tag -]
        return [expr {[lindex $path [expr [llength $path]-1]]=="L"}]
}

proc RichText::search { w from origpath include_this_line forward_or_back } {
        # puts "from = $from, origpath = $origpath, include_this_line = $include_this_line, forward_or_back = $forward_or_back"
        set textw [$w text]
        regexp {([^\.]*)\.(.*)} $from dummy froml fromc
        if {$forward_or_back=="back"} {
            set increment -1
            set line_comparison {$line > 0}
            set char_initialisation {set char $numchars}
            set char_comparison {$char > 0}
        } else {
            set increment 1
            set numlines [lindex [split [$textw index end] .] 0]
            set line_comparison {$line <= $numlines}
            set char_initialisation {set char 0}
            set char_comparison {$char <= $numchars}
        }
        # puts "numlines = $numlines"
        set origtag [join $origpath -]
        set textw [$w.text text]
        set line $froml
        set char $fromc
        if {!$include_this_line} { incr line $increment }
        for {} $line_comparison {incr line $increment} {
           set numchars [lindex [split [$textw index "$line.0 lineend"] .] 1]
           if {$line!=$froml} $char_initialisation
           for {} $char_comparison {incr char $increment} {
              set tags [$textw tag names $line.$char]
              #puts "line = $line, char = $char, tags = $tags, origtag = $origtag"
              foreach tag $tags {
                 if [info exists tagdone($tag)] continue
                 set tagdone($tag) 1
                 if [regexp ^OBJECT:.*\$ $tag] {
                     set object $tag
                 }
                 if {[info exists object] && [regexp ^TMWHERE:(.*)\$ $tag blah pathtag]} {
                        if {[tag_is_leaf $pathtag] && $pathtag!=$origtag} {
              # puts "found different tag: $pathtag"
                           return [list $object [split $pathtag -]]
                        }
                 }
              }
           }
        }
        error "search: no object found"
}

proc RichText::search_from_object { w subobject include_this_line forward_or_back } {
        set origobject [lindex $subobject 0]
        set origpath [lindex $subobject 1]
        set origrange [$w range_of_subobject $origobject $origpath]
        if {$forward_or_back=="back"} {
            set from [lindex $origrange 0]
        } else {
            set from [lindex $origrange 1]
        }
        RichText::search $w $from $origpath $include_this_line $forward_or_back
}



proc RichText::cursor_at { w where } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set newcursor [$w subobject_at $where $where]
        # puts "where = $where, newcursor = $newcursor"
        if {$newcursor=="" || $cursor==$newcursor} { return }
        set newpath [lindex $newcursor 1]
        # check the selected subobject ends in L, i.e. that it is cursorable.
        if {[lindex $newpath [expr [llength $newpath]-1]]!="L"} { return }

        set? cursor $newcursor
}

proc RichText::focus_at { w where } {
        upvar #0 config$w config
        upvar #0 $config(focusVar) focus
        set oldfocus $focus
        set newfocus [$w subobject_at $where $where]
        # puts "where = $where, newfocus = $newfocus"
        if {$newfocus=="" || $focus==$newfocus} { return }
        set newpath [lindex $newfocus 1]
        # check the selected subobject ends in L, i.e. that it is focusable.
        if {[lindex $newpath [expr [llength $newpath]-1]]!="L"} { return }

        if [catch {set? focus $newfocus}] {
            set focus $oldfocus
        }
}

proc RichText::focus_at_cursor { w } {
        upvar #0 config$w config
        upvar #0 $config(focusVar) focus
        upvar #0 $config(cursorVar) cursor
        set oldfocus $focus
        if [catch {set? focus $cursor}] {
            set focus $oldfocus
        }
}



proc RichText::cursor_up { w } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set old_cursor $cursor
        # puts "cursor = $cursor"
        catch {set? cursor [RichText::search_from_object $w $old_cursor 0 back]}
        $w text yview "DECORATE:cursor.first - 2 lines"
}
proc RichText::cursor_down { w } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set old_cursor $cursor
        # puts "cursor = $cursor"
        catch {set? cursor [RichText::search_from_object $w $old_cursor 0 forward]}
        $w text yview "DECORATE:cursor.first - 2 lines"
}
proc RichText::cursor_left { w } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set old_cursor $cursor
        catch {set? cursor [RichText::search_from_object $w $old_cursor 1 back]}
        $w text yview "DECORATE:cursor.first - 2 lines"
}
proc RichText::cursor_right { w } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        # puts "cursor = $cursor"
        set old_cursor $cursor
        catch {set? cursor [RichText::search_from_object $w $old_cursor 1 forward]}
        $w text yview "DECORATE:cursor.first - 2 lines"
}

proc RichText::cursor_home { w } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set? cursor [RichText::search $w 1.0 dummy 1 forward]
        $w text yview "DECORATE:cursor.first"
}


proc RichText::focus_change { w args } {
        global errorInfo
        upvar #0 config$w config
        upvar #0 $config(focusVar) focus
        upvar #0 $config(cursorVar) cursor
        set textw [$w text]
        catch {$textw tag remove DECORATE:focus 1.0 end}
        if [catch {$w tag_subobject $focus DECORATE:focus} err] {
            puts "error in tag_subobject $errorInfo"; error $err $errorInfo
        }
        update
        set? cursor $focus
        $textw tag raise DECORATE:focus DECORATE:cursor
        $textw yview "DECORATE:focus.first - 2 lines"
}

proc RichText::cursor_change { w args } {
        global errorInfo
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set textw [$w text]
        catch {$textw tag remove DECORATE:cursor 1.0 end}
        if [catch {$w tag_subobject $cursor DECORATE:cursor} err] {
            # puts "error in tag_subobject $errorInfo"; error $err $errorInfo
        }
        $textw tag raise DECORATE:focus DECORATE:cursor
#       $textw yview "DECORATE:cursor.first - 2 lines"
}



#--------------------------------------------------------------------------
# The bindings - leaffocus
#--------------------------------------------------------------------------

bind RichTextLeafMovement <ButtonPress-1> {
        focus_goTo %W
        RichText::cursor_at [winfo parent [winfo parent %W]] [%W index @%x,%y]
}
bind RichTextLeafMovement <B1-Motion> {
        RichText::cursor_at [winfo parent [winfo parent %W]] [%W index @%x,%y]
}
bind RichTextLeafMovement <ButtonRelease-1> {
        # puts "in RichTextLeafMovement <ButtonRelease-1>"
        RichText::focus_at [winfo parent [winfo parent %W]] [%W index @%x,%y]
}
bind RichTextLeafMovement <Up> {
        RichText::cursor_up [winfo parent [winfo parent %W]]
}
bind RichTextLeafMovement <Left> {
        RichText::cursor_left  [winfo parent [winfo parent %W]]
}
bind RichTextLeafMovement <Down> {
        RichText::cursor_down  [winfo parent [winfo parent %W]]
}
bind RichTextLeafMovement <Right> {
        RichText::cursor_right  [winfo parent [winfo parent %W]]
}
bind RichTextLeafMovement <Home> {
        RichText::cursor_home  [winfo parent [winfo parent %W]]
}
bind RichTextLeafMovement <Return> {
        RichText::focus_at_cursor  [winfo parent [winfo parent %W]]
}
bind RichTextLeafMovement <FocusIn> { 
        %W tag configure DECORATE:cursor -borderwidth 4
        %W tag configure DECORATE:focus -borderwidth 4 -relief ridge 
}
bind RichTextLeafMovement <FocusOut> { 
        %W tag configure DECORATE:cursor -borderwidth 2
        %W tag configure DECORATE:focus -borderwidth 2 -relief ridge 
}


proc RichText::bind_for_leaffocus { w } {
        upvar #0 $w data
        set textw [$w.text text]
        upvar #0 config$w config
        upvar #0 $config(focusVar) focus
        upvar #0 $config(cursorVar) cursor

        bindtags $textw [list RichTextLeafMovement TextScan $textw [winfo toplevel $w] all]

        foreach trace [trace vinfo focus] {
            eval trace vdelete focus $trace
        }
        foreach trace [trace vinfo cursor] {
            eval trace vdelete cursor $trace
        }
        set focus ""
        trace variable focus w "RichText::focus_change $w"
        trace variable cursor w "RichText::cursor_change $w"
        $textw tag remove DECORATE:sel 1.0 end
        $textw tag remove sel 1.0 end
        $textw tag configure DECORATE:focus \
                -background #b2dfee \
                -relief ridge \
                -borderwidth 2
        $textw tag configure DECORATE:cursor \
                -background [$textw cget -bg] \
                -relief ridge \
                -borderwidth 3
}



#--------------------------------------------------------------------------
# The bindings - leafedit
#--------------------------------------------------------------------------

proc RichText::editfocus_change { w args } {
        global errorInfo
        upvar #0 config$w config
        upvar #0 $config(focusVar) focus
        upvar #0 $config(cursorVar) cursor
        set textw [$w text]
        catch {$textw tag remove DECORATE:focus 1.0 end}
        if [catch {$w tag_subobject $focus DECORATE:focus} err] {
            puts "error in tag_subobject $errorInfo"; error $err $errorInfo
        }
        $w.text.b config -state normal
        bindtags $textw [list RichTextLeafEdit TextConstrainedEdit TextScan $w [winfo toplevel $w] all]
        set ranges [$textw tag ranges DECORATE:focus]
        set ranges_length [llength $ranges]
        set ranges_last [lindex $ranges [expr {$ranges_length-1}]]
        $textw mark set LEFT_LIMIT "[lindex $ranges 0] + 1 c"
        $textw mark gravity LEFT_LIMIT left
        $textw mark set RIGHT_LIMIT "$ranges_last - 1 c"
        $textw mark gravity RIGHT_LIMIT right
        $w.text.b mark set insert LEFT_LIMIT
        update
        
        set? cursor $focus
        $textw tag raise DECORATE:focus DECORATE:cursor
        $textw yview "DECORATE:focus.first - 2 lines"
        $textw tag configure DECORATE:focus
                -background white
}



bind RichTextLeafEdit <ButtonPress-1> {
        focus_goTo %W
        RichText::cursor_at [winfo parent [winfo parent %W]] [%W index @%x,%y]
}
bind RichTextLeafEdit <B1-Motion> {
        RichText::cursor_at [winfo parent [winfo parent %W]] [%W index @%x,%y]
}
bind RichTextLeafEdit <ButtonRelease-1> {
        RichText::focus_at [winfo parent [winfo parent %W]] [%W index @%x,%y]
}
bind RichTextLeafEdit <Control-Up> {
        RichText::cursor_up [winfo parent [winfo parent %W]] ; break
}
bind RichTextLeafEdit <Control-Left> {
        RichText::cursor_left  [winfo parent [winfo parent %W]] ; break 
}
bind RichTextLeafEdit <Control-Down> {
        RichText::cursor_down  [winfo parent [winfo parent %W]] ; break 
}
bind RichTextLeafEdit <Control-Right> {
        RichText::cursor_right  [winfo parent [winfo parent %W]] ; break 
}
bind RichTextLeafEdit <Control-Home> {
        RichText::cursor_home  [winfo parent [winfo parent %W]] ; break
}
bind RichTextLeafEdit <Control-Return> {
        RichText::focus_at_cursor  [winfo parent [winfo parent %W]] ; break
}


proc RichText::bind_for_leafedit { w } {
        upvar #0 $w data
        set textw [$w.text text]
            # until a focus is chosen, we disable the entire widget
        $w.text.b config -state disabled
        upvar #0 config$w config
        upvar #0 $config(focusVar) focus
        upvar #0 $config(cursorVar) cursor

        bindtags $textw [list RichTextLeafMovement TextScan $w [winfo toplevel $w] all]

        foreach trace [trace vinfo focus] {
            eval trace vdelete focus $trace
        }
        foreach trace [trace vinfo cursor] {
            eval trace vdelete cursor $trace
        }
        set focus ""
        trace variable focus w "RichText::editfocus_change $w"
        trace variable cursor w "RichText::cursor_change $w"
        $textw tag remove DECORATE:sel 1.0 end
        $textw tag remove sel 1.0 end
        $textw tag configure DECORATE:focus \
                -background #b2dfee \
                -relief ridge \
                -borderwidth 1
        $textw tag configure DECORATE:cursor \
                -background [$textw cget -bg] \
                -relief ridge \
                -borderwidth 2
}


