(****************************************************************************)
(*                                                                          *)
(*               Copyright 1993, 1994 University of Cambridge               *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : ppboxes.sig                                              *)
(* DESCRIPTION   : Signatures for pretty-printing boxes.                    *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 7th December 1993                                        *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 25th November 1994                                       *)
(****************************************************************************)

(* use "hol90_mlpretty_runtime/src/prettyp/ppboxes.sig"; *)

signature PPSTREAM =
sig
   type ppstream
   val mk_ppstream : {consumer:string->unit,flush:unit->unit,linewidth:int} ->
                     ppstream
   val dest_ppstream : ppstream ->
                       {consumer:string->unit,flush:unit->unit,linewidth:int}
   val clear_ppstream : ppstream -> unit
   val flush_ppstream : ppstream -> unit
end;

signature PPBOXES =
sig
   structure PPStream : PPSTREAM
   structure Scale   : sig
                          val char_size : int
                          val num_of_chars : int -> int
                       end
   type ppbox
   datatype ppindent = ABSOLUTE of int | RELATIVE of int
   val pp_empty_box  : ppbox
   val pp_string_box : string * int -> ppbox
   val pp_h_box      : int -> ppbox list -> ppbox
   val pp_v_box      : ppindent * int -> ppbox list -> ppbox
   val pp_hv_box     : int * ppindent * int -> ppbox list -> ppbox
   val pp_hov_box    : int * ppindent * int -> ppbox list -> ppbox
   val pp_h_sepbox   : ppbox -> (int * ppbox) list -> ppbox
   val pp_v_sepbox   : ppbox -> ((ppindent * int) * ppbox) list -> ppbox
   val pp_hv_sepbox  : ppbox -> ((int * ppindent * int) * ppbox) list -> ppbox
   val pp_hov_sepbox : ppbox -> ((int * ppindent * int) * ppbox) list -> ppbox
   val write_ppbox   : ppbox -> PPStream.ppstream -> unit
   val ppbox_to_string : ppbox -> int -> string
end;

signature SCALE =
sig
   val char_size : int
   val num_of_chars : int -> int
end


signature LABEL_PPBOXES =
sig
   structure PPStream : PPSTREAM
   structure Scale   : SCALE
   type 'a ppbox
   datatype ppindent = ABSOLUTE of int | RELATIVE of int
   val pp_empty_box  : 'a ppbox
   val pp_string_box : 'a -> string * int -> 'a ppbox
   val pp_h_box      : 'a -> int -> 'a ppbox list -> 'a ppbox
   val pp_v_box      : 'a -> ppindent * int -> 'a ppbox list -> 'a ppbox
   val pp_hv_box     : 'a -> int * ppindent * int -> 'a ppbox list -> 'a ppbox
   val pp_hov_box    : 'a -> int * ppindent * int -> 'a ppbox list -> 'a ppbox
   val pp_h_sepbox   : 'a -> 'a ppbox -> (int * 'a ppbox) list -> 'a ppbox
   val pp_v_sepbox   : 'a -> 'a ppbox ->
                       ((ppindent * int) * 'a ppbox) list -> 'a ppbox
   val pp_hv_sepbox  : 'a -> 'a ppbox ->
                       ((int * ppindent * int) * 'a ppbox) list -> 'a ppbox
   val pp_hov_sepbox : 'a -> 'a ppbox ->
                       ((int * ppindent * int) * 'a ppbox) list -> 'a ppbox
   val write_ppbox   : 'a ppbox -> PPStream.ppstream -> unit
   val ppbox_to_string : 'a ppbox -> int -> string

   type 'a text
   val commit_ppbox  : int -> 'a ppbox -> 'a text
   val write_text    : 'a text -> PPStream.ppstream -> unit
   val is_in_text    : int * int -> 'a text * int * int -> bool
   val text_to_string : 'a text -> string
   
   type pos
   val mk_pos : (int * int) -> pos
   val dest_pos : pos -> (int * int)

   (*-----------------------------------------------------------
    * label_at
    *
    *-----------------------------------------------------------*)
    
   exception Label_at
   val label_at      : pos -> 'a text -> 'a
   
   (*-----------------------------------------------------------
    * text_to_texttree
    *
    * Convert the text object to a tree structure where each node
    * of the tree is tagged with a positioned label.
    *-----------------------------------------------------------*)
    
   type 'a textitem
   val mk_textitem : {From: pos,To: pos,Label: 'a,Text: string Portable.option} -> 'a textitem
   val dest_textitem :  'a textitem -> {From: pos,To: pos,Label: 'a,Text: string Portable.option}
   datatype 'a texttree = NODE of ('a textitem * 'a texttree list)
   val text_to_texttree    : 'a text * pos -> 'a texttree Portable.option;
   
end;


signature LABEL_STRING_TABLE = sig
   type label
   val attrib : string -> string -> string
   val lookup : label * string -> (label * string * int) list
   val is : string -> string -> bool
   exception StringTable

end;


signature STRING_TABLE = sig
   type label
   val attrib : string -> string -> string
   val lookup : string -> string * int
   val is : string -> string -> bool
   exception StringTable

end;


