
structure HolTermPathReduction: HolTermPathReduction_sig =
struct
    open HolTermPaths;

    type access_data = int option;
    type matcher = 
		(HolTermPaths.path * term) -> 
		(access_data * (HolTermPaths.path * term));
    type accessor = {	    
		matcher: matcher,
		access_op: string,
		unique_name: string
    };
    type access = {
     		accessor: accessor,
		data: int option
    }
    type path = access list;
		
    fun repeat_collect f fm = 
        let val (x,y) = f fm 
            val (xt,y') = repeat_collect f y
        in  (x::xt,y')
        end handle _ => ([],fm);;
    
    fun repeat_discard f fm = 
    	let val (x,y) = f fm in repeat_discard f y handle _ => (x,y) end;
    
    fun or (f1,f2) fm = 
    	(f1 fm) handle _ => (f2 fm);
    
    exception MATCH_FAIL;
    
    local
    fun SIMPLE_aux [] (path,tm) =
          (NONE,(path,tm))
      | SIMPLE_aux (elem::more_pieces) ((pathhd::more_path),tm) =
        if (elem = pathhd) 
        then case elem of 
            BVAR => SIMPLE_aux more_pieces (more_path,(bvar tm))
          | BODY => SIMPLE_aux more_pieces (more_path,(body tm))
          | RATOR => SIMPLE_aux more_pieces (more_path,(rator tm))
          | RAND => SIMPLE_aux more_pieces (more_path,(rand tm))
          | _ => raise MATCH_FAIL
        else raise MATCH_FAIL
      | SIMPLE_aux _ _ =
          raise MATCH_FAIL
    in
    fun SIMPLE tester pieces (path,tm) =
        if (tester tm) then (SIMPLE_aux pieces (path,tm))
        else (raise MATCH_FAIL);
    end;
        
        
    fun STRIP tester pieces (path,tm) =
       let 
          val (vlist,res) = (repeat_collect (SIMPLE tester pieces)) (path,tm)
       in if (length vlist > 1) then ((hd vlist),res) else raise MATCH_FAIL
       end;
    
    local
    fun ASSOC_aux n tester (path,tm) =
       (if (n > 0) 
       then if (tester tm) 
            then (SOME n, snd (SIMPLE tester [RATOR,RAND] (path,tm)))
            else (SOME n, (path,tm))
       else raise MATCH_FAIL)
       handle _ =>
       let val (_,rest) = (SIMPLE tester [RAND] (path,tm))
       in
           ASSOC_aux (n + 1) tester rest
       end;
    in
    val ASSOC = ASSOC_aux 0
    end;
    
    val accessors_ref = ref [
    (* forall statements *)
     { 
    	access_op="snd o strip_forall",
    	unique_name="STRIP_FORALL",
    	matcher=STRIP is_forall [RAND,BODY]
     },
     { 
    	access_op="#Body o dest_forall",
    	unique_name="BODY_FORALL",
    	matcher=SIMPLE is_forall [RAND,BODY]
     },
     { 
    	access_op="#Bvar o dest_forall",
    	unique_name="BVAR_FORALL",
    	matcher=SIMPLE is_forall [RAND,BVAR]
     },
    
    (* exists statements *)
     { 
    	access_op="snd o strip_forall",
    	unique_name="STRIP_EXIST",
    	matcher=STRIP is_exists [RAND,BODY]
     },
     { 
    	access_op="#Body o dest_forall",
    	unique_name="BODY_EXIST",
    	matcher=SIMPLE is_exists [RAND,BODY]
     },
     { 
    	access_op="#Bvar o dest_forall",
    	unique_name="BVAR_EXIST",
    	matcher=SIMPLE is_exists [RAND,BVAR]
     },
    
    
    (* equality terms *)
     { 
    	access_op="lhs",
    	unique_name="LHS",
    	matcher=SIMPLE is_eq [RATOR,RAND]
     },
     { 
    	access_op="rhs",
    	unique_name="RHS",
    	matcher=SIMPLE is_eq [RAND] 
     },
    
    (* paired terms *)
     { 
    	access_op="#fst o dest_pair",
    	unique_name="FST",
    	matcher=SIMPLE is_pair [RATOR,RAND] 
     },
     { 
    	access_op="#snd o dest_pair",
    	unique_name="RHS",
    	matcher=SIMPLE is_pair [RAND] 
     },
    
    (* let terms *)
     { 
    	access_op="bvar o #func o dest_let",
    	unique_name="BVAR_LET",
    	matcher=SIMPLE is_let [RATOR,RAND,BVAR] 
     },
     { 
    	access_op="body o #func o dest_let",
    	unique_name="BODY_LET",
    	matcher=SIMPLE is_pair [RATOR,RAND,BODY] 
     },
     { 
    	access_op="#arg o dest_let",
    	unique_name="VAL_LET",
    	matcher=SIMPLE is_let [RAND] 
     },
    
    (* pabs terms *)
     { 
    	access_op="#body o dest_pabs",
    	unique_name="BODY_PABS",
    	matcher=repeat (or (SIMPLE is_pabs [RAND,BODY],SIMPLE is_abs [BODY]))
     },
    
    (* conjunct terms *)
     { 
    	access_op="el %n o conjuncts",
    	unique_name="CONJUNCTS",
    	matcher=ASSOC is_conj
     },
     { 
    	access_op="#conj1 o dest_conj",
    	unique_name="CONJ1",
    	matcher=SIMPLE is_conj [RATOR,RAND] 
     },
     { 
    	access_op="#conj2 o dest_conj",
    	unique_name="CONJ2",
    	matcher=SIMPLE is_conj [RAND] 
     },
    
    (* disjunct terms *)
     { 
    	access_op="#disj1 o dest_disj",
    	unique_name="DISJ1",
    	matcher=SIMPLE is_disj [RATOR,RAND] 
     },
     { 
    	access_op="#disj2 o dest_disj",
    	unique_name="DISJ2",
    	matcher=SIMPLE is_disj [RAND] 
     },
    
    (* implication terms *)
     { 
    	access_op="snd o strip_imp",
    	unique_name="STRIP_IMP",
    	matcher=STRIP is_imp [RATOR,RAND]
     },
     { 
    	access_op="#ant o dest_imp",
    	unique_name="ANTECEDENT",
    	matcher=SIMPLE is_imp [RATOR,RAND]
     },
     { 
    	access_op="#conseq o dest_imp",
    	unique_name="CONSEQUENT",
    	matcher=SIMPLE is_imp [RAND]
     },
    
    (* conditional terms *)
     { 
    	access_op="#cond o dest_cond",
    	unique_name="CONDITION_COND",
    	matcher=SIMPLE is_cond [RATOR,RATOR,RAND] 
     },
     { 
    	access_op="#larm o dest_cond",
    	unique_name="LARM_COND",
    	matcher=SIMPLE is_cond [RATOR,RAND] 
     },
     { 
    	access_op="#rarm o dest_cond",
    	unique_name="RARM_COND",
    	matcher=SIMPLE is_cond [RAND] 
     },
    
    (* function applications *)
     { 
    	access_op="rator",
    	unique_name="RATOR",
    	matcher=SIMPLE is_comb [RATOR] 
     },
     { 
    	access_op="rand",
    	unique_name="RAND",
    	matcher=SIMPLE is_comb [RAND] 
     },
    
    (* abstractions *)
     { 
    	access_op="bvar",
    	unique_name="BVAR",
    	matcher=SIMPLE is_abs [BVAR] 
     },
     { 
    	access_op="body",
    	unique_name="BODY",
    	matcher=SIMPLE is_abs [BODY] 
     }
    ];

    fun add_accessor a = (accessors_ref := (a::(!accessors_ref)));;
    fun accessors a = (!accessors_ref);

    exception REDUCE_PATH;	
    fun reduce_term_path_once (full_path,tm) =
        tryfind (fn (accessor' as {access_op,
    		unique_name,
    		matcher}) => 
    	   let val (data,rest) = matcher (full_path,tm)
    	   in ({accessor=accessor',data=data},rest)
	   end
        ) (accessors());
    
    
    val reduce_term_path = (repeat_collect reduce_term_path_once);;
end;
    
(*        
open HolTermPaths;
open HolTermPathReduction;
reduce_term_path ([RAND,BODY],(--`!x. x = 1`--));
reduce_term_path ([RAND],(--`P /\ Q /\ R /\ R'`--));;
reduce_term_path ([RATOR,RAND],(--`P /\ Q /\ R /\ R'`--));;
reduce_term_path ([RAND,RATOR,RAND],(--`P /\ Q /\ R /\ R'`--));;
reduce_term_path ([RAND,RAND,RATOR,RAND],(--`P /\ Q /\ R /\ R'`--));;
reduce_term_path ([RAND,RAND,RAND],(--`P /\ Q /\ R /\ R'`--));;
reduce_term_path ([RAND,BODY,RAND,BODY],(--`!x y. (x = 1) /\ (y = 1)`--));
reduce_term_path ([RAND,BODY,RATOR,RAND],(--`!x. x = 1`--));
reduce_term_path ([RAND,BODY,RAND],(--`!x. x = 1`--));
(--`!x::(\x. x < 3). F`--);
*)

fun resq_accessors (is_resq_forall: term -> bool,
		    is_resq_exists: term -> bool,
		    is_resq_select: term -> bool,
		    is_resq_abs: term -> bool) = 
[
     { 
    	access_op="snd o strip_resq_forall",
    	unique_name="STRIP_RESQ_FORALL",
    	matcher=STRIP is_resq_forall [RAND,BODY]
     },
     { 
    	access_op="#3 o dest_resq_forall",
    	unique_name="BODY_RESQ_FORALL",
    	matcher=SIMPLE is_resq_forall [RAND,BODY]
     },
     { 
    	access_op="#2 o dest_resq_forall",
    	unique_name="PRED_RESQ_FORALL",
    	matcher=SIMPLE is_resq_forall [RATOR,RAND]
     },
     { 
    	access_op="#1 o dest_resq_forall",
    	unique_name="BVAR_RESQ_FORALL",
    	matcher=SIMPLE is_resq_forall [RAND,BVAR]
     },
     { 
    	access_op="snd o strip_resq_exists",
    	unique_name="STRIP_RESQ_EXISTS",
    	matcher=STRIP is_resq_exists [RAND,BODY]
     },
     { 
    	access_op="#3 o dest_resq_exists",
    	unique_name="BODY_RESQ_EXISTS",
    	matcher=SIMPLE is_resq_exists [RAND,BODY]
     },
     { 
    	access_op="#2 o dest_resq_exists",
    	unique_name="PRED_RESQ_EXISTS",
    	matcher=SIMPLE is_resq_exists [RATOR,RAND]
     },
     { 
    	access_op="#1 o dest_resq_exists",
    	unique_name="BVAR_RESQ_EXISTS",
    	matcher=SIMPLE is_resq_exists [RAND,BVAR]
     }
		    
];;
		    



		    val accessors: HolTermPathReduction.accessor list
		end =
		
		ResqAccessors_sig =
signature Resq = 
sig
	val accessors: HolTermPathReduction.accessor list;
end;



val (dest_resq_exists,dest_resq_exists,dest_resq_select,dest_resq_abstract) =
    let fun dest_resq_quan cons s =
         let val check = assert (fn c => #Name(dest_const c) = cons)
         in
           fn tm => (let val {Rator = op1 ,Rand = rand1} = dest_comb tm
                     val {Rator = op2, Rand = c1} = dest_comb op1
                     val _ = check op2
                     val {Bvar = c2,Body = c3} = dest_abs rand1
                     in
                       (c2,c1,c3)
                     end)
        end
    in
    ((dest_resq_quan "RES_EXISTS" "dest_resq_exists"),
     (dest_resq_quan "RES_EXISTS" "dest_resq_exists"),
     (dest_resq_quan "RES_SELECT" "dest_resq_select"),
     (dest_resq_quan "RES_ABSTRACT" "dest_resq_abstract"))
    end ;
val is_resq_exists = can dest_resq_exists;
val is_resq_exists = can dest_resq_exists;
val is_resq_select = can dest_resq_select;
val is_resq_abstract = can dest_resq_abstract;


	val STRIP_RESQ_EXISTS_access = { 
		access_op="snd o strip_resq_exists",
		unique_name="STRIP_RESQ_EXISTS",
		tester=is_resq_exists,
		matcher=[RATOR,RAND,BODY]
	};



