(*--------------------------------------------------------------------------*)
(*                  Copyright (c) Donald Syme 1992                          *)
(*                  All rights reserved                                     *)
(*                                                                          *)
(* Donald Syme, hereafter referred to as `the Author', retains the copyright*)
(* and all other legal rights to the Software contained in this file,       *)
(* hereafter referred to as `the Software'.                                 *)
(*                                                                          *)
(* The Software is made available free of charge on an `as is' basis. No    *)
(* guarantee, either express or implied, of maintenance, reliability,       *)
(* merchantability or suitability for any purpose is made by the Author.    *)
(*                                                                          *)
(* The user is granted the right to make personal or internal use of the    *)
(* Software provided that both:                                             *)
(* 1. The Software is not used for commercial gain.                         *)
(* 2. The user shall not hold the Author liable for any consequences        *)
(*    arising from use of the Software.                                     *)
(*                                                                          *)
(* The user is granted the right to further distribute the Software         *)
(* provided that both:                                                      *)
(* 1. The Software and this statement of rights are not modified.           *)
(* 2. The Software does not form part or the whole of a system distributed  *)
(*    for commercial gain.                                                  *)
(*                                                                          *)
(* The user is granted the right to modify the Software for personal or     *)
(* internal use provided that all of the following conditions are observed: *)
(* 1. The user does not distribute the modified software.                   *)
(* 2. The modified software is not used for commercial gain.                *)
(* 3. The Author retains all rights to the modified software.               *)
(*                                                                          *)
(* Anyone seeking a licence to use this software for commercial purposes is *)
(* invited to contact the Author.                                           *)
(*--------------------------------------------------------------------------*)





(*--------------------------------------------------------------------
 use "packages/TkGoalProof/src/repl_goaltree.sig"; 
 use "packages/TkGoalProof/src/repl_goaltree.sml";
 
 open GoalTrees;
 open GTrees;
 open GTrees.BwdProofs;
 val s = set_goals ([([],--`T /\ T`--)],BWDTHENL [BWDPROOF [BWDTAC ""]]);;
 active_foci s;
 val f = first_focus s;
 at_THENL_start s f;
 propositions_at_THENL_start s f;
 current_tac s f;
 val s = expand true (CONJ_TAC,"CONJ_TAC") f s;
 active_foci s;
 val f = next_focus s f;
 at_THENL_start s f;
 current_tac s f;
 propositions_at_THENL_start s f;
 propositions_at_focus s f;
 current_tac s f;
 focused_case_at_THENL_start s f;
 refocus_at_THENL_start 2 s f;
 focus_is_opened s f;
 focus_is_active s f;

 val s = expand true (REWRITE_TAC [],"REWRITE_TAC []") f s;
 val f = next_focus s f;
 at_THENL_start s f;
 propositions_at_THENL_start s f;
 current_tac s f;
 val s = expand true (REWRITE_TAC [],"ASM_REWRITE_TAC []") f s;
 val x = top_is_proved s;
 final_thm s;
 val p = entire_proof s;
 val s = set_goals ([([],--`T /\ T`--)],p);;
 val s = set_goals ([([],--`T /\ T`--)],[BWDTHENL [BWDPROOF [BWDTAC "CONJ_TAC",BWDTAC "REWRITE_TAC []"]]);;
 val s = set_goals ([([],--`T /\ T`--)],[BWDTHENL [BWDPROOF [BWDTAC "CONJ_TAC",BWDTHENL
   [BWDPROOF [BWDTAC ".."], BWDPROOF [BWDTAC ".."],BWDPROOF [BWDTAC ".."]]]]);;
 val f = first_focus s;
 current_tac s f;
 at_THENL_start s f;
 propositions_at_THENL_start s f;
 current_tac s f;
 val s = expand true (CONJ_TAC,"CONJ_TAC") f s;
 THENL_mismatch s;
 val f = next_focus s f;
 trim s;
 val s = split_at_focus f s;
 val f = next_focus s f;

 THENL_mismatch s;
 adjust_THENLs s;



 val s = set_goalsI (([],--`T /\ T`--),BWDPROOF[BWDTAC ""]);;
 val f = first_focusI s;
 at_THENL_startI s f;
 propositions_at_THENL_startI s f;
 expandI true (CONJ_TAC,"CONJ_TAC") f s;
 propositions_at_focusI override s f;
 next_focusI s f;
 expandI true (REWRITE_TAC [],"REWRITE_TAC []") f s;
 next_focusI s f;
 expandI true (REWRITE_TAC [],"ASM_REWRITE_TAC []") f s;
 top_is_provedI s;
 final_thmI s;
 proof_so_farI s;
 layout_proof (proof_so_farI s);

 *-------------------------------------------------------------------*)

structure GTrees : GTrees_sig = struct

structure BwdProofs = BwdProofs;
structure PartialBwdProofs = PartialBwdProofs;
open BwdProofs;
open PartialBwdProofs;


fun issome (SOME x) = true
  | issome NONE = false

fun GTREES_ERR(func,mesg) =
     raise HOL_ERR{origin_structure = "GoalTrees",
             origin_function = func,
             message = mesg}
fun WRAP(HOL_ERR{origin_structure,origin_function,message},func) = 
     raise HOL_ERR{origin_structure=origin_structure,
                   origin_function=func^origin_function,
                   message=message}
  | WRAP (Fail x,func) = raise Fail (func ^ " - " ^ x)
  | WRAP (x,_) = raise x

fun replicate (x,0) = []
  | replicate (x,n) = x::replicate (x,n-1)

fun replace (1,h::t,new) = 
    new::t
  | replace (n,h::t,new) =
    if (n <= 0) 
    then GTREES_ERR("replace","non-positive index")
    else h::(replace (n-1,t,new))
  | replace _ =
    GTREES_ERR("replace","index too large")

fun remove (1,h::t) = t
  | remove (n,h::t) =
    if (n <= 0) 
    then GTREES_ERR("remove","non-positive index")
    else h::(remove (n-1,t))
  | remove _ =
    GTREES_ERR("remove","index too large")

fun insert_after (0,l,new) = new::l
  | insert_after (n,h::t,new) =
    if (n <= 0) 
    then GTREES_ERR("insert_after","non-positive index")
    else h::(insert_after (n-1,t,new))
  | insert_after _ =
    GTREES_ERR("insert_after","index too large")
fun insert_before (n,l,new) = 
    insert_after(n-1,l,new) handle e => WRAP(e,"insert_before")



(* -------------------------------------------------------------------------
 * Propositions
 *--------------------------------------------------------------------------*)

datatype proposition = PROVED of thm | POSED of goal
fun is_posed (PROVED _) = false
  | is_posed (POSED _) = true
fun is_proved (PROVED _) = true
  | is_proved (POSED _) = false
fun dest_proved (PROVED x) = x
  | dest_proved (POSED _) =
      GTREES_ERR("dest_proved", "")
fun dest_posed (POSED x) = x
  | dest_posed (PROVED _) =
      GTREES_ERR("dest_posed","");

(* -------------------------------------------------------------------------
 * Operations on proofs
 *--------------------------------------------------------------------------*)


fun pop n (BWDPROOF l) = el n l
fun pops (BWDPROOF l) = l
fun plength (BWDPROOF l) = length l
fun pindex ((case',pos),BWDTHENL cases) = 
    (pop pos (el case' cases)
     handle _ => GTREES_ERR("pindex","bad focus"))
  | pindex _ = GTREES_ERR("pindex","bad focus")
fun preplace (n,BWDPROOF l,r) = BWDPROOF (replace (n,l,r))
fun pinsert_before (n,BWDPROOF l,r) = BWDPROOF (insert_before (n,l,r))
fun pinsert_after (n,BWDPROOF l,r) = BWDPROOF (insert_after (n,l,r))
fun premove (n,BWDPROOF l) = BWDPROOF (remove (n,l))

fun op_replace ((case',pos),BWDTHENL cases,bwdop) =
    (BWDTHENL (replace (case',cases,preplace (pos,el case' cases,bwdop)))
     handle e => WRAP(e,"op_replace"))
  | op_replace _ = GTREES_ERR("op_replace","bad focus")
fun op_remove ((case',pos),BWDTHENL cases) =
    (BWDTHENL (replace (case',cases,premove (pos,el case' cases)))
     handle e => WRAP(e,"op_remove"))
  | op_remove _ = GTREES_ERR("op_remove","bad focus")
fun op_insert_before ((case',pos),BWDTHENL cases,bwdop) =
    (BWDTHENL (replace (case',cases,pinsert_before (pos,el case' cases,bwdop)))
     handle e => WRAP(e,"op_insert_before"))
  | op_insert_before _ = GTREES_ERR("pinsert","bad focus")
fun op_insert_after ((case',pos),BWDTHENL cases,bwdop) =
    (BWDTHENL (replace (case',cases,pinsert_after (pos,el case' cases,bwdop)))
     handle e => WRAP(e,"op_insert_after"))
  | op_insert_after _ = GTREES_ERR("op_insert_after","bad focus")

type focus = (int * int) list;

fun op_modify f (p:bwdop,[]) = f p
  | op_modify f (proof,(index::rest)) =
        let val bwdop = pindex (index,proof)
            val bwdop' = op_modify f (bwdop,rest)
        in op_replace (index,proof,bwdop')
        end

fun op_lookup (bwdop,[]) = bwdop
  | op_lookup (proof,(index::rest)) = 
       op_lookup (pindex (index,proof),rest)

(* -------------------------------------------------------------------------
 * Some higher-level operations on bwdproofs/bwdops
 *--------------------------------------------------------------------------*)

fun op_next' (p,[],f) = f
  | op_next' (BWDTHENL cases,((casenum,pos)::rest),f) = 
       let val case' = el casenum cases
       in if (pos = plength case') 
          then op_next' (pop pos case',rest,f)
          else op_next' (pop pos case',rest,SOME (pop (pos+1) case'))
       end
  | op_next' _ = 
        GTREES_ERR("op_next","bad focus")
fun op_next (p,focus) = op_next' (p,focus,NONE)

fun op_prev' (p,[],f) = f
  | op_prev' (BWDTHENL cases,((casenum,pos)::rest),f) = 
       let val case' = el casenum cases
       in if (pos = 1) 
          then op_next' (pop pos case',rest,f)
          else op_next' (pop pos case',rest,SOME (pop (pos-1) case'))
       end
  | op_prev' _ = 
        GTREES_ERR("op_prev","bad focus")
fun op_prev (p,focus) = op_prev' (p,focus,NONE)

fun op_insert_space_before_focus (bwdop,focus) =
   let val (front,back) = front_n_back focus
       fun insertor bwdop = op_insert_before (back,bwdop,BWDTAC "........")
   in op_modify insertor (bwdop,front)
   end

fun op_insert_space_after_focus (bwdop,focus) = 
   let val (front,back) = front_n_back focus
       fun insertor bwdop = op_insert_after (back,bwdop,BWDTAC "........")
   in op_modify insertor (bwdop,front)
   end

fun op_split_at_focus (bwdop,focus,n) =
   let fun splitter bwdop = 
         if (n=1) then bwdop else BWDTHENL (replicate (BWDPROOF [bwdop],n))
   in op_modify splitter (bwdop,focus)
   end

(* -------------------------------------------------------------------------
 * Backward proofs, with no idea of undo 
 *--------------------------------------------------------------------------*)

datatype gtree = 
    UNOPENED of goal
  | OPENED of goal * gtree list * (thm list -> thm);
fun is_unopened (OPENED _) = false
  | is_unopened (UNOPENED _) = true
fun is_opened (OPENED _) = true
  | is_opened (UNOPENED _) = false
exception DEST_UNOPENED
exception DEST_OPENED
fun dest_opened (OPENED x) = x
  | dest_opened _ = raise DEST_OPENED
fun dest_unopened (UNOPENED x) = x
  | dest_unopened _ = raise DEST_UNOPENED
type pstate = BwdProofs.bwdop * gtree list;
fun modify_proof f (a,b) = (f(a,b),b)

fun node_is_proved (UNOPENED _) = false
  | node_is_proved (OPENED (_,subtrees,_)) = all node_is_proved subtrees;
fun node_is_unproved x = not (node_is_proved x)

fun goal_at_node (UNOPENED gl) = gl
  | goal_at_node (OPENED (gl,_,_)) = gl;
fun proposition_at_node node = 
   if (node_is_proved node) 
   then PROVED (mk_thm (goal_at_node node))
   else POSED (goal_at_node node);


fun single x = [x];

(* -------------------------------------------------------------------------
 * nodes_from_proof
 * modify_nodes_from_proof
 *
 * Each application of a tactic opens one or more nodes in the tree,
 * producing more nodes.  This function returns the nodes produced
 * by applying a whole series of tactics to a list of nodes.
 * Completed nodes (i.e. nodes with no unopened subnodes)
 * are not returned.
 *
 * If the focus is on the right of a THEN or THENL, then the entire left
 * side must have been applied, producing a set of subgoals which the
 * right hand side operates on.  This function finds those subgoals.
 * 
 * Return the list of sub-goaltrees made by applying the given proof.
 * It fails if the proof has not actually been applied - i.e.
 * if there is a mismatch between opened nodes and tactic applications,
 * or between THENL branch counts.
 *
 *
 * modify_nodes_from_proof replaces the set of nodes 
 * with a new set of nodes.  f must
 * return the same number of nodes.  Used when tactics are actually
 * applied.
 *--------------------------------------------------------------------------*)

(*  (op @) (e, [x1, ..., xn])  ===>  ((e @ x1) @ x2) ... @ xn
    for operators that associate to the left (TAIL RECURSIVE)*)
fun foldl f  =
  let fun itl ([],e)  = e
        | itl (a::l,e) = itl (l,f(a,e))
  in  itl end;

exception PROOF_NOT_FULLY_APPLIED
exception THENL_MISMATCH
fun nodes_from_bwdop (BWDTAC _, subtrees) = 
     (flatten (map (#2 o dest_opened) subtrees)
      handle DEST_OPENED => raise PROOF_NOT_FULLY_APPLIED)
  | nodes_from_bwdop (BWDTHENL cases,gtrees) =
      if (length gtrees = length cases)
      then flatten (map nodes_from_bwdproof  (combine (map pops cases,map single gtrees)))
      else raise THENL_MISMATCH
and nodes_from_bwdproof x = foldl nodes_from_bwdop x


val bwdop_is_applied = can nodes_from_bwdop

fun distribute new_subnodes [] = []
  | distribute new_subnodes (h::t) =
       let val (gl,unopened_subnodes,validation) = dest_opened h
           val (front,back) = split_after (length unopened_subnodes) new_subnodes
       in OPENED (gl,front,validation)::distribute back t
       end;

fun apply_tactic_to_nodes f gtrees =
   let val subnodes = flatten (map (#2 o dest_opened) gtrees)
       val new_subnodes = f subnodes
   in distribute new_subnodes gtrees
   end;

fun modify_nodes_from_bwdop f (BWDTAC _, gtrees) =
      if exists is_unopened gtrees
      then (* tactic has not been applied, otherwise nodes would be opened *)
         raise PROOF_NOT_FULLY_APPLIED
      else apply_tactic_to_nodes f gtrees
  | modify_nodes_from_bwdop f (BWDTHENL cases,gtrees) =
      if (length gtrees = length cases)
      then flatten (map (modify_nodes_from_bwdproof f) (combine (map pops cases,map single gtrees)))
      else raise THENL_MISMATCH
and modify_nodes_from_bwdproof f = foldl (modify_nodes_from_bwdop f)


(* -------------------------------------------------------------------------
 * active_foci
 * 
 * Return the foci in the proof which are "active", in the sense
 * that a tactic may be expanded at that focus.  This is how we generate
 * the list of possible next-foci.
 *
 * next_focus
 *
 * Find the focus which looks most like the previous focus.  Done by sorting
 * all the active foci according to better_match.
 *
 * focus_match_score
 *
 * "score" the focus against the old focus, i.e. work out how well it
 * matches.  Returns a pair (length,last).  
 *    - <length> indicates the length of the match
 * of cases *and* positions, with a final location counting if the
 * case matches but the position doesn't.
 * (e.g. oldfocus = [(1,2),(2,3)], newfocus = [(1,2),(3,1)] scores length 1) 
 *     - <casediff> indicates the "case difference" of match just
 * beyond the last matching case.
 * (e.g. oldfocus = [(1,2),(2,2)], newfocus = [(1,2),(3,1)] scores casediff 1) 
 * (e.g. oldfocus = [(1,2),(2,2)], newfocus = [(1,2),(1,1)] scores casediff -1) 
 *
 * A higher length-match is preferred when choosing a focus given the
 * previous focus, and low positive casediffs are preferred amongst
 * choices with the same length match.
 *--------------------------------------------------------------------------*)

fun number_up n ((m,h)::t) = (map (fn x => (n,m)::x) h)::(number_up (n+1) t)
  | number_up n [] = []

fun active_foci (BWDTAC _, gtrees) = 
      (* if one gtree is opened all should be.  this indicates the
         tactic has been applied.  Whether or not the goals have been
         solved this is definitely not an active focus. If there are no goals
         then it is also not an active focus - it is in fact a tactic which 
         needs to be trimmed. *)
      if (exists is_opened gtrees orelse null gtrees) 
      then ([]:focus list)
      else [[]]
  | active_foci (BWDTHENL cases,gtrees) =
      flatten (number_up 1 (map (active_pos 1) (combine (map pops cases,map single gtrees))))
and active_pos n ([],gtrees) = (n,[])
  | active_pos n ((h::t),gtrees) = 
        let val step_results = nodes_from_bwdop (h,gtrees)
        in active_pos (n+1) (t,step_results)
        end
        handle PROOF_NOT_FULLY_APPLIED => (n,active_foci (h,gtrees))


infix focus_lt
fun [] focus_lt _ = true
  | ((cse1,pos1)::t1) focus_lt [] = false
  | ((cse1:int,pos1:int)::t1) focus_lt ((cse2,pos2)::t2) = 
       cse1 < cse2 orelse
       (cse1 = cse2 andalso pos1 < pos2) orelse
       (cse1 = cse2 andalso pos1=pos2 andalso t1 focus_lt t2)

fun next_focus (proof,gtrees) focus = 
   let val active = active_foci (proof,gtrees)
       val best = first (fn x => focus focus_lt x) active
                  handle _ => el 1 active
   in best
   end
   handle e => WRAP(e,"next_focus")

fun first_focus pstate = 
   next_focus pstate []

fun focus_is_active pstate focus = 
    mem focus (active_foci pstate)
    handle _ => false


(* -------------------------------------------------------------------------
 * nodes_at_focus
 * modify_nodes_at_focus
 *
 * nodes_at_focus returns the nodes acted on by the RHS of a THEN
 * action.  Completed nodes (i.e. nodes with no unopened subnodes)
 * are not returned.
 *
 * modify_nodes_at_focus produce a new goal tree by 
 * acting on the nodes returned by nodes_at_focus.
 *--------------------------------------------------------------------------*)

fun state_at_focus pstate [] =
      pstate
  | state_at_focus (BWDTHENL cases, gtrees) ((case',pos)::rest) = 
      state_at_pos (pops (el case' cases), [el case' gtrees]) (pos,rest)
  | state_at_focus _ _ = GTREES_ERR("state_at_focus","bad focus through BWDTAC")
and state_at_pos (seq,gtrees) (pos,rest) =
      if (pos = 1) 
      then state_at_focus (hd seq,gtrees) rest
      else let val step_results = nodes_from_bwdop (hd seq,gtrees)
           in state_at_pos (tl seq,step_results) (pos-1,rest)
           end
      handle PROOF_NOT_FULLY_APPLIED => 
      GTREES_ERR("state_at_pos","bad focus: not on active focus")
fun nodes_at_focus pstate = #2 o (state_at_focus pstate)

fun modify_nodes_at_focus f (_, gtrees) [] =
      f gtrees
  | modify_nodes_at_focus f (BWDTHENL cases, gtrees) ((case',pos)::rest) = 
      let val gtree' = el 1 (modify_nodes_at_pos f (pops (el case' cases),[el case' gtrees]) (pos,rest))
      in replace (case',gtrees,gtree')
      end
  | modify_nodes_at_focus _ _ _ = GTREES_ERR("modify_nodes_at_focus","bad focus")
and modify_nodes_at_pos f (seq,gtrees) (pos,rest) = 
      if (pos = 1) 
      then modify_nodes_at_focus f (hd seq,gtrees) rest
      else modify_nodes_from_bwdop 
              (fn nodes => modify_nodes_at_pos f (tl seq,nodes) (pos-1,rest))
              (hd seq,gtrees)
      handle PROOF_NOT_FULLY_APPLIED => 
      GTREES_ERR("modify_nodes_at_pos","bad focus: not on active focus")
      

fun propositions_at_focus pstate focus = map proposition_at_node (nodes_at_focus pstate focus);

fun focus_is_opened pstate focus = 
    exists is_opened (nodes_at_focus pstate focus)
    handle _ => false
   
(* -------------------------------------------------------------------------
 * at_THENL_start
 * nodes_at_THENL_start
 *
 * Users want to see goals in two situations:
 *      (a) After the first part of an "a THEN b" has been executed, and the
 * tactic b is to be applied to all resulting subgoals.  The focus is
 * then on the first operation in b.
 *      (b) The first part of an "a THENL [ b1; b2...]" has been
 * executed, and we want to extract all the subgoals produced,
 * even though we will only operate on one of them.  The focus
 * is on the first operation in one of b1, b2 etc.
 *
 * These are completely separate cases!  
 *
 * The at_THENL_start functions determines if the
 * action is a head action just after a THENL.  
 * If it is the first action it is counted as a THENL.
 *
 * nodes_at_focus returns the nodes acted on by the RHS of a THEN
 * action.  Completed nodes (i.e. nodes with no unopened subnodes)
 * are not returned.
 *
 * modify_nodes_at_focus
 * 
 * Produce a new goal tree by acting on the relevant nodes.
 * using the given function
 *
 * is_last_applied_THENL
 *
 * To fetch the goals produced by a THENL, we keep checking if the 
 * THENL we've just run (i.e. run the lhs of) is the last 
 * such THENL moved through by the focus.
 *
 * focused_case_at_THENL_start
 *
 * Work out which of the propositions returned by nodes_at_THENL_start is
 * the actual node being focused on.
 *
 * refocus_at_THENL_start
 *
 * Return a new focus which points to the first tactic of the given case
 * of the THENL.
 *--------------------------------------------------------------------------*)

fun nodes_at_THENL_start (bwdop, gtrees) focus =
   let val (front,(case',pos)) = front_n_back focus
   in nodes_at_focus (bwdop,gtrees) front
   end
      
fun focused_case_at_THENL_start (bwdop,gtrees) focus =
   let val (front,(case',pos)) = front_n_back focus
   in case'
   end

fun at_THENL_start (proof,gstk) focus = 
   let val (front,(case',pos)) = front_n_back focus
   in (pos = 1)
   end

fun refocus_at_THENL_start newcase (bwdop,gtrees) focus =
   let val (front,(case',pos)) = front_n_back focus
   in front @ [(newcase,1)]
   end

fun propositions_at_THENL_start pstate focus = 
    map proposition_at_node (nodes_at_THENL_start pstate focus)


(* -------------------------------------------------------------------------
 * THENL_mismatch
 *
 * Check the THENL branches match up throughout the proof.  Also check
 * for single-length THENL branches.
 *--------------------------------------------------------------------------*)

fun THENL_mismatch (BWDTAC _, gtrees) = false
  | THENL_mismatch (BWDTHENL cases, gtrees) =
      length cases <> length gtrees orelse
      exists THENL_mismatch_bwdproof (combine (cases,map single gtrees))
and THENL_mismatch_bwdproof (BWDPROOF [], gtrees) = false
  | THENL_mismatch_bwdproof (BWDPROOF (h::t), gtrees) =
      THENL_mismatch_bwdproof (BWDPROOF t, nodes_from_bwdop (h,gtrees))
      handle PROOF_NOT_FULLY_APPLIED => THENL_mismatch (h, gtrees)
           | THENL_MISMATCH => true

      

fun chop n alist = fst (Lib.split_after n alist) handle _ => alist
fun adjust_THENLs (BWDTAC tac, gtrees) = BWDTAC tac
  | adjust_THENLs (BWDTHENL cases, gtrees) =
      let val new_cases = 
         if (length cases <= length gtrees)
         then cases @ replicate (el 1 cases, length gtrees - length cases)
         else chop (length gtrees) cases
      in BWDTHENL (map adjust_THENLs_bwdproof (combine (new_cases,map single gtrees)))
      end
and adjust_THENLs_bwdproof (BWDPROOF [], gtrees) = BWDPROOF []
  | adjust_THENLs_bwdproof (BWDPROOF (h::t), gtrees) =
      case (h,gtrees) of
         (BWDTHENL cases,[_]) =>
            adjust_THENLs_bwdproof (BWDPROOF (pops (el 1 cases) @ t), gtrees)
       | _ =>   
         (BWDPROOF (h::pops (adjust_THENLs_bwdproof (BWDPROOF t, nodes_from_bwdop (h,gtrees))))
          handle PROOF_NOT_FULLY_APPLIED => BWDPROOF (adjust_THENLs (h, gtrees)::t)
               | THENL_MISMATCH => BWDPROOF (adjust_THENLs (h, gtrees)::t))
      


(* -------------------------------------------------------------------------
 * split_at_focus
 *
 * split_at_focus converts the proof from a THEN operation to
 * a THENL operation at the given focus.  The focus must be on the RHS
 * of a THEN, i.e the pos in the last field of the focus can't be zero.
 *--------------------------------------------------------------------------*)

fun split_at_focus focus (bwdop,gtrees) =
   let val nodes = nodes_at_focus (bwdop,gtrees) focus
   in op_split_at_focus (bwdop,focus,length nodes)
   end
   handle x => WRAP(x,"split_at_focus")

(* -------------------------------------------------------------------------
 * delete_at_focus
 *
 * delete the tactic about to be applied at the current focus.
 * contract THEN/THENL branches accordingly.
 *--------------------------------------------------------------------------*)

fun delete_at_focus focus (bwdop,gtrees) =
   let val (front,back) = front_n_back focus
       fun removalist bwdop = op_remove (back,bwdop)
   in op_modify removalist (bwdop,front)
   end
   handle x => WRAP(x,"delete_at_focus")

fun insert_space_before_focus focus (bwdop,gtrees) = 
   op_insert_space_before_focus (bwdop,focus)
fun insert_space_after_focus focus (bwdop,gtrees) = 
   op_insert_space_after_focus (bwdop,focus)

(* -------------------------------------------------------------------------
 * entire_proof
 * entire_partial_proof
 * executed_proof
 * 
 * Return the proof from the proof state, either as a full or partial
 * proof.  In executed_proof trim off all un-run portions first.
 *
 * trim
 * cleanup
 * 
 * "trim" removes *all* un-run parts of a proof and replaces them with
 * a single tactic application "........"
 *
 * "cleanup" removes all unneeded parts (i.e. parts which apply to
 * no goals because some previous tactic solved them).  "cleanup" is
 * invoked after every tactic application (see "expand").
 *--------------------------------------------------------------------------*)



fun soft_bwdproof (BWDPROOF (h::t),gtrees) =
      (let val lresults = nodes_from_bwdop (h,gtrees)
           val PBWDPROOF (focus,l) = soft_bwdproof (BWDPROOF t,lresults)
       in PBWDPROOF(focus+1,bwdop_to_hard_pbwdop h::l)
       end
       handle PROOF_NOT_FULLY_APPLIED => 
       PBWDPROOF (1,(soft_bwdop (h,gtrees))::map bwdop_to_soft_pbwdop t))
   | soft_bwdproof (BWDPROOF [],gtrees) = PBWDPROOF (1,[])
and soft_bwdop (BWDTAC tac,gtrees) = 
       PBWDTAC tac
  | soft_bwdop (BWDTHENL cases,gtrees) = 
       PBWDTHENL (map soft_bwdproof (combine (cases, map single gtrees)))

fun trim_bwdproof (BWDPROOF (h::t),gtrees) =
       let val trimh = trim_bwdop (h,gtrees)
       in BWDPROOF (trimh::pops (trim_bwdproof (BWDPROOF t,nodes_from_bwdop (h,gtrees))))
          handle PROOF_NOT_FULLY_APPLIED => 
          BWDPROOF [trimh]
       end
   | trim_bwdproof (BWDPROOF [],gtrees) = BWDPROOF []
and trim_bwdop (BWDTAC tac,gtrees) = 
       if exists is_opened gtrees
       then BWDTAC tac
       else BWDTAC "........"
  | trim_bwdop (BWDTHENL cases,gtrees) = 
       BWDTHENL (map trim_bwdproof (combine (cases, map single gtrees)))

exception CLEANUP
fun cleanup_bwdproof (BWDPROOF (h::t),gtrees) =
      (let val cleanuph = cleanup_bwdop (h,gtrees)
       in BWDPROOF (cleanuph::pops (cleanup_bwdproof (BWDPROOF t,nodes_from_bwdop (h,gtrees))))
          handle PROOF_NOT_FULLY_APPLIED => BWDPROOF (cleanuph::t)
               | THENL_MISMATCH => BWDPROOF (cleanuph::t)
       end
       handle CLEANUP => 
          BWDPROOF [])
   | cleanup_bwdproof (BWDPROOF [],gtrees) = BWDPROOF []
and cleanup_bwdop (_,[]) = 
       raise CLEANUP
  | cleanup_bwdop (BWDTAC tac,gtrees) = 
       BWDTAC tac
  | cleanup_bwdop (BWDTHENL cases,gtrees) = 
       BWDTHENL (map cleanup_bwdproof (combine (cases, map single gtrees)))
       handle _ => BWDTHENL cases

fun strip_first_thenl (BWDTHENL [x]) = x
  | strip_first_thenl _ = GTREES_ERR("strip_first_thenl","implementation error")
fun add_first_thenl x = (BWDTHENL [x])

fun trim (proof,gtrees) = add_first_thenl (trim_bwdproof (strip_first_thenl proof,gtrees))
fun cleanup (proof,gtrees) = add_first_thenl (cleanup_bwdproof (strip_first_thenl proof,gtrees))
val entire_partial_proof = soft_bwdop 
val executed_proof = trim_bwdop 
val entire_proof = pbwdop_to_bwdop o entire_partial_proof;

(* -------------------------------------------------------------------------
 *
 *--------------------------------------------------------------------------*)

fun current_tac (proof,_) f = 
   case (op_lookup (proof,f)) of
      BWDTAC x => x
    | _ => GTREES_ERR("current_tac","not on tactic node");


fun top_is_proved (_, [gtree]) = node_is_proved gtree
fun initial_goal (_,[gtree]) = goal_at_node gtree

fun set_goals (goals,proof) = 
   let val bool = Type.mk_type{Tyop = "bool", Args = []}
       fun is_bool tm = (Term.type_of tm = bool)
       fun is_prop (asl,w) = all is_bool (w::asl)
   in if (all is_prop goals)
      then (proof,map UNOPENED goals)
      else GTREES_ERR("set_goals","not a proposition; new goal not added")
   end;
val empty_goal = set_goals([([],--`T`--)],BWDTHENL [BWDPROOF [BWDTAC "........"]])

fun validate (OPENED (goal,subgoals,validation)) =
   validation (map validate subgoals)
  | validate _ = GTREES_ERR("validate","proof not finished!");
fun final_thm (_,[gtree]) = validate gtree;


(* -------------------------------------------------------------------------
 * override
 * 
 * if no new subgoals have been produced, i.e. the subgoals have all
 * been proved by applying the tactic, then 
 * simply change the tactic at the focus.
 * 
 * if new subgoals have been produced, then we need to decide what the
 * next operation in the proof which will effect the subgoals is.  If
 * it exists, then we don't need to create a THENL [...] branch.  
 *
 * If it doesn't exist then we create a THENL [...] branch of the
 * right length at the current focus.
 * 
 * We count the goals produced by pretending we do a THEN and calling
 * propositions_at_focus.  
 *--------------------------------------------------------------------------*)

fun override {oldproof,focus,tac,newgtrees} =
       let val opened_nodes = nodes_at_focus (oldproof,newgtrees) focus
           val unopened_nodes = flatten (map (#2 o dest_opened) opened_nodes)
           val oldproof_with_tac = op_modify (fn _ => BWDTAC tac) (oldproof,focus)
           val (front,(case',pos)) = front_n_back focus
           val next_focus = front @ [(case',pos+1)]
       in
          if (length unopened_nodes = 0 orelse (issome (op_next (oldproof,focus))))
          then oldproof_with_tac
          else op_split_at_focus (op_insert_space_after_focus (oldproof_with_tac,focus),next_focus,length unopened_nodes)
       end;


(* -------------------------------------------------------------------------
 * expand
 *--------------------------------------------------------------------------*)

fun expand_nodes (tactic,fail_on_no_change) gtrees =
   let val results = map (Tactical.VALID tactic o goal_at_node) gtrees
   in
       if (fail_on_no_change andalso (flatten (fst (split results)) = map goal_at_node gtrees))
       then GTREES_ERR("expand","goals did not change")
       else map (fn ((glist,validation),gtree) => OPENED (goal_at_node gtree,map UNOPENED glist,validation)) (combine (results,gtrees))
   end
   
fun expand fail_on_no_change (tactic,tac) focus (proof,gtrees) =
     let val newgtrees = modify_nodes_at_focus 
                             (expand_nodes (tactic,fail_on_no_change)) 
                             (proof,gtrees) focus
         val newproof = cleanup (override {oldproof=proof,tac=tac,focus=focus,newgtrees=newgtrees}, newgtrees)
     in (newproof,newgtrees)
     end;

fun close focus (proof,gtrees) =
    
     let fun closer gtree = UNOPENED (goal_at_node gtree)
         val newgtree = el 1 (modify_nodes_at_focus (map closer) (proof,gtrees) focus)
     in (proof,[newgtree])
     end;

       

end;


