

signature EliminateTactics_sig = sig

   (* ---------------------------------------------------------------------
    * ELIMINATE_TAC : tactic
    * VAR_ELIMINATE_TAC : tactic
    * TERM_ELIMINATE_TAC : tactic
    *
    * ELIMINATE_TAC eliminates variables and seemingly safe terms
    * VAR_ELIMINATE_TAC eliminates variables only - ultra safe
    * TERM_ELIMINATE_TAC eliminates terms aggressively,
    * regardless of whether the elimination will get rid of every
    * instance of the free variables in the term.  It also allows
    * the elimination of constants.
    *---------------------------------------------------------------------->*)

    val ELIMINATE_TAC : tactic
    val TERM_ELIMINATE_TAC : tactic
    val VAR_ELIMINATE_TAC : tactic

   (* ================================================================= *)
   (* ABBREV_TAC: term->term->tactic                                    *)
   (* ***********                                                       *)
   (*                                                                   *)
   (* Synopsis: abbreviates a term in the goal by adding an assumption  *)
   (*    that the term equals a variable, and substituting the variable *)
   (*    for all occurrences of the term.                               *)
   (*                                                                   *)
   (* Description: A theorem of the form:                               *)
   (* 		|- ?v. v = tm                                        *)
   (*    is generated, CHOOSE_THEN specializes to v, or if not free,    *)
   (*    a quoted variant of v, and the resultant theorem is used to    *)
   (*    substitute var for all instances of tm in the goal.  Lastly,   *)
   (*    the theorem is added to the assumption list.                   *)
   (*                                                                   *)
   (*               {a1,...,an}, gl                                     *)
   (*    ==========================================  ABBREV_TAC tm v    *)
   (*     {a1[v/tm],...,an[v/tm],(--`tm=v`--)}, gl[v/tm]                *)
   (*                                                                   *)
   (* Failure: fails if arguments do not type match, or 2nd argument is *)
   (*    not a free variable.                                           *)
   (*                                                                   *)
   (* Uses: Substitutes a variable for a large or frequent subterm in   *)
   (*       the goal, to simplify the detection of patterns, etc.       *)
   (*       The original goal is retrieved by substituting with the     *)
   (*       assumption which defines the introduced variable, and       *)
   (*       discarding it.                                              *)
   (*                                                                   *)
   (* Written by: Brian Graham 26.11.90                                 *)
   (*             from a suggestion by Jeff Joyce in Aug, 90.           *)
   (* ================================================================= *)

   val ABBREV_TAC : term -> term -> tactic

    structure GeneralElimination : sig
        type subst_acceptor (* {redex:term, residue:term} -> term list -> bool *)
        val XELIMINATE_TAC : subst_acceptor -> tactic
        
        val accept_eliminatible_var : subst_acceptor
        val accept_eliminatible_term : subst_acceptor
        val accept_any_reused_subst : subst_acceptor
        
        val or : subst_acceptor * subst_acceptor -> subst_acceptor
    end

end;



(*-----------------------------------------------------------------------
 * use "hol90_eliminate/src/eliminate.sig";
 * use "hol90_eliminate/src/eliminate.sml";
 *
 * g `(x = 1) /\ (y = x) ==> (y = 1)`;
 * open EliminateTactics;
 * e (REPEAT STRIP_TAC THEN ELIMINATE_TAC THEN ELIMINATE_TAC)
 *   - should prove goal (eliminating variable)
 *
 * open EliminateTactics;
 *
 * g `(f x = 1) /\ (y = f (x:'a)) ==> (y = 1)`;
 * e (REPEAT STRIP_TAC THEN ELIMINATE_TAC THEN ELIMINATE_TAC)
 *   - should prove goal (eliminating term)
 *
 * g `(f x = 1) /\ (y = f (x:'a)) ==> (y = 1)`;
 * e (REPEAT STRIP_TAC THEN ELIMINATE_TAC THEN VAR_ELIMINATE_TAC)
 *   - should fail (can't eliminate term using VAR_ELIMINATE_TAC)
 *
 * g `(f x = 1) /\ (y = f (x:'a)) ==> (y = 1)`;
 * e (REPEAT STRIP_TAC THEN VAR_ELIMINATE_TAC THEN ELIMINATE_TAC)
 *   - should succeed (eliminate variable first)
 *
 * g `(f x = g x) /\ (y = f (x:'a)) ==> (y = 1)`;
 * e (REPEAT STRIP_TAC THEN ELIMINATE_TAC THEN ELIMINATE_TAC)
 *   - should fail
 *
 * e (REPEAT STRIP_TAC THEN ELIMINATE_TAC THEN TERM_ELIMINATE_TAC)
 *   - should succeed,but leave unproveable goal
 *
 * g `(z-1 = x-2) /\ (x+1 = z+2) /\ ((x,y) = (3,6)) ==> ((x,y) = (z+1,y))`;;
 * e (REPEAT STRIP_TAC THEN ELIMINATE_TAC)
 *   - should fail
 * e (REPEAT STRIP_TAC THEN TERM_ELIMINATE_TAC)
 *
 *----------------------------------------------------------------------*)
structure EliminateTactics : EliminateTactics_sig  =
struct

open Rsyntax;

(* ---------------------------------------------------------------------
 * eliminates_occurrences: term list -> (term * term) -> term -> bool
 * eliminates_all_occurrences: term list -> (term * term) -> term list -> bool
 *
 * Check whether the substitution repl/candidate eliminates all occurrences
 * of all free variables in candidate, when the substitution is applied
 * to a term or a list of terms.  The free variables
 * of candidate should be already computed in the input frees_in_redex.
 *----------------------------------------------------------------------*)

fun eliminates_all_occurrences_tm frees_in_redex subst_spec tm =
   let 
      val after_subst = subst [subst_spec] tm in
      null (intersect (free_vars after_subst) frees_in_redex)
   end;

fun eliminates_all_occurrences_tml frees_in_redex subst_spec tml =
   all (eliminates_all_occurrences_tm frees_in_redex subst_spec) tml;


(* ---------------------------------------------------------------------
 * accept_eliminatible_term : (term,term) -> term list -> bool
 * accept_any_reused_subst : (term,term) -> term list -> bool
 * accept_no_term : (term,term) -> term list -> bool
 *
 * Different substitution-acceptance functions.  Only called
 * when candidate is not a variable.
 * Each decides whether to accept a substitution {redex=candidate,residue=repl},
 * when it is applied applied to the list of terms "terms".
 *---------------------------------------------------------------------->*)

structure GeneralElimination = struct

type subst_acceptor = {redex:term, residue:term} -> term list -> bool;

fun or (f1,f2) x y = f1 x y orelse f2 x y;

fun accept_eliminatible_var (subst_spec:{redex:term,residue:term}) terms =  
    is_var (#redex subst_spec) andalso 
    not (mem (#redex subst_spec) (free_vars (#residue subst_spec)));

fun accept_eliminatible_term subst_spec terms =  
   let 
      val frees_in_redex = free_vars (#redex subst_spec) in
      not (null frees_in_redex) andalso
      exists (free_in (#redex subst_spec)) terms andalso
      eliminates_all_occurrences_tml frees_in_redex subst_spec terms
   end;

fun accept_any_reused_subst (subst_spec:{redex:term,residue:term}) terms =  
      exists (free_in (#redex subst_spec)) terms

(* ---------------------------------------------------------------------
 * choose_subst_asms
 *    chooses from a set of theorems a subset appropriate to use as
 * a basis for elimination - i.e. a set of equality substitutions.
 * Two subsets are returned - the set to use, and the set to not to use
 * Assumptions of the form x = x are thrown away all together.
 * 
 * asm_thms:    the assumptions as theorems 				
 * gl:          the goal as a term 					
 * no_terms:    completely disallows the elimination of terms other than 		
 *	      	variables 					
 * any_terms:   allows elimination of terms which have 		
 *		free variables being used in some other context 	
 *
 *---------------------------------------------------------------------->*)

local
fun captured candidate disallowed =
    not (null (intersect disallowed (free_vars candidate)));
in
fun choose_subst_asms term_acceptor asm_thms gl  =
   let 
      fun choose_rec [] chosen disallowed = ([],[])
        | choose_rec (hdasm::tlasm) chosen disallowed =
         let val c = concl hdasm
             val terms = gl::(map concl (subtract asm_thms [hdasm]))
             fun throw_away () = 
                  choose_rec tlasm chosen disallowed
             fun dont_accept () = 
               let val (tlnosubs, tlsubs)=choose_rec tlasm chosen disallowed in
                  (hdasm::tlnosubs, tlsubs) 
               end
             fun accept_lhs () =
               let val (tlnosubs, tlsubs)=
                  choose_rec 
                   tlasm
                   (union chosen (free_vars (lhs c))) 
                   (union disallowed (free_vars (rhs c))) in
                 (tlnosubs, hdasm::tlsubs) 
               end 
             fun accept_rhs () =
               let val (tlnosubs, tlsubs)=
                 choose_rec 
                   tlasm
                   (union chosen (free_vars (rhs c))) 
                   (union disallowed (free_vars (lhs c))) in
                 (tlnosubs, (SYM hdasm)::tlsubs) 
               end in
          if (is_eq c) andalso null (intersect chosen (free_vars c)) 
          then
(* The following lines determine whether we use a particular assumption as the basis for *)
(* an elimination or not *)
            if (lhs c = rhs c) then throw_away()
	    else if (is_var (lhs c)) andalso not (mem (lhs c) disallowed) then accept_lhs()
            else if (is_var (rhs c)) andalso not (mem (rhs c) disallowed) then accept_rhs()
            else if not (captured (lhs c) disallowed) andalso term_acceptor (lhs c |-> rhs c) terms then accept_lhs()
            else if not (captured (rhs c) disallowed) andalso term_acceptor (rhs c |-> lhs c) terms then accept_rhs()
	    else dont_accept()
         else dont_accept()
       end in
      choose_rec asm_thms [] []
   end;
end;

(* ---------------------------------------------------------------------
 * XELIMINATE_TAC: ( (term * term) -> term list -> bool ) -> tactic
 *
 *---------------------------------------------------------------------->*)

local 
exception ACCEPT_EQ_TAC_FAIL;
fun ACCEPT_EQ_TAC (asms,gl) = 
    if (is_eq gl) andalso (rhs gl = lhs gl) 
    then ACCEPT_TAC (REFL (lhs gl)) (asms,gl)
    else raise ACCEPT_EQ_TAC_FAIL;
in
fun XELIMINATE_TAC term_acceptor (asms,gl) =
   POP_ASSUM_LIST (fn thms =>
      let val (nosub, sub) = choose_subst_asms term_acceptor thms gl in
         if (null sub) 
         then raise (HOL_ERR {
                        message="no terms found to eliminate",
                        origin_function="XELIMINATE_TAC",
                        origin_structure="Eliminate"})
         else (EVERY (rev (map (ASSUME_TAC o (SUBS sub)) nosub)) 
               THEN SUBST_TAC sub
               THEN TRY ACCEPT_EQ_TAC)
      end
   ) (asms,gl)
   handle HOL_ERR{message,origin_function,origin_structure} => 
          raise HOL_ERR {
                   message="XELIMINATE_TAC -- "^message,
                   origin_function=origin_function,
		   origin_structure=origin_structure};
end; (* local *)

end; (* struct General *)

open GeneralElimination;

(* ---------------------------------------------------------------------
 * ELIMINATE_TAC : tactic
 * VAR_ELIMINATE_TAC : tactic
 * TERM_ELIMINATE_TAC : tactic
 *
 * ELIMINATE_TAC eliminates variables and seemingly safe terms
 * VAR_ELIMINATE_TAC eliminates variables only - ultra safe
 * TERM_ELIMINATE_TAC eliminates terms aggressively,
 * regardless of whether the elimination will get rid of every
 * instance of the free variables in the term.  It also allows
 * the elimination of constants.
 *---------------------------------------------------------------------->*)

val ELIMINATE_TAC = XELIMINATE_TAC (or (accept_eliminatible_var,accept_eliminatible_term));

val VAR_ELIMINATE_TAC = XELIMINATE_TAC accept_eliminatible_var;

val TERM_ELIMINATE_TAC = XELIMINATE_TAC accept_any_reused_subst;


(* ================================================================= *)
(* ABBREV_TAC: term->term->tactic                                    *)
(* ***********                                                       *)
(*                                                                   *)
(* Synopsis: abbreviates a term in the goal by adding an assumption  *)
(*    that the term equals a variable, and substituting the variable *)
(*    for all occurrences of the term.                               *)
(*                                                                   *)
(* Description: A theorem of the form:                               *)
(* 		|- ?v. v = tm                                        *)
(*    is generated, CHOOSE_THEN specializes to v, or if not free,    *)
(*    a quoted variant of v, and the resultant theorem is used to    *)
(*    substitute var for all instances of tm in the goal.  Lastly,   *)
(*    the theorem is added to the assumption list.                   *)
(*                                                                   *)
(*               {a1,...,an}, gl                                     *)
(*    ==========================================  ABBREV_TAC tm v    *)
(*     {a1[v/tm],...,an[v/tm],(--`tm=v`--)}, gl[v/tm]                *)
(*                                                                   *)
(* Failure: fails if arguments do not type match, or 2nd argument is *)
(*    not a free variable.                                           *)
(*                                                                   *)
(* Uses: Substitutes a variable for a large or frequent subterm in   *)
(*       the goal, to simplify the detection of patterns, etc.       *)
(*       The original goal is retrieved by substituting with the     *)
(*       assumption which defines the introduced variable, and       *)
(*       discarding it.                                              *)
(*                                                                   *)
(* Written by: Brian Graham 26.11.90                                 *)
(*             from a suggestion by Jeff Joyce in Aug, 90.           *)
(* ================================================================= *)

fun ABBREV_TAC tm var =
 if (is_var var)
 then (if (type_of tm = type_of var)
       then (CHOOSE_THEN (fn th => SUBST_ALL_TAC th THEN ASSUME_TAC th)
                         (EXISTS (mk_exists{Bvar=var,Body=mk_eq{lhs=tm,rhs=var}}
				 ,tm) (REFL tm)))
       else raise Fail "")
 else raise Fail ""

 handle HOL_ERR{message,origin_function,origin_structure} => 
          raise HOL_ERR {
                   message="ABBREV_TAC -- "^message,
                   origin_function=origin_function,
		   origin_structure=origin_structure}
      | _ =>
          raise HOL_ERR {
                   message="ABBREV_TAC",
                   origin_function="ABBREV_TAC",
		   origin_structure="EliminateTactics"};
end;

open EliminateTactics;


