(*============================================================================*)
(* Definition to support schemas =>                                           *)
(*============================================================================*)

new_theory "Z";

(*--------------------------------------------------------------------------*)
(* Load "eliminate" library  						    *)
(*--------------------------------------------------------------------------*)

prim_load_library Lib.interpret {lib=find_library "eliminate",theory="-"};
open EliminateTactics;
prim_load_library Lib.interpret {lib=pred_set_lib,theory="-"};
open Gspec;
open Fset_conv;
open Set_ind;
add_theory_to_sml "pred_set";
add_theory_to_sml "pair";

prim_load_library Lib.interpret {lib=taut_lib,theory="-"};
prim_load_library Lib.interpret {lib=reduce_lib,theory="-"};
prim_load_library Lib.interpret {lib=arith_lib,theory="-"};
(*----------------------------------------------------------------------------*)
(* Rule and Tactic for simplifying terms of the form `x IN {...|...}`         *)
(*----------------------------------------------------------------------------*)

val SET_SPEC_RULE = CONV_RULE(DEPTH_CONV SET_SPEC_CONV)
and SET_SPEC_TAC  = CONV_TAC(DEPTH_CONV SET_SPEC_CONV);

(*----------------------------------------------------------------------------*)
(* Some proof utilities =>                                                      *)
(*----------------------------------------------------------------------------*)

fun APPLY_ASMS_TAC f =
 POP_ASSUM_LIST
  ( fn assums => MAP_EVERY ASSUME_TAC (rev (map f assums)));

val REWRITE_ASMS_TAC = APPLY_ASMS_TAC o REWRITE_RULE;

fun REWRITE_ALL_TAC thl = 
 REWRITE_ASMS_TAC thl THEN ASM_REWRITE_TAC [] THEN REWRITE_TAC thl;

(*----------------------------------------------------------------------------*)
(* RW_ASM_THEN ttac [f1;...;fn] f =                                           *)
(*  ASSUM_LIST( fn thl => ttac(REWRITE_RULE[f1 thl;...;fn thl](f thl)))            *)
(*----------------------------------------------------------------------------*)

fun RW_ASM_THEN ttac fl f =
 ASSUM_LIST( fn thl => ttac(REWRITE_RULE(map ( fn f => f thl) fl)(f thl)));

(*----------------------------------------------------------------------------*)
(* POP_ASSUMS n f = f[a1;...;an],                                             *)
(*                                                                            *)
(* where a1,...,an are the last n assumptions, which are popped =>              *)
(*----------------------------------------------------------------------------*)

fun POP_ASSUMS n f =
 if n=0
 then ALL_TAC
 else if n=1
 then POP_ASSUM( fn th => f[th])
 else POP_ASSUM( fn th => POP_ASSUMS (n-1) ( fn l => f (th::l)));

fun ITER n (tac:tactic)  = 
 if n < 0 then raise (Fail "ITER")
 else if n = 0 then ALL_TAC 
               else tac THEN ITER (n-1) tac;

(*----------------------------------------------------------------------------*)
(*  Resolution with filters =>                                                  *)
(*  Code written for HOL90 by chou@cs.ucla.edu => Ported to HOL88 by MJCG =>      *)
(*----------------------------------------------------------------------------*)

fun FILTER_STRIP_ASSUME_TAC (f : term -> bool) th =
  if (f (concl th)) then (STRIP_ASSUME_TAC th) else (ALL_TAC) ;

fun FILTER_IMP_RES_TAC (f : term -> bool) th g =
  IMP_RES_THEN (REPEAT_GTCL IMP_RES_THEN (FILTER_STRIP_ASSUME_TAC f)) th g
  handle _ => ALL_TAC g ;

fun FILTER_RES_TAC (f : term -> bool) g =
  RES_THEN (REPEAT_GTCL IMP_RES_THEN (FILTER_STRIP_ASSUME_TAC f)) g
  handle _ => ALL_TAC g ;

fun no_imp (tm) = not (free_in (--`==>`--) tm) ;

val LITE_IMP_RES_TAC = FILTER_IMP_RES_TAC no_imp;

(*----------------------------------------------------------------------------*)
(* The following constants are not defined in the logic,                      *)
(* but are translated away by the Z preprocessor => There are declared as       *)
(* constants here to prevent them being defined in descendents of "Z" =>        *)
(*----------------------------------------------------------------------------*)


new_constant{Name="NOT", Ty=(==`:bool->bool`==)};

new_infix{Name="AND",       Ty=(==`:bool->bool->bool`==), Prec=2};
new_infix{Name="OR",        Ty=(==`:bool->bool->bool`==), Prec=2};
new_infix{Name="IMPLIES",   Ty=(==`:bool->bool->bool`==), Prec=2};

new_binder_definition("FORALL", (--`FORALL = ($! :('a -> bool) -> bool)`--));
new_binder_definition("EXISTS", (--`EXISTS = ($! :('a -> bool) -> bool)`--));
new_binder_definition("SCHEMA_FORALL", (--`SCHEMA_FORALL = ($! :('a -> bool) -> bool)`--));
new_binder_definition("SCHEMA_EXISTS", (--`SCHEMA_EXISTS = ($! :('a -> bool) -> bool)`--));

new_infix{Name="SEQ",       Ty=(==`:bool->bool->bool`==), Prec=2};
new_constant{Name="HIDE",  Ty=(==`:bool->bool->bool`==)};
new_constant{Name="DELTA",  Ty=(==`:bool->bool`==)};
new_constant{Name="XI",     Ty=(==`:bool->bool`==)};
new_constant{Name="theta",  Ty=(==`:bool->'a`==)};
new_constant{Name="sig",    Ty=(==`:bool->bool`==)};
new_constant{Name="pred",   Ty=(==`:bool->bool`==)};

(* hide_constant "S"; *)
(* So `S` can be used as a set variable, as in ZRM *)

(*----------------------------------------------------------------------------*)
(* Some additional set theory theorems =>                                       *)
(*----------------------------------------------------------------------------*)


val DIFF_UNION = save_thm("DIFF_UNION", prove(
  (--`!s t (u:'a->bool). s DIFF (t UNION u) = (s DIFF t) DIFF u`--),
  REPEAT STRIP_TAC
   THEN REWRITE_TAC[EXTENSION,IN_DIFF,IN_UNION]
   THEN GEN_TAC
   THEN EQ_TAC
   THEN REPEAT STRIP_TAC
   THEN ASM_REWRITE_TAC[]
   THEN RES_TAC));

(*----------------------------------------------------------------------------
 * Schemas are represented by functions from tuples to booleans.  The
 * variables in the tuples are the free variables in the schema.
 *
 * On input,
 * To convert schema-descriptions to schema representations:
 *     DELTA S -> \(freevarstuple...freevarstuple'). <<S> ZAND <S'>>
 *     XI S -> \(freevarstuple...freevarstuple'). SCHEMA [<S>;<S'>] [] [freevarstuple = freevarstuple']
 *     SCHEMA schemas dec body -> (\(freevartuple). SCHEMA schemas dec body)
 *     SchemaName[primes] -> (\freevarstuple[primes]. SchemaName freevarstuple[primes])
 *     S1 OR S2 -> \(union (freevars S1) (freevars S2)). (S1 (freevars S1)) ZOR (S2 (freevars S2))
 *     S1 AND S2 -> \(union (freevars S1) (freevars S2)). (S1 (freevars S1)) ZAND (S2 (freevars S2))
 *     S1 NOT S2 -> \(union (freevars S1) (freevars S2)). (S1 (freevars S1)) ZNOT (S2 (freevars S2))
 *     FORALL (var::dec) S -> !var::dec. <S>
 *     FORALL BS S -> !(v1...vn). <BS> ==> <S>
 *     EXISTS (var::dec) S -> ?var. <var::dec> ==> <S>
 *     EXISTS BS S -> ?(v1...vn). <BS> ==> <S>
 *     S HIDE -> \(v1...vn). FST (S (v1...vn))
 *     sig S -> FST (S (v1...vn))
 * ???
 *     HIDE (var::dec) S -> \(subtract (freevars S) var). HIDE [...] (?<var::dec> CONJL [...])
 *     HIDE (schema) S -> \(subtract (freevars S) var). HIDE [...] (?(v1::decv1)...(vn::decvn). CONJL [...])
 *     SCHEMA_FORALL BS S -> \(subtract (freevars S) BS). !B1... FST(BS (B1...)) ==> BOTH  SCHEMA_FORALL [...] (!BL. ...)
 *     SCHEMA_EXISTS BL S -> \(subtract (freevars S) BL). SCHEMA_EXISTS [...] (!BL. ...)
 * To "use" a schema:
 *     SchemaRep -> BOTH (SchemaRep boundvartuple)
 * Schemas get "used" inside schemas and in terms
 * schema declarations are scanned for included sc
 *----------------------------------------------------------------------------*)

val CONJL =
new_recursive_definition {
 name = "CONJL",
 fixity = Prefix,
 rec_axiom = theorem "list" "list_Axiom",
 def = (--`(CONJL [] = T) /\ (CONJL(CONS b bl) = b /\ CONJL bl)`--)
};

val BOTH = new_definition("BOTH",(--`BOTH = UNCURRY $/\`--));
val BOTH_CLAUSES = save_thm("BOTH_CLAUSES",prove(
   (--`!t. (BOTH (T,t) = t) /\
           (BOTH (t,T) = t) /\ 
           (BOTH (F,t) = F) /\ 
           (BOTH (t,F) = F)`--),
   REWRITE_TAC [definition "pair" "UNCURRY_DEF", BOTH]));

val ZOR =  new_infix_definition("ZOR",
    (--`ZOR (decl1,body1) (decl2,body2) = 
	      ((decl1 /\ decl2),(body1 \/ body2))`--),300);;

val ZAND =  new_infix_definition("ZAND",
    (--`ZAND (decl1,body1) (decl2,body2) = 
	      ((decl1 /\ decl2),(body1 /\ body2))`--),400);;

val ZIMPLIES =  new_infix_definition("ZIMPLIES",
    (--`ZIMPLIES (decl1,body1) (decl2,body2) = 
	      ((decl1 /\ decl2),(body1 ==> body2))`--),200);;

val ZNOT =  new_definition("ZNOT",
    (--`ZNOT ((decl:bool),(body:bool)) = (decl,~body)`--));;


val SCHEMA = new_definition("SCHEMA",
         (--`SCHEMA schemas decs body = (CONJL (MAP FST schemas) /\ CONJL decs,
                                         CONJL (MAP SND schemas) /\ CONJL body)`--));


(*----------------------------------------------------------------------------*)
(*----------------------------------------------------------------------------*)

val NEQ = 
  new_infix_definition("NEQ",(--`$=/= (x:'a) y = ~(x = y)`--),200);

val CHOICE = save_thm("CHOICE", prove(
  (--`!(s:'a->bool). (s =/= {}) = CHOICE s IN s`--),
  GEN_TAC
   THEN REWRITE_TAC[NEQ]
   THEN EQ_TAC
   THEN REWRITE_TAC[CHOICE_DEF]
   THEN REPEAT STRIP_TAC
   THEN IMP_RES_TAC MEMBER_NOT_EMPTY));

val NOT_IN = 
  new_infix_definition("NOT_IN",(--`$NOT_IN (x:'a) s = ~(x IN s)`--),450);

(*----------------------------------------------------------------------------*)
(* Constants for restricting quantifiers to sets.                            *)
(*----------------------------------------------------------------------------*)

val MAPLET = 
   new_infix_definition("MAPLET",(--`|-> (x:'a) (y:'b) = (x,y)`--),500);

val dom =
   new_definition("dom",
         (--`dom(R:('a # 'b)->bool) = {x | ?y. (x |-> y) IN R}`--));

val dom_UNION = save_thm("dom_UNION", prove(
  (--`!X Y:('a # 'b)->bool. dom(X UNION Y) = dom X UNION dom Y`--),
  REWRITE_TAC[dom,EXTENSION,IN_UNION]
   THEN REPEAT GEN_TAC
   THEN CONV_TAC(DEPTH_CONV SET_SPEC_CONV)
   THEN EQ_TAC
   THEN REPEAT STRIP_TAC
   THENL
    [DISJ1_TAC,
     DISJ2_TAC,
     ALL_TAC,
     ALL_TAC]
   THEN EXISTS_TAC (--`y:'b`--)
   THEN ASM_REWRITE_TAC[]));

val dom_EMPTY = save_thm("dom_EMPTY", prove(
  (--`dom({}:('a # 'b)->bool) = {}`--),
  REWRITE_TAC[dom,NOT_IN_EMPTY,EXTENSION]
   THEN SET_SPEC_TAC
   THEN REWRITE_TAC[]));

val dom_SING = save_thm("dom_SING", prove(
  (--`!(x:'a)(y:'b). dom{x |-> y} = {x}`--),
  REWRITE_TAC[dom,EXTENSION,IN_UNION,IN_SING,MAPLET]
   THEN REPEAT GEN_TAC
   THEN CONV_TAC(DEPTH_CONV SET_SPEC_CONV)
   THEN REWRITE_TAC[PAIR_EQ]
   THEN EQ_TAC
   THEN REPEAT STRIP_TAC
   THEN EXISTS_TAC (--`y:'b`--)
   THEN ASM_REWRITE_TAC[]));

val dom_SING_Cor = save_thm("dom_SING_Cor", prove(
  (--`!(x:'a)(y:'b). x IN dom{x |-> y}`--),
  REWRITE_TAC[dom_SING,IN_SING]));

val ran = new_definition("ran",
       (--`ran(R:('a # 'b)->bool) = {y | ?x. (x|->y) IN R}`--));

val ran_EMPTY = save_thm("ran_EMPTY", prove(
  (--`ran({}:('a # 'b)->bool) = {}`--),
  REWRITE_TAC[ran,NOT_IN_EMPTY,EXTENSION]
   THEN SET_SPEC_TAC
   THEN REWRITE_TAC[]));

val ran_SING = save_thm("ran_SING", prove(
  (--`!(x:'a)(y:'b). ran{x |-> y} = {y}`--),
  REWRITE_TAC[ran,EXTENSION,IN_UNION,IN_SING,MAPLET]
   THEN REPEAT GEN_TAC
   THEN CONV_TAC(DEPTH_CONV SET_SPEC_CONV)
   THEN REWRITE_TAC[PAIR_EQ]
   THEN EQ_TAC
   THEN REPEAT STRIP_TAC
   THEN EXISTS_TAC (--`x:'a`--)
   THEN ASM_REWRITE_TAC[]));

val ran_UNION = save_thm("ran_UNION", prove(
  (--`!X Y:('a # 'b)->bool. ran(X UNION Y) = ran X UNION ran Y`--),
  REWRITE_TAC[ran,EXTENSION,IN_UNION]
   THEN REPEAT GEN_TAC
   THEN CONV_TAC(DEPTH_CONV SET_SPEC_CONV)
   THEN EQ_TAC
   THEN REPEAT STRIP_TAC
   THENL
    [DISJ1_TAC,
     DISJ2_TAC,
     ALL_TAC,
     ALL_TAC]
   THEN EXISTS_TAC (--`x'':'a`--)
   THEN ASM_REWRITE_TAC[]));


val PSET = new_definition("PSET",(--`PSET(X:'a->bool) = {Y | Y SUBSET X}`--));

val DIFF_IN_PSET = save_thm("DIFF_IN_PSET", prove(
  (--`! s t:'a->bool. (s DIFF t) IN PSET s`--),
  REPEAT STRIP_TAC
   THEN REWRITE_TAC[IN_DIFF,IN_UNION,PSET,SUBSET_DEF]
   THEN SET_SPEC_TAC
   THEN ASM_REWRITE_TAC[IN_DIFF]
   THEN REPEAT STRIP_TAC
   THEN ASM_REWRITE_TAC[]));

val CROSS = new_infix_definition("CROSS",
   (--`>< (X:'a->bool) (Y:'b->bool) = {(x,y) | x IN X /\  y IN Y}`--),300);

val BREL = new_infix_definition("BREL",
 (--`<-> (X:'a->bool) (Y:'b->bool) = PSET(X >< Y)`--),900);

val PFUN = new_infix_definition("PFUN",
 (--`-+> (X:'a->bool) (Y:'b->bool) = 
  {f | f IN (X <-> Y) /\  
       !x y1 y2. (x|->y1) IN f /\  (x|->y2) IN f ==> (y1=y2)}`--),800);

val TFUN = new_infix_definition("TFUN",
 (--`--> (X:'a->bool) (Y:'b->bool) = 
  {f | f IN (X -+> Y) /\  (dom f = X)}`--),800);

val domPfun = save_thm("domPfun", prove(
  (--`!f (X:'a->bool) (Y:'b->bool) x. f IN (X -+> Y) /\  x IN dom f ==> x IN X`--),
  REWRITE_TAC[MAPLET,dom,PFUN,BREL,PSET,CROSS,SUBSET_DEF]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN RES_TAC
   THEN ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 3)
   THEN ELIMINATE_TAC
   THEN ASM_REWRITE_TAC[]));

val domPfunIN = save_thm("domPfunIN", prove(
  (--`!f (X:'a->bool) (Y:'b->bool). f IN (X -+> Y)  ==> dom f IN PSET X`--),
  REWRITE_TAC[PSET,SUBSET_DEF]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN IMP_RES_TAC domPfun));

val SING_Pfun = save_thm("SING_Pfun", prove(
  (--`!(x:'a) (y:'b) X Y. x IN X /\  y IN Y ==> {x |-> y} IN (X -+> Y)`--),
  REWRITE_TAC[MAPLET,PFUN,BREL,PSET,CROSS,SUBSET_DEF,IN_SING]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN REWRITE_ASMS_TAC[IN_SING,PAIR_EQ]
   THEN ASM_REWRITE_TAC[]
   THEN EXISTS_TAC (--`x:'a`--)
   THEN EXISTS_TAC (--`y:'b`--)
   THEN ASM_REWRITE_TAC[]));

val UNION_SING_Pfun = save_thm("UNION_SING_Pfun", prove(
  (--`!f (X:'a->bool) (Y:'b->bool) x y.
    f IN (X -+> Y) /\  x IN X /\  y IN Y /\  ~(x IN dom f) 
    ==> 
    (f UNION {x |-> y}) IN (X -+> Y)`--),
  REWRITE_TAC[dom,PFUN,BREL,PSET,CROSS,SUBSET_DEF,MAPLET]
   THEN SET_SPEC_TAC
   THEN REWRITE_TAC[IN_UNION,IN_SING,PAIR_EQ]
   THEN REPEAT STRIP_TAC
   THEN RES_TAC
   THENL
    [EXISTS_TAC (--`x':'a`--)
      THEN EXISTS_TAC (--`y':'b`--)
      THEN ASM_REWRITE_TAC[],
     EXISTS_TAC (--`x:'a`--)
      THEN EXISTS_TAC (--`y:'b`--)
      THEN ASM_REWRITE_TAC[],
     ELIMINATE_TAC
      THEN RES_TAC,
     ELIMINATE_TAC
      THEN RES_TAC,
     ASM_REWRITE_TAC[]]));

val UNION_SING_IN_P = save_thm("UNION_SING_IN_P", prove(
  (--`!f (X:'a->bool) (Y:'b->bool) x y.
    f IN (X -+> Y) /\  x IN X /\  y IN Y
    ==> 
    dom(f UNION {x |-> y}) IN PSET X`--),
  REWRITE_TAC[dom,PFUN,BREL,PSET,CROSS,SUBSET_DEF,MAPLET]
   THEN SET_SPEC_TAC
   THEN REWRITE_TAC[IN_UNION,IN_SING,PAIR_EQ]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN RES_TAC
   THEN ASM_REWRITE_TAC[]
   THEN ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 3)
   THEN ASM_REWRITE_TAC[]));

val domTotalFun = save_thm("domTotalFun", prove(
  (--`!f (X:'a->bool) (Y : 'b->bool). f IN (X --> Y) ==> (dom f = X)`--),
  REWRITE_TAC
   [TFUN,PFUN,BREL,CROSS,MAPLET,dom,PSET,SUBSET_DEF,EXTENSION]
   THEN BETA_TAC
   THEN CONV_TAC(DEPTH_CONV SET_SPEC_CONV)
   THEN REPEAT STRIP_TAC
   THEN EQ_TAC
   THEN REPEAT STRIP_TAC
   THEN RES_TAC));

val IN_Fun_Pfun = save_thm("IN_Fun_Pfun", prove(
  (--`!f (X:'a->bool) (Y:'b->bool). f IN (X --> Y) ==> f IN (X -+> Y)`--),
  REWRITE_TAC[TFUN]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN ASM_REWRITE_TAC[]));

val ranPfun = save_thm("ranPfun", prove(
  (--`!f (X:'a->bool) (Y:'b->bool) y. f IN (X -+> Y) /\  y IN ran f ==> y IN Y`--),
  REWRITE_TAC[MAPLET,ran,PFUN,BREL,PSET,CROSS,SUBSET_DEF]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN RES_TAC
   THEN ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 3)
   THEN ELIMINATE_TAC
   THEN ASM_REWRITE_TAC[]));

val ranPfunIN = save_thm("ranPfunIN", prove(
  (--`!f (X:'a->bool) (Y:'b->bool). f IN (X -+> Y)  ==> ran f IN PSET Y`--),
  REWRITE_TAC[ran,PSET,MAPLET,PFUN,BREL,CROSS,SUBSET_DEF]
   THEN SET_SPEC_TAC
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN RES_TAC
   THEN IMP_RES_TAC PAIR_EQ
   THEN ELIMINATE_TAC
   THEN ASM_REWRITE_TAC[]));

val Ap = new_definition("Ap",(--`Ap(f:('a # 'b)->bool)x = @y. (x,y) IN f`--));

val ZAPPLY = new_infix_definition("ZAPPLY",
    (--`ZAPPLY = (Ap: (('a # 'b)->bool) -> 'a -> 'b)`--),800);

val Ap_UNION1 = save_thm("Ap_UNION1", prove(
  (--`!(x1:'a) (x2:'a). !(v:'b) X.
    ~(x1 = x2) ==> (((X UNION { (x1 |-> v) }) ZAPPLY x2) = (X ZAPPLY x2))`--),
  REWRITE_TAC[ZAPPLY,dom,MAPLET,Ap,IN_UNION,IN_SING,PAIR_EQ]
   THEN CONV_TAC(DEPTH_CONV SET_SPEC_CONV)
   THEN CONV_TAC(DEPTH_CONV NOT_EXISTS_CONV)
   THEN REPEAT STRIP_TAC
   THEN POP_ASSUM(ASSUME_TAC o GSYM)
   THEN ASM_REWRITE_TAC[]));

val Ap_UNION2 = save_thm("Ap_UNION2", prove(
  (--`!(x:'a) (v:'b) X.
    ~(x IN dom X) ==> ((X UNION {x|->v}) ZAPPLY x = v)`--),
  REWRITE_TAC[ZAPPLY,dom,MAPLET,Ap,IN_UNION,IN_SING,PAIR_EQ]
   THEN CONV_TAC(DEPTH_CONV SET_SPEC_CONV)
   THEN CONV_TAC(DEPTH_CONV NOT_EXISTS_CONV)
   THEN REPEAT STRIP_TAC
   THEN ASM_REWRITE_TAC[SELECT_REFL]));

val Ap_SING = save_thm("Ap_SING", prove(
  (--`!(x:'a) (v:'b). {x|->v} ZAPPLY x = v`--),
  REWRITE_TAC[ZAPPLY,dom,MAPLET,Ap,IN_UNION,IN_SING,PAIR_EQ]
   THEN CONV_TAC(DEPTH_CONV SET_SPEC_CONV)
   THEN CONV_TAC(DEPTH_CONV NOT_EXISTS_CONV)
   THEN REPEAT STRIP_TAC
   THEN ASM_REWRITE_TAC[SELECT_REFL]));

val ApFun = save_thm("ApFun", prove(
  (--`!f (X:'a->bool) (Y:'b->bool) x.
    (f IN (X -+> Y)) /\ 
    (x IN dom f)
    ==> 
    !y. (f ZAPPLY x = y) = (x,y) IN f`--),
  REWRITE_TAC[ZAPPLY,Ap,MAPLET,dom,PFUN,BREL,PSET,CROSS,SUBSET_DEF]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN EQ_TAC
   THEN REPEAT STRIP_TAC
   THENL
    [ELIMINATE_TAC
      THEN ASSUM_LIST
      ( fn thl => ASSUME_TAC(EXISTS((--`?y.((x:'a),(y:'b)) IN f`--),(--`y:'b`--))(el 1 thl)))
      THEN POP_ASSUM(ASSUME_TAC o SELECT_RULE)
      THEN RES_TAC
      THEN ASM_REWRITE_TAC[],
     ASSUM_LIST
      ( fn thl => ASSUME_TAC(EXISTS((--`?y.((x:'a),(y:'b)) IN f`--),(--`y:'b`--))(el 2 thl)))
      THEN POP_ASSUM(ASSUME_TAC o SELECT_RULE)
      THEN RES_TAC]));

val domFun = save_thm("domFun", prove(
  (--`!f (x:'a) (y:'b). (x,y) IN f ==> x IN dom f`--),
  REWRITE_TAC[dom,MAPLET]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN EXISTS_TAC (--`y:'b`--)
   THEN ASM_REWRITE_TAC[]));

val ApFunCor = save_thm("ApFunCor", prove(
  (--`!f (X:'a->bool) (Y:'b->bool) x y.
    (f IN (X -+> Y)) /\  (x,y) IN f ==> (f ZAPPLY x = y)`--),
  REPEAT STRIP_TAC
   THEN IMP_RES_TAC domFun
   THEN IMP_RES_TAC ApFun));

val ApIN = save_thm("ApIN", prove(
  (--`!(X:'a->bool) (Y:'b->bool) f x. f IN (X -+> Y)  /\  x IN dom f ==> (f ZAPPLY x) IN Y`--),
  REPEAT STRIP_TAC
   THEN IMP_RES_TAC ApFun
   THEN ASSUM_LIST
         (ASSUME_TAC o REWRITE_RULE[] o SPEC (--`(f:('a # 'b)->bool)ZAPPLY x`--) o el 2)
   THEN ASSUM_LIST(MAP_EVERY(UNDISCH_TAC o concl))
   THEN REWRITE_TAC[dom,PSET,CROSS,BREL,PFUN,MAPLET,SUBSET_DEF]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN RES_TAC
   THEN REWRITE_ASMS_TAC[PAIR_EQ]
   THEN ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 6)
   THEN ELIMINATE_TAC
   THEN ASSUM_LIST(ACCEPT_TAC o el 4)));

fun delete (h::t) x = if (x = h) then delete t x else h::(delete t x)
  | delete [] x = [];;
val IN_dom_ran = save_thm("IN_dom_ran", prove(
  (--`!f (X:'a->bool) (Y:'b->bool) x y.
    f IN (X -+> Y) /\  (x,y) IN f ==> x IN X /\  y IN Y`--),
  REWRITE_TAC[PSET,CROSS,BREL,PFUN,MAPLET,SUBSET_DEF]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN RES_TAC
   THEN REWRITE_ASMS_TAC[PAIR_EQ]
   THEN ASSUM_LIST( fn thl => REWRITE_TAC(delete thl (el 4 thl)))));

val IN_P = save_thm("IN_P", prove(
  (--`!x f (X:'a->bool) (Y:'b->bool). x IN dom f /\  f IN (X -+> Y) ==> x IN X`--),
  REWRITE_TAC[dom,PFUN,BREL,CROSS,PSET,MAPLET,SUBSET_DEF]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN RES_TAC
   THEN ASSUM_LIST(ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 3)
   THEN ASM_REWRITE_TAC[]));

val domAp = save_thm("domAp", prove(
  (--`!(X:'a->bool) (Y:'b->bool) f. 
   (f IN (X -+> Y)) ==> !x. x IN dom f = (x, f ZAPPLY x) IN f`--),
  REWRITE_TAC[ZAPPLY,Ap,MAPLET,dom,PFUN,BREL,PSET,CROSS,SUBSET_DEF]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN EQ_TAC
   THEN REPEAT STRIP_TAC
   THENL
    [ASSUM_LIST
      ( fn thl => ASSUME_TAC(EXISTS((--`?y.((x:'a),(y:'b)) IN f`--),(--`y:'b`--))(el 1 thl)))
      THEN POP_ASSUM(ASSUME_TAC o SELECT_RULE)
      THEN RES_TAC
      THEN ASM_REWRITE_TAC[],
     EXISTS_TAC (--`(@y. ((x:'a),(y:'b)) IN f)`--)
      THEN ASM_REWRITE_TAC[]]));

(* Domain Antirestriction *)
val DOM_ANTIRESTRICT = new_infix_definition("DOM_ANTIRESTRICT",
 (--`<+ (S':'a->bool) (R:('a # 'b)->bool) = 
              {(x |-> y) | ~(x IN S') /\  (x |-> y) IN R}`--),800);

val RAN_ANTIRESTRICT = new_infix_definition("RAN_ANTIRESTRICT",
 (--`+> (R:('a # 'b)->bool)  (T':'b->bool) = 
          {(x|->y) | (x |-> y) IN R /\  ~(y IN T')}`--),800);

val RangeAntiResSING = save_thm("RangeAntiResSING", prove(
  (--`!(x:'a) (y:'b). {x |-> y} +> {y} = {}`--),
  REPEAT GEN_TAC
   THEN REWRITE_TAC[RAN_ANTIRESTRICT,MAPLET,IN_SING,EXTENSION,NOT_IN_EMPTY]
   THEN SET_SPEC_TAC
   THEN REWRITE_TAC[PAIR_EQ]
   THEN REPEAT STRIP_TAC
   THEN RES_TAC));

val RangeAntiResPfun = save_thm("RangeAntiResPfun", prove(
  (--`!(X:'a->bool) (Y:'b->bool) f (y:'b). f IN (X -+> Y) ==> (f +> {y}) IN (X -+> Y)`--),
  REPEAT GEN_TAC
   THEN REWRITE_TAC[PFUN,BREL,CROSS,PSET,RAN_ANTIRESTRICT,MAPLET,NOT_IN_EMPTY,SUBSET_DEF,IN_SING]
   THEN SET_SPEC_TAC
   THEN SET_SPEC_TAC
   THEN REWRITE_TAC[PAIR_EQ]
   THEN REPEAT STRIP_TAC
   THEN RES_TAC
   THEN ELIMINATE_TAC
   THENL
    [EXISTS_TAC (--`x':'a`--)
      THEN EXISTS_TAC (--`y'':'b`--)
      THEN ASM_REWRITE_TAC[],
     ELIMINATE_TAC
      THEN RES_TAC]));

val domRangeAntiResPfun = save_thm("domRangeAntiResPfun", prove(
  (--`!(X:'a->bool) (Y:'b->bool) f (y:'b). f IN (X -+> Y) ==> dom(f +> {y}) IN PSET X`--),
  REPEAT GEN_TAC
   THEN REWRITE_TAC[dom,PFUN,BREL,CROSS,PSET,RAN_ANTIRESTRICT,MAPLET,NOT_IN_EMPTY,SUBSET_DEF,IN_SING]
   THEN SET_SPEC_TAC
   THEN SET_SPEC_TAC
   THEN REWRITE_TAC[PAIR_EQ]
   THEN REPEAT STRIP_TAC
   THEN RES_TAC
   THEN ELIMINATE_TAC
   THEN ASSUM_LIST(ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 3)
   THEN ASM_REWRITE_TAC[]));

val REL_OVERRIDE = new_infix_definition("REL_OVERRIDE",
(--`REL_OVERRIDE (f:('a#'b)->bool) (g:('a#'b)->bool)=(dom g <+ f) UNION g`--),800);

val OverrideIsFun = save_thm("OverrideIsFun", prove(
  (--`!(X:'a->bool) (Y:'b->bool) f g.
    (f IN (X --> Y)) /\ 
    (g IN (X --> Y))
    ==> 
    ((f REL_OVERRIDE g) IN (X --> Y))`--),
  REWRITE_TAC[dom,MAPLET,REL_OVERRIDE,DOM_ANTIRESTRICT,TFUN,PFUN,BREL,PSET,CROSS,SUBSET_DEF]  
   THEN SET_SPEC_TAC
   THEN REWRITE_TAC[IN_UNION]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN (ELIMINATE_TAC ORELSE ALL_TAC)
   THEN ASM_REWRITE_TAC[PAIR_EQ]
   THENL
    [EXISTS_TAC (--`x:'a`--)
      THEN EXISTS_TAC (--`y:'b`--)
      THEN ASM_REWRITE_TAC[]
      THEN RES_TAC
      THEN IMP_RES_TAC PAIR_EQ
      THEN ELIMINATE_TAC
      THEN ASM_REWRITE_TAC[],
     RES_TAC
      THEN ELIMINATE_TAC
      THEN ASM_REWRITE_TAC[PAIR_EQ]
      THEN EXISTS_TAC (--`x':'a`--)
      THEN EXISTS_TAC (--`y:'b`--)
      THEN ASM_REWRITE_TAC[],
     ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 6)
      THEN ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 5)
      THEN ASSUM_LIST
            ( fn thl => (STRIP_ASSUME_TAC o 
                    REWRITE_RULE[SYM(el 4 thl),SYM(el 3 thl)])
                   (el 8 thl))
      THEN ASSUM_LIST
            ( fn thl => (STRIP_ASSUME_TAC o 
                    REWRITE_RULE[SYM(el 3 thl),SYM(el 2 thl)])
                   (el 6 thl))
      THEN RES_TAC,
     ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 4)
      THEN ELIMINATE_TAC
      THEN ASSUM_LIST
            (STRIP_ASSUME_TAC          o 
             CONV_RULE NOT_EXISTS_CONV o
             SET_SPEC_RULE             o 
             REWRITE_RULE[dom]         o 
             el 3)
      THEN RES_TAC,
     ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 3)
      THEN ELIMINATE_TAC
      THEN ASSUM_LIST
            (STRIP_ASSUME_TAC          o 
             CONV_RULE NOT_EXISTS_CONV o
             SET_SPEC_RULE             o 
             REWRITE_RULE[dom]         o 
             el 2)
      THEN RES_TAC,
     RES_TAC,
     REWRITE_TAC[EXTENSION]
      THEN SET_SPEC_TAC
      THEN GEN_TAC
      THEN EQ_TAC
      THEN REPEAT STRIP_TAC
      THENL
       [ELIMINATE_TAC
         THEN ASSUM_LIST(ASSUME_TAC o GSYM o SET_SPEC_RULE o REWRITE_RULE[EXTENSION] o el 5)
         THEN ASM_REWRITE_TAC[]
         THEN EXISTS_TAC (--`y':'b`--)
         THEN ASM_REWRITE_TAC[],
        EXISTS_TAC (--`y:'b`--)
         THEN ASM_REWRITE_TAC[],
        EXISTS_TAC (--`y:'b`--)
         THEN ASM_REWRITE_TAC[]]]));

val OverrideIsPfun = save_thm("OverrideIsPfun", prove(
  (--`!(X:'a->bool) (Y:'b->bool) f g.
    (f IN (X -+> Y)) /\ 
    (g IN (X -+> Y))
    ==> 
    ((f REL_OVERRIDE g) IN (X -+> Y))`--),
  REWRITE_TAC[dom,MAPLET,REL_OVERRIDE,DOM_ANTIRESTRICT,PFUN,BREL,PSET,CROSS,SUBSET_DEF]  
   THEN SET_SPEC_TAC
   THEN REWRITE_TAC[IN_UNION]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN (ELIMINATE_TAC ORELSE ALL_TAC)
   THEN ASM_REWRITE_TAC[PAIR_EQ]
   THENL
    [EXISTS_TAC (--`x:'a`--)
      THEN EXISTS_TAC (--`y:'b`--)
      THEN ASM_REWRITE_TAC[]
      THEN RES_TAC
      THEN IMP_RES_TAC PAIR_EQ
      THEN ELIMINATE_TAC
      THEN ASM_REWRITE_TAC[],
     RES_TAC
      THEN ELIMINATE_TAC
      THEN ASM_REWRITE_TAC[PAIR_EQ]
      THEN EXISTS_TAC (--`x':'a`--)
      THEN EXISTS_TAC (--`y:'b`--)
      THEN ASM_REWRITE_TAC[],
     ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 6)
      THEN ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 5)
      THEN ASSUM_LIST
            ( fn thl => (STRIP_ASSUME_TAC o 
                    REWRITE_RULE[SYM(el 4 thl),SYM(el 3 thl)])
                   (el 8 thl))
      THEN ASSUM_LIST
            ( fn thl => (STRIP_ASSUME_TAC o 
                    REWRITE_RULE[SYM(el 3 thl),SYM(el 2 thl)])
                   (el 6 thl))
      THEN RES_TAC,
     ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 4)
      THEN ELIMINATE_TAC
      THEN ASSUM_LIST
            (STRIP_ASSUME_TAC          o 
             CONV_RULE NOT_EXISTS_CONV o
             SET_SPEC_RULE             o 
             REWRITE_RULE[dom]         o 
             el 3)
      THEN RES_TAC,
     ASSUM_LIST(STRIP_ASSUME_TAC o REWRITE_RULE[PAIR_EQ] o el 3)
      THEN ELIMINATE_TAC
      THEN ASSUM_LIST
            (STRIP_ASSUME_TAC          o 
             CONV_RULE NOT_EXISTS_CONV o
             SET_SPEC_RULE             o 
             REWRITE_RULE[dom]         o 
             el 2)
      THEN RES_TAC,
     RES_TAC]));
     
val domOverride = save_thm("domOverride", prove(
  (--`!(X:'a->bool) (Y:'b->bool) f g.
    (f IN (X -+> Y)) /\ 
    (g IN (X -+> Y))
    ==> 
    (dom(f REL_OVERRIDE g) = dom f UNION dom g)`--),
  REWRITE_TAC[MAPLET,dom,REL_OVERRIDE,DOM_ANTIRESTRICT,PFUN,BREL,PSET,CROSS,SUBSET_DEF]  
   THEN SET_SPEC_TAC
   THEN REWRITE_TAC[EXTENSION,IN_UNION]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN ASM_REWRITE_TAC[PAIR_EQ]
   THEN EQ_TAC
   THEN REPEAT STRIP_TAC
   THENL
    [DISJ1_TAC
      THEN EXISTS_TAC (--`y:'b`--)
      THEN ASM_REWRITE_TAC[],
     DISJ2_TAC
      THEN EXISTS_TAC (--`y:'b`--)
      THEN ASM_REWRITE_TAC[],
     ASM_CASES_TAC (--`?y.((x:'a),(y:'b)) IN g`--)
      THENL
       [POP_ASSUM STRIP_ASSUME_TAC
         THEN EXISTS_TAC (--`y':'b`--)
         THEN ASM_REWRITE_TAC[],
        EXISTS_TAC (--`y:'b`--)
         THEN DISJ1_TAC
         THEN EXISTS_TAC (--`x:'a`--)
         THEN EXISTS_TAC (--`y:'b`--)
         THEN ASM_REWRITE_TAC[]],
     EXISTS_TAC (--`y:'b`--)
      THEN ASM_REWRITE_TAC[]]));

val ApOverride1 = save_thm("ApOverride1", prove(
  (--`!f g (X:'a->bool) (Y:'b->bool) x.
    (f IN (X -+> Y)) /\ 
    (g IN (X -+> Y)) /\ 
    (x IN (dom f DIFF dom g))
    ==> 
    ((f REL_OVERRIDE g)ZAPPLY x = f ZAPPLY x)`--),
  REWRITE_TAC[MAPLET,dom,IN_DIFF]
   THEN REPEAT STRIP_TAC
   THEN IMP_RES_TAC domOverride
   THEN REWRITE_ASMS_TAC[GSYM(REWRITE_RULE[MAPLET]dom)]
   THEN ASSUM_LIST
        ( fn thl => (ASSUME_TAC                      o 
                REWRITE_RULE[IN_UNION,el 6 thl] o
                SPEC_ALL                        o
                REWRITE_RULE[EXTENSION])
               (el 2 thl))
   THEN ASSUM_LIST( fn thl => ASSUME_TAC(REWRITE_RULE thl (SPEC_ALL OverrideIsPfun)))
   THEN ASSUM_LIST
         ( fn thl => ASSUME_TAC(REWRITE_RULE(thl@[MAPLET,dom]) (SPEC_ALL ApFun)))
   THEN ASSUM_LIST
         ( fn thl => ASSUME_TAC
                 (REWRITE_RULE thl 
                   (SPEC_ALL(SPEC (--`(f:('a # 'b)->bool) REL_OVERRIDE g`--) ApFun))))
   THEN ASM_REWRITE_TAC[REL_OVERRIDE,DOM_ANTIRESTRICT,IN_UNION]
   THEN REWRITE_TAC[MAPLET] (* Needed because of bug in ASM_REWRITE_TAC *)
   THEN SET_SPEC_TAC
   THEN DISJ1_TAC
   THEN ASSUM_LIST
         (STRIP_ASSUME_TAC o SET_SPEC_RULE o REWRITE_RULE[dom] o el 10)
   THEN EXISTS_TAC (--`x:'a`--)
   THEN EXISTS_TAC (--`y:'b`--)
   THEN ASM_REWRITE_TAC[GSYM MAPLET,PAIR_EQ]));

val ApOverride2 = save_thm("ApOverride2", prove(
  (--`!f g (X:'a->bool) (Y:'b->bool) x.
    (f IN (X -+> Y)) /\ 
    (g IN (X -+> Y)) /\ 
    (x IN dom g)
    ==> 
    ((f REL_OVERRIDE g) ZAPPLY x = g ZAPPLY x)`--),
  REPEAT STRIP_TAC
   THEN IMP_RES_TAC domOverride
   THEN ASSUM_LIST
        ( fn thl => (ASSUME_TAC                      o 
                REWRITE_RULE[IN_UNION,el 5 thl] o
                SPEC_ALL                        o
                REWRITE_RULE[EXTENSION])
               (el 2 thl))
   THEN ASSUM_LIST( fn thl => ASSUME_TAC(REWRITE_RULE thl (SPEC_ALL OverrideIsPfun)))
   THEN ASSUM_LIST
         ( fn thl => 
           ASSUME_TAC(REWRITE_RULE thl (SPEC_ALL(SPEC (--`g:('a # 'b)->bool`--) ApFun))))
   THEN ASSUM_LIST
         ( fn thl => ASSUME_TAC
                 (REWRITE_RULE thl 
                   (SPEC_ALL(SPEC (--`(f:('a # 'b)->bool) REL_OVERRIDE g`--) ApFun))))
   THEN ASM_REWRITE_TAC[REL_OVERRIDE,DOM_ANTIRESTRICT,IN_UNION]
   THEN SET_SPEC_TAC
   THEN DISJ2_TAC
   THEN IMP_RES_TAC domAp));

val BIGOR = new_infix_definition("BIGOR", 
   (--`|\/| f1 f2 (x:'a) = (f1 x) \/ (f2 x)`--),300);

val BIGAND = new_infix_definition("BIGAND",
   (--`|/\| f1 f2 (x:'a) = (f1 x) /\  (f2 x)`--),400);

val BIGIMP = new_infix_definition("BIGIMP",
   (--`|==>| f1 f2 (x:'a) = (f1 x) ==> (f2 x)`--),200);

val BIGNOT = new_definition("BIGNOT",
    (--`BIGNOT f (x:'a) = ~(f x)`--));


val NN = new_definition("NN",(--`NN = {n | n >= 0}`--));

val NN_1 = new_definition("NN_1",(--`NN_1 = {n | n > 0}`--));

val TO = new_infix_definition("TO",(--`TO m n i = m <= i /\  i <= n`--),500);


use "src/arith-tools.sml";

val IncInterval = save_thm("IncInterval", prove(
  (--`(1 TO (n+1)) = (1 TO n) |\/| ((n+1) TO (n+1))`--),
  CONV_TAC(FUN_EQ_CONV)
   THEN REPEAT GEN_TAC
   THEN REWRITE_TAC[TO,BIGOR]
   THEN ARITH_TAC));

val UnitInterval = save_thm("UnitInterval", prove(
  (--`(n TO n) x = (x = n)`--),
  REWRITE_TAC[TO]
   THEN ARITH_TAC));

val IntervalDIFFLemma = save_thm("IntervalDIFFLemma", prove(
  (--`!f (X:'a->bool) n x (v:'a).
    (f IN (NN_1 --> X) /\  (1 TO n)x 
    ==> x IN (dom f) DIFF dom {(n+1) |-> v})`--),
  REWRITE_TAC
   [MAPLET,dom,NN_1,TO,IN_DIFF,IN_SING,dom_SING,
    TFUN,PFUN,PSET,CROSS,BREL,SUBSET_DEF,EXTENSION,EQ_IMP_THM,PAIR_EQ]
   THEN BETA_TAC
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN REWRITE_ASMS_TAC[ARITH_PROVE (--`(1 <= n) = (n > 0)`--)]
   THEN RES_TAC
   THENL
    [EXISTS_TAC (--`y:'a`--)
      THEN ASM_REWRITE_TAC[],
     ELIMINATE_TAC
      THEN IMP_RES_TAC(ARITH_PROVE (--`((n+1) <= n) ==> F`--))]));

val IN_NN = save_thm("IN_NN", prove(
  (--`!n. n IN NN`--),
  REWRITE_TAC[NN]
   THEN SET_SPEC_TAC
   THEN ARITH_TAC));

val IN_NN_1 = save_thm("IN_NN_1", prove(
  (--`!n. (n+1) IN NN_1`--),
  REWRITE_TAC[NN_1]
   THEN SET_SPEC_TAC
   THEN ARITH_TAC));

val IntervalApLemma1 = save_thm("IntervalApLemma1", prove(
  (--`!f (X:'a->bool) n x v.
    f IN (NN_1 --> X) /\  (1 TO n)x  /\  v IN X 
    ==> 
    ((f REL_OVERRIDE {(n+1) |-> v}) ZAPPLY x = f ZAPPLY x)`--),
  REPEAT STRIP_TAC
   THEN IMP_RES_TAC IntervalDIFFLemma
   THEN POP_ASSUM(ASSUME_TAC o SPEC_ALL)
   THEN ASSUME_TAC(SPEC (--`n:num`--) IN_NN_1)
   THEN IMP_RES_TAC SING_Pfun
   THEN IMP_RES_TAC IN_Fun_Pfun
   THEN IMP_RES_TAC ApOverride1));

val IntervalApLemma2 = save_thm("IntervalApLemma2", prove(
  (--`!f (X:'a->bool) n v.
    f IN (NN_1 --> X) /\  v IN X 
    ==> 
    ((f REL_OVERRIDE {(n+1) |-> v}) ZAPPLY (n+1) = v)`--),
  REPEAT STRIP_TAC
   THEN IMP_RES_TAC IntervalDIFFLemma
   THEN POP_ASSUM(ASSUME_TAC o SPEC_ALL)
   THEN ASSUME_TAC(SPEC (--`n:num`--) IN_NN_1)
   THEN ASSUME_TAC(ISPECL[(--`n+1`--),(--`v:'a`--)]dom_SING_Cor)
   THEN LITE_IMP_RES_TAC SING_Pfun
   THEN LITE_IMP_RES_TAC IN_Fun_Pfun
   THEN LITE_IMP_RES_TAC ApOverride2
   THEN ASSUM_LIST(ACCEPT_TAC o REWRITE_RULE[Ap_SING] o el 2)));

val IntervalSINGLemma = save_thm("IntervalSINGLemma", prove(
  (--`!n x (v:'a). (1 TO n)x  ==> ~(x IN (dom {(n+1) |-> v}))`--),
  REWRITE_TAC[MAPLET,TO,IN_SING,dom_SING]
   THEN ARITH_TAC));

val UnitINTERVAL = save_thm("UnitINTERVAL", prove(
  (--`x IN (n TO n) = (x = n)`--),
   REWRITE_TAC[TO,SPECIFICATION]
   THEN SET_SPEC_TAC
   THEN ARITH_TAC));

val IN_INTERVAL = save_thm("IN_INTERVAL", prove(
  (--`x IN (m TO n) = m <= x /\  x <= n`--),
  REWRITE_TAC[TO,SPECIFICATION]
   THEN SET_SPEC_TAC
   THEN ARITH_TAC));

val IN_Interval = save_thm("IN_Interval", prove(
  (--`!i m n. i IN (m TO n) = (m TO n) i`--),
  REWRITE_TAC[TO,SPECIFICATION]
   THEN SET_SPEC_TAC
   THEN BETA_TAC
   THEN REWRITE_TAC[]));

val OverrideSingPfun = save_thm("OverrideSingPfun", prove(
  (--`!(X:'a->bool) f x.
    (f IN (NN_1 -+> X)) /\  x IN X
    ==> 
    ((f REL_OVERRIDE {(n+1) |-> x}) IN (NN_1 -+> X))`--),
  REPEAT STRIP_TAC
   THEN ASSUME_TAC(SPEC (--`n:num`--) IN_NN_1)
   THEN IMP_RES_TAC SING_Pfun
   THEN IMP_RES_TAC OverrideIsPfun));

val UNION_NN_1_SUC = save_thm("UNION_NN_1_SUC", prove(
  (--`!n. NN_1 UNION {n+1} = NN_1`--),
  GEN_TAC
   THEN REWRITE_TAC[EXTENSION,IN_UNION,NN_1,IN_SING]
   THEN SET_SPEC_TAC
   THEN ARITH_TAC));

val OverrideSingFun = save_thm("OverrideSingFun", prove(
  (--`!(X:'a->bool) f x.
    (f IN (NN_1 --> X)) /\  x IN X
    ==> 
    ((f REL_OVERRIDE {(n+1) |-> x}) IN (NN_1 --> X))`--),
  REWRITE_TAC[TFUN]
   THEN SET_SPEC_TAC
   THEN REPEAT STRIP_TAC
   THEN IMP_RES_TAC OverrideSingPfun
   THEN ASM_REWRITE_TAC[]
   THEN ASSUME_TAC(SPEC (--`n:num`--) IN_NN_1)
   THEN IMP_RES_TAC SING_Pfun
   THEN IMP_RES_TAC domOverride
   THEN ASM_REWRITE_TAC[dom_SING,UNION_NN_1_SUC]));

close_theory();;
export_theory();;





