(****************************************************************************)
(*                                                                          *)
(*               Copyright 1993, 1994 University of Cambridge               *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : ppboxes_sml-nj.sml                                       *)
(* DESCRIPTION   : Box abstraction for pretty-printing using the Oppen      *)
(*                 printer in Standard ML of New Jersey.                    *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 7th December 1993                                        *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 25th November 1994                                       *)
(****************************************************************************)

local
   structure PrettyPrint = System.PrettyPrint
   open PrettyPrint
in

functor PPBoxesFun (Scale : sig val char_size : int end) : PPBOXES =
struct

structure Scale =
struct

open Scale;

fun num_of_chars i =
   let val scaled = i div char_size
       val i' = scaled * char_size
   in  if (i' = i) then scaled else scaled + 1
   end;

end;

open Scale;

structure PPStream : PPSTREAM = PrettyPrint;

type ppbox = PPStream.ppstream -> unit;

(*--------------------------------------------------------------------------*)
(* Auxiliary functions.                                                     *)
(*                                                                          *)
(* replicate : int -> string -> string                                      *)
(* repeat    : int -> ('a -> 'b) -> 'a -> unit                              *)
(* increment : int -> int -> (int -> 'a -> 'b) -> 'a list -> unit           *)
(* relative  : int -> (int -> 'a -> int) -> 'a list -> unit                 *)
(*--------------------------------------------------------------------------*)

fun replicate n s =
   if (n < 1) then "" else s ^ (replicate (n - 1) s);

fun repeat n f x =
   if (n < 1) then () else (f x; repeat (n - 1) f x);

fun apply _ [] = ()
  | apply f (x::xs) = (f x; apply f xs);

fun increment (_:int) _ _ [] = ()
  | increment i n f (x::xs) = (f n x; increment i (n + i) f xs);

fun relative (_:int) _ [] = ()
  | relative n f (x::xs) = relative (f n x) f xs;

(*--------------------------------------------------------------------------*)
(* Type of indentations for pretty-printing.                                *)
(*                                                                          *)
(* ABSOLUTE n : indent n spaces from beginning of first line.               *)
(* RELATIVE n : indent n spaces from beginning of previous line.            *)
(*--------------------------------------------------------------------------*)

datatype ppindent = ABSOLUTE of int | RELATIVE of int;

(*--------------------------------------------------------------------------*)
(* pp_empty_box : ppstream -> unit                                          *)
(*                                                                          *)
(* Empty box.                                                               *)
(*--------------------------------------------------------------------------*)

fun pp_empty_box pps = ();

(*--------------------------------------------------------------------------*)
(* pp_string_box : string * int -> ppstream -> unit                         *)
(*                                                                          *)
(* Make a box from a string. The integer argument is the width of the box.  *)
(*--------------------------------------------------------------------------*)

fun pp_string_box (s,_) pps = add_string pps s;

(*--------------------------------------------------------------------------*)
(* pp_h_box : int -> (ppstream -> unit) list -> ppstream -> unit            *)
(*                                                                          *)
(* Horizontally formatted box with h spaces between subboxes.               *)
(*--------------------------------------------------------------------------*)

fun pp_h_box _ [] _ = ()
  | pp_h_box _ (box::[]) pps = box pps
  | pp_h_box h (box::boxes) pps =
   let val sep = replicate (num_of_chars h) " "
       fun f box = (add_string pps sep; box pps)
   in  (begin_block pps INCONSISTENT 0; box pps; apply f boxes; end_block pps)
   end;

(*--------------------------------------------------------------------------*)
(* pp_v_box : ppindent * int -> (ppstream -> unit) list -> ppstream -> unit *)
(*                                                                          *)
(* Vertically formatted box with indentation i and vertical separation v.   *)
(*--------------------------------------------------------------------------*)

fun pp_v_box (_,_) [] _ = ()
  | pp_v_box (_,_) (box::[]) pps = box pps
  | pp_v_box (i,v) (box::boxes) pps =
   let val {consumer,flush,linewidth} = dest_ppstream pps
       val ind = num_of_chars (case i of (ABSOLUTE n) => n | (RELATIVE n) => n)
       and inc = num_of_chars (case i of (ABSOLUTE _) => 0 | (RELATIVE n) => n)
       fun f n box =
          (repeat (num_of_chars v) add_newline pps;
           add_break pps (linewidth,n);
           box pps)
   in  (begin_block pps CONSISTENT 0;
        box pps;
        increment inc ind f boxes;
        end_block pps)
   end;

(*--------------------------------------------------------------------------*)
(* pp_hv_box :                                                              *)
(*    int * ppindent * int -> (ppstream -> unit) list -> ppstream -> unit   *)
(*                                                                          *)
(* Format subboxes horizontally if sufficient space, otherwise break the    *)
(* inconsistently.                                                          *)
(*                                                                          *)
(* Can't implement vertical spacing --- taken to be zero                    *)
(* Can't implement relative offsets --- taken as absolute                   *)
(*--------------------------------------------------------------------------*)

fun pp_hv_box (_,_,_) [] _ = ()
  | pp_hv_box (_,_,_) (box::[]) pps = box pps
  | pp_hv_box (h,i,(_:int)) (box::boxes) pps =
   let val ind = num_of_chars (case i of (ABSOLUTE n) => n | (RELATIVE n) => n)
       and inc = 0
       fun f n box = (add_break pps (num_of_chars h,n); box pps)
   in  (begin_block pps INCONSISTENT 0;
        box pps;
        increment inc ind f boxes;
        end_block pps)
   end;

(*--------------------------------------------------------------------------*)
(* pp_hov_box :                                                             *)
(*    int * ppindent * int -> (ppstream -> unit) list -> ppstream -> unit   *)
(*                                                                          *)
(* Format subboxes horizontally if sufficient space, otherwise format the   *)
(* subboxes vertically.                                                     *)
(*--------------------------------------------------------------------------*)

fun pp_hov_box (_,_,_) [] _ = ()
  | pp_hov_box (_,_,_) (box::[]) pps = box pps
  | pp_hov_box (h,i,v) (box::boxes) pps =
   let val ind = num_of_chars (case i of (ABSOLUTE n) => n | (RELATIVE n) => n)
       and inc = num_of_chars (case i of (ABSOLUTE _) => 0 | (RELATIVE n) => n)
       fun f n box = (repeat (num_of_chars v) (add_break pps) (0,0);
                      add_break pps (num_of_chars h,n); box pps)
   in  (begin_block pps CONSISTENT 0;
        box pps;
        increment inc ind f boxes;
        end_block pps)
   end;

(*--------------------------------------------------------------------------*)
(* pp_h_sepbox : (ppstream -> unit) ->                                      *)
(*               (int * (ppstream -> unit)) list ->                         *)
(*               (ppstream -> unit)                                         *)
(*                                                                          *)
(* As for pp_h_box but with separate control over the parameters for each   *)
(* subbox.                                                                  *)
(*--------------------------------------------------------------------------*)

fun pp_h_sepbox box [] = box
  | pp_h_sepbox box sepboxes = fn pps =>
   let fun f (h,box) = (add_string pps (replicate (num_of_chars h) " ");
                        (box pps : unit))
   in  (begin_block pps INCONSISTENT 0;
        box pps;
        apply f sepboxes;
        end_block pps)
   end;

(*--------------------------------------------------------------------------*)
(* pp_v_sepbox : (ppstream -> unit) ->                                      *)
(*               ((ppindent * int) * (ppstream -> unit)) list ->            *)
(*               (ppstream -> unit)                                         *)
(*                                                                          *)
(* As for pp_v_box but with separate control over the parameters for each   *)
(* subbox.                                                                  *)
(*--------------------------------------------------------------------------*)

fun pp_v_sepbox box [] = box
  | pp_v_sepbox box sepboxes = fn pps =>
   let val {consumer,flush,linewidth} = dest_ppstream pps
       fun f ind ((i,v),box) =
          let val ind' = case i of (ABSOLUTE n) => num_of_chars n
                                 | (RELATIVE n) => (ind + num_of_chars n)
          in  (repeat (num_of_chars v) add_newline pps;
               add_break pps (linewidth,ind');
               (box pps : unit);
               ind')
          end
   in  (begin_block pps CONSISTENT 0;
        box pps;
        relative 0 f sepboxes;
        end_block pps)
   end;

(*--------------------------------------------------------------------------*)
(* pp_hv_sepbox : (ppstream -> unit) ->                                     *)
(*                ((int * ppindent * int) * (ppstream -> unit)) list ->     *)
(*                (ppstream -> unit)                                        *)
(*                                                                          *)
(* As for pp_hv_box but with separate control over the parameters for each  *)
(* subbox.                                                                  *)
(*                                                                          *)
(* Can't implement vertical spacing --- taken to be zero                    *)
(* Can't implement relative offsets --- taken as absolute                   *)
(*--------------------------------------------------------------------------*)

fun pp_hv_sepbox box [] = box
  | pp_hv_sepbox box sepboxes = fn pps =>
   let fun f ((h,i,(_:int)),box) =
          let val ind = num_of_chars (case i of (ABSOLUTE n) => n
                                              | (RELATIVE n) => n)
          in  (add_break pps (num_of_chars h,ind); (box pps : unit))
          end
   in  (begin_block pps INCONSISTENT 0;
        box pps;
        apply f sepboxes;
        end_block pps)
   end;

(*--------------------------------------------------------------------------*)
(* pp_hov_sepbox : (ppstream -> unit) ->                                    *)
(*                 ((int * ppindent * int) * (ppstream -> unit)) list ->    *)
(*                 (ppstream -> unit)                                       *)
(*                                                                          *)
(* As for pp_hov_box but with separate control over the parameters for each *)
(* subbox.                                                                  *)
(*--------------------------------------------------------------------------*)

fun pp_hov_sepbox box [] = box
  | pp_hov_sepbox box sepboxes = fn pps =>
   let fun f ind ((h,i,v),box) =
          let val ind' = case i of (ABSOLUTE n) => num_of_chars n
                                 | (RELATIVE n) => (ind + num_of_chars n)
          in  (repeat (num_of_chars v) (add_break pps) (0,0);
               add_break pps (num_of_chars h,ind'); (box pps : unit); ind')
          end
   in  (begin_block pps CONSISTENT 0;
        box pps;
        relative 0 f sepboxes;
        end_block pps)
   end;

(*--------------------------------------------------------------------------*)
(* write_ppbox : (ppstream -> unit) -> ppstream -> unit                     *)
(*                                                                          *)
(* Write a box to ppstream.                                                 *)
(*--------------------------------------------------------------------------*)

fun write_ppbox box pps = box pps;

end; (* PPBoxesFun *)

end;
