(*--------------------------------------------------------------------------*)
(*                  Copyright (c) Donald Syme 1992                          *)
(*                  All rights reserved                                     *)
(*                                                                          *)
(* Donald Syme, hereafter referred to as `the Author', retains the copyright*)
(* and all other legal rights to the Software contained in this file,       *)
(* hereafter referred to as `the Software'.                                 *)
(*                                                                          *)
(* The Software is made available free of charge on an `as is' basis. No    *)
(* guarantee, either express or implied, of maintenance, reliability,       *)
(* merchantability or suitability for any purpose is made by the Author.    *)
(*                                                                          *)
(* The user is granted the right to make personal or internal use of the    *)
(* Software provided that both:                                             *)
(* 1. The Software is not used for commercial gain.                         *)
(* 2. The user shall not hold the Author liable for any consequences        *)
(*    arising from use of the Software.                                     *)
(*                                                                          *)
(* The user is granted the right to further distribute the Software         *)
(* provided that both:                                                      *)
(* 1. The Software and this statement of rights are not modified.           *)
(* 2. The Software does not form part or the whole of a system distributed  *)
(*    for commercial gain.                                                  *)
(*                                                                          *)
(* The user is granted the right to modify the Software for personal or     *)
(* internal use provided that all of the following conditions are observed: *)
(* 1. The user does not distribute the modified software.                   *)
(* 2. The modified software is not used for commercial gain.                *)
(* 3. The Author retains all rights to the modified software.               *)
(*                                                                          *)
(* Anyone seeking a licence to use this software for commercial purposes is *)
(* invited to contact the Author.                                           *)
(*--------------------------------------------------------------------------*)




(*----------------------------------------------------------------------------
 - new_theory "myZ";
 - Globals.library_path := (!Globals.library_path)@["../hol90_Z/"];
 - prim_load_library Lib.interpret {lib = find_library"pred_set",theory="-"};
 - prim_load_library Lib.interpret {lib = find_library"res_quan",theory="-"};
 - prim_load_library Lib.interpret {lib = find_library"pair",theory="-"};
 - prim_load_library Lib.interpret {lib = find_library"Z",theory="-"};
 - load_library {lib = find_library"BirthdayBook",theory="-"};
 - use "../hol90_Z/src/Z.sig";  use "../hol90_Z/src/Z.sml";  
 -
 - Z.register_defined_schema {name="BirthdayBook",theory="BirthdayBook"};
  Z.register_defined_schema {name="AddBirthday",theory="BirthdayBook"};
  Z.register_defined_schema {name="FindBirthday",theory="BirthdayBook"};
 - Z.defined_schemas ();
 - val s1 = rhs (concl (definition "BirthdayBook" "AddBirthday"));;
 - Z.dest_SCHEMA s1;
 - Z.mk_SCHEMA (Z.dest_SCHEMA s1) = s1;
 - Z.dest_schema_and (hd (#schemas (Z.dest_SCHEMA s1)));
 - Zxi_delta.dest_delta (hd (#schemas (Z.dest_SCHEMA s1)));
 - val s2 = rhs (concl (definition "BirthdayBook" "BirthdayBook"));;
 - Z.dest_SCHEMA s2;
 - Zxi_delta.dest_delta (Zxi_delta.mk_delta "BirthdayBook");
 - Zxi_delta.dest_xi (Zxi_delta.mk_xi "BirthdayBook");
 - val BB = Z.mk_schema_usage (#const (const_decl "BirthdayBook"));
 - val BB = Z.mk_schema_usage (#const (const_decl "BirthdayBook'"));
 - Z.dest_schema_usage BB;;
 - Z.dest_schema_usage BB';;
 - Z.mk_pred_schema_forall {body=(--`T`--),schema=(--`BirthdayBook`--)};
 - Z.dest_pred_schema_forall it;
 -----------------------------------------------------------------------------*)


structure Z : Z_sig =
struct

   open Pair_syn        (* Jim's paired syntax functions *)

   fun replicate x 0 = []
     | replicate x n = (x::replicate x (n-1))

(*----------------------------------------------------------------------------
 - Varstructs are canonicalized (sorted) tuples
 -
 -----------------------------------------------------------------------------*)

type varstruct = term;

fun varlt v1 v2 = #Name(dest_var v1) < #Name(dest_var v2);

fun dest_varstruct tm  = strip_pair tm;
fun mk_varstruct l = list_mk_pair (sort varlt l);
fun is_varstruct tm = all is_var (strip_pair tm) handle _ => false;
fun varstruct_union vs1 vs2 =
   mk_varstruct (
    map Psyntax.mk_var
      (union (map Psyntax.dest_var (dest_varstruct vs1)) 
             (map Psyntax.dest_var (dest_varstruct vs2))));

type varl = term list;
fun dest_varl l = l;
fun mk_varl l = (sort varlt l);
fun varl_union vs1 vs2 =
   mk_varl (union vs1 vs2);

(*----------------------------------------------------------------------------
 -
 -
 -----------------------------------------------------------------------------*)

fun repeat_count f x = 
      let val a = f x val (r,n) = repeat_count f a in (r,n+1) end
      handle _ => (x,0)
fun takeprime s = 
   let val (f,b) = front_n_back (explode s) in 
   if (b="'") then (implode f) else raise (Fail "takeprime") end;
fun takeprime_var tm = 
   mk_var{Name=takeprime(#Name(dest_var tm)),Ty=type_of tm};
fun takeprime_varl tm = 
   mk_varl(map takeprime_var(dest_varl tm));
fun takeprime_varstruct tm = 
   mk_varstruct(map takeprime_var(dest_varstruct tm));

fun primes n = implode (replicate "'" n);
fun prime (s,n) = s ^ (primes n);
val prime' = op ^;
fun prime_var_prim f (tm,n) = 
   let val {Name,Ty} = dest_var tm in mk_var{Name=f (Name,n),Ty=Ty} end;
val prime_var = prime_var_prim prime
val prime_var' = prime_var_prim prime'

fun prime_varl_prim f (tm,s) = 
   mk_varl (map (fn tm => f (tm,s)) tm);
val prime_varl = prime_varl_prim prime_var;
val prime_varl' = prime_varl_prim prime_var';

fun prime_varstruct_prim f (tm,s) = 
   mk_varstruct (map (fn tm => f (tm,s)) (dest_varstruct tm));
val prime_varstruct = prime_varstruct_prim prime_var;
val prime_varstruct' = prime_varstruct_prim prime_var';

val unprime = repeat_count takeprime
val unprime_var = repeat_count takeprime_var
val unprime_varl = repeat_count takeprime_varl
val unprime_varstruct = repeat_count takeprime_varstruct

(*----------------------------------------------------------------------------
 -
 -
 -----------------------------------------------------------------------------*)

fun z2op oper = mk_const{Name=oper,Ty=(==`:(bool # bool) -> bool`==)}
val zboth = z2op "BOTH"
val zand = z2op "AND"
val zor = z2op "ZOR"
val zimp = z2op "ZIMPLIES"
val znot = z2op "ZNOT";


fun dest_operator oper tm =
   let val (con,args) = strip_comb tm 
       val _ = if (#Name (dest_const con) = oper) then () else raise (Fail ("dest_" ^ oper))
   in
     args
   end;
fun dest_2op oper tm =
   let val [c1,c2] = dest_operator oper tm in
    (c1,c2)
   end
val dest_zand = dest_2op "ZAND";
val dest_zor = dest_2op "ZOR";
val dest_zimp = dest_2op "ZIMP";
fun dest_znot tm = let val [c] = dest_operator "ZNOT" tm in c end
fun dest_zboth tm = 
   let val [c] = dest_operator "BOTH" tm in 
     c
   end
fun mk_zand (s1,s2) = list_mk_comb(zand,[s1,s2]);
fun mk_zor (s1,s2) = list_mk_comb(zor,[s1,s2]);
fun mk_zimp (s1,s2) = list_mk_comb(zimp,[s1,s2]);
fun mk_znot s = mk_comb{Rator=znot,Rand=s};
fun mk_zboth s = mk_comb{Rator=zboth,Rand=s};

(*----------------------------------------------------------------------------
 - Table of defined schemas constants.  Includes primed constants.
 -
 -----------------------------------------------------------------------------*)

val defined_schemas_ref = ref ([] : (string * thm) list)

fun defined_schemas () = (!defined_schemas_ref);
fun is_defined_schema s =
   exists (curry (op =) s o #1) (defined_schemas())
fun definition_for_defined_schema s = assoc s (defined_schemas())
fun varstruct_for_defined_schema s =
   #Bvar (dest_pabs (rhs (concl (definition_for_defined_schema s))))
fun register_defined_schema' {name,thm} =
   defined_schemas_ref := (name,thm)::(!defined_schemas_ref)

(*----------------------------------------------------------------------------
 - force_schema_constant
 -
 - DESCRIPTION
 -
 - Called to bring a new (primed) schema constant into existence,
 - if it doesn't already exist.
 - Simulates an infinite family of constants in the same manner
 - as natural numbers and strings.  HOL should provide a mechanism
 - to extend the infinte families supported.
 -
 - NOTES
 -
 - Doesn't call new_definition as we don't want primed constant
 - defns to appear in the theory file.  The constants will
 - appear as necessary.
 -
 - SIDE EFFECTS
 -
 - Switches to draft_mode if in proof_mode.
 -----------------------------------------------------------------------------*)

fun force_schema_constant name =
   if (is_defined_schema name)
   then ()
   else
     let val (raw,p) = unprime name
         val _ = (assert is_defined_schema raw handle _ => raise (Fail "force_schema_constant"))
     in (
      (if not (is_constant name)
       then
         let val defn = rhs (concl (definition_for_defined_schema raw))
             val ty = type_of defn
             val _ = if not (draft_mode()) then extend_theory "-" else ()
         in
             new_constant {Name=name,Ty=ty}
         end
  else ());
      (let val defn = rhs (concl (definition_for_defined_schema raw))
           val {Bvar=vs,Body=bdy} = dest_pabs defn
           val vsl = strip_pair vs
           val newvsl = prime_varl(vsl,p)
           val newvs = list_mk_pair newvsl
           val subs = map2 (curry (op |->)) vsl newvsl
           val newbdy = subst subs bdy
           val newdefn = mk_pabs{Bvar=newvs,Body=newbdy}
           val newty = type_of newdefn
           val newthm = mk_thm([],mk_eq{lhs=mk_const{Name=name,Ty=newty},
                                        rhs=newdefn})
       in
          register_defined_schema' {name=name,thm=newthm}
       end))
     end;

(*----------------------------------------------------------------------------
 -
 -
 -----------------------------------------------------------------------------*)

fun register_defined_schema {name,theory} =
   let val _ = register_defined_schema' {name=name,thm=definition theory name}
               handle _ => raise (Fail "register_defined_schema: schema not found")
       val _ = force_schema_constant (prime (name,1)) handle _ => ()
   in
     ()
   end;

(*----------------------------------------------------------------------------
 - Mutually recursive schema recognisers
 -
 -----------------------------------------------------------------------------*)

fun dest_SCHEMA tm =
  let val {Body=body,Bvar=varstruct} = dest_pabs tm
      val (con,[schemas,decs,bod]) = strip_comb body 
      val schemas' = map dest_schema_application (#els (dest_list schemas))
      val decs' = #els (dest_list decs)
      val bod' = #els (dest_list bod)
  in 
     if (#Name (dest_const con) = "SCHEMA")
     then {schemas=schemas',decs=decs',body=bod'}
     else raise (Fail "dest_SCHEMA")
  end
   
and mk_SCHEMA {schemas,decs,body} =
  let val schemas' = mk_list{els=map mk_schema_application schemas,ty=(==`:bool # bool`==)}
      val decs' = mk_list{els=decs,ty=(==`:bool`==)}
      val body' = mk_list{els=body,ty=(==`:bool`==)}
  in
   mk_pabs{Bvar=mk_varstruct (union (free_vars schemas') (free_vars decs')),
           Body=list_mk_comb(#const (const_decl "SCHEMA"),[schemas',decs',body'])}
  end
and is_SCHEMA tm = can dest_SCHEMA tm

(*----------------------------------------------------------------------------
 -
 -
 -----------------------------------------------------------------------------*)

and dest_schema_2op oper tm =
   let val {Body=body,Bvar=varstruct} = dest_pabs tm
       val (s1,s2) = dest_2op oper body
   in
       (dest_schema_application s1,dest_schema_application s2)
   end

and mk_schema_2op oper (schema1,schema2) =
   let val varstruct1 = varstruct_for_schema schema1
       val varstruct2 = varstruct_for_schema schema2
       val varstruct = varstruct_union varstruct1 varstruct2
   in
     mk_pabs{Bvar=varstruct,
             Body=list_mk_comb(z2op oper,[
                    mk_comb{Rator=schema1,Rand=varstruct1},
               mk_comb{Rator=schema2,Rand=varstruct2}
                  ])
           }
   end

and is_schema_2op oper = can (dest_schema_2op oper)

and mk_schema_or tm = mk_schema_2op "ZOR" tm
and dest_schema_or tm = dest_schema_2op "ZOR" tm
and is_schema_or tm = is_schema_2op "ZOR" tm

and mk_schema_and tm = mk_schema_2op "ZAND" tm
and dest_schema_and tm = dest_schema_2op "ZAND"  tm
and is_schema_and tm = is_schema_2op "ZAND" tm

and mk_schema_imp tm = mk_schema_2op "ZIMP" tm
and dest_schema_imp tm = dest_schema_2op "ZIMP" tm
and is_schema_imp tm = is_schema_2op "ZIMP" tm

and dest_schema_not tm =
   let val {Body=body,Bvar=varstruct} = dest_pabs tm
       val s = dest_znot  body
   in
       dest_schema_application s
   end

and mk_schema_not schema =
   let val varstruct = varstruct_for_schema schema
   in
     mk_pabs{Bvar=varstruct,
             Body=mk_comb{Rator=znot,
                          Rand=mk_comb{Rator=schema,Rand=varstruct}}}
   end
and is_schema_not tm = can dest_schema_not tm


(*
and dest_schema_hide_var tm =
and mk_schema_hide_var (var,schema) =
val is_schema_hide_var = can dest_schema_hide_var

and dest_schema_hide_schema tm =
and mk_schema_hide_schema (var,schema) =
val is_schema_hide_schema = can dest_schema_hide_schema

*)

(*----------------------------------------------------------------------------
 -
 -
 -----------------------------------------------------------------------------*)

and dest_pred_schema_binder dest_pbinder tm =
    let val {Bvar,Body} = dest_pbinder tm
        val {ant=schema_app,conseq=body} = dest_imp Body
        val schema = dest_schema_usage schema_app
        val _ = assert is_schema schema handle _ => raise (Fail "dest_pred_schema_binder")
    in
        {schema=schema,body=body}
    end
and mk_pred_schema_binder mk_pbinder {schema,body} =
    let val vs = varstruct_for_schema schema
        val bdy' = mk_imp{ant=mk_schema_usage schema,conseq=body}
    in
       mk_pbinder{Body=bdy',Bvar=vs}
    end
and is_pred_schema_binder dest_pbinder tm = can (dest_pred_schema_binder dest_pbinder) tm

and dest_pred_schema_forall tm = dest_pred_schema_binder dest_pforall tm
and mk_pred_schema_forall v = mk_pred_schema_binder mk_pforall v
and is_pred_schema_forall tm = can (dest_pred_schema_binder dest_pforall) tm

and dest_pred_schema_exists tm = dest_pred_schema_binder dest_pexists tm
and mk_pred_schema_exists v = mk_pred_schema_binder mk_pexists v
and is_pred_schema_exists tm = can (dest_pred_schema_binder dest_pexists) tm


(*----------------------------------------------------------------------------
 -
 -
 -----------------------------------------------------------------------------*)

and sig_for_schema tm =
   raise (Fail "sig_for_schema")

and theta_for_schema tm =
   raise (Fail "theta_for_schema")

and dest_schema_application tm =
   let val {Rator=schema,Rand=varstruct} = dest_comb tm
       val _ = (assert is_schema schema handle _ => raise (Fail "dest_schema_application"))
   in
       if (varstruct_for_schema schema = varstruct)
       then schema
       else raise (Fail "dest_schema_application")
   end

and is_schema_application tm = can dest_schema_application tm
and mk_schema_application schema = 
   mk_comb{Rator=schema,Rand=varstruct_for_schema schema}

and dest_schema_usage tm =
   dest_schema_application (dest_zboth tm)
and is_schema_usage tm = can dest_schema_usage tm
and mk_schema_usage schema = 
   mk_zboth (mk_schema_application schema)

and dest_schema_constant tm =
   let val name = #Name (dest_const tm)
       val _ = (assert is_defined_schema name handle _ => raise (Fail "dest_schema_constant"))
   in
       name
   end
and is_schema_constant tm = can dest_schema_constant tm
and mk_schema_constant name = 
   let val _ = force_schema_constant name
       val _ = (assert is_defined_schema name handle _ => raise (Fail "mk_schema_constant"))
   in
      #const (const_decl name)
   end

and is_schema tm = 
   is_schema_constant tm orelse
   is_SCHEMA tm orelse
   is_schema_or tm orelse
   is_schema_and tm orelse
   is_schema_imp tm orelse
   is_schema_not tm orelse
   false

and varstruct_for_schema tm = 
   #Bvar (dest_pabs tm)
   handle _ => varstruct_for_defined_schema (dest_schema_constant tm)
   handle _ => raise (Fail "varstruct_for_schema")


end;


(*----------------------------------------------------------------------------
 -
 -
 -----------------------------------------------------------------------------*)

structure Zxi_delta : Zxi_delta_sig = struct

open Z;
open Pair_syn;

fun dest_delta tm =
   let val (s,s') = dest_schema_and tm
       val name = dest_schema_constant s
       val name' = dest_schema_constant s'
   in
       if (prime (name,1) = name')
       then name
       else raise (Fail "dest_delta")
   end

fun mk_delta name =
   let val _ = (assert is_defined_schema name handle _ => raise (Fail "mk_delta"))
       val name' = prime(name,1)
       val varstruct = varstruct_for_defined_schema name
       val varstruct' = varstruct_for_defined_schema name'
       val delta_varstruct = list_mk_pair ((strip_pair varstruct)@strip_pair varstruct') 
   in
       mk_schema_and(mk_schema_constant name,mk_schema_constant name')
   end

fun is_delta tm = can dest_delta tm


fun dest_xi tm =
   let val {schemas=[s,s'],decs=[],body=[eql]} = dest_SCHEMA tm
       val {lhs=l,rhs=r} = dest_eq eql 
       val name = dest_schema_constant s
       val name' = dest_schema_constant s'
   in
       if (prime (name,1) = name') andalso (prime_varstruct (l,1) = r)
       then name
       else raise (Fail "dest_xi")
   end

fun mk_xi name =
   let val _ = (assert is_defined_schema name handle _ => raise (Fail "mk_xi"))
       val name' = prime(name,1)
       val varstruct = varstruct_for_defined_schema name
       val varstruct' = varstruct_for_defined_schema name'
       val xi_varstruct = list_mk_pair ((strip_pair varstruct)@strip_pair varstruct') 
   in
       mk_SCHEMA {
         schemas=[mk_schema_constant name,mk_schema_constant name'],
         decs = [],
         body = [mk_eq{lhs=varstruct,rhs=varstruct'}]
       }
   end

fun is_xi tm = can dest_xi tm

end;


(*============================================================================*)
(* Some proof utilities.                                                      *)
(*============================================================================*)
(*
val ASM_F_TAC = IMP_RES_TAC(DISCH_ALL(ASSUME (Parse.--`F`--)));

fun APPLY_ASMS_TAC f =
 POP_ASSUM_LIST
  (fn assums => MAP_EVERY ASSUME_TAC (rev (map f assums)));

val REWRITE_ASMS_TAC = APPLY_ASMS_TAC o REWRITE_RULE;

fun REWRITE_ALL_TAC thl = 
 REWRITE_ASMS_TAC thl THEN ASM_REWRITE_TAC [] THEN REWRITE_TAC thl;

val simp_thms = [SCHEMA,CONJL,FORALL_RESTRICT,EXISTS_RESTRICT];

val SIMP_TAC =
 APPLY_ASMS_TAC (BETA_RULE o REWRITE_RULE simp_thms)
  THEN ASM_REWRITE_TAC simp_thms
  THEN BETA_TAC;

(*----------------------------------------------------------------------------*)
(* RW_ASM_THEN ttac [f1;...;fn] f =                                           *)
(*  ASSUM_LIST(fn thl. ttac(REWRITE_RULE[f1 thl,...,fn thl](f thl)))            *)
(*----------------------------------------------------------------------------*)

fun RW_ASM_THEN ttac fl f =
 ASSUM_LIST(fn thl => ttac(REWRITE_RULE(map (fn f => f thl) fl)(f thl)));

(*----------------------------------------------------------------------------*)
(* POP_ASSUMS n f = f[a1,...,an],                                             *)
(*                                                                            *)
(* where a1,...,an are the last n assumptions, which are popped.              *)
(*----------------------------------------------------------------------------*)

fun POP_ASSUMS n f =
 if n=0
 then ALL_TAC
 else if n=1
 then POP_ASSUM(fn th => f[th])
 else POP_ASSUM(fn th => POP_ASSUMS (n-1) (fn l => f (th::l)));

fun ITER n (tac:tactic)  = 
 if n < 0 then raise (Fail "ITER")
 if n = 0 then ALL_TAC 
          else tac THEN ITER (n-1) tac;

(*----------------------------------------------------------------------------*)
(* Generalized beta-reduction (useful for reducing schema applications).      *)
(*----------------------------------------------------------------------------*)

val GEN_BETA_TAC = CONV_TAC(DEPTH_CONV GEN_BETA_CONV);

(*----------------------------------------------------------------------------*)
(* Rule and Tactic for simplifying terms of the form `x IN {...|...}`         *)
(*----------------------------------------------------------------------------*)

val SET_SPEC_RULE = CONV_RULE(DEPTH_CONV SET_SPEC_CONV)
and SET_SPEC_TAC  = CONV_TAC(DEPTH_CONV SET_SPEC_CONV);

val simp sc =
 (let val th1 = 
   (RAND_CONV(RATOR_CONV(RAND_CONV(UNWIND_AUTO_CONV THENC PRUNE_CONV))) 
     ORELSEC ALL_CONV) sc
      val th2 =
   RIGHT_CONV_RULE
    (REWRITE_CONV([SCHEMA,CONJL,PAIR_EQ]) THENC REWRITE_CONV[]) 
    th1
      val tyl = 
   filter 
    (fn t => (fst(dest_const(rator(rator t))) = "IN") handle _ => false)
    (conjuncts(rhs(concl th2)))
  in
  RIGHT_CONV_RULE(REWRITE_CONV(map ASSUME tyl))th2
 ) handle _ => raise (Fail "simp");
*)


