#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------





# Procedure:    widget:check
#
# Synopsis:
#       Internal procedure to validate the content of a widget.
#
# Usage:
#c      widget:check pathName
#
# Parameters:
#c      pathName
#               Path name of a widget.
#
# Return value:
#       None.
#
# Description:
#       `widget:check' causes the Validate event to occur on a given single
#       widget.  (It uses widget_event (q.v.) to cause the event.)
#
#       If the validation succeeds, widget:check returns.  Otherwise, it
#       uses `widget:error' (q.v.) to display an error message, and resignals
#       the error.

proc widget:check w {
        set status [catch {widget_event $w Validate} message]
        if {$status != 0} {
                global errorInfo
                set info $errorInfo
                widget:error $w "$w: $message"
                error $message $info
        }
}

# Procedure:    widget:checkall
#
# Synopsis:
#       Internal procedure to validate the content of a widget tree.
#
# Usage:
#c      widget:checkall pathName
#
# Parameters:
#c      pathName
#               Path name of a widget.
#
# Return value:
#       None.
#

# Description:
#       `widget:checkall' is used to validate an entire portion of the
#       widget tree at once.  It runs `widget:check' on the specified widget
#       and on all of its subordinates.  It returns normally if all the
#       checks succeed, and otherwise throws an error (and displays a message)
#       detailing the check that failed.

proc widget:checkall w {
        widget:check $w
        foreach c [winfo children $w] {
                if {$c != [winfo toplevel $c]} {
                        widget:checkall $c
                }
        }
}

# Procedure:    widget:error
#
# Synopsis:
#       Internal procedure to report a validation error on a widget.
#
# Usage:
#c      widget:error pathName message
#
# Parameters:
#c      pathName
#               Path name of a widget.
#c      message
#               Error message to display
#
# Return value:
#       None.
#
# Description:
#       widget:error makes a modal dialog box describing an error in
#       validating the contents of a widget, and waits for the dialog to be
#       dismissed by the user.

proc widget:error {w message} {
        global widget:error$w
        modalDialog transient choicebox $w.error \
                -text $message \
                -buttons OK \
                -icon rtfm \
                -textvariable widget:error$w
        widget_waitVariable widget:error$w
        unset widget:error$w
        modalDialog.end $w.error
}

# Procedure:    widget_bind
#
# Synopsis:
#       Establish an event handler on a widget.
#
# Usage:        widget_bind pathName event ?action?
#       -or-    widget_bind Class event ?action?
#
# Parameters:
#c      pathName
#               Path name of a widget
#c      Class
#               Widget class
#c      event
#               Event to catch.  May be one of:
#c                      Destroy GainFocus
#c                      LoseFocus UpdateContent
#c                      Validate Unmap
#c      action
#               Tcl command to execute when the specified event occurs
#               on the specified widget.  If the binding is for a class, the
#               widget name will be appended to `action'.
#               If `action' is the null string, any existing binding is
#               removed.  If `action' begins with a plus sign, the specified
#               action is appended to the set of bindings for the widget.
#
# Return value:
#       The Tcl command that will be executed when the specified event occurs.
#
# Description:
#       `widget_bind' establishes a new binding for a given event on a
#       specified widget.  It is used to manage events that do not correspond
#       to X events, or that must be multiply dispatched.
#
#       The events, and their meanings, are as follows.
#
#c      Destroy
#               Widget has been destroyed.
#c      GainFocus
#               Widget has gained the keyboard focus by
#               means of `focus_goTo.'  This is a more
#               restrictive event than <FocusIn>, which
#               seems to happen almost at random.
#       LoseFocus
#               Widget has lost the keyboard focus by
#               means of `focus_goTo.'  This is a more
#               restrictive event than <FocusOut>, which
#               happens for various reasons outside the
#               application's control.
#c      UpdateContent
#               Widget has been requested to update its
#               content because a command is to be executed.
#c      Validate
#               Widget has been requested to check that its
#               content meets constraints.
#c      Unmap
#               Widget has been unmapped from the screen.

proc widget_bind {w event {string ?}} {
        global widget_priv
        case $event in {
                { Destroy GainFocus 
                  LoseFocus UpdateContent 
                  Validate Unmap } {
                }
                default {
                        error "widget_bind: unknown event $event"
                }
        }
        if {$string == "?"} {
                if [info exists widget_priv(event,$event,$w)] {
                        set string $widget_priv(event,$event,$w)
                } else {
                        set string ""
                }
        } elseif {$string == ""} {
                catch {unset widget_priv(event,$event,$w)}
        } else {
                if {[string index $string 0] == "+"} {
                        set string [string range $string 1 end]
                        if [info exists widget_priv(event,$event,$w)] {
                                append string \n
                                append string $widget_priv(event,$event,$w)
                        }
                }
                set widget_priv(event,$event,$w) $string
        }
        return $string
}

# Procedure:    widget_addBinding
#
# Synopsis:
#       Add a binding to the list of bindings for a given widget and event.
#
# Usage:
#c      widget_addBinding pathName event action
#
# Parameters:
#c      pathName
#               Path name of a widget
#c      event
#               Event for which to watch (see widget_bind)
#c      action
#               Tcl command to execute when the specified event occurs.
#
# Return value:
#       Complete set of bindings for the specified widget and event.
#
# Description:
#       widget_addBinding adds an action to the list of actions for a
#       specified event on a given widget.  It differs from calling widget_bind
#       with an event beginning with a plus sign in that if the specified
#       action is already on the list of actions to perform, it will not be
#       added a second time.

proc widget_addBinding {w event string} {
        global widget_priv
        if [info exists widget_priv(event,$event,$w)] {
                set curBinding $widget_priv(event,$event,$w)
        } else {
                set curBinding ""
        }
        if {[string first $curBinding $string] >= 0} {
                return $curBinding
        } else {
                return [widget_bind $w $event +$string]
        }
}

# Procedure:    widget_event
#
# Synopsis:
#       Cause an event on a widget.
#
# Usage:
#c      widget_event pathName event
#
# Parameters:
#c      pathName
#               Path name of a widget
#c      event
#               Event that has occurred (see widget_bind)
#
# Return value:
#       None.
#
# Description:
#       widget_event is used when one of the events monitored by widget_bind
#       occurs.  It executes the actions bound to the event for the widget,
#       and for its class.

proc widget_event {w event} {
        global widget_priv
        set action {}
        if [info exists widget_priv(event,$event,$w)] {
                set action $widget_priv(event,$event,$w)\n
        }
        if {[winfo exists $w] 
            && [info exists widget_priv(event,$event,[winfo class $w])]} {
                append action "$widget_priv(event,$event,[winfo class $w]) $w\n"
        }
        if {$action != ""} {
                return [uplevel #0 $action]
        } else {
                return 0
        }
}

# Procedure:    widget:destroy
#
# Synopsis:
#       Internal procedure executed in response to all <Destroy> events
#       to clean up after widget destruction.
#
# Usage:
#c      widget:destroy pathName
#
# Parameters:
#c      pathName
#               Path name of a widget.
#
# Return value:
#       None.
#
# Description:
#       `widget:destroy' is called when a <Destroy> X event is received for
#       a given widget.  It executes all the `Destroy' bindings that have
#       been established using `widget_bind' for the given widget (and its
#       class).
#
#       Finally all `widget_priv' entries belonging to the given widget
#       are deleted.

proc widget:destroy w {
        global widget_priv
        global widget_type
        widget_event $w Destroy
        widget_unsetPriv widget_priv $w {
                event,Destroy event,GainFocus event,LoseFocus,
                event,Unmap event,UpdateContent event,Validate
        }
}

# Procedure:    widget:unmap
#
# Synopsis:
#       Internal procedure executed in response to all <Unmap> events
#       to clean up after widget disappearance.
#
# Usage:
#c      widget:unmap pathName
#
# Parameters:
#c      pathName
#               Path name of a widget.
#
# Return value:
#       None.
#
# Description:
#       `widget:unmap' is called when an <Unmap> X event is received for
#       a given widget.  It executes all the `Unmap' bindings that have
#       been established using `widget_bind' for the given widget (and its
#       class).

proc widget:unmap w {
        global widget_priv
        widget_event $w Unmap
}

# Procedure:    widget_waitVariable
#
# Synopsis:
#       Wait for a variable
#
# Usage:
#c      widget_waitVariable name
#
# Parameters:
#       name
#               Name of a variable
#
# Return value:
#       None specified.
#
# Description:
#       The `widget_waitVariable' is identical to the Tk command, `tkwait
#       variable', except that it records the name of the variable in an
#       array named `widget_waitVariables'.  The rationale is that a `tkwait
#       variable' hangs the process if the application is destroyed while
#       the `tkwait' is pending.

proc widget_waitVariable {vname} {
        global widget_waitVariables
        global widget_appDestroyed
        if {![info exists widget_waitVariables($vname)]} {
                set widget_waitVariables($vname) 1
        } else {
                incr widget_waitVariables($vname)
        }
        uplevel 1 tkwait variable [list $vname]
        set content [uplevel #0 set $vname]
        if {[incr widget_waitVariables($vname) -1] == 0} {
                unset widget_waitVariables($vname)
        }
        if {[info exists widget_appDestroyed]} {
                error \
"widget_waitVariable $vname terminated prematurely:
   application has been destroyed."
        }
}

# Procedure:    widget:destroyApp
#
# Synopsis:
#       Throw errors at widget_waitVariable when an application is destroyed.
#
# Usage:
#c      widget:destroyApp
#
# Parameters:
#       None
#
# Return value:
#       None specified.
#
# Description:
#       widget:destroyApp is called when the application root window is
#       destroyed.  It sets the values of all variables active in
#       widget_waitVariable, so that all the waits will terminate.

proc widget:destroyApp {} {
        global widget_waitVariables
        global widget_appDestroyed
        set widget_appDestroyed 1
        if [catch {array names widget_waitVariables} names] return
        foreach vname $names {
                widget:destroyApp2 $vname
        }
}
proc widget:destroyApp2 {vname} {
        upvar #0 $vname var
        set var "@APP@DESTROYED@"
}

# Procedure:    widget_checkAndDestroy
#
# Synopsis:
#       Destroy a widget, if the widget exists.
#
# Usage:
#c      widget_checkAndDestroy pathName
#
# Parameters:
#c      pathName
#               Path name of a widget to destroy.
#
# Return value:
#       None.
#
# Description:
#       widget_checkAndDestroy checks if a specified widget exists.  If it
#       exists, it is destroyed, otherwise, nothing happens.
#
# Notes:
#       The commonest use of `widget_checkAndDestroy' is so that destroying a
#       widget may destroy its parent.  For example, destroying a widget
#       that has been packed into a `transient' frame should destroy the
#       top-level window as well.
#
# Example:
#c      widget_addBinding $w.child Destroy \
#
#c              "widget_checkAndDestroy $w"

proc widget_checkAndDestroy w {
        if [catch {winfo exists $w} exists] return
        if {$exists && ($w == "." || [info commands $w] == $w)} {
                destroy $w
        }
}

# Procedure:    widget_unsetPriv
#
# Synopsis:
#       Service procedure that cleans up working storage for a widget.
#
# Usage:
#c      widget_unsetPriv array widgetName prefices
#
# Parameters:
#c      array
#               The array (e.g., widget_priv, focus_priv) to be
#               cleaned up.
#c      widgetName
#               The widget whose entries are to be cleaned up
#c      prefices
#               The prefices to the widget name that form
#               keys for the array.
#
# Return value:
#       None.

proc widget_unsetPriv {aname w prefices} {
        upvar #0 $aname a
        foreach p $prefices {
                if [info exists a($p,$w)] {
                        unset a($p,$w)
                }
        }
}


