#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------





# Procedure:    transient
#
# Synopsis:
#       Create a transient top-level window, and pack a widget inside it.
#
# Usage:
#c      transient widget pathName args
#
# Parameters:
#c      widget
#               Name of a command that creates a widget, e.g., `label',
#               `fileselect'.
#c      pathName
#               Path name of a toplevel window to be created.
#c      args
#               Arguments to the widget creation command.
#
# Return value:
#       Name of the newly-created transient window.
#
# Description:
#       The `transient' procedure makes a top-level window to hold a given
#       widget.  It then creates the widget, packs it in the window, and
#       makes the window transient, centering it relative to its parent's
#       top-level window.
#
#       The widget is always named `$pathName.b'.  The class name of the
#       transient window is `Transient.'

        # Default look and feel:
option add *Transient.borderWidth 2 widgetDefault
option add *Transient.relief raised widgetDefault

bind Transient <FocusIn> { toplevel_focusIn %W %d }

proc transient {args} {
        set afterw 0
        set configs ""
        set widget ""
        set withfeedback 1
        set decorate 0
        for {set i 0} {$i<[llength $args]} {incr i} {
            case [lindex $args $i] -withfeedback {
                incr i
                set withfeedback [lindex $args $i]
            } -decorate {
                incr i
                set decorate [lindex $args $i]
            } default {
                if $afterw {
                    lappend configs [lindex $args $i]
                    incr i
                    lappend configs [lindex $args $i]
                } else {
                    if [regexp {\..*} [lindex $args $i]] {
                        set w [lindex $args $i]
                        set afterw 1
                    } else {
                        lappend widget [lindex $args $i]
                    }
                }
            }
        }
        if [winfo exists $w] { error "$w already exists" }
        toplevel $w -class Transient
        wm withdraw $w

        if {$withfeedback} {
            global gui_flags
            fontcheck label $w.feedback \
                -height 1 \
                -width 40 \
                -anchor w \
                -relief sunken \
                -font $gui_flags(font,feedback)
            pack $w.feedback -side bottom -expand no -fill x
        }
        if [catch {uplevel 1 $widget $w.b $configs} err] {
            destroy $w
            global errorInfo
            error $err $errorInfo
        }
        
        pack append $w $w.b {expand padx 4 pady 4}
        $w config -background [lindex [$w.b config -background] 4]
        transient_make $w $decorate
        widget_addBinding $w.b Destroy "
                after 1 widget_checkAndDestroy $w
        "
        return $w
}

# Procedure: transient_make
#
# Synopsis:
#       Apply the `transient window' hints to a top-level window.
#
# Usage:
#c      transient_make pathName
#
# Parameters:
#c      pathName
#               Path name of a top-level window.
#
# Return value:
#       The path name is returned.
#
# Description:
#       The transient_make procedure arranges for a window to be made
#       transient.  It is marked as `transient', owned by the top-level
#       window of its parent.  It is centered relative to the top-level
#       window of its parent.  It is withdrawn and repainted so that
#       it appears in the right place on the screen with the correct
#       window manager decorations (often, undecorated).
#
#       Note: Now allow decorated transients for auto-positioning
#       non-modal windows.  Decorations are helpful for letting
#       the user move the window out of the way and for destroying it.

proc transient_make { w decorate } {
        wm withdraw $w
        update idletasks
        set p [winfo parent $w]
        set t [winfo toplevel $p]
        set cx [expr [winfo rootx $t]+[winfo width $t]/2]
        set cy [expr [winfo rooty $t]+[winfo height $t]/2]
        set x [expr $cx-[winfo reqwidth $w]/2]
        set y [expr $cy-[winfo reqheight $w]/2]
        if {$x < 0} { set x 0 }
        if {$y < 0} { set y 0 }
        wm group $w $t
        if !$decorate {
            wm transient $w $t
        }
        wm positionfrom $w user
        wm geometry $w +$x+$y
        wm deiconify $w
        tkwait visibility $w
        update
}


