#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




 #########################################################################
 #                                                                       #
 # Copyright (C) 1993, 1994 by General Electric Company.                 #
 # All rights reserved.                                                  #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File: focusmgr.tcl

# Description:
#       Experimental focus manager for Tk

#       This code is intended to handle a variety of focus management issues
#       for Tcl/Tk.  It allows the TAB and SHIFT-TAB keys to traverse forward
#       and backward over the components of a top-level window.

#       The following assumptions are made about focusing.

#       A widget is considered `focusable' if either it or its class has
#       a binding for some event of one of the following types.
#       - <KeyPress>, <Key>
#       - <KeyRelease>

#       Moreover, a widget is not focusable if it is unmapped, or if its state
#       is `disabled'.

#       Widgets are traversed in preorder on the widget tree; widgets that
#       are not mapped are not visited.  Since `winfo children' returns the
#       children in opposite order to their creation, the traversal is in
#       opposite order to `winfo children'.

#       Canvas widgets get special handling, because not only the widget 
#       but also the focusable component must be identified.

# Global variables:
#c      focus_priv(current,$w)
#               Currently focused item among the children of toplevel
#               window $w.
#c      focus_priv(confine,$w)
#               Set to 1 if focus is to be confined so that it doesn't leave
#               widget $w (or its children) once it's directed there.
#c      focus_priv(skip,$w)
#               Set to 1 if focus is never to be directed to widget $w
#               via keyboard traversal.  Set to 0 if keyboard traversal
#               should ignore whether $w's state is `disabled' when
#               directing the focus thither.

 # focusAux.nw,v 1.2 1995/07/07 15:30:21 drs1004 Exp
 # /homes/drs1004/repository/tkaux/src/focusAux.nw,v
 # focusAux.nw,v
# Revision 1.2  1995/07/07  15:30:21  drs1004
# release TkHolWorkbench_0_3ap1
#
# Revision 1.1  1995/06/09  18:10:40  drs1004
# documentation and cleanup of unwanted things.
#
# Revision 1.1  1995/05/24  17:09:24  drs1004
# tosrc
#
# Revision 1.1.1.1  1994/12/07  10:17:29  donald
# First import after some small chanegs.
#
 # Revision 1.20  1994/10/27  18:29:42  kennykb
 # Release 2.0 -- 10-27-94.  To be uploaded to archive sites.
 #
 # Revision 1.19  1994/10/27  18:27:39  kennykb
 # Updated legal notices prior to release.
 #
 # Revision 1.18  1994/09/15  14:02:47  kennykb
 # Fixed performance bug where cleaning up after destroyed widgets takes
 # O(n**4) time.  Should now be linear.
 #
 # Revision 1.17  1994/01/20  19:10:03  kennykb
 # Fixed bug where destroying the only focusable window crashes app
 # with `focus_priv(current,.) doesn't exist'.
 #
 # Revision 1.16  1993/11/01  18:20:46  kennykb
 # Beta release to be announced on comp.lang.tcl
 #
 # Revision 1.15  1993/10/27  15:52:49  kennykb
 # Package for alpha release to the Net, and for MMACE 931101 release.
 #
 # Revision 1.14  1993/10/26  21:41:38  kennykb
 # Added `focus:unmap' procedure, to manage refocus after the focus widget
 # is unmapped.
 #
 # Revision 1.13  1993/10/25  16:14:02  kennykb
 # Added code to handle the case where the window that owns the focus
 # is destroyed.  Needed in the 93-11-01 MMACE release.
 #
 # Revision 1.12  1993/10/21  21:31:59  kennykb
 # Made changes to allow for KP_Tab as well as Tab, since it appears that
 # certain X displays have two Tab keys.
 #
 # Revision 1.11  1993/10/20  19:10:47  kennykb
 # Alpha release #1 was thawed for bug fixes in tk 3.3.  Now frozen again at this
 # point.
 #
 # Revision 1.10  1993/10/20  18:44:05  kennykb
 # Made changes so that tk 3.3 will traverse the widgets in stacking order.
 # Tk 3.2 will still traverse them in order of creation.
 #
 # Fixed copyright notice so that it doesn't look like structured commentary.
 #
 # Revision 1.9  1993/10/14  18:15:42  kennykb
 # Cleaned up alignment of log messages, to avoid problems extracting
 # structured commentary.
 #
 # Revision 1.8  1993/10/14  18:06:59  kennykb
 # Added GE legal notice to head of file in preparation for release.
 #
 # Revision 1.7  1993/10/14  14:02:02  kennykb
 # Alpha release #1 frozen at this point.
 #
 # Revision 1.6  1993/10/14  13:35:54  kennykb
 # Changed to use the UpdateContent event when asking focused window to
 # update itself.
 #
 # Revision 1.5  1993/07/22  21:12:53  kennykb
 # Corrected typo in focus_skip, and God knows how it ever worked.
 #
 # Revision 1.4  1993/07/20  19:17:12  kennykb
 # Improved structured comments.
 # Changed modules through `g' in the alphabet to follow `:' and `_' naming
 # conventions.
 #
 # Revision 1.3  1993/07/19  18:49:24  kennykb
 # Renamed all button_ commands to either button. or button:, in
 # conformance with new module naming conventions.
 #
 # Revision 1.2  1993/07/16  15:58:00  kennykb
 # Renamed all commands that start with `wiget.' to either `widget_' or
 # `widget:'.  Added the code for creating composite widgets.
 #
 # Revision 1.1  1993/06/03  15:28:43  kennykb
 # Initial revision
 #

# Procedure:    focus_bindForTraversal
#
# Synopsis:
#       Establish <Tab>-traversal bindings for any widget or widget class.
#
# Usage:
#c      focus_bindForTraversal pathNameOrClass ?-controlonly"
#
# Parameters:
#c      pathNameOrClass
#               Path name of a widgt for which to establish bindings,
#               class name of a widget class for which to establish
#               default bindings, or the keyword `all'.
#
# Options:
#c      -controlonly
#               Do not bind Tab and Shift-Tab, but only the keys with
#               Control depressed.
#
# Description:
#       focus_bindForTraversal sets up bindings for keyboard actions in
#       a wide variety of widgets.  It binds the following keys:
#
#       (<Tab>), <Control-Tab>:
#               Focus the next widget.
#       (<Shift-Tab>), <Control-Shift-Tab>:
#               Focus the previous widget.
#       (<Linefeed>, <Return>, <Enter>):
#
#       <Control-Linefeed>, <Control-Return>, <Control-Enter>:
#
#       <Control-j>, <Control-m>:
#               Invoke the default button for the widget's top-level window.
#
#       The keys in parentheses are bound only if the `-controlonly'
#       option is not specified.
#
# Notes:
#       The user does not usually need to call this function; the `init.tcl'
#       initialization does it for most widgets.

proc focus_bindForTraversal {w {controlonly ""}} {
        bind $w <Control-Tab> "focus_goToNext %W"
        bind $w <Control-Shift-Tab> "focus_goToPrev %W"
        catch {bind $w <Control-KP_Tab> "focus_goToNext %W"}
        catch {bind $w <Control-Shift-KP_Tab> "focus_goToPrev %W"}
        bind $w <Control-Key-j> {button_invokeDefault %W}
        bind $w <Control-Key-m> {button_invokeDefault %W}
        catch { bind $w <Control-Key-Linefeed> {button_invokeDefault %W} }
        catch { bind $w <Control-Key-Return> {button_invokeDefault %W} }
        catch { bind $w <Control-Key-KP_Enter> {button_invokeDefault %W} }
        catch { bind $w <Control-Key-Enter> {button_invokeDefault %W} }
                bind $w <Tab> "focus_goToNext %W"
                bind $w <Shift-Tab> "focus_goToPrev %W"
                catch { bind $w <KP_Tab> "focus_goToNext %W" }
                catch { bind $w <Shift-KP_Tab> "focus_goToPrev %W" }
        if {$controlonly != "-controlonly"} {
                catch { bind $w <Key-Linefeed> {button_invokeDefault %W} }
                catch { bind $w <Key-Return> {button_invokeDefault %W} }
                catch { bind $w <Key-KP_Enter> {button_invokeDefault %W} }
                catch { bind $w <Key-Enter> {button_invokeDefault %W} }
        }
}

# Procedure:    focus_confine
#
# Synopsis:
#       Confine <Tab> traversal to a particular widget and its children.
#
# Usage:
#c      focus_confine pathName ?value?
#
# Parameters:
#c      pathName
#               Path name of a widget
#c      value
#               A Boolean value (0 or 1).  Default is 1
#
# Return value:
#       None.
#
# Description:
#
#       focus_confine accepts the path name of a composite widget.  It is
#       used to confine keyboard traversals with <Tab> and <Shift-Tab> to that
#       part of the application.  A <Tab> or <Shift-Tab> that originates in
#       one of the widgets in pathName's subtree will always go to another
#       widget in pathName's subtree.
#
#       The second argument describes whether the focus shall be confined.
#       A value of 1 (default) causes the focus to be confined; a value
#       of 0 causes the focus to be unconfined again.
#
# Notes:
#
#       The focus is always confined to top-level windows; <Tab> and
#       <Shift-Tab> never cross among different top-level windows.

proc focus_confine {w {flag 1}} {
        global focus_priv
        if {$flag} {
                set focus_priv(confine,$w) $flag
                widget_addBinding $w Destroy "focus:destroy $w"
        } else {
                unset focus_priv(confine,$w)
        }
}

# Procedure:    focus:current
#
# Synopsis:
#       Direct focus to a window or canvas item.
#
# Usage:
#c      focus:current focusItem ?-force?
#
# Parameters:
#c      focusItem
#               A window to focus, a two-item list giving a canvas
#               and item to focus, or the keyword `none.'
#
# Options:
#c      -force
#               If this flag is present, the GainFocus handler for the item
#               is executed irrespective of whether it already had the focus.
#
# Return value:
#       None.
#
# Description:
#       `focus:current' makes the given item the `current focus' for its
#       toplevel window.  If the `current focus' has changed, GainFocus
#       and LoseFocus procedures are executed.  The `-force' flag forces
#       the GainFocus procedure for the specified window to be executed 
#       unconditionally; this flag is used on return from modal dialog
#       boxes to allow an item to update itself even though it never
#       formally lost the default focus.
#
# Notes:
#       This procedure is usually called in response to a <Button> event
#       (in the click-to-focus model) or an <Enter> event (in the 
#       focus-follows-mouse model).  
#       
#       A <Tab> or <Shift-Tab> event will also call it indirectly, as
#       may any other event that explicitly redirects the focus.
#
#       Applications implemented using this focus manager usually
#       follow a hybrid focus model, where the focus follows the mouse among
#       the top level windows, but is directed among individual
#       widgets by explicit user action.
#
#       The user should not call `focus:current' directly; `focus_goTo' or
#       one of its variants should be used when the focus changes.
#
# See also:
#c      focus_goTo
#
#c      focus_goToFirst
#
#c      focus_goToNext
#
#c      focus_goToPrev

proc focus:current {item {force ""}} {
        global focus_priv
        if {$item == "none"} return
        set w [lindex $item 0]
        if {[info exists focus_priv(current,[winfo toplevel $w])]} {
                set oldfocus $focus_priv(current,[winfo toplevel $w])
        } else {
                set oldfocus "none"
        }
        if {$item != $oldfocus || $force == "-force"} {
                if {$oldfocus != "none" && $oldfocus != $item} {
                        widget_event [lindex $oldfocus 0] LoseFocus
                }
                set t [winfo toplevel $w]
                set focus_priv(current,$t) $item
                widget_addBinding $w Destroy "focus:destroy $w"
                widget_addBinding $w Unmap "focus:unmap $w"
                widget_event [lindex $item 0] GainFocus
                focus default $w
        }
}

# Procedure:    focus:destroy
#
# Synopsis:
#       Internal procedure to clean up focus management when a
#       window is destroyed.
#
# Usage:
#c      focus:destroy pathName
#
# Parameters:
#c      pathName
#               Path name of a widget
#
# Return value:
#       None.
#
# Description:
#       focus:destroy is invoked whenever a focusable widget is destroyed.
#       It cleans up any `focus_priv' entries that exist for the widget.
#       If the widget owns the focus, it queues an idle task to find
#       a good place to move the focus.
#
# Bugs:
#       Something similar is needed for Unmap events.

proc focus:destroy w {
        global focus_priv
        if {$w == [focus]} {
                focus:destroyFocused $w
        }
        widget_unsetPriv focus_priv $w { confine current skip }
}

# Procedure: focus:destroyFocused
#
# Synopsis:
#       Handle the case where the current focus window is destroyed.
#
# Usage:
#c      focus:destroyFocused pathName
#
# Parameters:
#c      pathName
#               Path name of the window being destroyed.
#
# Return value:
#       None specified.
#
# Description:
#       focus:destroyFocused is invoked when a <Destroy> event detects that
#       the window that owns the focus is being destroyed.
#       It accumulates a list of the window's parents, up to its top level
#       window.  After a trip through the event handler, 
#       focus:refocusAfterDestroy is called with this list as its parameter.
#       It locates the `most appropriate' window to receive focus, and does
#       a `focus_goTo' to refocus it.

proc focus:destroyFocused w {
        global focus_priv
        set ancestors {}
        while {$w != [winfo toplevel $w]} {
                set w [winfo parent $w]
                lappend ancestors $w
        }
        catch {unset focus_priv(current,$w)}
        after 1 focus:refocusAfterDestroy [list $ancestors]
}

# Procedure: focus_findCurrent
#
# Synopsis:
#       Find the currently focused window or canvas item.
#
# Usage:
#c      focus_findCurrent
#
# Parameters:
#       None.
#
# Return value:
#       The return value is one of
#       - `none', indicating that there is no current focus.
#       - The path name of the window that has the focus.
#       - A two-element list consisting of the path name of a canvas that has
#         the focus and the item name within the canvas of the item that
#         has the focus
#
# Description:
#       focus_findCurrent locates the focus in order that it can be saved
#       temporarily (e.g., in order to force a temporary change of focus
#       to execute focusIn and focusOut handlers).

proc focus_findCurrent {} {
        set w [focus]
        if {[winfo exists $w] \
            && [info commands $w] == $w \
            && [winfo class $w] == "Canvas"} {
                lappend w [$w focus]
        }
        return $w
}

# Procedure:    focus_findDefault
#
# Synopsis:
#       Locate the default focus in a top-level window
#
# Usage:
#c      focus_findDefault pathName
#
# Parameters:
#c      pathName
#               Path name of a window
#
# Return value:
#       Path name of the default focus in the toplevel window of
#       which $pathName is a descendant.
#
# Description:
#       `focus_findDefault' locates and returns the current default
#       focus associated with any given window.

proc focus_findDefault w {
        global focus_priv
        set t [winfo toplevel $w]
        if {![info exists focus_priv(current,$t)]} {
                focus:current $t
        }
        if {![info exists focus_priv(current,$t)]} {
                return "none"
        }
        return $focus_priv(current,$t)
}               

# Procedure:    focus:findItem
#
# Synopsis:
#       Internal procedure that determines whether a widget can accept focus.
#
# Usage:
#c      focus:findItem pathName ?direction"
#
# Parameters;
#c      pathName
#               Path name of a window
#c      direction
#               `first' or `last'.  Default is `first'.
#
# Return value:
#       The return value will be one of
#       - `none', indicating that the widget does not accept focus.
#       - A widget path name.
#       - A two-element list comprising a widget path name and
#         the name of a canvas item within the widget.
#
# Description:
#       The focus:findItem procedure determines whether the widget passed
#       to it as a parameter can accept the keyboard focus.  If the
#       widget is a canvas, it also determines which item within the
#       canvas should receive the focus.  It returns the null string
#       if the widget cannot accept focus, the name of the widget if
#       the widget can accept focus on its own behalf, or a list of
#       the widget name and the item name if a canvas item accepts
#       focus on behalf of the canvas.
#
#       The second argument indicates a direction of traversal.  If it
#       is `first', then the first item within a canvas capable of
#       accepting the focus is returned.  If it is `last', then the
#       last item capable of accepting focus is returned.
#
# Notes:
#       Focus is NOT directed to the returned item; focus:findItem simply
#       determines which item is accepting focus.
#
#       focus:findItem does NOT traverse child widgets; use `focus_goTo' to
#       move the focus.

proc focus:findItem {w {direction first}} {

        global focus_priv
        
        # NOTE: Do *not* call this procedure while waiting for a window
        # to become visible; do a `tkwait visibility' if necessary.

        # An unmapped window is never focusable.

        if {![winfo ismapped $w]} {
                return "none"
        }

        # A widget that is marked to be skipped, or not to be skipped,
        # follows the marking.

        if [info exists focus_priv(skip,$w)] {
                set skip $focus_priv(skip,$w)
        } else {
                set skip ""
        }

        # A widget that has a state, and whose state is `disabled',
        # is never focusable.

        if {$skip == ""} {
                set skip 0
                if {![catch {$w config -state} c]} {
                        set state [lindex $c 4]
                        if {$state == "disabled"} {
                                set skip 1
                        }
                }
        }

        # A widget that is marked "skip-over" is never focusable

        if {$skip} {
                return "none"
        }

        set first "none"
        set last "none"

        # A widget is focusable if it or its class has at least one
        # binding on a Key- or KeyRelease- event

        foreach binding [concat [bind $w] [bind [winfo class $w]]] {
                if [regexp {Key|KeyRelease} $binding] {
                        set first $w
                        set last $w
                }
        }

        # A canvas widget may also have focusable items.  Check for them

        if {[winfo class $w] == "Canvas"} {
                foreach item [$w find all] {
                        if [focus:focusable $w $item] {
                                if {$first == "none"} {
                                        set first [list $w $item]
                                }
                                set last [list $w $item]
                        }
                }
        }

        case $direction in {
                first { return $first }
                last { return $last }
                default {
                        error "\
unknown direction $direction: must be first or last.\
"
                }
        }
}

# Procedure:    focus:first
#
# Synopsis:
#       Internal procedure that locates the first focusable item within
#       a window.
#
# Usage:
#c      focus:first pathName
#
# Parameters:
#c      pathName
#               Path name of a window
#
# Return value:
#       The return value is one of
#       - `none', indicating that the widget does not accept focus.
#       - A widget path name
#       - A two-element list comprising a widget path name and
#         the name of a canvas item within the widget.
#
# Description:
#       The focus:first procedure locates the first focusable item
#       within a window, and returns its path name, and possibly its
#       canvas item ID.
#
# Notes:
#       The focus:first procedure does NOT direct the focus to the
#       given item; it merely indicates which item it is.
#
# See also:
#c      focus_goTo

proc focus:first w {
        global tk_version

        # If the window itself or an item in it is focusable, return it.

        set r [focus:findItem $w first]
        if {$r != "none"} {
                return $r
        }

        # Traverse the window children, returning anything that's focusable

        set kids [winfo children $w]
        if {$tk_version >= 3.3} {
                set init {set i 0}
                set cond {$i < [llength $kids]}
                set reinit {incr i}
        } else {
                set init {set i [expr [llength $kids]-1]}
                set cond {$i >= 0}
                set reinit {incr i -1}
        }
        for $init $cond $reinit {
                set kid [lindex $kids $i]
                if {$kid != [winfo toplevel $kid]} {
                        set r [focus:first $kid]
                        if {$r != "none"} {
                                return $r
                        }
                }
        }

        # Nothing focusable was found.

        return "none"
}       

# Procedure:    focus:focusable
#
# Synopsis:
#       Internal procedure that determines whether a canvas item accepts
#       the keyboard focus.
#
# Usage:
#c      focus:focusable pathName tagOrId
#
# Arguments:
#c      pathName
#               Path name of a canvas widget
#c      tagOrId
#               Tag name or item name
#
#
# Return value:
#       1 if the specified item or items accept keyboard input, 0
#       otherwise
#
# Description:
#
#       The `focus:focusable' procedure determines whether a given
#       canvas item has a binding allowing it to accept keyboard
#       input.  It returns 1 if it does, and 0 if it doesn't.

proc focus:focusable {canvas tagOrId} {
        foreach id [$canvas find withtag $tagOrId] {
                foreach binding [$canvas bind $id] {
                        if [regexp {Key|KeyRelease} $binding] {
                                return 1
                        }
                }
                foreach tag [lindex [$canvas itemconfigure $id -tags] 4] {
                        foreach binding [$canvas bind $tag] {
                                if [regexp {Key|KeyRelease} $binding] {
                                        return 1
                                }
                        }
                }
        }
        return 0
}

# Procedure:    focus_goTo
#
# Synopsis:
#       Direct the focus to a particular widget or canvas item.
#
# Usage:
#c      focus_goTo pathName ?-force?
#
#c      focus_goTo {pathName tagOrID} ?-force?
#
# Parameters:
#c      pathName
#               Name of a widget that accepts the keyboard focus.
#c      tagOrID
#               Tag or ID of a canvas item that accepts the keyboard focus.
#
# Options:
#c      -force
#               If this flag is present, the GainFocus handler for the item
#               is executed irrespective of whether it already had the focus.
#               This flag is used for return from modal dialog boxes so that
#               an item can update itself even though it never lost the
#               default focus.
#
# Return value:
#       None.
#
# Description:
#       focus_goTo accepts the description of an item in the user interface
#       and directs focus there.

proc focus_goTo {item {force ""}} {

        # Make sure we're directing focus to an extant window.

        set w [lindex $item 0]

        if {$w != "none"} {
                if {![winfo exists $w]
                    || [info commands $w] != $w} {
                        set item [focus_findDefault .]
                }
        }

        # Record focus relative to top-level window.

        focus:current $item $force

        # Direct focus to the window

        set w [lindex $item 0]
        focus $w

        # Direct focus to the appropriate canvas item

        if {[llength $item] >= 2} {
                $w focus [lindex $item 1]
        } elseif {$item != "none" && [winfo class $w] == "Canvas"} {
                $w focus {}
        }

}

# Procedure:    focus_goToFirst
#
# Synopsis:
#       Direct focus to the first item within a window.
#
# Usage:
#c      focus_goToFirst pathName
#
#c      focus_goToFirst {pathName tagOrID}
#
# Parameters:
#c      pathName
#               Path name of a widget that accepts keyboard focus.
#c      tagOrID
#               Tag or ID of a canvas item within the widget that accepts
#               keyboard focus.
#
# Return value:
#       None.
#
# Description:
#       focus_goToFirst accepts the name of an item in the user interface.
#       It determines the logical `first item' and directs the focus there.
#       It is used to handle the <Tab> key for navigation among widgets.

proc focus_goToFirst {item} {
        focus_goTo [focus:first $item]
}

# Procedure:    focus_goToNext
#
# Synopsis:
#       Advance focus to next logical item.
#
# Usage:
#c      focus_goToNext pathName
#
#c      focus_goToNext {pathName tagOrID}
#
# Parameters:
#c      pathName
#               Path name of a widget that accepts keyboard focus.
#c      tagOrID
#               Tag or ID specifying a canvas item within pathName that
#               accepts the keyboard focus.
#
# Return value:
#       None.
#
# Description:
#       focus_goToNext accepts the name of an item in the user interface.
#       It determines the logical `next item' and directs the focus there.
#       It is used to handle the Tab key for navigation among widgets.

proc focus_goToNext {item} {
        focus_goTo [focus:next $item]
}

# Procedure:    focus_goToPrev
#
# Synopsis:
#       Direct focus to the previous logical item.
#
# Usage:
#c      focus_goToPrev pathName
#
#c      focus_goToPrev {pathName itemName}
#
# Parameters:
#c      pathName
#               Path name of a widget that accepts keyboard focus.
#c      tagOrID
#               Tag or ID specifying a canvas item within pathName that
#               accepts the keyboard focus.
#
# Return value:
#       None.
#
# Description:
#       focus_goToPrev accepts the name of an item in the user interface.
#       It determines the logical `previous item' and directs the focus there.
#       It is used to handle the Shift-Tab key for navigation among widgets.

proc focus_goToPrev {item} {
        focus_goTo [focus:prev $item]
}

# Procedure:    focus:last
#
# Synopsis:
#       Internal procedure to find the last focusable item within a widget.
#
# Usage:
#c      focus:last pathName
#
# Parameters:
#c      pathName
#               Path name of a widget.
#
# Return value:
#       The return value is one of
#       - `none', indicating that the widget does not accept focus.
#       - A widget path name
#       - A two-element list comprising a widget path name and
#         the name of a canvas item within the widget.
#
# Description:
#       The focus:last procedure locates the last focusable item
#       within a window, and returns its path name, and possibly its
#       canvas item ID.
#
# Notes:
#       The focus:last procedure does NOT direct the focus to the
#       given item; it merely indicates which item it is.
#
# See also:
#c      focus_goTo
#
#c      focus_goToFirst
#
#c      focus_goToNext
#
#c      focus_goToPrev

proc focus:last w {
        global tk_version

        # Traverse the window children, returning anything that's focusable

        set kids [winfo children $w]
        if {$tk_version < 3.3} {
                set init {set i 0}
                set cond {$i < [llength $kids]}
                set reinit {incr i}
        } else {
                set init {set i [expr [llength $kids]-1]}
                set cond {$i >= 0}
                set reinit {incr i -1}
        }
        for $init $cond $reinit {
                set kid [lindex $kids $i]
                if {$kid != [winfo toplevel $kid]} {
                        set r [focus:last $kid]
                        if {$r != "none"} {
                                return $r
                        }
                }
        }

        # If the window itself or an item in it is focusable, return it.

        set r [focus:findItem $w last]
        if {$r != "none"} {
                return $r
        }

        # Nothing focusable was found.

        return "none"
}       

# Procedure:    focus:next
#
# Synopsis:
#       Internal procedure to locate the next focusable item relative to
#       a widget or canvas item.
#
# Usage:
#c      focus:next pathName
#
#c      focus:next {pathName tagOrID}
#
# Parameters:   
#c      pathName
#               Path name of a widget that accepts keyboard focus.
#c      tagOrID
#               Tag or ID specifying a canvas item within pathName that
#               accepts the keyboard focus.
#
# Return value:
#       The return value is one of
#       - `none', indicating that the widget does not accept focus.
#       - A widget path name
#       - A two-element list comprising a widget path name and
#         the name of a canvas item within the widget.
#
# Description:
#       Given a focusable item, this procedure locates the next
#       focusable item within the item's toplevel window.
#
# Notes:
#       Focus is NOT directed to either item.
#
# See also:
#c      focus_goTo
#
#c      focus_goToFirst
#
#c      focus_goToNext
#
#c      focus_goToPrev


proc focus:next a {
        global focus_priv
        global tk_version

        set w [lindex $a 0]
        set item [lindex $a 1]

        # Traverse among canvas items.  A canvas is followed by its items;
        # each item is followed by the next younger item.

        if {[winfo class $w] == "Canvas"} {
                set items [$w find all]
                if {$item == ""} {
                        set index -1
                } else {
                        set index [lsearch $items $item]
                }
                if {$item == "" || $index >= 0} {
                        set items [lrange $items [expr $index+1] end]
                        foreach id $items {
                                if [focus:focusable $w $id] {
                                        return [list $w $id]
                                }
                        }
                }
        }

        # A widget is followed by its children.

        set kids [winfo children $w]
        if {$tk_version >= 3.3} {
                set init {set i 0}
                set cond {$i < [llength $kids]}
                set reinit {incr i}
        } else {
                set init {set i [expr [llength $kids]-1]}
                set cond {$i >= 0}
                set reinit {incr i -1}
        }
        for $init $cond $reinit {
                set kid [lindex $kids $i]
                if {$kid != [winfo toplevel $kid]} {
                        set r [focus:first $kid]
                        if {$r != "none"} {
                                return $r
                        }
                }
        }

        # Traverse the sister, aunt, great-aunt, ... widgets

        while {$w != [winfo toplevel $w] \
            && ![info exists focus_priv(confine,$w)]} {
                set p [winfo parent $w]
                set windows [winfo children $p]
                set index [lsearch $windows $w]

                # Traverse first the younger siblings

                if {$tk_version >= 3.3} {
                        set init {set i [expr $index+1]}
                        set cond {$i < [llength $windows]}
                        set reinit {incr i}
                } else {
                        set init {set i [expr $index-1]}
                        set cond {$i >= 0}
                        set reinit {incr i -1}
                }

                for $init $cond $reinit {
                        set kid [lindex $windows $i]
                        if {$kid != [winfo toplevel $kid]} {
                                set r [focus:first $kid]
                                if {$r != "none"} {
                                        return $r
                                }
                        }
                }

                # Go to the aunts.

                set w $p
        }

        # Got to the top level window -- wrap around to its first
        #       focusable child

        return [focus:first $w]
}

# Procedure:    focus:prev
#
# Synopsis:
#       Internal procedure to locate the previous focusable item
#       relative to a widget or canvas item.
#
# Usage:
#c      focus:prev pathName
#
#c      focus:prev {pathName tagOrID}
#
# Parameters:   
#c      pathName
#               Path name of a widget that accepts keyboard focus.
#c      tagOrID
#               Tag or ID specifying a canvas item within pathName that
#               accepts the keyboard focus.
#
# Return value:
#       The return value is one of
#       - `none', indicating that the widget does not accept focus.
#       - A widget path name
#       - A two-element list comprising a widget path name and
#         the name of a canvas item within the widget.
#
# Description:
#       Given a focusable item, this procedure locates the previous
#       focusable item within the item's toplevel window.
#
# Notes:
#       Focus is NOT directed to either item.
#
# See also:
#c      focus_goTo
#
#c      focus_goToFirst
#
#c      focus_goToNext
#
#c      focus_goToPrev

proc focus:prev a {

        global focus_priv
        global tk_version

        set w [lindex $a 0]
        set item [lindex $a 1]

        # If looking at a canvas item, traverse to the next older item.

        if {[winfo class $w] == "Canvas" && $item != ""} {
                set items [$w find all]
                set index [lsearch $items $item]
                for {set i [expr $index-1]} {$i >= 0} {incr i -1} {
                        set id [lindex $items $i]
                        if [focus:focusable $w $id] {
                                return [list $w $id]
                        }
                }
                if {[focus:findItem $w first] == $w} {
                        return $w
                }
        }

        # A widget is preceded by its older siblings, which are then preceded
        # by their parents, aunts, grandparents, great-aunts, ...

        while {$w != [winfo toplevel $w] \
            && ![info exists focus_priv(confine,$w)]} {
                set p [winfo parent $w]
                set windows [winfo children $p]
                set index [lsearch $windows $w]

                # Traverse the elder siblings

                if {$tk_version < 3.3} {
                        set init {set i [expr $index+1]}
                        set cond {$i < [llength $windows]}
                        set reinit {incr i}
                } else {
                        set init {set i [expr $index-1]}
                        set cond {$i >= 0}
                        set reinit {incr i -1}
                }

                for $init $cond $reinit {
                        set id [lindex $windows $i]
                        if {$id != [winfo toplevel $id]} {
                                set r [focus:last $id]
                                if {$r != "none"} {
                                        return $r
                                }
                        }
                }

                # Parent widget

                set w $p
                set r [focus:findItem $w last]
                if {$r != "none"} {
                        return $r
                }

                # Next trip does the aunts.
        }

        # Got to the top level window.  If it's focusable, return it, otherwise
        # do the last focusable child.

        set r [focus:findItem $w last]
        if {$r != "none"} {
                return $r
        }
        return [focus:last $w]
}

# Procedure:    focus:refocusAfterDestroy
#
# Synopsis:
#       Recover the focus if the current focus window is destroyed.
#
# Usage:
#c      focus:refocusAfterDestroy pathNames
#
# Parameters:
#c      pathNames
#               List of path names of candidate windows for refocus.
#
# Return value:
#       None specified.
#
# Description:
#       focus:refocusAfterDestroy is called as an `after' handler
#       when the current focus window is destroyed.  If the `focus' is
#       still set to `none', it locates the nearest window in the hierarchy
#       to the one that was destroyed, and focuses it.

proc focus:refocusAfterDestroy {ancestors} {
        foreach w $ancestors {
                if {[focus] != "none"} break
                if {[winfo exists $w] && [info commands $w] == $w} {
                        focus_goToFirst $w
                }
        }
}

# Procedure:    focus_skip
#
# Synopsis:
#       Arrange so that <Tab> traversal skips over a particular widget.
#
# Usage:
#c      focus_skip pathName ?value?
#
# Parameters:
#c      pathName
#               Path name of a widget
#c      value
#               A Boolean value (0 or 1).
#
# Return value:
#       None.
#
# Description:
#       focus_skip accepts the path name of a widget.
# 
#       In the two-argument form, it arranges to use the second argument in
#       place of the `disabled' state in determining whether the Tab
#       and Shift-Tab keys should skip the widget.  0 means never skip
#       the widget, irrespective of the `disabled' state; 1 means
#       always skip the widget, irrespective of the `disabled' state.
#
#       The one-argument form undoes the effect of a previous two-argument
#       call and allows the `disabled' state to control the tabbing again.

proc focus_skip {w {flag ""}} {
        global focus_priv
        if {$flag != ""} {
                set focus_priv(skip,$w) $flag
                widget_addBinding $w Destroy "focus:destroy $w"
        } else {
                unset focus_priv(skip,$w)
        }
}

# Procedure:    focus:unmap
#
# Synopsis:
#       Internal procedure to handle unmapping of a window
#
# Usage:
#c      focus:unmap pathName
#
# Parameters:
#c      pathName
#               Path name of a window being unmapped.
#
# Return value:
#       None specified.
#
# Description:
#       focus:unmap is called when a window that has owned the focus is
#       unmapped.  If the window still owns the focus, and if the window's
#       top-level window is still mapped, focus:unmap moves the focus to
#       the previous focusable object.

proc focus:unmap w {
        if {[winfo exists $w] && $w == [focus]
            && [winfo ismapped [winfo toplevel $w]]} {
                focus_goToPrev $w
        }
}

# Procedure:    focus_update
#
# Synopsis:
#       Make sure that the currently focused window has saved its state.
#
# Usage:
#c      focus_update
#
# Parameters:
#       None.
#
# Return value:
#       None.
#
# Description:
#       focus_update temporarily sets focus to `none', and then resets it
#       to the current focus.  It is used to make sure that the currently
#       focused window has saved its state.

proc focus_update {} {
        return [widget_event [focus] UpdateContent]
}

